/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.validation.validator;

import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.wst.validation.internal.core.Message;
import org.eclipse.wst.validation.internal.provisional.core.IMessage;
import org.eclipse.wst.validation.internal.provisional.core.IReporter;
import org.eclipse.wst.validation.internal.provisional.core.IValidator;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel;

import benten.cat.validation.validator.messages.TranslationValidationReporterMessages;

/**
 * 検証メッセージを追加するための翻訳リポーターです。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings( { "restriction" })
public class TranslationValidationReporter {

	/**
	 * Benten 設定初期化のためのメッセージ。
	 */
	protected static final TranslationValidationReporterMessages fMsg = new TranslationValidationReporterMessages();

	/**
	 * 翻訳バリデーター。
	 */
	private final IValidator validator;

	/**
	 * メッセージを伝えるレポーター。
	 */
	private final IReporter reporter;

	/**
	 * 対象となるファイル。
	 */
	private final IFile file;

	/**
	 * 対象となる DOM モデル。
	 */
	private final IDOMModel model;

	/**
	 * コンストラクター。
	 * @param validator 翻訳バリデーター
	 * @param reporter メッセージを伝えるレポーター
	 * @param file 対象となるファイル
	 * @param model 対象となるモデル
	 */
	public TranslationValidationReporter(final IValidator validator, final IReporter reporter, final IFile file,
			final IDOMModel model) {
		this.validator = validator;
		this.reporter = reporter;
		this.file = file;
		this.model = model;
	}

	/**
	 * 検証結果のクリア。
	 */
	public void clear() {
		if (file == null) {
			return;
		}
		if (reporter != null) {
			reporter.removeAllMessages(validator, file);
		}
	}

	/**
	 * モデルの取得。
	 * @return モデル
	 */
	public IDOMModel getModel() {
		return model;
	}

	/**
	 * メッセージ･リストの取得。
	 * @return メッセージ･リスト
	 */
	@SuppressWarnings("unchecked")
	public List<IMessage> getMessages() {
		return reporter.getMessages();
	}

	/**
	 * 情報の追加。
	 * @param targetOffset ターゲットのオフセット
	 * @param length 文字数
	 * @param id メッセージ・プロパティーの id
	 * @param params 埋め込むパラメーター
	 */
	public void addInfo(final int targetOffset, final int length, final String id, final String... params) {
		add(IMessage.LOW_SEVERITY, targetOffset, length, id, params);
	}

	/**
	 * 警告の追加。
	 * @param targetOffset ターゲットのオフセット
	 * @param length 文字数
	 * @param id メッセージ・プロパティーの id
	 * @param params 埋め込むパラメーター
	 */
	public void addWarning(final int targetOffset, final int length, final String id, final String... params) {
		add(IMessage.NORMAL_SEVERITY, targetOffset, length, id, params);
	}

	/**
	 * エラーの追加。
	 * @param targetOffset ターゲットのオフセット
	 * @param length 文字数
	 * @param id メッセージ・プロパティーの id
	 * @param params 埋め込むパラメーター
	 */
	public void addError(final int targetOffset, final int length, final String id, final String... params) {
		add(IMessage.HIGH_SEVERITY, targetOffset, length, id, params);
	}

	/**
	 * メッセージの追加。
	 * @param serverity 重大度
	 * @param targetOffset ターゲットのオフセット
	 * @param length 文字数
	 * @param id メッセージ・プロパティーの id
	 * @param params 埋め込むパラメーター
	 */
	protected void add(final int serverity, final int targetOffset, final int length, final String id,
			final String... params) {

		final Message message = new MessageForThisClassLoader();
		message.setSeverity(serverity);
		message.setBundleName("message"); //$NON-NLS-1$
		message.setTargetObject(file);
		message.setGroupName(fMsg.getAddGroupName());
		message.setOffset(targetOffset);
		message.setLength(length);
		message.setLineNo(getLineNumber(model.getStructuredDocument(), targetOffset));
		message.setId(id);
		message.setParams(params);
		reporter.addMessage(validator, message);
	}

	/**
	 * 行番号の取得。
	 *
	 * @param iDoc ドキュメント・オブジェクト。
	 * @param targetOffset オフセット。
	 * @return 行番号。
	 */
	private int getLineNumber(final IDocument iDoc, final int targetOffset) {
		try {
			return iDoc.getLineOfOffset(targetOffset) + 1;
		} catch (final BadLocationException e) {
			return IMessage.LINENO_UNSET;
		}
	}

	/**
	 * メッセージ・クラス。
	 * <p>
	 * 空継承しているのは、setBundleName で指定したプロパティー・ファイルをロードするときに
	 * メッセージ・クラスから getClass().getClassLoader() し、そのクラス・ローダーから
	 * プラグインのリソースが読み込まれるためです。
	 */
	private static class MessageForThisClassLoader extends Message {
	}
}
