/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.dialogs;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.model.WorkbenchLabelProvider;

/**
 * リソース選択ダイアログ。
 *
 * <UL>
 * <LI>ワークスペース以下のリソースを選択する際に利用されるダイアログの基底クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractResourceSelectionDialog extends SelectionDialog {

	/** ツリー・ビューアー */
	private TreeViewer treeViewer;

	/** ツリー */
	private Tree tree;

	/** リソースの配列 */
	private final IResource[] resources;

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param resources リソース
	 */
	public AbstractResourceSelectionDialog(final Shell parentShell, final IResource... resources) {
		super(parentShell);
		this.resources = resources;
		setShellStyle(getShellStyle() | SWT.RESIZE);
	}

	/**
	 * 選択を受け入れるリソースか判定。
	 * @param element 要素
	 * @return 受け入れる場合は true
	 */
	abstract protected boolean acceptSelection(final IResource element);

	@Override
	protected Point getInitialSize() {
		return new Point(400, 400);
	}

	@Override
	public void create() {
		super.create();

		final List<?> initialSelections = getInitialElementSelections();
		if (initialSelections.isEmpty()) {
			getOkButton().setEnabled(false);
		} else {
			final IResource resource = (IResource) initialSelections.get(0);
			if (resource != null) {
				updateOkButton(resource);

				for (final IResource res : resources) {
					try {
						if (res.isAccessible() == false) {
							// 非オープン状態の場合はスキップ。
							continue;
						}
						res.accept(new IResourceVisitor() {
							public boolean visit(final IResource c) throws CoreException {
								if (c != null && acceptSelection(c)) {
									if (c.getLocation().toOSString().equals(resource.getLocation().toOSString())) {

										// ツリーの子を指定しても選択状態とならないため親を指定
										//IStructuredSelection selection = new StructuredSelection(c);
										final IStructuredSelection selection = new StructuredSelection(res);

										treeViewer.setSelection(selection, true);
										return false;
									}
								}
								return true;
							}
						});
					} catch (final CoreException e) {
						throw new IllegalStateException(e);
					}
				}
			}
		}
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		final Composite composite = (Composite) super.createDialogArea(parent);
		createMessageArea(composite);

		treeViewer = new TreeViewer(composite, SWT.BORDER);
		tree = treeViewer.getTree();
		tree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		treeViewer.setContentProvider(new TreeContentProvider());
		treeViewer.setLabelProvider(new WorkbenchLabelProvider());
		treeViewer.addFilter(new ViewerFilter() {
			@Override
			public boolean select(final Viewer viewer, final Object parentElement, final Object element) {
				if (element instanceof IProject) {
					final IProject project = (IProject) element;
					if (!project.isAccessible()) {
						return false;
					}
				}
				return true;
			}
		});
		treeViewer.setInput(resources);
		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				final IStructuredSelection sselection = (IStructuredSelection) event.getSelection();
				final IResource firstElement = (IResource) sselection.getFirstElement();
				updateOkButton(firstElement);
			}
		});
		return composite;
	}

	@Override
	protected void okPressed() {
		final IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
		final Object firstElement = selection.getFirstElement();
		final List<Object> list = new ArrayList<Object>();
		list.add(firstElement);
		setResult(list);
		super.okPressed();
	}

	/**
	 * OK ボタンの更新。
	 * @param resource リソース
	 */
	private void updateOkButton(final IResource resource) {
		getOkButton().setEnabled(acceptSelection(resource));
	}

	/**
	 * ツリー・コンテンツ・プロバイダー。
	 */
	private class TreeContentProvider implements IStructuredContentProvider, ITreeContentProvider {

		/**
		 * {@inheritDoc}
		 */
		public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
		}

		/**
		 * {@inheritDoc}
		 */
		public void dispose() {
		}

		/**
		 * {@inheritDoc}
		 */
		public Object[] getElements(final Object inputElement) {
			return getChildren(inputElement);
		}

		/**
		 * {@inheritDoc}
		 */
		public Object[] getChildren(final Object parentElement) {
			if (parentElement instanceof IProject[]) {
				final IProject[] projects = (IProject[]) parentElement;
				return projects;
			}
			if (parentElement instanceof IContainer) {
				try {
					final IContainer container = (IContainer) parentElement;
					if (container.isAccessible()) {
						final List<IResource> list = new LinkedList<IResource>();
						for (final IResource member : container.members()) {
							if (member.getType() == IResource.FOLDER || acceptSelection(member)) {
								list.add(member);
							}
						}
						return list.toArray();
					}
				} catch (final CoreException e) {
					e.printStackTrace();
				}
			}
			return new Object[0];
		}

		/**
		 * {@inheritDoc}
		 */
		public Object getParent(final Object element) {
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		public boolean hasChildren(final Object element) {
			return getChildren(element).length > 0;
		}
	}
}
