/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.text;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterProcessor;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * TEXT を処理します。
 * 
 * @author IGA Tosiki
 */
public class BentenTwaFilterTextProcessor implements BentenTwaFilterProcessor {
	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".utf8") || file.getName().toLowerCase().endsWith(".utf-8")) { //$NON-NLS-1$ //$NON-NLS-2$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliffFromSource(final File sourceFile, final BlancoXliffFile xliffFile,
			final HelpTransUnitId id, final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffBody body = xliffFile.getBody();

		// テキストを表現します。
		xliffFile.setDatatype("x-text"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);
		final String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		final SentencePartitionList sentenceList = new SentencePartitionList(text);
		if (sentenceList.size() > 1) {
			for (final String sentence : sentenceList) {
				id.incrementSubSeq(sentenceList.size());
				createTransUnit(body, id, sentence, resultInfo);
			}
		} else {
			createTransUnit(body, id, text, resultInfo);
		}
		id.incrementSeq();
	}

	/**
	 * trans-unit の作成。
	 * 
	 * <UL>
	 * <LI>このメソッドは HTML -> XLIFF で利用されます。
	 * </UL>
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source,
			BentenProcessResultInfo resultInfo) {
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliffAndSource(final File sourceFile, final File xliffFile,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BentenXliff xliff = BentenXliff.loadInstance(xliffFile);

		final Iterator<BlancoXliffTransUnit> units = xliff.getAllTransUnitList().iterator();

		final StringBuilder target = new StringBuilder();

		while (units.hasNext()) {
			resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

			final BlancoXliffTransUnit unit = units.next();
			final BlancoXliffTarget t = unit.getTarget();

			if (t != null && t.getTarget() != null && !t.getTarget().equals("")) { //$NON-NLS-1$
				target.append(t.getTarget());
			} else {
				target.append(unit.getSource());
			}

			if (!HelpTransUnitId.hasContinue(unit.getId())) {
				break;
			}
		}

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		outStream.write(target.toString().getBytes("UTF-8"));//$NON-NLS-1$
		outStream.flush();
		byte[] bytes = outStream.toByteArray();

		return bytes;
	}
}
