/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.cat.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.cat.core.valueobject.BentenMarkNoTransableProcessInput;

/**
 * Apache Antタスク [BentenMarkNoTransable]のクラス。
 *
 * 翻訳対象外マークのための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.cat.core]にBentenMarkNoTransableBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenmarknotransable&quot; classname=&quot;benten.twa.cat.core.BentenMarkNoTransableTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenMarkNoTransableTask extends Task {
    /**
     * 翻訳対象外マークのための Ant タスク。
     */
    protected BentenMarkNoTransableProcessInput fInput = new BentenMarkNoTransableProcessInput();

    /**
     * フィールド [xliffdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldXliffdirProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * マーク対象の XLIFF が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setXliffdir(final String arg) {
        fInput.setXliffdir(arg);
        fIsFieldXliffdirProcessed = true;
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * マーク対象の XLIFF が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getXliffdir() {
        return fInput.getXliffdir();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenMarkNoTransableTask begin.");

        // 項目番号[1]、アトリビュート[xliffdir]は必須入力です。入力チェックを行います。
        if (fIsFieldXliffdirProcessed == false) {
            throw new BuildException("必須アトリビュート[xliffdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- xliffdir:[" + getXliffdir() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenMarkNoTransableProcessインタフェースを実装して benten.twa.cat.coreパッケージに BentenMarkNoTransableProcessImplクラスを作成することにより解決できる場合があります。
            final BentenMarkNoTransableProcess proc = new BentenMarkNoTransableProcessImpl();
            if (proc.execute(fInput) != BentenMarkNoTransableBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
