/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.tmx.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.tmx.core.valueobject.BentenMergeTmxProcessInput;
import benten.twa.tmx.messages.BentenMergeTmxMessages;
import blanco.commons.util.BlancoFileUtil;
import blanco.tmx.BlancoTmxParser;
import blanco.tmx.BlancoTmxSerializer;
import blanco.tmx.BlancoTmxUtil;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;

/**
 * TMX フラグメントのマージ
 *
 * <pre>
 * TMX フラグメントを TMX にマージします。
 *   1.  TMX の内容を、別の TMX にマージします。
 * </pre>
 * 
 * <UL>
 * <LI>マージの際に、source、target のいずれも完全に一致する登録がすでに存在する場合には、重複エントリとみなして追加しません。
 * </UL>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳メモリー断片抽出・マージ機能: TMXフラグメント・マージ機能」に対応します。
 *
 * @author IGA Tosiki
 */
public class BentenMergeTmxProcessImpl implements BentenMergeTmxProcess {
	/**
	 * TMXマージ機能のためのメッセージ。
	 */
	protected static final BentenMergeTmxMessages fMsg = new BentenMergeTmxMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenMergeTmxProcessInput fInput = null;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenMergeTmxProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenMergeTmxProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File fileSourcefile = new File(fInput.getSourcefile());
		if (fileSourcefile.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE011(fInput.getSourcefile()));
		}
		if (fileSourcefile.isFile() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE012(fInput.getSourcefile()));
		}
		if (fileSourcefile.canRead() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE013(fInput.getSourcefile()));
		}

		final File fileTargetfile = new File(fInput.getTargetfile());
		if (fileTargetfile.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE014(fInput.getSourcefile()));
		}
		if (fileTargetfile.isFile() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE015(fInput.getSourcefile()));
		}
		if (fileTargetfile.canRead() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE016(fInput.getSourcefile()));
		}
		if (fileTargetfile.canWrite() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE017(fInput.getSourcefile()));
		}

		if (progress(fMsg.getCoreP011(input.getSourcefile()))) {
			return 6;
		}
		final BlancoTmx tmxIn = new BlancoTmxParser().parse(fileSourcefile);

		if (progress(fMsg.getCoreP012(input.getTargetfile()))) {
			return 6;
		}
		final BlancoTmx tmxOut = new BlancoTmxParser().parse(fileTargetfile);
		final int originalSize = tmxOut.getBody().getTuList().size();

		if (progress(fMsg.getCoreP013())) {
			return 6;
		}

		// source と target をあわせて string キーとして扱います。
		final Map<String, String> mapTmxOut = new HashMap<String, String>(16384);
		for (final BlancoTmxTu tu : tmxOut.getBody().getTuList()) {
			// 与えられたロケールをもとに検索します。
			final BlancoTmxTuv tuvSource = BlancoTmxUtil.getTuvByLang(tu, input.getTranssourcelang());
			final BlancoTmxTuv tuvTarget = BlancoTmxUtil.getTuvByLang(tu, input.getTranstargetlang());

			final String source = (tuvSource == null ? "" : tuvSource.getSeg()); //$NON-NLS-1$
			final String target = (tuvTarget == null ? "" : tuvTarget.getSeg()); //$NON-NLS-1$
			// source と target をあわせて string キーとして登録します。
			mapTmxOut.put("source:[" + source + "], target:[" + target + "]", "exist"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}

		if (progress(fMsg.getCoreP014())) {
			return 6;
		}
		for (final BlancoTmxTu tu : tmxIn.getBody().getTuList()) {
			// 与えられたロケールをもとに検索します。
			final BlancoTmxTuv tuvSource = BlancoTmxUtil.getTuvByLang(tu, input.getTranssourcelang());
			final BlancoTmxTuv tuvTarget = BlancoTmxUtil.getTuvByLang(tu, input.getTranstargetlang());

			final String source = (tuvSource == null ? "" : tuvSource.getSeg()); //$NON-NLS-1$
			final String target = (tuvTarget == null ? "" : tuvTarget.getSeg()); //$NON-NLS-1$
			if (mapTmxOut.get("source:[" + source + "], target:[" + target + "]") != null) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				// すでに一致する登録が存在します。
				// マージの際に、source、target のいずれも完全に一致する登録がすでに存在する場合には、重複エントリとみなして追加しません。
			} else {
				tmxOut.getBody().getTuList().add(tu);
			}
		}

		if (progress(fMsg.getCoreP015())) {
			return 6;
		}

		Collections.sort(tmxOut.getBody().getTuList(), new BentenTmxTuComparator());
		BentenTmxTuUtil.removeDuplicatedTuv(tmxOut.getBody().getTuList());
		final int afterSize = tmxOut.getBody().getTuList().size();

		fResultInfo.setUnitCount(fResultInfo.getUnitCount() + Math.max(afterSize - originalSize, 0));

		if (progress(fMsg.getCoreP016(input.getTargetfile()))) {
			return 6;
		}

		try {
			final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			new BlancoTmxSerializer().serialize(tmxOut, outStream);
			outStream.flush();

			if (BlancoFileUtil.bytes2FileIfNecessary(outStream.toByteArray(), fileTargetfile) == 0) {
				// 変更無し。
				getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
			} else {
				// 更新のあったファイル数をカウントします。
				getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			}
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE022(fileTargetfile.getAbsolutePath(), e.toString()));
		}

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}
}
