/*******************************************************************************
 * Copyright (c) 2000, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Max Weninger (max.weninger@windriver.com) - Bug 131895 [Edit] Undo in compare
 *     Max Weninger (max.weninger@windriver.com) - Bug 72936 [Viewers] Show line numbers in comparision
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.compare;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;

import org.eclipse.compare.ICompareContainer;
import org.eclipse.compare.internal.CompareUIPlugin;
import org.eclipse.compare.internal.MergeViewerAction;
import org.eclipse.compare.internal.Utilities;
import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.IHandler;
import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IPainter;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.ITextViewerExtension5;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.IUndoManagerExtension;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.WhitespaceCharacterPainter;
import org.eclipse.jface.text.source.CompositeRuler;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.jface.text.source.ISharedTextColors;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRuler;
import org.eclipse.jface.text.source.LineNumberRulerColumn;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.VerifyKeyListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorActionDelegate;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.menus.CommandContributionItem;
import org.eclipse.ui.menus.CommandContributionItemParameter;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditorPreferenceConstants;
import org.eclipse.ui.texteditor.ChangeEncodingAction;
import org.eclipse.ui.texteditor.FindReplaceAction;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.IElementStateListener;
import org.eclipse.ui.texteditor.ITextEditor;

import benten.cat.glossary.ui.actions.ShowGlossaryAction;
import benten.cat.tm.ui.actions.ShowTranslationAction;
import benten.cat.ui.CatUiPlugin;
import benten.cat.ui.viewers.messages.TranslationSourceViewerMessages;

/**
 * マージ・ソース・ビューアー。
 *
 * <UL>
 * <LI>このクラスは {@link org.eclipse.compare.internal.MergeSourceViewer} 由来のものです。
 * <LI>Benten 固有部分にはコメントに [Benten] と記載しています。
 * <LI>比較エディター上の左右それぞれのソース・ビューアー部分です。
 * <LI>コンテキスト・メニューに「翻訳アシスト」、「TM の表示」、「用語集の表示」を追加しています。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
/**
 * Wraps a JFace SourceViewer and add some convenience methods.
 */
@SuppressWarnings("restriction")
public class MergeSourceViewer implements ISelectionChangedListener,

// [Benten] 追加
		IPreferenceChangeListener,

		ITextListener, IMenuListener, IOperationHistoryListener, IAdaptable {

	public static final String UNDO_ID = "undo"; //$NON-NLS-1$
	public static final String REDO_ID = "redo"; //$NON-NLS-1$
	public static final String CUT_ID = "cut"; //$NON-NLS-1$
	public static final String COPY_ID = "copy"; //$NON-NLS-1$
	public static final String PASTE_ID = "paste"; //$NON-NLS-1$
	public static final String DELETE_ID = "delete"; //$NON-NLS-1$
	public static final String SELECT_ALL_ID = "selectAll"; //$NON-NLS-1$
	public static final String FIND_ID = "find"; //$NON-NLS-1$
	public static final String GOTO_LINE_ID = "gotoLine"; //$NON-NLS-1$
	public static final String CHANGE_ENCODING_ID = "changeEncoding"; //$NON-NLS-1$

	// [Benten] 追加
	protected static final TranslationSourceViewerMessages fMsg = new TranslationSourceViewerMessages();
	public static final int CONTENTASSIST_PROPOSALS = ISourceViewer.CONTENTASSIST_PROPOSALS;
	public static final int SHOW_TRANSLATION = 1001;
	public static final int SHOW_GLOSSARY = 1002;
	public static final Map<Character, Integer> keyBindMap = new HashMap<Character, Integer>() {
		/**
		 * Serializable のバージョン ID 対応。
		 */
		private static final long serialVersionUID = 1L;

		{
			put(' ', CONTENTASSIST_PROPOSALS);
			put('t', SHOW_TRANSLATION);
			put('g', SHOW_GLOSSARY);
		}
	};
	private final IPainter lineBreakPainter;

	class TextOperationAction extends MergeViewerAction {

		private final int fOperationCode;

		TextOperationAction(final int operationCode, final boolean mutable, final boolean selection,
				final boolean content) {
			this(operationCode, null, mutable, selection, content);
		}

		public TextOperationAction(final int operationCode, final String actionDefinitionId, final boolean mutable,
				final boolean selection, final boolean content) {
			super(mutable, selection, content);
			if (actionDefinitionId != null)
				setActionDefinitionId(actionDefinitionId);
			fOperationCode = operationCode;
			update();
		}

		@Override
		public void run() {
			// [Benten] 変更
			/*
			if (isEnabled()) {
				getSourceViewer().doOperation(fOperationCode);
			}
			*/
			doOperation(fOperationCode);
		}

		@Override
		public boolean isEnabled() {
			// [Benten] 変更
			//return fOperationCode != -1 && getSourceViewer().canDoOperation(fOperationCode);
			return canDoOperation(fOperationCode);
		}

		@Override
		public void update() {
			this.setEnabled(isEnabled());
		}
	}

	// [Benten] 追加
	public boolean canDoOperation(final int fOperationCode) {
		final StyledText widget = getSourceViewer().getTextWidget();
		if (widget != null) {
			if (fOperationCode == SHOW_TRANSLATION) {
				return widget.getSelectionCount() > 0;
			} else if (fOperationCode == SHOW_GLOSSARY) {
				return widget.getSelectionCount() > 0;
			}
		}
		return fOperationCode != -1 && getSourceViewer().canDoOperation(fOperationCode);
	}

	// [Benten] 追加
	public void doOperation(final int fOperationCode) {
		if (canDoOperation(fOperationCode)) {
			if (fOperationCode == SHOW_TRANSLATION) {
				runAction(new ShowTranslationAction());
				return;
			} else if (fOperationCode == SHOW_GLOSSARY) {
				runAction(new ShowGlossaryAction());
				return;
			}
			getSourceViewer().doOperation(fOperationCode);
		}
	}

	// [Benten] 追加
	private void runAction(final IEditorActionDelegate action) {
		action.setActiveEditor(null, CatUiPlugin.getDefault().getActiveEditorPart());
		action.selectionChanged(null, getSourceViewer().getSelection());
		action.run(null);
	}

	/**
	 * TODO_: The only purpose of this class is to provide "Go to Line" action in
	 * TextMergeViewer. The adapter should be removed as soon as we implement
	 * embedded TextEditor in a similar way JDT has it done for Java compare.
	 */
	class TextEditorAdapter implements ITextEditor {

		public void close(final boolean save) {
			// defining interface method
		}

		public void doRevertToSaved() {
			// defining interface method
		}

		public IAction getAction(final String actionId) {
			// defining interface method
			return null;
		}

		public IDocumentProvider getDocumentProvider() {
			return new IDocumentProvider() {

				public void aboutToChange(final Object element) {
					// defining interface method
				}

				public void addElementStateListener(final IElementStateListener listener) {
					// defining interface method
				}

				public boolean canSaveDocument(final Object element) {
					// defining interface method
					return false;
				}

				public void changed(final Object element) {
					// defining interface method
				}

				public void connect(final Object element) throws CoreException {
					// defining interface method
				}

				public void disconnect(final Object element) {
					// defining interface method
				}

				public IAnnotationModel getAnnotationModel(final Object element) {
					// defining interface method
					return null;
				}

				public IDocument getDocument(final Object element) {
					return MergeSourceViewer.this.getSourceViewer().getDocument();
				}

				public long getModificationStamp(final Object element) {
					// defining interface method
					return 0;
				}

				public long getSynchronizationStamp(final Object element) {
					// defining interface method
					return 0;
				}

				public boolean isDeleted(final Object element) {
					// defining interface method
					return false;
				}

				public boolean mustSaveDocument(final Object element) {
					// defining interface method
					return false;
				}

				public void removeElementStateListener(final IElementStateListener listener) {
					// defining interface method
				}

				public void resetDocument(final Object element) throws CoreException {
					// defining interface method
				}

				public void saveDocument(final IProgressMonitor monitor, final Object element,
						final IDocument document, final boolean overwrite) throws CoreException {
					// defining interface method
				}
			};
		}

		public IRegion getHighlightRange() {
			// defining interface method
			return null;
		}

		public ISelectionProvider getSelectionProvider() {
			return MergeSourceViewer.this.getSourceViewer().getSelectionProvider();
		}

		public boolean isEditable() {
			// defining interface method
			return false;
		}

		public void removeActionActivationCode(final String actionId) {
			// defining interface method
		}

		public void resetHighlightRange() {
			// defining interface method
		}

		public void selectAndReveal(final int start, final int length) {
			selectAndReveal(start, length, start, length);
		}

		/*
		 * @see org.eclipse.ui.texteditor.AbstractTextEditor#selectAndReveal(int, int, int, int)
		 */
		private void selectAndReveal(final int selectionStart, final int selectionLength, final int revealStart,
				final int revealLength) {

			final ISelection selection = getSelectionProvider().getSelection();
			if (selection instanceof ITextSelection) {
				final ITextSelection textSelection = (ITextSelection) selection;
				if (textSelection.getOffset() != 0 || textSelection.getLength() != 0)
					markInNavigationHistory();
			}

			final StyledText widget = MergeSourceViewer.this.getSourceViewer().getTextWidget();
			widget.setRedraw(false);
			{
				adjustHighlightRange(revealStart, revealLength);
				MergeSourceViewer.this.getSourceViewer().revealRange(revealStart, revealLength);

				MergeSourceViewer.this.getSourceViewer().setSelectedRange(selectionStart, selectionLength);

				markInNavigationHistory();
			}
			widget.setRedraw(true);
		}

		/*
		 * @see org.eclipse.ui.texteditor.AbstractTextEditor#markInNavigationHistory()
		 */
		private void markInNavigationHistory() {
			getSite().getPage().getNavigationHistory().markLocation(this);
		}

		/*
		 * @see org.eclipse.ui.texteditor.AbstractTextEditor#adjustHighlightRange(int, int)
		 */
		private void adjustHighlightRange(final int offset, final int length) {

			if (MergeSourceViewer.this instanceof ITextViewerExtension5) {
				final ITextViewerExtension5 extension = (ITextViewerExtension5) MergeSourceViewer.this;
				extension.exposeModelRange(new Region(offset, length));
			} else if (!isVisible(MergeSourceViewer.this.getSourceViewer(), offset, length)) {
				MergeSourceViewer.this.getSourceViewer().resetVisibleRegion();
			}
		}

		/*
		 * @see org.eclipse.ui.texteditor.AbstractTextEditor#isVisible(ISourceViewer, int, int)
		 */
		private/*static*/final boolean isVisible(final ITextViewer viewer, final int offset, final int length) {
			if (viewer instanceof ITextViewerExtension5) {
				final ITextViewerExtension5 extension = (ITextViewerExtension5) viewer;
				final IRegion overlap = extension.modelRange2WidgetRange(new Region(offset, length));
				return overlap != null;
			}
			return viewer.overlapsWithVisibleRegion(offset, length);
		}

		public void setAction(final String actionID, final IAction action) {
			// defining interface method
		}

		public void setActionActivationCode(final String actionId, final char activationCharacter,
				final int activationKeyCode, final int activationStateMask) {
			// defining interface method
		}

		public void setHighlightRange(final int offset, final int length, final boolean moveCursor) {
			// defining interface method
		}

		public void showHighlightRangeOnly(final boolean showHighlightRangeOnly) {
			// defining interface method
		}

		public boolean showsHighlightRangeOnly() {
			// defining interface method
			return false;
		}

		public IEditorInput getEditorInput() {
			if (MergeSourceViewer.this.fContainer.getWorkbenchPart() instanceof IEditorPart)
				return ((IEditorPart) MergeSourceViewer.this.fContainer.getWorkbenchPart()).getEditorInput();
			return null;
		}

		public IEditorSite getEditorSite() {
			// defining interface method
			return null;
		}

		public void init(final IEditorSite site, final IEditorInput input) throws PartInitException {
			// defining interface method
		}

		public void addPropertyListener(final IPropertyListener listener) {
			// defining interface method
		}

		public void createPartControl(final Composite parent) {
			// defining interface method
		}

		public void dispose() {
			// defining interface method
		}

		public IWorkbenchPartSite getSite() {
			return MergeSourceViewer.this.fContainer.getWorkbenchPart().getSite();
		}

		public String getTitle() {
			// defining interface method
			return null;
		}

		public Image getTitleImage() {
			// defining interface method
			return null;
		}

		public String getTitleToolTip() {
			// defining interface method
			return null;
		}

		public void removePropertyListener(final IPropertyListener listener) {
			// defining interface method
		}

		public void setFocus() {
			// defining interface method
		}

		@SuppressWarnings("unchecked")
		public Object getAdapter(final Class adapter) {
			// defining interface method
			return null;
		}

		public void doSave(final IProgressMonitor monitor) {
			// defining interface method
		}

		public void doSaveAs() {
			// defining interface method
		}

		public boolean isDirty() {
			// defining interface method
			return false;
		}

		public boolean isSaveAsAllowed() {
			// defining interface method
			return false;
		}

		public boolean isSaveOnCloseNeeded() {
			// defining interface method
			return false;
		}
	}

	private final ResourceBundle fResourceBundle;
	private final ICompareContainer fContainer;
	private final SourceViewer fSourceViewer;
	private Position fRegion;
	private boolean fEnabled = true;
	@SuppressWarnings("unchecked")
	private final HashMap fActions = new HashMap();
	private IDocument fRememberedDocument;

	private boolean fAddSaveAction = true;
	private boolean isConfigured = false;

	// line number ruler support
	private final IPropertyChangeListener fPreferenceChangeListener;
	private boolean fShowLineNumber = false;
	private LineNumberRulerColumn fLineNumberColumn;
	@SuppressWarnings("unchecked")
	private final List textActions = new ArrayList();
	private CommandContributionItem fSaveContributionItem;

	public MergeSourceViewer(final SourceViewer sourceViewer, final ResourceBundle bundle,
			final ICompareContainer container) {
		Assert.isNotNull(sourceViewer);
		fSourceViewer = sourceViewer;
		fResourceBundle = bundle;
		fContainer = container;

		// [Benten] 追加
		lineBreakPainter = new WhitespaceCharacterPainter(fSourceViewer);
		preferenceChange(null);
		CatUiPlugin.getDefault().addPreferenceChangeListner(this);

		final MenuManager menu = new MenuManager();
		menu.setRemoveAllWhenShown(true);
		menu.addMenuListener(this);
		final StyledText te = getSourceViewer().getTextWidget();
		te.setMenu(menu.createContextMenu(te));
		fContainer.registerContextMenu(menu, getSourceViewer());

		// for listening to editor show/hide line number preference value
		fPreferenceChangeListener = new IPropertyChangeListener() {
			public void propertyChange(final PropertyChangeEvent event) {
				MergeSourceViewer.this.handlePropertyChangeEvent(event);
			}
		};
		EditorsUI.getPreferenceStore().addPropertyChangeListener(fPreferenceChangeListener);
		fShowLineNumber = EditorsUI.getPreferenceStore().getBoolean(
				AbstractDecoratedTextEditorPreferenceConstants.EDITOR_LINE_NUMBER_RULER);
		if (fShowLineNumber) {
			updateLineNumberRuler();
		}

		final IOperationHistory history = getHistory();
		if (history != null)
			history.addOperationHistoryListener(this);

		// don't add save when in a dialog, IWorkbenchPart is null in dialog containers
		fAddSaveAction = fContainer.getWorkbenchPart() != null;

		// [Benten] 追加
		getSourceViewer().appendVerifyKeyListener(new VerifyKeyListener() {
			public void verifyKey(final VerifyEvent event) {
				if (event.stateMask == SWT.CTRL) {
					final Integer operationCode = keyBindMap.get((char) event.keyCode);
					if (operationCode != null) {
						if (canDoOperation(operationCode)) {
							doOperation(operationCode);
						}
						event.doit = false;
					}
				}
			}
		});
	}

	// [Benten] 追加
	/**
	 * {@inheritDoc}
	 */
	public void preferenceChange(final PreferenceChangeEvent event) {
		Display.getCurrent().asyncExec(new Runnable() {
			public void run() {
				try {
					final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
					final boolean show = store.getBoolean(CatUiPlugin.CatUiPreference.SHOW_LINE_BREAK.name());
					if (fSourceViewer != null) {
						if (show) {
							fSourceViewer.addPainter(lineBreakPainter);
						} else {
							fSourceViewer.removePainter(lineBreakPainter);
						}
					}
				} catch (final RuntimeException e) {
					CatUiPlugin.getDefault().log(e);
				}
			}
		});
	}

	public void rememberDocument(final IDocument doc) {
		//		if (doc != null && fRememberedDocument != null) {
		//			System.err.println("MergeSourceViewer.rememberDocument: fRememberedDocument != null: shouldn't happen"); //$NON-NLS-1$
		//		}
		fRememberedDocument = doc;
	}

	public IDocument getRememberedDocument() {
		return fRememberedDocument;
	}

	public void hideSaveAction() {
		fAddSaveAction = false;
	}

	public void setFont(final Font font) {
		final StyledText te = getSourceViewer().getTextWidget();
		if (te != null)
			te.setFont(font);
		if (fLineNumberColumn != null) {
			fLineNumberColumn.setFont(font);
			layoutViewer();
		}
	}

	public void setBackgroundColor(final Color color) {
		final StyledText te = getSourceViewer().getTextWidget();
		if (te != null)
			te.setBackground(color);
		if (fLineNumberColumn != null)
			fLineNumberColumn.setBackground(color);
	}

	public void setForegroundColor(final Color color) {
		final StyledText te = getSourceViewer().getTextWidget();
		if (te != null)
			te.setForeground(color);
	}

	public void setEnabled(final boolean enabled) {
		if (enabled != fEnabled) {
			fEnabled = enabled;
			final StyledText c = getSourceViewer().getTextWidget();
			if (c != null) {
				c.setEnabled(enabled);
				final Display d = c.getDisplay();
				c.setBackground(enabled ? d.getSystemColor(SWT.COLOR_LIST_BACKGROUND) : null);
			}
		}
	}

	public boolean getEnabled() {
		return fEnabled;
	}

	public void setRegion(final Position region) {
		fRegion = region;
	}

	public Position getRegion() {
		return fRegion;
	}

	public boolean isControlOkToUse() {
		final StyledText t = getSourceViewer().getTextWidget();
		return t != null && !t.isDisposed();
	}

	public void setSelection(final Position position) {
		if (position != null)
			getSourceViewer().setSelectedRange(position.getOffset(), position.getLength());
	}

	public void setLineBackground(final Position position, final Color c) {
		final StyledText t = getSourceViewer().getTextWidget();
		if (t != null && !t.isDisposed()) {
			final Point region = new Point(0, 0);
			getLineRange(position, region);

			region.x -= getDocumentRegionOffset();

			try {
				t.setLineBackground(region.x, region.y, c);
			} catch (final IllegalArgumentException ex) {
				// silently ignored
			}
		}
	}

	public void resetLineBackground() {
		final StyledText t = getSourceViewer().getTextWidget();
		if (t != null && !t.isDisposed()) {
			final int lines = getLineCount();
			t.setLineBackground(0, lines, null);
		}
	}

	/*
	 * Returns number of lines in document region.
	 */
	public int getLineCount() {
		final IRegion region = getSourceViewer().getVisibleRegion();

		final int length = region.getLength();
		if (length == 0)
			return 0;

		final IDocument doc = getSourceViewer().getDocument();
		int startLine = 0;
		int endLine = 0;

		final int start = region.getOffset();
		try {
			startLine = doc.getLineOfOffset(start);
		} catch (final BadLocationException ex) {
			// silently ignored
		}
		try {
			endLine = doc.getLineOfOffset(start + length);
		} catch (final BadLocationException ex) {
			// silently ignored
		}

		return endLine - startLine + 1;
	}

	public int getViewportLines() {
		final StyledText te = getSourceViewer().getTextWidget();
		final Rectangle clArea = te.getClientArea();
		if (!clArea.isEmpty())
			return clArea.height / te.getLineHeight();
		return 0;
	}

	public int getViewportHeight() {
		final StyledText te = getSourceViewer().getTextWidget();
		final Rectangle clArea = te.getClientArea();
		if (!clArea.isEmpty())
			return clArea.height;
		return 0;
	}

	/*
	 * Returns lines
	 */
	public int getDocumentRegionOffset() {
		final int start = getSourceViewer().getVisibleRegion().getOffset();
		final IDocument doc = getSourceViewer().getDocument();
		if (doc != null) {
			try {
				return doc.getLineOfOffset(start);
			} catch (final BadLocationException ex) {
				// silently ignored
			}
		}
		return 0;
	}

	public int getVerticalScrollOffset() {
		final StyledText st = getSourceViewer().getTextWidget();
		final int lineHeight = st.getLineHeight();
		return getSourceViewer().getTopInset() - ((getDocumentRegionOffset() * lineHeight) + st.getTopPixel());
	}

	/*
	 * Returns the start line and the number of lines which correspond to the given position.
	 * Starting line number is 0 based.
	 */
	public Point getLineRange(final Position p, final Point region) {

		final IDocument doc = getSourceViewer().getDocument();

		if (p == null || doc == null) {
			region.x = 0;
			region.y = 0;
			return region;
		}

		final int start = p.getOffset();
		final int length = p.getLength();

		int startLine = 0;
		try {
			startLine = doc.getLineOfOffset(start);
		} catch (final BadLocationException e) {
			// silently ignored
		}

		int lineCount = 0;

		if (length == 0) {
			//			// if range length is 0 and if range starts a new line
			//			try {
			//				if (start == doc.getLineStartOffset(startLine)) {
			//					lines--;
			//				}
			//			} catch (BadLocationException e) {
			//				lines--;
			//			}

		} else {
			int endLine = 0;
			try {
				endLine = doc.getLineOfOffset(start + length - 1); // why -1?
			} catch (final BadLocationException e) {
				// silently ignored
			}
			lineCount = endLine - startLine + 1;
		}

		region.x = startLine;
		region.y = lineCount;
		return region;
	}

	/*
	 * Scroll TextPart to the given line.
	 */
	public void vscroll(int line) {

		final int srcViewSize = getLineCount();
		final int srcExtentSize = getViewportLines();

		if (srcViewSize > srcExtentSize) {

			if (line < 0)
				line = 0;

			final int cp = getSourceViewer().getTopIndex();
			if (cp != line)
				getSourceViewer().setTopIndex(line + getDocumentRegionOffset());
		}
	}

	@SuppressWarnings("unchecked")
	public void addAction(final String actionId, final MergeViewerAction action) {
		fActions.put(actionId, action);
	}

	public IAction getAction(final String actionId) {
		IAction action = (IAction) fActions.get(actionId);
		if (action == null) {
			action = createAction(actionId);
			if (action == null)
				return null;
			if (action instanceof MergeViewerAction) {
				final MergeViewerAction mva = (MergeViewerAction) action;
				if (mva.isContentDependent())
					getSourceViewer().addTextListener(this);
				if (mva.isSelectionDependent())
					getSourceViewer().addSelectionChangedListener(this);

				// [Benten] 追加
				if (actionId.equals(String.valueOf(CONTENTASSIST_PROPOSALS))) {
					action.setText(fMsg.getLabel15());
				} else if (actionId.equals(String.valueOf(SHOW_TRANSLATION))) {
					action.setText(fMsg.getLabel16());
				} else if (actionId.equals(String.valueOf(SHOW_GLOSSARY))) {
					action.setText(fMsg.getLabel17());
				} else {

					Utilities.initAction(action, fResourceBundle, "action." + actionId + "."); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			addAction(actionId, action);
		}
		if (action instanceof MergeViewerAction) {
			final MergeViewerAction mva = (MergeViewerAction) action;
			if (mva.isEditableDependent() && !getSourceViewer().isEditable())
				return null;
		}
		return action;
	}

	protected IAction createAction(final String actionId) {
		if (UNDO_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.UNDO, IWorkbenchCommandConstants.EDIT_UNDO, true,
					false, true);
		if (REDO_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.REDO, IWorkbenchCommandConstants.EDIT_REDO, true,
					false, true);
		if (CUT_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.CUT, IWorkbenchCommandConstants.EDIT_CUT, true, true,
					false);
		if (COPY_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.COPY, IWorkbenchCommandConstants.EDIT_COPY, false,
					true, false);
		if (PASTE_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.PASTE, IWorkbenchCommandConstants.EDIT_PASTE, true,
					false, false);
		if (DELETE_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.DELETE, IWorkbenchCommandConstants.EDIT_DELETE, true,
					false, false);
		if (SELECT_ALL_ID.equals(actionId))
			return new TextOperationAction(ITextOperationTarget.SELECT_ALL, IWorkbenchCommandConstants.EDIT_SELECT_ALL,
					false, false, false);

		// [Benten] 追加
		if (String.valueOf(CONTENTASSIST_PROPOSALS).equals(actionId))
			return new TextOperationAction(CONTENTASSIST_PROPOSALS, null, true, true, false);
		if (String.valueOf(SHOW_TRANSLATION).equals(actionId))
			return new TextOperationAction(SHOW_TRANSLATION, null, false, true, false);
		if (String.valueOf(SHOW_GLOSSARY).equals(actionId))
			return new TextOperationAction(SHOW_GLOSSARY, null, false, true, false);
		return null;
	}

	@SuppressWarnings("unchecked")
	public void selectionChanged(final SelectionChangedEvent event) {
		final Iterator e = fActions.values().iterator();
		while (e.hasNext()) {
			final Object next = e.next();
			if (next instanceof MergeViewerAction) {
				final MergeViewerAction action = (MergeViewerAction) next;
				if (action.isSelectionDependent())
					action.update();
			}
		}
	}

	public void textChanged(final TextEvent event) {
		updateContentDependantActions();
	}

	@SuppressWarnings("unchecked")
	void updateContentDependantActions() {
		final Iterator e = fActions.values().iterator();
		while (e.hasNext()) {
			final Object next = e.next();
			if (next instanceof MergeViewerAction) {
				final MergeViewerAction action = (MergeViewerAction) next;
				if (action.isContentDependent())
					action.update();
			}
		}
	}

	/*
	 * Allows the viewer to add menus and/or tools to the context menu.
	 */
	@SuppressWarnings("unchecked")
	public void menuAboutToShow(final IMenuManager menu) {

		menu.add(new Separator("undo")); //$NON-NLS-1$
		addMenu(menu, UNDO_ID);
		addMenu(menu, REDO_ID);
		menu.add(new GroupMarker("save")); //$NON-NLS-1$
		if (fAddSaveAction)
			addSave(menu);
		menu.add(new Separator("file")); //$NON-NLS-1$

		menu.add(new Separator("ccp")); //$NON-NLS-1$
		addMenu(menu, CUT_ID);
		addMenu(menu, COPY_ID);
		addMenu(menu, PASTE_ID);
		addMenu(menu, DELETE_ID);
		addMenu(menu, SELECT_ALL_ID);

		menu.add(new Separator("edit")); //$NON-NLS-1$
		addMenu(menu, CHANGE_ENCODING_ID);
		menu.add(new Separator("find")); //$NON-NLS-1$
		addMenu(menu, FIND_ID);

		menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));

		menu.add(new Separator("text")); //$NON-NLS-1$
		for (final Iterator iterator = textActions.iterator(); iterator.hasNext();) {
			final IAction action = (IAction) iterator.next();
			menu.add(action);
		}

		// [Benten] 追加
		menu.add(new Separator("benten")); //$NON-NLS-1$
		addMenu(menu, String.valueOf(CONTENTASSIST_PROPOSALS));
		addMenu(menu, String.valueOf(SHOW_TRANSLATION));
		addMenu(menu, String.valueOf(SHOW_GLOSSARY));

		menu.add(new Separator("rest")); //$NON-NLS-1$

		// update all actions
		// to get undo redo right
		updateActions();
	}

	private void addMenu(final IMenuManager menu, final String actionId) {
		final IAction action = getAction(actionId);
		if (action != null)
			menu.add(action);
	}

	private void addSave(final IMenuManager menu) {
		final ICommandService commandService = (ICommandService) fContainer.getWorkbenchPart().getSite().getService(
				ICommandService.class);
		final Command command = commandService.getCommand(IWorkbenchCommandConstants.FILE_SAVE);

		final IHandler handler = command.getHandler();
		if (handler != null) {
			if (fSaveContributionItem == null) {
				fSaveContributionItem = new CommandContributionItem(new CommandContributionItemParameter(fContainer
						.getWorkbenchPart().getSite(), null, command.getId(), CommandContributionItem.STYLE_PUSH));
			}
			// save is an editable dependent action, ie add only when edit
			// is possible
			if (handler.isHandled() && getSourceViewer().isEditable())
				menu.add(fSaveContributionItem);
		}
	}

	/**
	 * The viewer is no longer part of the UI, it's a wrapper only. The disposal
	 * doesn't take place while releasing the editor pane's children. The method
	 * have to be called it manually. The wrapped viewer is disposed as a
	 * regular viewer, while disposing the UI.
	 */
	public void dispose() {
		getSourceViewer().removeTextListener(this);
		getSourceViewer().removeSelectionChangedListener(this);
		EditorsUI.getPreferenceStore().removePropertyChangeListener(fPreferenceChangeListener);

		final IOperationHistory history = getHistory();
		if (history != null)
			history.removeOperationHistoryListener(this);
	}

	/**
	 * update all actions independent of their type
	 *
	 */
	@SuppressWarnings("unchecked")
	public void updateActions() {
		final Iterator e = fActions.values().iterator();
		while (e.hasNext()) {
			final Object next = e.next();
			if (next instanceof MergeViewerAction) {
				final MergeViewerAction action = (MergeViewerAction) next;
				action.update();
			} else if (next instanceof FindReplaceAction) {
				final FindReplaceAction action = (FindReplaceAction) next;
				action.update();
			} else if (next instanceof ChangeEncodingAction) {
				final ChangeEncodingAction action = (ChangeEncodingAction) next;
				action.update();
			}
		}
	}

	public void configure(final SourceViewerConfiguration configuration) {
		if (isConfigured)
			getSourceViewer().unconfigure();
		isConfigured = true;
		getSourceViewer().configure(configuration);
	}

	/**
	 * specific implementation to support a vertical ruler
	 * @param x x
	 * @param y y
	 * @param width width
	 * @param height height
	 */
	public void setBounds(final int x, final int y, final int width, final int height) {
		if (getSourceViewer().getControl() instanceof Composite) {
			((Composite) getSourceViewer().getControl()).setBounds(x, y, width, height);
		} else {
			getSourceViewer().getTextWidget().setBounds(x, y, width, height);
		}
	}

	/**
	 * handle show/hide line numbers from editor preferences
	 * @param event event
	 */
	protected void handlePropertyChangeEvent(final PropertyChangeEvent event) {

		final String key = event.getProperty();

		if (key.equals(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_LINE_NUMBER_RULER)) {
			final boolean b = EditorsUI.getPreferenceStore().getBoolean(
					AbstractDecoratedTextEditorPreferenceConstants.EDITOR_LINE_NUMBER_RULER);
			if (b != fShowLineNumber) {
				toggleLineNumberRuler();
			}
		} else if (key.equals(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_LINE_NUMBER_RULER_COLOR)) {
			updateLineNumberColumnPresentation(true);
		}
	}

	/**
	 * Hides or shows line number ruler column based of preference setting
	 */
	private void updateLineNumberRuler() {
		final IVerticalRuler v = getVerticalRuler(getSourceViewer());
		if (v != null && v instanceof CompositeRuler) {
			final CompositeRuler c = (CompositeRuler) v;

			if (!fShowLineNumber) {
				if (fLineNumberColumn != null) {
					c.removeDecorator(fLineNumberColumn);
				}
			} else {
				if (fLineNumberColumn == null) {
					fLineNumberColumn = new LineNumberRulerColumn();
					updateLineNumberColumnPresentation(false);
				}
				c.addDecorator(0, fLineNumberColumn);
			}
		}
	}

	/**
	 * Calls the <code>getVerticalRuler</code> method of the given viewer.
	 *
	 * TODO_: don't use reflection
	 *
	 * @param viewer the viewer to call <code>getVerticalRuler</code> on
	 */
	private IVerticalRuler getVerticalRuler(final SourceViewer viewer) {
		try {
			final Method method = SourceViewer.class.getDeclaredMethod("getVerticalRuler", new Class[] {}); //$NON-NLS-1$
			method.setAccessible(true);
			return (IVerticalRuler) method.invoke(viewer, new Object[] {});
		} catch (final Throwable e) {
			CompareUIPlugin.log(e);
		}
		return null;
	}

	private void updateLineNumberColumnPresentation(final boolean refresh) {
		if (fLineNumberColumn == null)
			return;
		RGB rgb = getColorFromStore(EditorsUI.getPreferenceStore(),
				AbstractDecoratedTextEditorPreferenceConstants.EDITOR_LINE_NUMBER_RULER_COLOR);
		if (rgb == null)
			rgb = new RGB(0, 0, 0);
		final ISharedTextColors sharedColors = getSharedColors();
		fLineNumberColumn.setForeground(sharedColors.getColor(rgb));
		if (refresh) {
			fLineNumberColumn.redraw();
		}
	}

	private void layoutViewer() {
		final Control parent = getSourceViewer().getControl();
		if (parent instanceof Composite && !parent.isDisposed())
			((Composite) parent).layout(true);
	}

	private ISharedTextColors getSharedColors() {
		return EditorsUI.getSharedTextColors();
	}

	private RGB getColorFromStore(final IPreferenceStore store, final String key) {
		RGB rgb = null;
		if (store.contains(key)) {
			if (store.isDefault(key))
				rgb = PreferenceConverter.getDefaultColor(store, key);
			else
				rgb = PreferenceConverter.getColor(store, key);
		}
		return rgb;
	}

	/**
	 * Toggles line number ruler column.
	 */
	private void toggleLineNumberRuler() {
		fShowLineNumber = !fShowLineNumber;

		updateLineNumberRuler();
	}

	@SuppressWarnings("unchecked")
	public void addTextAction(final IAction textEditorPropertyAction) {
		textActions.add(textEditorPropertyAction);
	}

	@SuppressWarnings("unchecked")
	public void addAction(final String id, final IAction action) {
		fActions.put(id, action);
	}

	private IOperationHistory getHistory() {
		if (PlatformUI.getWorkbench() == null) {
			return null;
		}
		return PlatformUI.getWorkbench().getOperationSupport().getOperationHistory();
	}

	public void historyNotification(final OperationHistoryEvent event) {
		// This method updates the enablement of all content operations
		// when the undo history changes. It could be localized to UNDO and REDO.
		final IUndoContext context = getUndoContext();
		if (context != null && event.getOperation().hasContext(context)) {
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					updateContentDependantActions();
				}
			});
		}
	}

	private IUndoContext getUndoContext() {
		final IUndoManager undoManager = getSourceViewer().getUndoManager();
		if (undoManager instanceof IUndoManagerExtension)
			return ((IUndoManagerExtension) undoManager).getUndoContext();
		return null;
	}

	/**
	 * @return the wrapped viewer
	 */
	public SourceViewer getSourceViewer() {
		return fSourceViewer;
	}

	@SuppressWarnings("unchecked")
	public Object getAdapter(final Class adapter) {
		if (adapter == ITextEditor.class) {
			return new TextEditorAdapter();
		}
		return null;
	}
}
