/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.preference;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

import benten.ui.UiPlugin;

/**
 * Benten プロジェクト・プロパティー定義。
 *
 * <UL>
 * <LI>Benten 全体で使用するプロジェクト・プロパティー設定定義クラスです。
 * <LI>設定するプラグインが存在しない場合、取得できる値は null、false などの規定の値です。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class BentenProjectProperty {

	/** プロジェクト・プロパティー保管キー */
	public static enum ProjectProperty {

		/** 翻訳対象 id */
		TRANS_TARGET_ID,

		/** 翻訳対象インポート元ディレクトリー */
		TRANS_TARGET_IMPORT_FROM_DIR,

		/** 翻訳対象インポート済み */
		TRANS_TARGET_IMPORTED,

		/** 翻訳対象インポートの確認 */
		PROMPT_TRANS_TARGET_IMPORTED,

		/** XLIFF へ変換済み */
		XLIFF_CONVERTED,

		/** XLIFF へ変換の確認 */
		PROMPT_XLIFF_CONVERTED,

		/** TM プロジェクト設定の使用 */
		USE_TM_PROJECT_SETTINGS,

		/** 用語集プロジェクト設定の使用 */
		USE_GLOSSAY_PROJECT_SETTINGS,

		/** 翻訳元の言語 */
		TRANS_SOURCE_LANG,

		/** 翻訳先の言語 */
		TRANS_TARGET_LANG,
	}

	/**
	 * プロジェクトのプロパティー・ストアを取得。
	 * @param project プロジェクト
	 * @return プロジェクトのプロパティー・ストア
	 */
	public static IPreferenceStore getStore(final IProject project) {
		final String pluginId = UiPlugin.getDefault().getPluginId();
		return new RealtimeSavedPreferenceStore(new ProjectScope(project), pluginId);
	}

	/**
	 * 値をセットしたときに、すぐに永続化するストア・クラス。
	 * <UL>
	 * <LI>現在、{@link #setValue(String, String)} および {@link #setValue(String, boolean)}
	 * による値のセットのみ即時永続化をサポートしています。その他の setter メソッドによる永続化はサポートされません。
	 * </UL>
	 */
	private static class RealtimeSavedPreferenceStore extends ScopedPreferenceStore {

		/**
		 * コンストラクター。
		 * @param context ストアするスコープ。
		 * @param qualifier プリファランス・ノードをルックアップする際に利用。
		 */
		public RealtimeSavedPreferenceStore(final IScopeContext context, final String qualifier) {
			super(context, qualifier);
		}

		@Override
		public void setValue(final String name, final boolean value) {
			super.setValue(name, value);
			saveQuietly();
		}

		@Override
		public void setValue(final String name, final String value) {
			super.setValue(name, value);
			saveQuietly();
		}

		/**
		 * 非同期に保管。
		 */
		public void saveQuietly() {
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					try {
						save();
					} catch (final Exception e) {
						UiPlugin.getDefault().log(e);
					}
				}
			});
		}
	}
}
