/*!
  \file
  \brief VAڑ (Linux)

  \author Satofumi KAMIMURA

  $Id$

  \todo K؂ȉӏ tcdrain() Ăяoׂ
*/

#include "SerialCtrl.h"
#include "RingBuffer.h"
#include <sys/poll.h>
#include <termios.h>
#include <fcntl.h>
#include <errno.h>

using namespace beego;


/*!
  \brief SerialCtrl ̓NX
*/
struct SerialCtrl::pImpl {
  std::string error_message;
  int fd;
  struct termios sio;           /*!< ʐM */
  struct pollfd nfds;           /*!< ^CAEg */
  RingBuffer<char> ring_buffer;

  pImpl(void) : error_message("not connected."), fd(-1) {
    nfds.fd = fd;
    nfds.events = 0;
    nfds.revents = 0;
  }

  ~pImpl(void) {
    disconnect();
  }

  void disconnect(void) {
    if (fd >= 0) {
      close(fd);
      fd = -1;
    }
    ring_buffer.clear();
    error_message = "disconnected.";
  }

  bool changeBaudrate(long baudrate) {
    long baudrate_value = -1;
    switch (baudrate) {

    case 4800:
      baudrate_value = B4800;
      break;

    case 9600:
      baudrate_value = B9600;
      break;

    case 19200:
      baudrate_value = B19200;
      break;

    case 38400:
      baudrate_value = B38400;
      break;

    case 57600:
      baudrate_value = B57600;
      break;

    case 115200:
      baudrate_value = B115200;
      break;

    default:
      char* message = new char [80 + 12];
      sprintf(message, "Invalid baudrate value: %ld", baudrate);
      error_message = message;
      delete [] message;
      return false;
      break;
    }

    cfsetospeed(&sio, baudrate_value);
    cfsetispeed(&sio, baudrate_value);
    tcsetattr(fd, TCSADRAIN, &sio);
    flush();

    return true;
  }

  void update(int timeout, int size = 0) {

    enum { BufferSize = 1024 };
    int require_size = (size > 0) ? size : BufferSize;
    int filled = 0;
    while (filled < require_size) {
      if (poll(&nfds, 1, (size == 0) ? 0 : timeout) <= 0) {
	break;			// timeout
      }
      char data[BufferSize];
      int read_n = require_size - filled;
      int n = read(fd, data, (read_n > BufferSize) ? BufferSize : read_n);
      if (n <= 0) {
	return;
      }
      ring_buffer.put(data, n);
      filled += n;
    }
  }

  bool isLf(char ch) {
    return ((ch == '\r') || (ch == '\n')) ? true : false;
  }

  void flush(void) {
    tcflush(fd, TCIOFLUSH);
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl) {
}


SerialCtrl::~SerialCtrl(void) {
}


const char* SerialCtrl::what(void) {
  return pimpl->error_message.c_str();
}


bool SerialCtrl::connect(const char* device, long baudrate) {

  // foCXɐڑ
  int fd = open(device, O_RDWR);
  if (fd < 0) {
    pimpl->error_message = std::string(device) + " : " + strerror(errno);
    return false;
  }
  pimpl->fd = fd;

  // VAݒ
  tcgetattr(fd, &pimpl->sio);
  pimpl->sio.c_iflag = IGNPAR;
  pimpl->sio.c_oflag = 0;
  pimpl->sio.c_cflag = CS8 | CREAD | CLOCAL;
  pimpl->sio.c_lflag = 0;

  pimpl->sio.c_cc[VMIN] = 0;
  pimpl->sio.c_cc[VTIME] = 0;
  tcsetattr(fd, TCSAFLUSH, &pimpl->sio);
  clear();

  // ^CAEgݒ
  pimpl->nfds.fd = fd;
  pimpl->nfds.events = POLLIN | POLLPRI | POLLERR | POLLHUP | POLLNVAL;
  pimpl->nfds.revents = 0;

  // {[[gݒ
  bool ret = pimpl->changeBaudrate(baudrate);
  if (ret == false) {
    return false;
  }

  pimpl->error_message = "no error";
  return true;
}


void SerialCtrl::disconnect(void) {
  pimpl->disconnect();
}


bool SerialCtrl::isConnected(void) {
  return (pimpl->fd < 0) ? false : true;
}


bool SerialCtrl::changeBaudrate(long baudrate) {
  return pimpl->changeBaudrate(baudrate);
}


int SerialCtrl::send(const char* data, int size) {
  if (isConnected() == false) {
    return 0;
  }
  return write(pimpl->fd, data, size);
}


int SerialCtrl::recv(char* data, int size, int timeout) {
  //printf("lin recv.\n");
  if ((isConnected() == false) || (size <= 0)) {
    return 0;
  }

  // vMf[^΁AԂ
  int filled = SerialCtrl::size();
  if (filled >= size) {
    pimpl->ring_buffer.get(data, size);
    return size;
  }

  // Młf[^Ԃ
  int left = size - filled;
  pimpl->update(timeout, left);

  // size() ĂԂ update(0) Ă΂邽
  filled = pimpl->ring_buffer.size();
  pimpl->ring_buffer.get(data, filled);

  return filled;
}


int SerialCtrl::getLine(std::string& data, int timeout) {

  data.clear();
  int n;
  do {
    char ch;
    n = recv(&ch, 1, timeout);
    if (n <= 0) {
      break;
    }
    if (pimpl->isLf(ch)) {
      break;
    }
    data.push_back(ch);
  } while (n > 0);

  return data.size();
}


int SerialCtrl::size(int timeout) {
  pimpl->update(timeout);
  return pimpl->ring_buffer.size();
}


void SerialCtrl::clear(void) {
  flush();
  pimpl->update(0);
  pimpl->ring_buffer.clear();
}


void SerialCtrl::skip(int timeout) {
  clear();

  char buffer[BUFSIZ];
  int total = 0;
  int n;

  enum { EachTimeout = 10 };
  do {
    n = recv(buffer, BUFSIZ, EachTimeout);
    total += EachTimeout;
  } while ((n > 0) && ((timeout > 0) && (total < timeout)));
}


void SerialCtrl::flush(void) {
  pimpl->flush();
}
