package net.sf.amateras.air.wizards;

import java.util.ArrayList;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.internal.resources.File;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * Choose resources to export as the AIR package.
 * 
 * @author Naoki Takezoe
 */
public class ExportAIRWizardPage extends WizardPage {

	private IProject project;
	private CheckboxTreeViewer treeViewer;
	private Text packageName;
	private Text storeType;
	private Text keystore;
	private Text storepass;
	private String descriptor;

	protected ExportAIRWizardPage(IProject project) {
		super("ExportAIRWizardPage");
		setTitle(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_TITLE"));
		setDescription(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_DESCRIPTION"));
		this.project = project;

		ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(this.project),
				AIRPlugin.PLUGIN_ID);
		descriptor = store.getString(AIRPlugin.PREF_DESCRIPTOR);
	}

	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout(1, false));

		treeViewer = new CheckboxTreeViewer(composite);
		treeViewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		treeViewer.setContentProvider(new WorkbenchContentProvider());
		treeViewer.setLabelProvider(new WorkbenchLabelProvider());
		treeViewer.setInput(project);
		treeViewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent arg0) {
				doValidate();
			}
		});
		checkDescription();

		// package name
		Composite exportPanel = new Composite(composite, SWT.NULL);
		exportPanel.setLayout(new GridLayout(2, false));
		exportPanel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_NAME"), exportPanel);
		packageName = new Text(exportPanel, SWT.BORDER);
		packageName.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		packageName.setText(project.getName() + ".air");
		packageName.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		// store type
		Group group = new Group(composite, SWT.NULL);
		group.setText(AIRPlugin.getResourceString("CERTIFICATE"));
		group.setLayout(new GridLayout(2, false));
		group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_STORE_TYPE"), group);
		storeType = new Text(group, SWT.BORDER);
		storeType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		storeType.setText("pkcs12");
		storeType.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		//KeyStore
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_KEY_STORE"), group);
		Composite panel = new Composite(group, SWT.NULL);
		panel.setLayout(new GridLayout(2, false));
		panel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		keystore = new Text(panel, SWT.BORDER);
		keystore.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		keystore.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});
		UIUtil.createBrowseFileButton(keystore, panel);

		//StorePass
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_STOREPASS"), group);
		storepass = new Text(group, SWT.BORDER);
		storepass.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		storepass.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		// create Strore
		UIUtil.createLabel("", group);
		Button button = new Button(group, SWT.PUSH);
		button.setText(AIRPlugin.getResourceString("CREATE_CERTIFICATE"));
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				CreateCertificateDialog dialog = new CreateCertificateDialog(getShell(), project,
						keystore, storepass);
				dialog.open();
				dialog = null;
			}
		});

		setControl(composite);
		doValidate();
	}

	private void checkDescription() {
		Widget o = treeViewer.getControl();

		Item[] items = null;
		if (o instanceof TreeItem) {
			items = ((TreeItem) o).getItems();
		} else if (o instanceof Tree) {
			items = ((Tree) o).getItems();
		} else {
			return;
		}

		for (int i = 0; i < items.length; i++) {
			Item item = items[i];
			if (item instanceof TreeItem) {
				setDefaultChecked(item);
			}
		}
	}

	private void setDefaultChecked(Item item) {
		if (item instanceof TreeItem) {
			Object data = item.getData();
			if (data != null) {
				if (data instanceof File) {
					String fileName = ((File) data).getName();

					if (fileName.equals(descriptor)) {
						((TreeItem) item).setChecked(true);

					} else if (fileName.endsWith(".html") || fileName.endsWith(".swf")) {
						((TreeItem) item).setChecked(true);
					}
				}
			}
		}
	}

	private void doValidate() {
		if (treeViewer.getCheckedElements().length == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_RESOURCE"));
			setPageComplete(false);
			return;
		}
		if (packageName.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_NAME"));
			setPageComplete(false);
			return;
		}
		if (storeType.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_STORE_TYPE"));
			setPageComplete(false);
			return;
		}
		if (keystore.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_STORE"));
			setPageComplete(false);
			return;
		}
		setErrorMessage(null);
		setPageComplete(true);
	}

	public boolean finish() {
		ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(this.project),
				AIRPlugin.PLUGIN_ID);
		String descriptor = store.getString(AIRPlugin.PREF_DESCRIPTOR);

		Object[] checkedElements = treeViewer.getCheckedElements();

		List<IResource> resources = new ArrayList<IResource>();
		for (int i = 0; i < checkedElements.length; i++) {
			if (checkedElements[i] instanceof IResource) {
				if (((IResource) checkedElements[i]).getName().endsWith(descriptor)) {
					resources.add(0, (IResource) checkedElements[i]);
				} else {
					resources.add((IResource) checkedElements[i]);
				}
			}
		}

		ArrayList<String> args = new ArrayList<String>();
		args.add("-package");
		args.add("-storetype");
		args.add(storeType.getText());
		args.add("-keystore");
		args.add(keystore.getText());
		args.add("-storepass");
		args.add(storepass.getText());
		args.add(packageName.getText());

		ADTRunner runner = new ADTRunner();

		return runner.run(project, args, resources.toArray(new IResource[resources.size()]));
	}
}
