package net.sf.amateras.air.wizards;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Map;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.wizards.newresource.ResourceMessages;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;

public abstract class AbstractNewProjectWizard extends Wizard implements INewWizard,IExecutableExtension {

	protected IProject newProject;
	protected AbstractProjectWizardPage page1;
	protected WizardPage page2;
	protected IConfigurationElement config;

	/**
	 * Creates the first wizard page.
	 * 
	 * @return the first wizard page
	 */
	protected abstract AbstractProjectWizardPage createPage1();

	/**
	 * Creates the second wizard page.
	 * 
	 * @return the second wizard page
	 */
	protected abstract WizardPage createPage2();

	protected abstract void addProjectSpecificParameters(Map params);

	protected abstract void createProjectSpecificResources(IProject project, Map params,
			IProgressMonitor monitor) throws Exception;

	/**
	 * Returns the nature id to add new project.
	 * 
	 * @return the nature id
	 */
	protected abstract String getNatureId();

	
	public void setInitializationData(IConfigurationElement config,
			String propertyName, Object data) throws CoreException {
		this.config = config;
	}
	
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		// Nothing to do.
	}

	public void addPages() {
		page1 = createPage1();
		addPage(page1);

		page2 = createPage2();
		if (page2 != null) {
			addPage(page2);
		}
	}

	public boolean performFinish() {
		createNewProject();
		
		if (newProject == null) {
			return false;
		}
		BasicNewProjectResourceWizard.updatePerspective(config);
		//        updatePerspective();
		//        selectAndReveal(newProject);

		return true;
	}
	
	

	protected void createFile(IProject project, String path, InputStream in, Map params,
			IProgressMonitor monitor) throws Exception {
		if (params == null) {
			IFile file = project.getFile(new Path(path));
			file.create(in, true, monitor);
			return;
		}
		try {
			byte[] buf = new byte[in.available()];
			in.read(buf);
			String xml = new String(buf, "UTF-8");

			// replaces variables
			Object[] entries = params.entrySet().toArray();
			for (int i = 0; i < entries.length; i++) {
				Map.Entry entry = (Map.Entry) entries[i];
				xml = xml.replaceAll("\\$\\{" + entry.getKey() + "\\}", (String) entry.getValue());
			}
			// creates file
			IFile file = project.getFile(new Path(path));
			file.create(new ByteArrayInputStream(xml.getBytes("UTF-8")), true, monitor);
		} finally {
			in.close();
		}
	}

	/**
	 * Creates a project resource given the project handle and description.
	 * 
	 * @param description
	 *            the project description to create a project resource for
	 * @param projectHandle
	 *            the project handle to create a project resource for
	 * @param monitor
	 *            the progress monitor to show visual progress with
	 * 
	 * @exception CoreException
	 *                if the operation fails
	 * @exception OperationCanceledException
	 *                if the operation is canceled
	 */
	private void createProject(IProjectDescription description, IProject projectHandle,
			IProgressMonitor monitor) throws CoreException, OperationCanceledException {
		try {
			monitor.beginTask("", 2000);//$NON-NLS-1$

			projectHandle.create(description, new SubProgressMonitor(monitor, 1000));

			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}

			projectHandle.open(IResource.BACKGROUND_REFRESH, new SubProgressMonitor(monitor, 1000));

		} finally {
			monitor.done();
		}
	}

	private IProject createNewProject() {
		if (newProject != null) {
			return newProject;
		}

		// get a project handle
		final IProject newProjectHandle = page1.getProjectHandle();

		// get a project descriptor
		URI location = null;
		if (!page1.useDefaults()) {
			location = page1.getLocationURI();
		}

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		final IProjectDescription description = workspace.newProjectDescription(newProjectHandle
				.getName());
		description.setLocationURI(location);

		final Map params = createParamMap(page2);
		addProjectSpecificParameters(params);

		//        // update the referenced project if provided
		//        if (referencePage != null) {
		//            IProject[] refProjects = referencePage.getReferencedProjects();
		//            if (refProjects.length > 0) {
		//				description.setReferencedProjects(refProjects);
		//			}
		//        }

		// create the new project operation
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor) throws CoreException {
				createProject(description, newProjectHandle, monitor);

				AIRPlugin.addNature(newProjectHandle, getNatureId());
				createResource(newProjectHandle, monitor, params);

			}
		};

		// run the new project creation operation
		try {
			getContainer().run(true, true, op);
		} catch (InterruptedException e) {
			return null;
		} catch (InvocationTargetException e) {
			// ie.- one of the steps resulted in a core exception
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				if (((CoreException) t).getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS) {
					MessageDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage,
							NLS.bind(ResourceMessages.NewProject_caseVariantExistsError,
									newProjectHandle.getName()));
				} else {
					ErrorDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage,
							null, // no special message
							((CoreException) t).getStatus());
				}
			} else {
				// CoreExceptions are handled above, but unexpected runtime
				// exceptions and errors may still occur.
				IDEWorkbenchPlugin.getDefault().getLog().log(
						new Status(IStatus.ERROR, IDEWorkbenchPlugin.IDE_WORKBENCH, 0,
								t.toString(), t));
				MessageDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage, NLS
						.bind(ResourceMessages.NewProject_internalError, t.getMessage()));
			}
			return null;
		}

		newProject = newProjectHandle;

		return newProject;
	}

	protected abstract Map createParamMap(WizardPage page);

	protected abstract void createResource(IProject newProjectHandle, IProgressMonitor monitor,
			Map params);
}
