package net.sf.amateras.air.propertyPages;

import java.util.ArrayList;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.builder.CompileProperty;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * create Build Setting dialog
 * @author ogawahideko
 *
 */
public class CreateBuildSettingDialog extends TitleAreaDialog {
	private CompileProperty compile;
	private Text command;

	private Button mxmlType;
	private Button as3Type;
	private Button chkExtention;
	private Text options;
	private Button chkOutput;
	private Text txtOutput;
	private Button chkLibraryPath;
	private Text txtLibraryPath;
	private Button chkSourcePath;
	private Text txtSourcePath;
	private Button chkIncludeClasses;
	private Text txtIncludeClasses;

	private IProject project;
	private ScopedPreferenceStore store;

	private List<CheckEntry> needText = new ArrayList<CheckEntry>();

	public CreateBuildSettingDialog(Shell parentShell, IProject project, CompileProperty compile) {
		super(parentShell);
		this.project = project;
		this.compile = compile;
		store = new ScopedPreferenceStore(new ProjectScope(project), AIRPlugin.PLUGIN_ID);
	}

	protected Control createDialogArea(Composite parent) {
		setTitle(AIRPlugin.getResourceString("BUILD_SETTING") + " - " + compile.getDirectory());

		Composite area = (Composite) super.createDialogArea(parent);
		Composite composite = new Composite(area, SWT.NULL);
		composite.setLayout(new GridLayout(1, false));
		composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Group groupType = new Group(composite, SWT.NULL);
		groupType.setLayout(new GridLayout(2, false));
		groupType.setText(AIRPlugin.getResourceString("BUILD_TYPE"));
		groupType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		mxmlType = new Button(groupType, SWT.RADIO);
		mxmlType.setText("AIRFlex");
		mxmlType.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setDefaultSetting(CompileProperty.MXML_TYPE);
			}
		});
		as3Type = new Button(groupType, SWT.RADIO);
		as3Type.setText("ActionScript");
		as3Type.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setDefaultSetting(CompileProperty.AS3_TYPE);
			}
		});

		chkExtention = new Button(composite, SWT.CHECK);
		chkExtention.setText(AIRPlugin.getResourceString("ENABLE_BUILD_EXTENSION"));
		chkExtention.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				refreshExtensionState();
			}
		});
		Group group = new Group(composite, SWT.NULL);
		group.setLayout(new GridLayout(2, false));
		group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		command = createText("Command", group, true);
		options = createText("Options", group, false);

		chkOutput = new Button(group, SWT.CHECK);
		chkOutput.setText("-output");
		txtOutput = new Text(group, SWT.BORDER);
		txtOutput.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		chkLibraryPath = new Button(group, SWT.CHECK);
		chkLibraryPath.setText("-library-path+=");
		txtLibraryPath = new Text(group, SWT.BORDER);
		txtLibraryPath.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		chkSourcePath = new Button(group, SWT.CHECK);
		chkSourcePath.setText("-source-path");
		txtSourcePath = new Text(group, SWT.BORDER);
		txtSourcePath.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		chkIncludeClasses = new Button(group, SWT.CHECK);
		chkIncludeClasses.setText("-include-classes");
		txtIncludeClasses = new Text(group, SWT.BORDER);
		txtIncludeClasses.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		return parent;
	}

	private void refreshExtensionState() {
		if (chkExtention.getSelection()) {
			command.setEnabled(true);
			options.setEnabled(true);
			chkOutput.setEnabled(true);
			txtOutput.setEnabled(true);
			chkLibraryPath.setEnabled(true);
			txtLibraryPath.setEnabled(true);
			chkSourcePath.setEnabled(true);
			txtSourcePath.setEnabled(true);
			chkIncludeClasses.setEnabled(true);
			txtIncludeClasses.setEnabled(true);
		} else {
			command.setEnabled(false);
			options.setEnabled(false);
			chkOutput.setEnabled(false);
			txtOutput.setEnabled(false);
			chkLibraryPath.setEnabled(false);
			txtLibraryPath.setEnabled(false);
			chkSourcePath.setEnabled(false);
			txtSourcePath.setEnabled(false);
			chkIncludeClasses.setEnabled(false);
			txtIncludeClasses.setEnabled(false);
		}

	}

	@Override
	protected Control createContents(Composite parent) {
		Control control = super.createContents(parent);
		if (compile.getCommandType() == CompileProperty.MXML_TYPE) {
			mxmlType.setSelection(true);
		} else {
			as3Type.setSelection(true);
		}

		if (compile.isExtention()) {
			chkExtention.setSelection(true);
			setTypeContents(compile.getCommandType(), compile.getCommand(), compile.getOption());
		} else {
			chkExtention.setSelection(false);
			setDefaultSetting(compile.getCommandType());
		}

		setASCompVisible(compile.getCommandType());

		doValidate();
		return control;
	}

	private void setTypeContents(int type, String commandValue, String optionValue) {
		command.setText(commandValue);
		String[] opt = optionValue.split(" ");
		options.setText("");
		chkOutput.setSelection(false);
		txtOutput.setText("");
		chkLibraryPath.setSelection(false);
		txtLibraryPath.setText("");
		chkSourcePath.setSelection(false);
		txtSourcePath.setText("");
		chkIncludeClasses.setSelection(false);
		txtIncludeClasses.setText("");
		for (int i = 0; i < opt.length; i++) {
			if (opt[i].equals("-output")) {
				chkOutput.setSelection(true);
				if (i <= opt.length) {
					i++;
					txtOutput.setText(replaceString(opt[i]));
					continue;
				}
			} else if (opt[i].startsWith("-library-path+=")) {
				chkLibraryPath.setSelection(true);
				txtLibraryPath.setText(replaceString(opt[i].substring("-library-path+=".length())));
			} else if (opt[i].startsWith("-source-path")) {
				chkSourcePath.setSelection(true);
				if (i <= opt.length) {
					i++;
					txtSourcePath.setText(replaceString(opt[i]));
					continue;
				}
			} else if (opt[i].startsWith("-include-classes")) {
				chkIncludeClasses.setSelection(true);
				if (i <= opt.length) {
					i++;
					txtIncludeClasses.setText(replaceString(opt[i]));
					continue;
				}

			} else {
				if (options.getText().length() > 0) {
					options.setText(options.getText() + " " + replaceString(opt[i]));
				} else {
					options.setText(replaceString(opt[i]));
				}
			}
		}

		refreshExtensionState();

	}

	private void setDefaultSetting(int type) {
		if (type == CompileProperty.MXML_TYPE) {
			setTypeContents(compile.getCommandType(), store
					.getDefaultString(AIRPlugin.PREF_FLEX_COMPILE_COMMAND), store
					.getDefaultString(AIRPlugin.PREF_FLEX_COMPILE_OPTION));
		} else {
			setTypeContents(compile.getCommandType(), store
					.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_COMMAND), store
					.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_OPTION));
		}
		setASCompVisible(type);
	}

	private void setASCompVisible(int type) {
		if (type == CompileProperty.MXML_TYPE) {
			chkSourcePath.setVisible(false);
			txtSourcePath.setVisible(false);
			chkIncludeClasses.setVisible(false);
			txtIncludeClasses.setVisible(false);
		} else {
			chkSourcePath.setVisible(true);
			txtSourcePath.setVisible(true);
			chkIncludeClasses.setVisible(true);
			txtIncludeClasses.setVisible(true);
		}
	}

	@Override
	protected void okPressed() {
		if (mxmlType.getSelection()) {
			compile.setCommandType(CompileProperty.MXML_TYPE);
		} else {
			compile.setCommandType(CompileProperty.AS3_TYPE);
		}
		compile.setExtention(chkExtention.getSelection());
		if (chkExtention.getSelection()) {
			compile.setCommand(command.getText());
			StringBuilder sb = new StringBuilder();
			if (options.getText().length() > 0) {
				sb.append(options.getText()).append(" ");
			}
			if (chkOutput.getSelection()) {
				sb.append(chkOutput.getText()).append(" ").append(txtOutput.getText()).append(" ");
			}
			if (chkSourcePath.getSelection() && chkSourcePath.isVisible()) {
				sb.append(chkSourcePath.getText()).append(" ").append(txtSourcePath.getText())
						.append(" ");
			}
			if (chkIncludeClasses.getSelection() && chkIncludeClasses.isVisible()) {
				sb.append(chkIncludeClasses.getText()).append(" ").append(
						txtIncludeClasses.getText()).append(" ");
			}
			if (chkLibraryPath.getSelection()) {
				sb.append(chkLibraryPath.getText()).append(txtLibraryPath.getText());
			}
			compile.setOption(sb.toString().trim());
		} else {
			if (compile.getCommandType() == CompileProperty.MXML_TYPE) {
				compile.setCommand(store.getDefaultString(AIRPlugin.PREF_FLEX_COMPILE_COMMAND));
				compile.setOption(store.getDefaultString(AIRPlugin.PREF_FLEX_COMPILE_OPTION));
			} else {
				compile.setCommand(store.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_COMMAND));
				compile.setOption(store.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_OPTION));
			}
		}
		super.okPressed();
	}

	private Text createText(String label, Composite parent, boolean isNeed) {
		return createText(label, parent, isNeed, SWT.BORDER);
	}

	private Text createText(String label, Composite parent, boolean isNeed, int textType) {
		UIUtil.createLabel(label, parent);
		Text text = new Text(parent, textType);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		if (isNeed) {
			needText.add(new CheckEntry(text, label));
			text.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					doValidate();
				}
			});
		}
		return text;
	}

	private void doValidate() {
		for (CheckEntry check : needText) {
			if (check.getText().getText().length() == 0) {
				setErrorMessage(getMessage("IS_NEED", check.getMessage()));
				getButton(IDialogConstants.OK_ID).setEnabled(false);
				return;
			}
		}
		setErrorMessage(null);
		getButton(IDialogConstants.OK_ID).setEnabled(true);
	}

	private String getMessage(String key, Object... args) {
		return AIRPlugin.getResourceString(key, args);
	}

	class CheckEntry {
		private Text text;
		private String message;

		CheckEntry(Text text, String message) {
			this.text = text;
			this.message = message;
		}

		public Text getText() {
			return text;
		}

		public void setText(Text text) {
			this.text = text;
		}

		public String getMessage() {
			return message;
		}

		public void setMessage(String message) {
			this.message = message;
		}
	}

	private String replaceString(String value) {
		try {
			value = value.replaceAll("\\$\\{PROJECT_NAME\\}", project.getName());
			value = value.replaceAll("\\$\\{DIRECTORY\\}", compile.getDirectory());
//			value = value.replaceAll("\\$\\{LIBRARY_PATH\\}", store.getString(
//					AIRPlugin.PREF_FLEX_LIBRARIES).replaceAll("\t", ","));
			return value;
		} catch (Exception ex) {
			ex.printStackTrace();
			return value;
		}
	}
}
