package net.sf.amateras.air.mxml.models;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sf.amateras.air.mxml.descriptor.BooleanPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.ColorChoosePropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.DoublePropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.IEditorValueDescriptor;
import net.sf.amateras.air.mxml.descriptor.ImagePropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.NumberPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.StringListPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.StringPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.WidthHeightPropertyDescriptor;
import net.sf.amateras.air.util.ColorUtil;
import net.sf.amateras.air.util.XMLUtil;

import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource;
import org.w3c.dom.Element;

/**
 * This is a base class for Flex component models.
 * 
 * @author Naoki Takezoe
 */
public abstract class AbstractModel implements IModel, IPropertySource {

	private Map<Object, ModelProperty> properties;

	private PropertyChangeSupport listeners = new PropertyChangeSupport(this);
	private Map<Object, Object> propertyAttributes = new LinkedHashMap<Object, Object>();
	private Map additionalAttributes = new HashMap();

	private List<Element> additionalChildElements = new ArrayList<Element>();

	public AbstractModel() {
		properties = PropertyManager.getProperties(getClass().getName());
		if (properties == null) {
			properties = new HashMap<Object, ModelProperty>();
			installModelProperty();
			PropertyManager.putProperties(getClass().getName(), properties);
		}
	}

	protected void installModelProperty() {
		addStringModelProperty(ID, CATEGORY_PROPERTY, null);
	}

	public void setAdditionalAttributes(Map additionalAttributes) {
		this.additionalAttributes = additionalAttributes;
	}

	public Map getAdditionalAttributes() {
		return this.additionalAttributes;
	}

	public void addAdditionalChildElement(Element childNode) {
		additionalChildElements.add(childNode);
	}

	public List<Element> getAdditionalChildElements() {
		return additionalChildElements;
	}

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		listeners.addPropertyChangeListener(listener);
	}

	public void firePropertyChange(String propName, Object oldValue, Object newValue) {
		listeners.firePropertyChange(propName, oldValue, newValue);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		listeners.removePropertyChangeListener(listener);
	}

	protected void addModelProperty(Object id, ModelProperty property) {
		properties.put(id, property);
	}

	public ModelProperty getModelProperty(Object id) {
		if (properties == null) {
			return null;
		}
		return properties.get(id);
	}

	public Map getModelProperties() {
		return properties;
	}

	public IPropertyDescriptor[] getPropertyDescriptors() {
		List<IPropertyDescriptor> descriptors = new ArrayList<IPropertyDescriptor>();
		for (Iterator ite = this.properties.values().iterator(); ite.hasNext();) {
			ModelProperty property = (ModelProperty) ite.next();
			IPropertyDescriptor descriptor = property.getDescriptor();
			if (descriptor != null) {
				descriptors.add(descriptor);
			}
		}
		return (IPropertyDescriptor[]) descriptors.toArray(new IPropertyDescriptor[descriptors
				.size()]);
	}

	@SuppressWarnings("unchecked")
	public Object getPropertyValue(Object id) {
		if (getModelProperty(id) == null) {
			//System.err.println("No ModelProperty id=" + id);
			return null;
		}
		IPropertyDescriptor descriptor = getModelProperty(id).getDescriptor();
		if (descriptor instanceof IEditorValueDescriptor) {
			try {
				return ((IEditorValueDescriptor) descriptor).getEditorValue(propertyAttributes
						.get(id));
			} catch (Exception ex) {
				System.err.println("UnMatch DataType. model=" + this + ",id=" + id + ", value="
						+ propertyAttributes.get(id).getClass());
				ex.printStackTrace();
			}
		}

		return propertyAttributes.get(id);

	}

	public boolean isDefaultValue(Object id, Object value) {
		if (value == null) {
			return true;
		} else {
			IPropertyDescriptor descriptor = getModelProperty(id).getDescriptor();
			if (descriptor instanceof IEditorValueDescriptor) {
				Object defaultValue = ((IEditorValueDescriptor) descriptor).getDefaultValue();
				if (value.equals(defaultValue)) {
					return true;
				}
			}
			return false;
		}
	}

	public boolean isPropertySet(Object id) {
		return properties.containsKey(id);
	}

	public void resetPropertyValue(Object id) {
		propertyAttributes.remove(id);
	}

	public void setPropertyValue(Object id, Object value) {
		setPropertyValue(id, value, true);
	}

	public void setPropertyValue(Object id, Object value, boolean isFirePropertyChange) {
		Object oldValue = propertyAttributes.get(id);
		propertyAttributes.put(id, value);
		if (isFirePropertyChange) {
			firePropertyChange(id.toString(), oldValue, value);
		}
	}

	public Object getEditableValue() {
		return this;
	}

	private boolean isLayoutKey(Object id) {
		if (id.equals("_" + X) || id.equals("_" + Y) || id.equals("_" + WIDTH)
				|| id.equals("_" + HEIGHT)) {
			return true;
		} else {
			return false;
		}
	}

	///////////////////////////////////////////////////////
	// To XML

	protected String getAttributesXML() {
		StringBuilder sb = new StringBuilder();
		StringBuilder layoutInfos = new StringBuilder();

		// properties
		Iterator iterator = getPropertyAttributes().keySet().iterator();
		while (iterator.hasNext()) {
			String key = (String) iterator.next();
			if (!isOutputAtribute(key.substring(1))) {
				continue;
			}
			if (isLayoutKey(key)) {
				addAttributeXml(layoutInfos, key);
			} else {
				addAttributeXml(sb, key);
			}
		}

		sb.append(layoutInfos.toString());
		return sb.toString();
	}

	protected boolean isOutputAtribute(String propertyName) {
		return true;
	}

	private void addAttributeXml(StringBuilder sb, String key) {
		Object value = getPropertyAttributes().get(key);
		if (value != null) {
			if (value instanceof RGB) {
				sb.append(" ").append(key.substring(1)).append("=\"");
				sb.append(ColorUtil.toHex((RGB) value)).append("\"");
			} else {
				if (value.toString().length() != 0) {
					sb.append(" ").append(key.substring(1)).append("=\"");
					sb.append(XMLUtil.escape(value.toString())).append("\"");
				}
			}
		}
	}

	protected String getAdditionalAttributesXML() {
		StringBuffer sb = new StringBuffer();
		Iterator ite = this.additionalAttributes.entrySet().iterator();
		while (ite.hasNext()) {
			Map.Entry entry = (Map.Entry) ite.next();
			sb.append(" ");
			sb.append((String) entry.getKey());
			sb.append("=\"");
			sb.append(XMLUtil.escape((String) entry.getValue()));
			sb.append("\"");
		}
		return sb.toString();
	}

	protected String getAdditionalChildElementsXML() {
		StringBuilder sb = new StringBuilder();
		if (additionalChildElements.size() != 0) {
			Iterator ite = additionalChildElements.iterator();
			while (ite.hasNext()) {
				sb.append(XMLUtil.nodeToXML((Element) ite.next()));
			}
			sb.append("\n");
		}
		return sb.toString();
	}

	public abstract String toMXML();

	///////////////////////////////////////////////////////
	// Property Methods

	protected void addStringModelProperty(String propertyName, String category, String defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new StringPropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addStringModelProperty(String propertyName, String category) {
		addStringModelProperty(propertyName, category, null);
	}

	protected void addBooleanModelProperty(String propertyName, String category,
			boolean defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new BooleanPropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addNumberModelProperty(String propertyName, String category, Integer defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new NumberPropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addNumberModelProperty(String propertyName, String category) {
		addNumberModelProperty(propertyName, category, null);
	}

	protected void addWidthHeightModelProperty(String propertyName, String category,
			String defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new WidthHeightPropertyDescriptor("_" + propertyName, propertyName, String
						.valueOf(defaultValue)), category));
	}

	protected void addDoubleModelProperty(String propertyName, String category, Double defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new DoublePropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addColorModelProperty(String propertyName, String category, RGB defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new ColorChoosePropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addColorModelProperty(String propertyName, String category) {
		addColorModelProperty(propertyName, category, null);
	}

	protected void addImageModelProperty(String propertyName, String category, String defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new ImagePropertyDescriptor("_" + propertyName, propertyName, defaultValue),
				category));
	}

	protected void addImageModelProperty(String propertyName, String category) {
		addImageModelProperty(propertyName, category, null);
	}

	protected void addListModelProperty(String propertyName, String category, String[] list) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new StringListPropertyDescriptor("_" + propertyName, propertyName, list), category));
	}
	
	protected void addListModelProperty(String propertyName, String category, String[] list, String defaultValue) {
		addModelProperty("_" + propertyName, new ModelProperty(propertyName,
				new StringListPropertyDescriptor("_" + propertyName, propertyName, list, defaultValue), category));
	}

	protected void addListModelProperty(String propertyName, String category) {
		addListModelProperty(propertyName, category, null);
	}

	protected void removePropertySheet(String propertyName) {
		ModelProperty mp = getModelProperty("_" + propertyName);
		mp.setDescriptor(null);
	}

	public Map getPropertyAttributes() {
		return propertyAttributes;
	}

	public void setAttribute(String propertyName, Object value) {
		setPropertyValue("_" + propertyName, value);
	}

	public Object getAttribute(String propertyName) {
		Object value = getPropertyValue("_" + propertyName);
		return value;
	}

	public String getAttributeToString(String propertyName) {
		Object value = getPropertyValue("_" + propertyName);
		if (value == null) {
			return "";
		} else {
			return value.toString();
		}
	}

	public int getAttributeToNumber(String propertyName) {
		Object value = getPropertyValue("_" + propertyName);
		return Integer.parseInt(value.toString());
	}

}
