package net.sf.amateras.air.mxml;

import java.io.ByteArrayInputStream;
import java.util.EventObject;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.mxml.action.IStructuredSelectionRefresh;
import net.sf.amateras.air.mxml.action.MXMLEditorAction;
import net.sf.amateras.air.mxml.editparts.MXMLEditPartFactory;
import net.sf.amateras.air.mxml.factory.MenuFactory;
import net.sf.amateras.air.mxml.factory.ModelFactory;
import net.sf.amateras.air.mxml.factory.TreeEditPartFactory;
import net.sf.amateras.air.mxml.models.IContainerModel;
import net.sf.amateras.air.mxml.models.RootModel;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.gef.ContextMenuProvider;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.KeyHandler;
import org.eclipse.gef.KeyStroke;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.ui.actions.DeleteAction;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.gef.ui.actions.RedoRetargetAction;
import org.eclipse.gef.ui.actions.UndoRetargetAction;
import org.eclipse.gef.ui.parts.ContentOutlinePage;
import org.eclipse.gef.ui.parts.GraphicalEditorWithPalette;
import org.eclipse.gef.ui.parts.GraphicalViewerKeyHandler;
import org.eclipse.gef.ui.parts.TreeViewer;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.IPageSite;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

/**
 * The graphical editor which provides WYSIWYG MXML editing.
 * 
 * @author Naoki Takezoe
 */
public class VisualMXMLEditor extends GraphicalEditorWithPalette {

	private RootModel root;
	private boolean savePreviouslyNeeded;
	private MXMLOutlinePage outline;

	private MenuFactory menuFactory;

	public VisualMXMLEditor() {
		super();
		this.root = new RootModel();
		setEditDomain(new DefaultEditDomain(this));
		menuFactory = new MenuFactory(this);
		getActionRegistry().registerAction(new UndoRetargetAction());
		getActionRegistry().registerAction(new RedoRetargetAction());
	}

	@Override
	protected PaletteRoot getPaletteRoot() {
		return ModelFactory.getPaletteRoot();
	}

	/**
	 * Updates {@link MXMLEditorAction}s.
	 * 
	 * @see MXMLEditorAction#update(IStructuredSelection)
	 */
	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		super.selectionChanged(part, selection);
		if (selection instanceof IStructuredSelection) {
			for (IStructuredSelectionRefresh action : menuFactory.getRefreshActions()) {
				action.update((IStructuredSelection) selection);
			}
		}
	}

	/**
	 * Returns an editor action.
	 */
	public IAction getAction(Object key) {
		return getActionRegistry().getAction(key);
	}

	@Override
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();
		GraphicalViewer viewer = getGraphicalViewer();
		viewer.setEditPartFactory(new MXMLEditPartFactory());
	}

	/**
	 * Returns the root model of the editing MXML.
	 * 
	 * @return the root model
	 */
	public IContainerModel getRootModel() {
		return this.root;
	}

	public void refreshGraphicalViewer() {
		this.root.removeAll();

		IEditorInput input = getEditorInput();
		if (input instanceof IFileEditorInput) {
			try {
				IFile file = ((IFileEditorInput) input).getFile();
				MXMLLoader.loadMXML(file.getContents(), root);
				GraphicalViewer viewer = getGraphicalViewer();
				if (this.root.getChildren().size() > 0) {
					viewer.setContents(this.root.getChildren().get(0));
					if (this.outline != null) {
						this.outline.getViewer().setContents(this.root.getChildren().get(0));
					}
				}

			} catch (Exception ex) {
				AIRPlugin.logException(ex);
				// TODO error
			}
		} else {
			// TODO error
		}
	}

	protected void initializeGraphicalViewer() {
		try {
			GraphicalViewer viewer = getGraphicalViewer();
			ScalableRootEditPart rootEditPart = new ScalableRootEditPart();
			viewer.setRootEditPart(rootEditPart);
			refreshGraphicalViewer();

			final DeleteAction deleteAction = new DeleteAction((IWorkbenchPart) this);
			deleteAction.setSelectionProvider(getGraphicalViewer());
			getActionRegistry().registerAction(deleteAction);
			getGraphicalViewer().addSelectionChangedListener(new ISelectionChangedListener() {
				public void selectionChanged(SelectionChangedEvent event) {
					deleteAction.update();
				}
			});

			// Create menu.
			MenuManager menuMgr = menuFactory.createGraphicalViewerMenu();
			viewer.setContextMenu(menuMgr);
			getEditorSite().registerContextMenu("#VisualMXMLEditor", menuMgr, viewer);

			// combined F2, DirectEditAction
			KeyHandler keyHandler = new KeyHandler();
			keyHandler.put(KeyStroke.getPressed(SWT.F2, 0), getActionRegistry().getAction(
					GEFActionConstants.DIRECT_EDIT));

			getGraphicalViewer().setKeyHandler(
					new GraphicalViewerKeyHandler(getGraphicalViewer()).setParent(keyHandler));

		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		super.init(site, input);
		setPartName(getEditorInput().getName());
	}

	public void doSave(IProgressMonitor monitor) {
		IEditorInput input = getEditorInput();
		if (input instanceof IFileEditorInput) {
			try {
				IFile file = ((IFileEditorInput) input).getFile();
				file.setContents(new ByteArrayInputStream(this.root.toMXML().getBytes("UTF-8")),
						false, true, monitor);
			} catch (Exception ex) {
				AIRPlugin.logException(ex);
			}
		} else {
			// TODO Can't save to the file.
		}

		setPartName(getEditorInput().getName());
		getCommandStack().markSaveLocation();
	}

	public void doSaveAs() {
		doSave(new NullProgressMonitor());
	}

	public boolean isSaveOnCloseNeeded() {
		return getCommandStack().isDirty();
	}

	public boolean isDirty() {
		return isSaveOnCloseNeeded();
	}

	public boolean isSaveAsAllowed() {
		return true;
	}

	private void setSavePreviouslyNeeded(boolean value) {
		savePreviouslyNeeded = value;
	}

	private boolean savePreviouslyNeeded() {
		return savePreviouslyNeeded;
	}

	public void commandStackChanged(EventObject event) {
		if (isDirty()) {
			if (!savePreviouslyNeeded()) {
				setSavePreviouslyNeeded(true);
				firePropertyChange(IEditorPart.PROP_DIRTY);
			}
		} else {
			setSavePreviouslyNeeded(false);
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
		super.commandStackChanged(event);
	}

	//	/**
	//	 * The action that opens the ActionScript view.
	//	 */
	//	private class OpenActionScriptViewAction extends Action {
	//		public OpenActionScriptViewAction(){
	//			super("Open ActionScript View", AIRPlugin.getImageDescriptor("icons/air.gif"));
	//		}
	//		public void run(){
	//			IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
	//			try {
	//				page.showView("net.sf.amateras.air.views.ActionScriptView");
	//			} catch(PartInitException ex){
	//			}
	//		}
	//	}

	public Object getAdapter(Class clazz) {
		if (IContentOutlinePage.class.equals(clazz)) {
			if (this.outline == null) {
				this.outline = new MXMLOutlinePage();
			}
			return outline;
		}
		return super.getAdapter(clazz);
	}

	public MenuFactory getMenuFactory() {
		return menuFactory;
	}

	//////////////////////////////////////////////////////////////////
	// Outline
	//////////////////////////////////////////////////////////////////

	private class MXMLOutlinePage extends ContentOutlinePage {

		public MXMLOutlinePage() {
			super(new TreeViewer());
		}

		public void init(IPageSite pageSite) {
			super.init(pageSite);
			menuFactory.setOutLinePageMenu(pageSite);
		}

		public void createControl(Composite parent) {
			super.createControl(parent);
			EditPartViewer viewer = getViewer();
			GraphicalViewer gviewer = getGraphicalViewer();
			viewer.setEditDomain(gviewer.getEditDomain());
			getViewer().setEditPartFactory(new TreeEditPartFactory());
			viewer.setContextMenu(new OutlineContextMenuProvider());
			viewer.setContents(getRootModel());
			getSelectionSynchronizer().addViewer(getViewer());
			//viewer.addSelectionChangedListener(this);
		}

		public EditPartViewer getViewer() {
			return super.getViewer();
		}

		public void dispose() {
			getSelectionSynchronizer().removeViewer(getViewer());
			super.dispose();
		}
	}

	private class OutlineContextMenuProvider extends ContextMenuProvider {

		public OutlineContextMenuProvider() {
			super(getGraphicalViewer());
		}

		public void buildContextMenu(IMenuManager menu) {
			menuFactory.setOutLineContextMenu(menu);
		}
	}

}
