package net.sf.amateras.air.launch;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * 
 * @author Naoki Takezoe
 */
public class LaunchAIRMainTab extends AbstractLaunchConfigurationTab {

	public static final String ATTR_PROJECT = "air.project";
	public static final String ATTR_TARGET = "air.target";
	public static final String ATTR_TARGET_ENABLE_LAUNCH = "air.target.enable.lanch";
	private Text textProject;
	private Text textDescriptor;
	private Button chkEnableLancherSetting;
	private boolean isValid = true;

	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout(1, false));
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		createFileGroup(composite);
		setControl(composite);
	}

	private void createFileGroup(Composite composite) {
		Group projectGroup = new Group(composite, SWT.NULL);
		projectGroup.setText("Project");
		projectGroup.setLayout(new GridLayout(2, false));
		projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		textProject = new Text(projectGroup, SWT.BORDER);
		textProject.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		textProject.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
			}
		});

		Button browseFile = new Button(projectGroup, SWT.PUSH);
		browseFile.setText("Browse...");
		browseFile.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				IProject project = UIUtil.browseProject();
				if (project != null) {
					textProject.setText(project.getName());
					ScopedPreferenceStore projectStore = new ScopedPreferenceStore(
							new ProjectScope(project), AIRPlugin.PLUGIN_ID);
					textDescriptor.setText(projectStore.getString(AIRPlugin.PREF_DESCRIPTOR));
					updateLaunchConfigurationDialog();
				}
			}
		});

		Group descriptorGroup = new Group(composite, SWT.NULL);
		descriptorGroup.setText("Application Descriptor");
		descriptorGroup.setLayout(new GridLayout(1, false));
		descriptorGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		textDescriptor = new Text(descriptorGroup, SWT.BORDER);
		textDescriptor.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		textDescriptor.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
			}
		});

		chkEnableLancherSetting = createCheckButton(descriptorGroup, "Enable Lancher setting");
		chkEnableLancherSetting.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (chkEnableLancherSetting.getSelection()) {
					textDescriptor.setEnabled(true);
				} else {
					textDescriptor.setEnabled(false);
				}
				updateLaunchConfigurationDialog();

			}
		});

	}

	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {

	}

	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			String project = configuration.getAttribute(ATTR_PROJECT, "");
			textProject.setText(project);

			if (isEnableDescriptorEx(configuration)) {
				chkEnableLancherSetting.setSelection(true);
				textDescriptor.setEnabled(true);
				textDescriptor.setText(configuration.getAttribute(ATTR_TARGET, ""));
			} else {
				chkEnableLancherSetting.setSelection(false);
				textDescriptor.setEnabled(false);
				IProject prj = ResourcesPlugin.getWorkspace().getRoot().getProject(project);
				textDescriptor.setText(getProjectDescriptor(prj));
			}

		} catch (Exception ex) {
			AIRPlugin.logException(ex);
		}
	}

	public static boolean isEnableDescriptorEx(ILaunchConfiguration configration)
			throws CoreException {
		String enable = configration.getAttribute(ATTR_TARGET_ENABLE_LAUNCH, "false");
		if (enable.equals("true")) {
			return true;
		} else {
			return false;
		}
	}

	public static String getProjectDescriptor(IProject project) {
		ScopedPreferenceStore projectStore = new ScopedPreferenceStore(new ProjectScope(project),
				AIRPlugin.PLUGIN_ID);
		return projectStore.getString(AIRPlugin.PREF_DESCRIPTOR);

	}

	/**
	 * perform Apply
	 * @param configuration
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		String projectName = textProject.getText();
		if (projectName.length() == 0) {
			setErrorMessage("Project not found.");
			isValid = false;
			return;
		}

		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		if (project == null || !project.exists()) {
			setErrorMessage("Project not found.");
			isValid = false;
			return;
		}

		//descriptor 
		String descriptor = null;
		if (chkEnableLancherSetting.getSelection()) {
			descriptor = textDescriptor.getText();//use lancher setting

		} else {
			descriptor = getProjectDescriptor(project);
		}
		if (descriptor == null || descriptor.length() == 0
				|| !project.getFile(new Path(descriptor)).exists()) {
			setErrorMessage("Application Descriptor not found.");
			isValid = false;
		}

		configuration.setAttribute(ATTR_TARGET_ENABLE_LAUNCH, chkEnableLancherSetting
				.getSelection() ? "true" : "false");
		configuration.setAttribute(ATTR_TARGET, descriptor);
		configuration.setAttribute(ATTR_PROJECT, projectName);

		setErrorMessage(null);
		isValid = true;
	}

	@Override
	public boolean isValid(ILaunchConfiguration launchConfig) {
		return isValid;
	}

	public String getName() {
		return "Main";
	}

}
