package net.sf.amateras.air.builder;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * Provides mxml compiler by fsch.
 * 
 * @author Naoki Takezoe
 */
public class MXMLCompiler {

	private IProject project;
	private Map<String, String> idMap = Collections.synchronizedMap(new HashMap<String, String>());
	private Pattern pattern;

	public MXMLCompiler(IProject project, Pattern pattern) {
		this.project = project;
		this.pattern = pattern;
	}

	public void setMessagePattern(Pattern pattern) {
		this.pattern = pattern;
	}

	public void dispose() {
		project = null;
		pattern = null;
		idMap.clear();
	}

	/**
	 * Requests compiling mxml file.
	 * 
	 * @param name the relative path of requested mxml file.
	 * @param fcsh fcsh process.
	 * @param moniter progress moniter.
	 * @throws IOException, {@link CoreException}
	 */
	public String requestCompileMXML(String name, FcshShell fcsh, IProgressMonitor monitor)
			throws IOException, CoreException {

		String command = null;

		if (idMap.containsKey(name)) {
			String id = (String) idMap.get(name);
			command = "compile " + id + "\n";

		} else {
			IPreferenceStore preferenceStore = AIRPlugin.getDefault().getPreferenceStore();
			String preCommand = preferenceStore.getString(AIRPlugin.PREF_FLEX_COMPILE_COMMAND);

			command = new StringBuilder(preCommand).append(" ").append(name).toString();

			//command = "mxmlc +configname=air " + name + "\n";

			ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(this.project),
					AIRPlugin.PLUGIN_ID);
			String libraries = store.getString(AIRPlugin.PREF_FLEX_LIBRARIES);
			if (libraries.length() > 0) {
				command += " -library-path+=" + libraries;
			}
			command += "\n";
		}

		monitor.setTaskName(String.format(AIRPlugin.getResourceString("COMPILE_START_COMPILE"),
				command));

		String result = fcsh.startCommand(command, monitor);
		Matcher matcher = pattern.matcher(result);
		if (matcher.find()) {
			String id = matcher.group(1);
			idMap.put(name, id);
		}

		project.refreshLocal(IResource.DEPTH_INFINITE, null);
		return result;
	}

	public String requestCompile(String compileCommand, FcshShell fcsh, IProgressMonitor monitor)
			throws IOException, CoreException {

		String command = null;

		if (idMap.containsKey(compileCommand)) {
			String id = (String) idMap.get(compileCommand);
			command = "compile " + id + "\n";

		} else {
			command = new StringBuilder(compileCommand).append("\n").toString();
		}

		monitor.setTaskName(String.format(AIRPlugin.getResourceString("COMPILE_START_COMPILE"),
				command));

		String result = fcsh.startCommand(command, monitor);
		Matcher matcher = pattern.matcher(result);
		if (matcher.find()) {
			String id = matcher.group(1);
			idMap.put(compileCommand, id);
		}

		project.refreshLocal(IResource.DEPTH_INFINITE, null);
		return result;
	}

}
