package net.sf.amateras.air.as;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DefaultIndentLineAutoEditStrategy;
import org.eclipse.jface.text.DocumentCommand;
import org.eclipse.jface.text.IDocument;

public class CodeFormattingStrategy extends DefaultIndentLineAutoEditStrategy {

	public void customizeDocumentCommand(IDocument d, DocumentCommand c) {
		if (c.length == 0 && c.text != null && isLineDelimiters(d, c.text))
			whenAfterNewLine(d, c);
		else if ("}".equals(c.text)) {
			whenAfterBracket(d, c);
		}
	}

	private boolean isLineDelimiters(IDocument d, String txt) {
		String[] delimiters = d.getLegalLineDelimiters();
		if (delimiters != null) {
			return endsWith(delimiters, txt) > -1;
		}
		return false;
	}


	private void whenAfterNewLine(IDocument document, DocumentCommand command) {
		int docLength = document.getLength();
		if (command.offset == -1 || docLength == 0) {
			return;
		}

		try {
			int p = (command.offset == docLength ? command.offset - 1 : command.offset);
			int line = document.getLineOfOffset(p);

			StringBuffer buf = new StringBuffer(command.text);
			if (command.offset < docLength && document.getChar(command.offset) == '}') {
				int indLine = findMatchingOpenBracket(document, line, command.offset, 0);
				if (indLine == -1) {
					indLine = line;
				}
				buf.append(getIndentOfLine(document, indLine));
			} else {
				int start = document.getLineOffset(line);
				int whiteend = findEndOfWhiteSpace(document, start, command.offset);
				buf.append(document.get(start, whiteend - start));
				if (getBracketCount(document, start, command.offset, true) > 0) {
					buf.append('\t');
				}
			}
			command.text = buf.toString();

		} catch (BadLocationException excp) {
			excp.printStackTrace();
		}
	}


	private int findMatchingOpenBracket(IDocument document, int line, int end,
			int closingBracketIncrease) throws BadLocationException {

		int start = document.getLineOffset(line);
		int brackcount = getBracketCount(document, start, end, false) - closingBracketIncrease;
		while (brackcount < 0) {
			line--;
			if (line < 0) {
				return -1;
			}
			start = document.getLineOffset(line);
			end = start + document.getLineLength(line) - 1;
			brackcount += getBracketCount(document, start, end, false);
		}
		return line;
	}


	private int getBracketCount(IDocument document, int start, int end, boolean ignoreCloseBrackets)
			throws BadLocationException {

		int begin = start;
		int bracketcount = 0;
		while (begin < end) {
			char curr = document.getChar(begin);
			begin++;
			switch (curr) {
			case '/':
				if (begin < end) {
					char next = document.getChar(begin);
					if (next == '*') {
						begin = getCommentEnd(document, begin + 1, end);
					} else if (next == '/') {
						begin = end;
					}
				}
				break;
			case '*':
				if (begin < end) {
					char next = document.getChar(begin);
					if (next == '/') {
						bracketcount = 0;
						begin++;
					}
				}
				break;
			case '{':
				bracketcount++;
				ignoreCloseBrackets = false;
				break;
			case '}':
				if (!ignoreCloseBrackets) {
					bracketcount--;
				}
				break;
			case '"':
			case '\'':
				begin = getStringEnd(document, begin, end, curr);
				break;
			default:
			}
		}
		return bracketcount;
	}


	private int getCommentEnd(IDocument document, int position, int end)
			throws BadLocationException {
		int currentPosition = position;
		while (currentPosition < end) {
			char curr = document.getChar(currentPosition);
			currentPosition++;
			if (curr == '*') {
				if (currentPosition < end && document.getChar(currentPosition) == '/') {
					return currentPosition + 1;
				}
			}
		}
		return end;
	}


	private String getIndentOfLine(IDocument document, int line) throws BadLocationException {
		if (line > -1) {
			int start = document.getLineOffset(line);
			int end = start + document.getLineLength(line) - 1;
			int whiteend = findEndOfWhiteSpace(document, start, end);
			return document.get(start, whiteend - start);
		}
		return "";
	}


	private int getStringEnd(IDocument document, int position, int end, char character)
			throws BadLocationException {
		int currentPosition = position;
		while (currentPosition < end) {
			char currentCharacter = document.getChar(currentPosition);
			currentPosition++;
			if (currentCharacter == '\\') {
				currentPosition++;
			} else if (currentCharacter == character) {
				return currentPosition;
			}
		}
		return end;
	}


	private void whenAfterBracket(IDocument document, DocumentCommand command) {
		if (command.offset == -1 || document.getLength() == 0) {
			return;
		}

		try {
			int p = (command.offset == document.getLength() ? command.offset - 1 : command.offset);
			int line = document.getLineOfOffset(p);
			int start = document.getLineOffset(line);
			int whiteend = findEndOfWhiteSpace(document, start, command.offset);

			if (whiteend == command.offset) {
				int indLine = findMatchingOpenBracket(document, line, command.offset, 1);
				if (indLine != -1 && indLine != line) {
					StringBuffer replaceText = new StringBuffer(getIndentOfLine(document, indLine));
					replaceText.append(document.get(whiteend, command.offset - whiteend));
					replaceText.append(command.text);
					command.length = command.offset - start;
					command.offset = start;
					command.text = replaceText.toString();
				}
			}
		} catch (BadLocationException excp) {
			excp.printStackTrace();
		}
	}

	private int endsWith(String[] searchStrings, String text) {
		int index = -1;
		for (int i = 0; i < searchStrings.length; i++) {
			if (text.endsWith(searchStrings[i])) {
				if (index == -1 || searchStrings[i].length() > searchStrings[index].length())
					index = i;
			}
		}
		return index;
	}

}
