/*--------------------------------------------------------------------
 * FILE:
 *     pgcluster.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at backend for the replication.
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */

/*--------------------------------------
 * INTERFACE ROUTINES
 *
 * setup/teardown:
 *      PGR_Close_Sock
 *      PGR_Free_Conf_Data
 * I/O call:
 *      PGR_Create_Socket_Connect
 *      PGR_Create_Socket_Bind
 *      PGR_Create_Acception
 * table handling:
 *      PGR_Get_Conf_Data
 *-------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"

#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/in.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <arpa/inet.h>
#include <sys/file.h>
#include <netdb.h>

#include "libpq/libpq.h"
#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "nodes/print.h"
#include "utils/guc.h"
#include "parser/parser.h"
#include "access/xact.h"
#include "replicate_com.h"

#ifdef HAVE_SIGPROCMASK
sigset_t	UnBlockSig,
			BlockSig,
			AuthBlockSig;

#else
int			UnBlockSig,
			BlockSig,
			AuthBlockSig;
#endif

static PGR_ConfData_Info * ConfData_Info;
static PGR_Log_Info * Log_Info;
static PGR_WatchDog_Info WatchDog_Info;

int PGR_Create_Socket_Connect(int * fdP, char * hostName , unsigned short portNumber);
void PGR_Close_Sock(int * sock);
int PGR_Create_Socket_Bind(int * fdP, char * hostName , unsigned short portNumber);
int PGR_Create_Acception(int fd, int * sockP, char * hostName , unsigned short portNumber);
int PGR_Free_Conf_Data(void);
int PGR_Get_Conf_Data(char * dir , char * fname);
void PGRset_recovery_packet_no(RecoveryPacket * packet, int packet_no);
unsigned int PGRget_ip_by_name(char * host);
void PGRsem_unlock( int semid, int sem_num );
void PGRsem_lock( int semid, int sem_num );
void PGRwrite_log_file(FILE * fp, const char * fmt,...);

void show_debug(const char * fmt,...);
void show_error(const char * fmt,...);

void PGR_Init_Com_Info(PGR_Com_Info * com_info);
/* watch dog */
void PGRsyn_quit(void);
int PGRmap_to_cluster_tbl(HostTbl * ptr);
int PGRmap_to_replication_server_tbl(ReplicateServerInfo * ptr);
int PGRmap_to_load_balancer_tbl(RecoveryTbl * ptr);
pid_t PGR_ACK_Main(uint16_t port);
pid_t PGR_SYN_Main(int syn_target, void * status_tbl);
void PGRset_start_life_check(void);
void PGRexit_life_check(int sig);
void PGRquit_lifecheck_child(int sig);
void PGRchild_wait(int sig);
int PGRsyn_init(void);

static char * get_string(char * buf);
static bool is_start_tag(char * ptr);
static bool is_end_tag(char * ptr);
static void init_conf_data(ConfDataType *conf);
static int get_key(char * key, char * str);
static int get_conf_key_value(char * key, char * value , char * str);
static int add_conf_data(char *table,int rec_no, char *key,char * value);
static int get_table_data(FILE * fp,char * table, int rec_no);
static int get_single_data(char * str);
static int get_conf_file(char * fname);
/* static function for watch dog */
static MapTable * get_map_by_pid(pid_t pid);
static pid_t do_life_check(MapTable * map, int timeout);
static int send_life_check(int sock, int send_timeout);
static int read_life_check(int sock, int recv_timeout);
static void set_map_status(MapTable *map,int status);
static void write_status_file(MapTable * map);
static void set_life_check_status( MapTable *map,int status);
static pid_t ack_loop(int fd, uint16_t port);
static void wait_start_life_check(int timeout);
char * Function;
/*--------------------------------------------------------------------
 * SYMBOL
 *     PGR_Create_Socket_Connect()
 * NOTES
 *     create new socket
 * ARGS
 *    int * fdP:
 *    char * hostName:
 *    unsigned short portNumber:
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGR_Create_Socket_Connect(int * fdP, char * hostName , unsigned short portNumber)
{

	int sock;
	size_t	len = 0;
	struct sockaddr_in addr;
	int one = 1;

	if ((*hostName == '\0') || (portNumber < 1000))
	{
		* fdP = -1;
		return STATUS_ERROR;
	}
	if ((*fdP = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		* fdP = -1;
		return STATUS_ERROR;
	}
	if ((setsockopt(*fdP, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))) == -1)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	if (setsockopt(*fdP, IPPROTO_TCP, TCP_NODELAY, (char *) &one, sizeof(one)) < 0)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	
	addr.sin_family = AF_INET;
	if (hostName[0] == '\0')
   		addr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		struct hostent *hp;

		hp = gethostbyname(hostName);
		if ((hp == NULL) || (hp->h_addrtype != AF_INET))
		{
			PGR_Close_Sock(fdP);
			return STATUS_ERROR;
		}
		memmove((char *) &(addr.sin_addr), (char *) hp->h_addr, hp->h_length);
	}

	addr.sin_port = htons(portNumber);
	len = sizeof(struct sockaddr_in);
	
	if ((sock = connect(*fdP,(struct sockaddr*)&addr,len)) < 0)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	
	return	STATUS_OK;
}

int
PGR_Create_Socket_Bind(int * fdP, char * hostName , unsigned short portNumber)
{

	int err;
	size_t	len = 0;
	struct sockaddr_in addr;
	int one = 1;

	if ((*fdP = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		return STATUS_ERROR;
	}
	if ((setsockopt(*fdP, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))) == -1)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	addr.sin_family = AF_INET;
	if (hostName[0] == '\0')
		addr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		struct hostent *hp;

		hp = gethostbyname(hostName);
		if ((hp == NULL) || (hp->h_addrtype != AF_INET))
		{
			PGR_Close_Sock(fdP);
			return STATUS_ERROR;
		}
		memmove((char *) &(addr.sin_addr), (char *) hp->h_addr, hp->h_length);
	}

	addr.sin_port = htons(portNumber);
	len = sizeof(struct sockaddr_in);
	
	err = bind(*fdP, (struct sockaddr *) & addr, len);
	if (err < 0)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	err = listen(*fdP, MAX_SOCKET_QUEUE );
	if (err < 0)
	{
		PGR_Close_Sock(fdP);
		return STATUS_ERROR;
	}
	return	STATUS_OK;
}

int
PGR_Create_Acception(int fd, int * sockP, char * hostName , unsigned short portNumber)
{
	int sock;
	struct sockaddr  addr;
	size_t	len = 0;
	int one = 1;

	len = sizeof(struct sockaddr);
	while ((sock = accept(fd,&addr,&len)) < 0)
	{
		PGR_Close_Sock(&fd);
		PGR_Create_Socket_Bind(&fd, hostName , portNumber);
	}
	
	if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *) &one, sizeof(one)) < 0)
	{
		return STATUS_ERROR;
	}
	if (setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *) &one, sizeof(one)) < 0)
	{
		return STATUS_ERROR;
	}
	*sockP = sock;

	return	STATUS_OK;
}

void
PGR_Close_Sock(int * sock)
{
	close( (int)*sock);
	*sock = -1;
}

static char *
get_string(char * buf)
{
	int i,len1,len2,start_flag;
	char *readp, *writep; 

	writep = readp = buf;
	i = len1 = 0;
	while (*(readp +i) != '\0')
	{
		if (!isspace(*(readp+ i)))
		{
			len1 ++;
		}
		i++;
	}
	start_flag = len2 = 0;
	while (*readp != '\0')
	{
		if (*readp == '#') 
		{
			*writep = '\0';
			break;
		}
		if (isspace(*readp))
		{
			if ((len2 >= len1) || (!start_flag))
			{
				readp++;
				continue;
			}
			*writep = *readp;
		}
		else
		{
			start_flag = 1;
			*writep = *readp;
			len2 ++;
		}
		readp ++;
		writep ++;
	}
	*writep = '\0';
	return buf;
}

static bool
is_start_tag(char * ptr)
{
	if ((*ptr == '<') && (*(ptr+1) != '/'))
	{
		return true;
	}
	return false;
}

static bool
is_end_tag(char * ptr)
{
	if ((*ptr == '<') && (*(ptr+1) == '/'))
	{
		return true;
	}
	return false;
}

static void
init_conf_data(ConfDataType *conf)
{
	memset(conf->table,0,sizeof(conf->table));
	memset(conf->key,0,sizeof(conf->key));
	memset(conf->value,0,sizeof(conf->value));
	conf->rec_no = 0;
	conf->last = NULL;
	conf->next = NULL;
}

static int
get_key(char * key, char * str)
{
	int offset = 1;
	char * ptr_s,*ptr_e;

	ptr_s = strchr(str,'<');
	if (ptr_s == NULL)
	{
		return STATUS_ERROR;
	}
	if (*(ptr_s+1) == '/')
	{
		offset = 2;
	}
	ptr_e = strchr(str,'>');
	if (ptr_e == NULL)
	{
		return STATUS_ERROR;
	}
	*ptr_e = '\0';
	strcpy(key,ptr_s + offset);
	*ptr_e = '>';
	return STATUS_OK;
}

static int
get_conf_key_value(char * key, char * value , char * str)
{
	int i;
	int len1,len2,start_flag;
	char * ptr_s,*ptr_e;

	if(get_key(key,str) == STATUS_ERROR)
	{
		return STATUS_ERROR;
	}
	ptr_e = strchr(str,'>');
	if (ptr_e == NULL)
	{
		return STATUS_ERROR;
	}
	ptr_s = ptr_e + 1;

	len1 = 0;
	while ((*ptr_s != '<') && (*ptr_s != '\0'))
	{
			if (! isspace(*ptr_s))
			{
				len1 ++;
			}
			ptr_s ++;
	}
	ptr_s = ptr_e + 1;
	i = len2 = start_flag = 0;
	while ((*ptr_s != '<') && (*ptr_s != '\0'))
	{
		if (isspace(*ptr_s))
		{
			if ((len2 >= len1) || (!start_flag))
			{
				ptr_s ++;
				continue;
			}
			*(value + i) = *ptr_s;
		}
		else
		{
			start_flag = 1;
			*(value + i) = *ptr_s;
			len2 ++;
		}
		i++;
		ptr_s ++;
	}
	*(value + i) = '\0';
	return STATUS_OK;
}

static int
add_conf_data(char *table,int rec_no, char *key,char * value)
{
	ConfDataType * conf_data;

	conf_data = (ConfDataType *)malloc(sizeof(ConfDataType));
	if (conf_data == NULL)
	{
		return STATUS_ERROR;
	}
	init_conf_data(conf_data);
	if (table != NULL)
	{
		memcpy(conf_data->table,table,sizeof(conf_data->table));
	}
	else
	{
		memset(conf_data->table,0,sizeof(conf_data->table));
	}
	strncpy(conf_data->key,key,sizeof(conf_data->key));
	strncpy(conf_data->value,value,sizeof(conf_data->value));
	conf_data->rec_no = rec_no;
	if (ConfData_Info->ConfData_Top == (ConfDataType *)NULL)
	{
		ConfData_Info->ConfData_Top = conf_data;
		conf_data->last = (char *)NULL;
	}
	if (ConfData_Info->ConfData_End == (ConfDataType *)NULL)
	{
		conf_data->last = (char *)NULL;
	}
	else
	{
		conf_data->last = (char *)ConfData_Info->ConfData_End;
		ConfData_Info->ConfData_End->next = (char *)conf_data;
	}
	ConfData_Info->ConfData_End = conf_data;
	conf_data->next = (char *)NULL;
	return STATUS_OK;
}

static int
get_table_data(FILE * fp,char * table, int rec_no)
{
	char buf[1024];
	char key_buf[1024];
	char value_buf[1024];
	int len = 0;
	char * ptr;

	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		/*
		 * pic up a data string
		 */
		ptr = get_string(buf);
		len = strlen(ptr);
		if (len == 0)
		{
			continue;
		}
		if (is_end_tag(ptr))
		{
			if(get_key(key_buf,ptr) == STATUS_ERROR)
			{
				return STATUS_ERROR;
			}
			if (!strcmp(key_buf,table))
			{
				return STATUS_OK;
			}
		}
		if (is_start_tag(ptr))
		{
			if(get_conf_key_value(key_buf,value_buf,ptr) == STATUS_ERROR)
			{
				return STATUS_ERROR;
			}
			add_conf_data(table,rec_no,key_buf,value_buf);
		}
	}
	return STATUS_ERROR;
}

static int
get_single_data(char * str)
{
	char key_buf[1024];
	char value_buf[1024];
	if(get_conf_key_value(key_buf,value_buf,str) == STATUS_ERROR)
	{
		return STATUS_ERROR;
	}
	add_conf_data(NULL,0,key_buf,value_buf);
	return STATUS_OK;
}


static int
get_conf_file(char * fname)
{
	FILE * fp = NULL;
	int len;
	char buf[1024];
	char key_buf[1024];
	char last_key_buf[1024];
	char *ptr;
	int rec_no = 0;

	/*
	 * configuration file open
	 */
	if ((fp = fopen(fname,"r")) == NULL)
	{
		return STATUS_ERROR;
	}
	/*
	 * configuration file read
	 */
	memset(last_key_buf,0,sizeof(last_key_buf));
	memset(key_buf,0,sizeof(key_buf));
	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		/*
		 * pic up a data string
		 */
		ptr = get_string(buf);
		len = strlen(ptr);
		if (len == 0)
		{
			continue;
		}
		if (is_start_tag(ptr))
		{
			if(get_key(key_buf,ptr) == STATUS_ERROR)
			{
				fclose(fp);
				return STATUS_ERROR;
			}
			if (strstr(ptr,"</") == NULL)
			{
				if (strcmp(last_key_buf,key_buf))
				{
					rec_no = 0;
					strcpy(last_key_buf,key_buf);
				}
				get_table_data(fp,key_buf,rec_no);
				rec_no ++;
			}
			else
			{
				get_single_data(ptr);
			}
		}
	}
	fclose(fp);
	return STATUS_OK;
}

int
PGR_Free_Conf_Data(void)
{
	ConfDataType * conf, *nextp;

	if (ConfData_Info->ConfData_Top == (ConfDataType *)NULL)
	{
		return STATUS_ERROR;
	}
	conf = ConfData_Info->ConfData_Top;

	while (conf != (ConfDataType *)NULL)
	{
		nextp = (ConfDataType*)conf->next;
		free (conf);
		conf = nextp;
	}
	ConfData_Info->ConfData_Top = ConfData_Info->ConfData_End = (ConfDataType *)NULL;
	return STATUS_OK;
}

int
PGR_Get_Conf_Data(char * dir , char * fname)
{

	int status;

	char * conf_file;
	if ((dir == NULL) || ( fname == NULL))
	{
		return STATUS_ERROR;
	}
	conf_file = malloc(strlen(dir) + strlen(fname) + 2);
	if (conf_file == NULL)
	{
		return STATUS_ERROR;
	}
	sprintf(conf_file,"%s/%s",dir,fname);

	ConfData_Info->ConfData_Top = ConfData_Info->ConfData_End = (ConfDataType * )NULL;
	status = get_conf_file(conf_file);
	free (conf_file);
	conf_file = NULL;

	return status;
}

void
PGRset_recovery_packet_no(RecoveryPacket * packet, int packet_no)
{
	if (packet == NULL)
	{
		return;
	}
	packet->packet_no = htons(packet_no) ;

}

unsigned int
PGRget_ip_by_name(char * host)
{
	struct hostent *hp = NULL;
	unsigned int ip = 0;
	unsigned char uc = 0;
	int i;

	if ((host == NULL) || (*host == '\0'))
	{
		return 0;
	}
	hp = gethostbyname( host );
	if (hp == NULL)
	{
		return 0;
	}
	for (i = 3 ; i>= 0 ; i --)
	{
		uc = (unsigned char)hp->h_addr_list[0][i];
		ip = ip | uc;
		if (i > 0)
		ip = ip << 8;
	}
	return ip;
}

void
PGRsem_unlock( int semid, int sem_num )
{
	int	status;
	int cnt = 10;
	struct sembuf sops;

	if (semid <= 0)
		return;
	sops.sem_num = sem_num;
	sops.sem_op = 1;
	sops.sem_flg = 0;

	do
	{
		status = semop(semid, &sops, 1);
		if ((status == -1) && (errno != EINTR))
		{
			usleep(cnt);
			cnt ++;
			if (cnt > 20)
				break;
		}
	} while (status == -1 && errno == EINTR);
}

void
PGRsem_lock( int semid, int sem_num )
{
	int cnt = 10;
	int	status;
	struct sembuf sops;

	if (semid <= 0)
		return;
	sops.sem_num = sem_num;
	sops.sem_op = -1;
	sops.sem_flg = 0;

	do
	{
		status = semop(semid, &sops, 1);
		if ((status == -1) && (errno != EINTR))
		{
			usleep(cnt);
			cnt ++;
			if (cnt > 20)
				break;
		}
	} while (status == -1 && errno == EINTR);
}

void
PGRwrite_log_file(FILE * fp, const char * fmt,...)
{
	char buf[256];
	char log[288];
	char * p;
	va_list ap;
	time_t t;

	if (fp == NULL)
	{
		return;
	}
	if (time(&t) < 0)
	{
		return;
	}
	snprintf(log,sizeof(log),"%s ",ctime(&t));
	p = strchr(log,'\n');
	if (p != NULL)
	{
		*p = ' ';
	}
	va_start(ap,fmt);
	vsnprintf(buf,sizeof(buf),fmt,ap);
	va_end(ap);
	strcat(log,buf);
	strcat(log,"\n");
	if (fputs(log,fp) >= 0)
	{
		fflush(fp);
	}
}


void
show_debug(const char * fmt,...)
{
	va_list ap;

	if (Log_Info->Debug_Print)
	{
		fprintf(stdout,"DEBUG:");
		va_start(ap,fmt);
		vfprintf(stdout,fmt,ap);
		va_end(ap);
		fprintf(stdout,"\n");
		fflush(stdout);
	}
}

void
show_error(const char * fmt,...)
{
	va_list ap;
	char buf[256];

	if (Log_Info->Debug_Print)
	{
		fprintf(stderr,"ERROR:");
		va_start(ap,fmt);
		vfprintf(stderr,fmt,ap);
		va_end(ap);
		fprintf(stderr,"\n");
		fflush(stderr);
	}
	if (Log_Info->Log_Print)
	{
		va_start(ap,fmt);
		vsnprintf(buf,sizeof(buf),fmt,ap);
		va_end(ap);
		PGRwrite_log_file(Log_Info->LogFp,buf);
	}
}

void 
PGR_Init_Com_Info(PGR_Com_Info * p)
{
	if (p != (PGR_Com_Info *)NULL)
	{
		/* init PGR_ConfData_Info data */
		ConfData_Info = &(p->ConfData_Info);
		ConfData_Info->ConfData_Top = (ConfDataType*)NULL;
		ConfData_Info->ConfData_End = (ConfDataType*)NULL;
		ConfData_Info->LifeCheck_Port_Number = 0;
		ConfData_Info->Recovery_Port_Number = 0;

		/* init PGR_Log_Info data */
		Log_Info = &(p->Log_Info);
		Log_Info->StatusFp = (FILE *)NULL;
		Log_Info->LogFp = (FILE *)NULL;
		Log_Info->PGRStatusFileName = NULL;
		Log_Info->PGRLogFileName = NULL;
		Log_Info->Log_Print = 0;
		Log_Info->Debug_Print = 0;
	}
	else
	{
		ConfData_Info = (PGR_ConfData_Info *)NULL;
		Log_Info = (PGR_Log_Info *)NULL;
	}
}

/*--------------------------------------
 * Watch Dog functions
 *--------------------------------------
 */


int
PGRsyn_init(void)
{
	union semun sem_arg;
	int sem_nums = 0;
	int i = 0;
	int size = 0;

	size = sizeof(MapTable) * (MAX_DB_SERVER+1);
	WatchDog_Info.MapTableShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (WatchDog_Info.MapTableShmid < 0)
	{
		show_error("shmget() failed. (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	WatchDog_Info.Map_Table = (MapTable *)shmat(WatchDog_Info.MapTableShmid,0,0);
	if (WatchDog_Info.Map_Table == (MapTable *)-1)
	{
		show_error("shmat() failed. (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(WatchDog_Info.Map_Table,0,size);

	size = sizeof(char);
	WatchDog_Info.LifeCheckStartShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (WatchDog_Info.LifeCheckStartShmid < 0)
	{
		show_error("shmget() failed. (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	WatchDog_Info.LifeCheckStartFlag = (char *)shmat(WatchDog_Info.LifeCheckStartShmid,0,0);
	if (WatchDog_Info.LifeCheckStartFlag == (char *)-1)
	{
		show_error("shmat() failed. (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	*WatchDog_Info.LifeCheckStartFlag = LIFE_CHECK_STOP;

	sem_nums = MAX_DB_SERVER + 4;
	if ((WatchDog_Info.SynSemID = semget(IPC_PRIVATE,sem_nums,IPC_CREAT | IPC_EXCL | 0600)) < 0)
	{
		show_error("semget() failed. (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	for ( i = 0 ; i < sem_nums ; i ++)
	{
		semctl(WatchDog_Info.SynSemID, i, GETVAL, sem_arg);
		sem_arg.val = 1;
		semctl(WatchDog_Info.SynSemID, i, SETVAL, sem_arg);
	}
	WatchDog_Info.LifeCheckTimeOut = 10;
	return STATUS_OK;
}

void
PGRsyn_quit(void)
{
	if (WatchDog_Info.Map_Table != (MapTable *)NULL)
	{
		shmdt((char *)WatchDog_Info.Map_Table);
		shmctl(WatchDog_Info.MapTableShmid,IPC_RMID,(struct shmid_ds *)NULL);
		WatchDog_Info.Map_Table = (MapTable *)NULL;
	}

	if (WatchDog_Info.LifeCheckStartFlag != (char *)NULL)
	{
		shmdt((char *)WatchDog_Info.LifeCheckStartFlag);
		shmctl(WatchDog_Info.LifeCheckStartShmid,IPC_RMID,(struct shmid_ds *)NULL);
		WatchDog_Info.LifeCheckStartFlag = (char *)NULL;
	}

	if (WatchDog_Info.SynSemID >= 0)
	{
		semctl(WatchDog_Info.SynSemID, 0, IPC_RMID);
		WatchDog_Info.SynSemID = -1;
	}
}

int
PGRmap_to_cluster_tbl(HostTbl * ptr)
{
	int cnt = 0;
	MapTable * map;
	map = WatchDog_Info.Map_Table;

	if ((ptr == (HostTbl*)NULL) || (map == (MapTable*)NULL))
	{
		return 0;
	}
	cnt = 0;
	while (ptr->useFlag != DATA_END)
	{
		cnt ++;
		map->useFlag = &(ptr->useFlag);
		map->hostName = (char *)(ptr->hostName);
		map->port = &(ptr->port);
		map->lifecheckPort = &(ptr->lifecheckPort);
		map->sock = 0;
		map->rec_no = cnt;
		set_life_check_status(map,LIFE_CHECK_TRY_COUNT);
		show_debug("use[%d] host[%s] port[%d] life[%d] rec[%d]",
				*(map->useFlag),
				map->hostName,
				*(map->port),
				*(map->lifecheckPort),
				map->rec_no);
		ptr ++;
		map ++;
	}
	map->useFlag = &(ptr->useFlag);
	return cnt;
}

int
PGRmap_to_replication_server_tbl(ReplicateServerInfo * ptr)
{
	int cnt = 0;
	MapTable * map;
	map = WatchDog_Info.Map_Table;

	if ((ptr == (ReplicateServerInfo*)NULL) || (map == (MapTable*)NULL))
	{
		return 0;
	}
	cnt = 0;
	while (ptr->useFlag != DATA_END)
	{
		cnt ++;
		map->useFlag = &(ptr->useFlag);
		map->hostName = (char *)(ptr->hostName);
		map->port = &(ptr->portNumber);
		map->lifecheckPort = &(ptr->lifecheckPortNumber);
		map->sock = 0;
		map->rec_no = cnt;
		set_life_check_status(map,LIFE_CHECK_TRY_COUNT);
		show_debug("use[%d] host[%s] port[%d] life[%d] rec[%d]",
				*(map->useFlag),
				map->hostName,
				*(map->port),
				*(map->lifecheckPort),
				map->rec_no);
		ptr ++;
		map ++;
	}
	map->useFlag = &(ptr->useFlag);
	return cnt;
}

int
PGRmap_to_load_balancer_tbl(RecoveryTbl * ptr)
{
	int cnt = 0;
	MapTable * map;
	map = WatchDog_Info.Map_Table;

	if ((ptr == (RecoveryTbl*)NULL) || (map == (MapTable*)NULL))
	{
		return 0;
	}
	cnt = 0;
	while (ptr->useFlag != DATA_END)
	{
		cnt ++;
		map->useFlag = &(ptr->useFlag);
		map->hostName = (char *)(ptr->hostName);
		map->port = &(ptr->port);
		map->lifecheckPort = &(ptr->lifecheckPort);
		map->sock = 0;
		map->rec_no = cnt;
		set_life_check_status(map,LIFE_CHECK_TRY_COUNT);
		show_debug("use[%d] host[%s] port[%d] life[%d] rec[%d]",
				*(map->useFlag),
				map->hostName,
				*(map->port),
				*(map->lifecheckPort),
				map->rec_no);
		ptr ++;
		map ++;
	}
	map->useFlag = &(ptr->useFlag);
	return cnt;
}

void
PGRexit_life_check(int sig)
{
	PGRsyn_quit();
	signal(SIGCHLD,SIG_DFL);
	if (sig != SIGCHLD)
	{
		signal(sig,SIG_IGN);
	}
	kill (0,sig);
	while (wait(NULL) > 0 )
		;
	exit(0);
}

void
PGRquit_lifecheck_child(int sig)
{
	MapTable * map = NULL; 
	pid_t pid = getpid();
	map = get_map_by_pid(pid);
	if (map != NULL)
	{
		if (map->sock > 0)
		{
			close(map->sock);
		}
		map->sock = -1;
		map->pid = 0;
	}
	exit(0);

}

static MapTable *
get_map_by_pid(pid_t pid)
{
	MapTable * map;
	map = WatchDog_Info.Map_Table;

	if (( map == NULL ) || (pid <= 0))
	{
		return (MapTable *)NULL;
	}
	while (*(map->useFlag) != DATA_END)
	{
		if (map->pid == pid)
		{
			return map;
		}
		map ++;
	}
	return (MapTable *)NULL;
}

static pid_t
do_life_check(MapTable * map, int timeout)
{
	pid_t pid;
	pid_t pgid = 0;
	int status;

	pgid = getpgid(0);
	if ((pid = fork()) != 0 )
	{
		return pid;
	}
	
	setpgid(0,pgid);
	pqsignal(SIGHUP,PGRquit_lifecheck_child);
	pqsignal(SIGTERM,PGRquit_lifecheck_child);
	pqsignal(SIGINT,PGRquit_lifecheck_child);
	pqsignal(SIGQUIT,PGRquit_lifecheck_child);
	PG_SETMASK(&AuthBlockSig);

	if (map == NULL)
	{
		exit(0);
	}
	if (*(map->useFlag) == DATA_ERR)
	{
		map->pid = 0;
		exit(0);
	}
	if (map->sock > 0)
	{
		close(map->sock);
	}
	status = PGR_Create_Socket_Connect(&(map->sock),map->hostName,*(map->lifecheckPort));
	if (status != STATUS_OK)
	{
		set_map_status(map,DATA_ERR);
		close(map->sock);
		map->sock = -1;
		map->pid = 0;
		exit(0);
	}
	for (;;)
	{
		status = send_life_check(map->sock, timeout);
		if (status != STATUS_OK)
		{
			set_map_status(map,DATA_ERR);
			close(map->sock);
			map->sock = -1;
			map->pid = 0;
			exit(0);
		}
		status = read_life_check(map->sock, timeout);
		if (status != STATUS_OK)
		{
			set_map_status(map,DATA_ERR);
			close(map->sock);
			map->sock = -1;
			map->pid = 0;
			exit(0);
		}
		set_life_check_status(map,LIFE_CHECK_TRY_COUNT);
		sleep(timeout);
	}
}

static int
send_life_check(int sock, int send_timeout)
{
	char life_check_data;
	int buf_size = sizeof(char);
	int s;
	int rtn;	
	fd_set	  wmask;
	struct timeval timeout;

	timeout.tv_sec = send_timeout;
	timeout.tv_usec = 0;

	life_check_data = 'L';
	/*
	 * Wait for something to happen.
	 */
	FD_ZERO(&wmask);
	FD_SET(sock,&wmask);
	rtn = select(sock+1, (fd_set *)NULL, &wmask, (fd_set *)NULL, &timeout);
	if (rtn && FD_ISSET(sock, &wmask))
	{
		for (;;)
		{
			s = send(sock,&life_check_data,buf_size ,0);
			if (s <= 0){
				if (errno == EINTR)
				{
					continue;
				}
				return STATUS_ERROR;
			}
			if (s == buf_size)
			{
				return STATUS_OK;
			}
		}
	}
	return STATUS_ERROR;
}

static int
read_life_check(int sock, int recv_timeout)
{
	char life_check_data;
	int buf_size = sizeof(char);
	int	rtn = 0;
	fd_set	  rmask;
	struct timeval timeout;
	int r = 0;

	timeout.tv_sec = recv_timeout;
	timeout.tv_usec = 0;

	/*
	 * Wait for something to happen.
	 */
	FD_ZERO(&rmask);
	FD_SET(sock,&rmask);
	rtn = select(sock+1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &timeout);
	if (rtn && FD_ISSET(sock, &rmask))
	{
		for (;;)
		{
			r = recv(sock,&life_check_data,buf_size, MSG_WAITALL);
			if (r <= 0)
			{
				if (errno == EINTR)
				{
					continue;
				}
				return STATUS_ERROR;
			}
			if (r == buf_size)
			{
				return STATUS_OK;
			}
		}
	}
	return STATUS_ERROR;
}

static void
set_map_status(MapTable *map,int status)
{
	if (map == NULL)
	{
		return;
	}
	PGRsem_lock(WatchDog_Info.SynSemID,map->rec_no);
	if (*(map->useFlag) != status)
	{
		*(map->useFlag) = status;
		write_status_file(map);
	}
	PGRsem_unlock(WatchDog_Info.SynSemID,map->rec_no);

}

static void
write_status_file(MapTable * map)
{
	if (Log_Info == NULL)
		return;
	switch(*(map->useFlag))
	{
		case DATA_FREE:
			PGRwrite_log_file(Log_Info->StatusFp,"port(%d) host:%s free",
					*(map->port),
					map->hostName);
			break;
		case DATA_INIT:
			PGRwrite_log_file(Log_Info->StatusFp,"port(%d) host:%s initialize",
					*(map->port),
					map->hostName);
			break;
		case DATA_USE:
			PGRwrite_log_file(Log_Info->StatusFp,"port(%d) host:%s start use",
					*(map->port),
					map->hostName);
			break;
		case DATA_ERR:
			PGRwrite_log_file(Log_Info->StatusFp,"port(%d) host:%s error",
					*(map->port),
					map->hostName);
			break;
		case DATA_END:
			PGRwrite_log_file(Log_Info->StatusFp,"port(%d) host:%s end",
					*(map->port),
					map->hostName);
			break;
	}
}

static void
set_life_check_status( MapTable *map,int status)
{
	if (map == NULL)
	{
		return;
	}
	PGRsem_lock(WatchDog_Info.SynSemID,map->rec_no);
	if (map->status != status)
	{
		map->status = status;
	}
	PGRsem_unlock(WatchDog_Info.SynSemID,map->rec_no);
}

void
PGRchild_wait(int sig)
{
	pid_t pid = 0;

	do {
		int ret;
		pid = waitpid(-1,&ret,WNOHANG);
	} while(pid > 0);
}

pid_t
PGR_ACK_Main(uint16_t port)
{
	int status;
	int fd = -1;
	int rtn;
	pid_t pid = 0;
	pid_t pgid = 0;

	pgid = getpgid(0);
	if ((pid = fork()) != 0 )
	{
		return pid;
	}
	setpgid(0,pgid);	
	pqsignal(SIGHUP,PGRexit_life_check);
	pqsignal(SIGTERM,PGRexit_life_check);
	pqsignal(SIGINT,PGRexit_life_check);
	pqsignal(SIGQUIT,PGRexit_life_check);
	pqsignal(SIGCHLD,PGRchild_wait);
	PG_SETMASK(&AuthBlockSig);
	status = PGR_Create_Socket_Bind(&fd, "", port);

	if (status != STATUS_OK)
	{
		return pid;
	}
	for (;;)
	{
		fd_set	  rmask;
		struct timeval timeout;

		timeout.tv_sec = 60;
		timeout.tv_usec = 0;

		/*
		 * Wait for something to happen.
		 */
		FD_ZERO(&rmask);
		FD_SET(fd,&rmask);
		rtn = select(fd+1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &timeout);
		if (rtn && FD_ISSET(fd, &rmask))
		{
			ack_loop(fd,port);
		}
	}
	exit(0);
}

static pid_t
ack_loop(int fd, uint16_t port)
{

	int count;
	int sock;
	int status = STATUS_OK;
	int timeout = 600;
	pid_t pid;
	pid_t pgid = 0;

	pgid = getpgid(0);
	if ((pid = fork()) != 0 )
	{
		return pid;
	}
	
	setpgid(0,pgid);
	pqsignal(SIGHUP,PGRquit_lifecheck_child);
	pqsignal(SIGTERM,PGRquit_lifecheck_child);
	pqsignal(SIGINT,PGRquit_lifecheck_child);
	pqsignal(SIGQUIT,PGRquit_lifecheck_child);
	PG_SETMASK(&AuthBlockSig);

	count = 0;
	while ((status = PGR_Create_Acception(fd,&sock,"",port)) != STATUS_OK)
	{
		close(sock);
		sock = -1;
		if ( count > MAX_RETRY_TIMES)
		{
			PGRquit_lifecheck_child(15);
		}
		count ++;
	}
	for(;;)
	{
		status = read_life_check(sock, timeout);
		if (status != STATUS_OK)
		{
			close(sock);
			break;
		}
		status = send_life_check(sock, timeout);
		if (status != STATUS_OK)
		{
			close(sock);
			break;
		}
	}
	PGRquit_lifecheck_child(15);
	return pid;
}

pid_t
PGR_SYN_Main(int syn_target, void * status_tbl)
{
	int cnt = 0;
	pid_t pid = 0;
	pid_t pgid = 0;
	int i = 0;
	int status = 0;
	int syn_timeout = 0;
	MapTable * map = (MapTable *)NULL;

	pgid = getpgid(0);
	if ((pid = fork()) != 0 )
	{
		return pid;
	}
	
	setpgid(0,pgid);
	pqsignal(SIGHUP,PGRexit_life_check);
	pqsignal(SIGTERM,PGRexit_life_check);
	pqsignal(SIGINT,PGRexit_life_check);
	pqsignal(SIGQUIT,PGRexit_life_check);
	signal(SIGCHLD,SIG_DFL);
	PG_SETMASK(&AuthBlockSig);

	syn_timeout = WatchDog_Info.LifeCheckTimeOut / LIFE_CHECK_TRY_COUNT;
	syn_timeout += 1;

	/* wait until target server is ready */
	wait_start_life_check(syn_timeout);

	/* map the target status table */
	switch (syn_target)
	{
		case SYN_TO_LOAD_BALANCER:
			cnt = PGRmap_to_load_balancer_tbl((RecoveryTbl *)status_tbl);
			break;
		case SYN_TO_CLUSTER_DB:
			cnt = PGRmap_to_cluster_tbl((HostTbl *) status_tbl);
			break;
		case SYN_TO_REPLICATION_SERVER:
			cnt = PGRmap_to_replication_server_tbl((ReplicateServerInfo *) status_tbl);
			break;
	}
	map = WatchDog_Info.Map_Table;
	if (map == (MapTable *)NULL)
	{
		show_error("target status table is not ready");
		PGRexit_life_check(15);
	}

	/* create child process */
	for (i = 0 ; i < cnt ; i ++)
	{
		do_life_check((map + i), syn_timeout);
	}

	/* check response from each target servers */
	for (;;)
	{
		i = 0;
		while(*((map + i)->useFlag) != DATA_END)
		{
			status = (map +i)->status;
			if (status < 0) 
			{
				if (*((map +i)->useFlag) == DATA_ERR)
				{
					/* already closed */
					i++;
					continue;
				}
				else
				{
					/* restart life check */
					do_life_check((map + i), syn_timeout);
					status = 1;
				}
			}
			if (status == 0)
			{
				/* the target server should be down */
				set_map_status((map + i),DATA_ERR);
				if ((map + i)->pid > 0)
				{
					kill((map + i)->pid,SIGTERM);
				}
			}
			status --;
			set_life_check_status((map + i),status);
			i ++;
		}
		sleep(WatchDog_Info.LifeCheckTimeOut);
	}
}

void
PGRset_start_life_check(void)
{
	if (WatchDog_Info.LifeCheckStartFlag == NULL)
	{
		return;
	}
	if (*WatchDog_Info.LifeCheckStartFlag == LIFE_CHECK_STOP)
	{
		*WatchDog_Info.LifeCheckStartFlag = LIFE_CHECK_START;
	}
}

static void
wait_start_life_check(int timeout)
{
	while(WatchDog_Info.LifeCheckStartFlag == NULL)
	{
		sleep(timeout);
	}
	while (*WatchDog_Info.LifeCheckStartFlag != LIFE_CHECK_START)
	{
		sleep(timeout);
	}

}

#endif /* USE_REPLICATION */
