/*--------------------------------------------------------------------
 * FILE:
 *     replicate.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at pgreplicate for the replication.
 *
 * Portions Copyright (c) 2003-2004, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"
#include "postgres_fe.h"

#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/msg.h>
#include <signal.h>

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"

#include <sys/socket.h>
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>

#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif
#include "access/xact.h"
#include "replicate_com.h"
#include "pgreplicate.h"

#define IPC_NMAXSEM (32)

/*--------------------------------------
 * PROTOTYPE DECLARATION
 *--------------------------------------
 */
static PGconn * pgr_createConn( char * host, char * port,char * database, char * userName );
static TransactionTbl * setTransactionTbl(HostTbl * host_ptr, ReplicateHeader * header);
static TransactionTbl * insertTransactionTbl( HostTbl * host_ptr, TransactionTbl * datap);
static TransactionTbl * getTransactionTbl( HostTbl * host_ptr, ReplicateHeader * header);
static void deleteTransactionTbl(HostTbl * host_ptr,ReplicateHeader * header);
static TransactionTbl * setPGconnTbl(HostTbl * host_ptr, ReplicateHeader * header, int use_flag);
static TransactionTbl * insertPGconnTbl( HostTbl * host_ptr, TransactionTbl * datap);
static TransactionTbl * getPGconnTbl( HostTbl * host_ptr, ReplicateHeader * header);
static void deletePGconnTbl(HostTbl * host_ptr,ReplicateHeader * header);
static HostTbl * deleteHostTbl(HostTbl * ptr);
static bool is_master_in_recovery(char * host, int port);
static void sem_quit(int semid);
static int send_cluster_status_to_load_balance(HostTbl * host_ptr,int status);
static void end_transaction_status(int recovery_status);
static void set_transaction_status(int status);
static void begin_transaction_status(int recovery_status);
static void check_transaction_status(ReplicateHeader * header,int recovery_status);
static void clearHostTbl(void);
static bool is_need_sync_time(ReplicateHeader * header);
static bool is_need_wait_answer(ReplicateHeader * header);
static void write_host_status_file(HostTbl * host_ptr);
static void delete_template(HostTbl * ptr, ReplicateHeader * header);
static char * check_copy_command(char * query);
static int read_answer(int dest);
static int next_replication_id(void);
static bool is_need_use_rlog(ReplicateHeader * header);
static bool is_autocommit_off(char * query);
static bool is_autocommit_on(char * query);
static unsigned int get_host_ip_from_tbl(char * host);
static unsigned int get_srcHost_ip_from_tbl(char * srcHost);
static int check_delete_transaction (HostTbl * host_ptr, ReplicateHeader * header);
static int send_func(HostTbl * host_ptr,ReplicateHeader * header, char * func,char * result);
static uint32_t get_oid(HostTbl * host_ptr,ReplicateHeader * header);
static int set_oid(HostTbl * host_ptr,ReplicateHeader * header, uint32_t oid);
static int replicate_lo( PGconn * conn, char cmdType, LOArgs * query);
static bool is_executed_query_in_origin( ReplicateHeader *header );
static bool is_executed_query( PGconn *conn, ReplicateHeader * header);
static int send_sync_data(PGconn *conn, ReplicateHeader * header, bool sync_command_flg, int current_cluster);
static int replicate_query(PGresult * res, PGconn * conn, HostTbl * host_ptr, ReplicateHeader * header, char *query, bool sync_command_flg, int current_cluster);
static TransactionTbl * get_conn( HostTbl * host_ptr, ReplicateHeader * header);
static TransactionTbl * recreate_conn( HostTbl* host_ptr, ReplicateHeader* header);
static void * thread_send_source(void * arg);
static void * thread_send_cluster(void * arg);
static int set_pgconn_queue(ReplicateHeader * header);

bool PGRis_same_host(char * host1, unsigned short port1 , char * host2, unsigned short port2);
HostTbl * PGRadd_HostTbl(HostTbl *  conf_data, int useFlag);
HostTbl * PGRget_master(void);
void PGRset_recovery_status(int status);
int PGRget_recovery_status(void);
int PGRcheck_recovered_host(void);
int PGRset_recovered_host(HostTbl * target,int useFlag);
int PGRinit_recovery(void);
void PGRreplicate_exit(int signal_arg);
int PGRsend_replicate_packet_to_server( HostTbl * host_ptr, ReplicateHeader * header, char *query , char * result, int type);
int PGRreplicate_packet_send_each_server( HostTbl * ptr, bool return_response, ReplicateHeader * header, char * query,int dest);
HostTbl * PGRget_HostTbl(char * hostName,int port);
int PGRset_queue(ReplicateHeader * header,char * query);
int PGRset_host_status(HostTbl * host_ptr,int status);
void PGRclear_transactions(void);
void PGRclear_connections(int use_flag);
int PGRset_replication_id(uint32_t id);
int PGRdo_replicate(int sock,ReplicateHeader *header, char * query);
int PGRreturn_result(int dest, int wait);
int PGRreplicate_packet_send( ReplicateHeader * header, char * query,int dest);
char * PGRread_packet(int sock, ReplicateHeader *header);
void PGRnotice_replication_server(char * hostName, unsigned short portNumber,unsigned short recoveryPortNumber, unsigned short lifecheckPortNumber, char * userName);
char * PGRread_query(int sock, ReplicateHeader *header);
int PGRwait_transaction_count_clear(void);
int PGRsync_oid(ReplicateHeader *header);
unsigned int PGRget_next_query_id(void);
int PGRget_pgconn_queue(void);

bool
PGRis_same_host(char * host1, unsigned short port1 , char * host2, unsigned short port2)
{
	unsigned int ip1, ip2;

	if ((host1[0] == '\0' ) || (host2[0] == '\0') ||
		( port1 != port2 ))
	{
		return false;
	}
	ip1 = PGRget_ip_by_name( host1);
	ip2 = PGRget_ip_by_name( host2);
	if ((ip1 == ip2) && (port1 == port2))
	{
		return true;
	}
	return false;
}

static PGconn *
pgr_createConn( char * host, char * port,char * database, char * userName )
{
	char *func = "pgr_createConn()";
	int cnt = 0;
	PGconn * conn = NULL;

#ifdef PRINT_DEBUG
	show_debug("%s:PQsetdbLogin host[%s] port[%s] db[%s] user[%s]",
		func,host,port,database,userName);
#endif
	conn = PQsetdbLogin(host, port, NULL, NULL, database, userName, NULL);
	/* check to see that the backend Connection was successfully made */
	cnt = 0;
	while (PQstatus(conn) == CONNECTION_BAD)
	{
		show_error("%s:PQsetdbLogin failed. close socket",func);
		if (conn != NULL)
		{
			PQfinish(conn);
		}
		conn = PQsetdbLogin(host, port, NULL, NULL, database, userName, NULL);
		if (cnt > PGR_CONNECT_RETRY_TIME )
		{
			if (conn != NULL)
			{
				PQfinish(conn);
			}
			show_error("%s:PQsetdbLogin  timeout",func);
			return (PGconn *)NULL;
		}
		cnt ++;
	}
#ifdef PRINT_DEBUG
	show_debug("%s:PQsetdbLogin ok",func);
#endif
	return conn;
}

static TransactionTbl *
setTransactionTbl(HostTbl * host_ptr, ReplicateHeader * header)
{
	char * func = "setTransactionTbl()";
	TransactionTbl * ptr = NULL;
	TransactionTbl work ;
	char port[8];
	char * hostName, *dbName, *userName;

	if ((host_ptr == NULL) || (header == NULL))
	{
		return (TransactionTbl *)NULL;
	}
	dbName = (char *)header->dbName;
	snprintf(port,sizeof(port),"%d", host_ptr->port);
	userName = (char *)header->userName;
	hostName = (char *)host_ptr->hostName;

	ptr = getTransactionTbl(host_ptr,header);
	if (ptr != NULL)
	{
		ptr->conn = pgr_createConn(hostName,port,dbName,userName);
		if (ptr->conn == NULL)
		{
			show_error("%s:Transaction is pooling but pgr_createConn failed",func);
			deleteTransactionTbl(host_ptr, header);
			PGRset_host_status(host_ptr,DATA_ERR);
			ptr = NULL;
		}
		return ptr;
	}

	memset(&work,0,sizeof(work));
	strncpy(work.host, hostName, sizeof(work.host));
	strncpy(work.srcHost, header->from_host, sizeof(work.srcHost));
	work.hostIP = PGRget_ip_by_name(hostName);
	work.port = host_ptr->port;
	work.srcHostIP = PGRget_ip_by_name(header->from_host);
	work.pid = ntohs(header->pid);
	strncpy(work.dbName,header->dbName,sizeof(work.dbName));
	strncpy(work.userName,header->userName,sizeof(work.userName));
	work.conn = pgr_createConn(hostName,port,dbName,userName);
	if (work.conn == NULL)
	{
		show_error("%s:New Transaction but pgr_createConn%s@%s failed",func,port,hostName);
		return (TransactionTbl *)NULL;
	}
	work.useFlag = DATA_USE ;
	ptr = insertTransactionTbl(host_ptr,&work);
	if (ptr == (TransactionTbl *)NULL)
	{
		show_error("%s:insertTransactionTbl failed",func);
		return (TransactionTbl *)NULL;
	}
	return ptr;
}

static TransactionTbl *
insertTransactionTbl( HostTbl * host_ptr, TransactionTbl * datap)
{
	char * func = "insertTransactionTbl()";
	TransactionTbl * workp = NULL;

	if ((host_ptr == (HostTbl *)NULL) || (datap == (TransactionTbl*)NULL))
	{
		show_error("%s:host table or transaction table is NULL",func);
		return (TransactionTbl *)NULL;
	}
	workp = Transaction_Tbl_End;
	if (workp == (TransactionTbl *)NULL)
	{
		workp = (TransactionTbl*)malloc(sizeof(TransactionTbl));
		if (workp == (TransactionTbl*)NULL)
		{
			show_error("%s:malloc() failed. (%s)",func,strerror(errno));
			return (TransactionTbl *)NULL;
		}
		Transaction_Tbl_Begin = workp;
	}
	else
	{
		workp->next = malloc(sizeof(TransactionTbl));
		if ( workp->next  == NULL)
		{
			show_error("%s:malloc() failed. (%s)",func,strerror(errno));
			return (TransactionTbl *)NULL;
		}
		workp = (TransactionTbl*)workp->next;
	}
	memset(workp,0,sizeof(TransactionTbl));
	workp->next = NULL;
	workp->last = (char *)Transaction_Tbl_End;
	Transaction_Tbl_End = workp;
	workp->hostIP = datap->hostIP;
	workp->port = datap->port;
	workp->pid = datap->pid;
	workp->srcHostIP = datap->srcHostIP;
	strncpy(workp->host,datap->host,sizeof(workp->host));
	strncpy(workp->srcHost,datap->srcHost,sizeof(workp->srcHost));
	strncpy(workp->dbName,datap->dbName,sizeof(workp->dbName));
	strncpy(workp->userName,datap->userName,sizeof(workp->userName));
	workp->conn = datap->conn;
	workp->useFlag = DATA_USE;
	return workp;
}

static TransactionTbl *
getTransactionTbl( HostTbl * host_ptr, ReplicateHeader * header)
{
	TransactionTbl * ptr = NULL;
	unsigned int host_ip,srcHost_ip;
	unsigned short pid = 0;


	if (Transaction_Tbl_Begin == (TransactionTbl *) NULL)
	{
		return (TransactionTbl * )NULL;
	}
	if ((host_ptr == (HostTbl *)NULL) ||
		(header == (ReplicateHeader *)NULL))
	{
		return (TransactionTbl * )NULL;
	}
	host_ip = get_host_ip_from_tbl(host_ptr->hostName);
	if (host_ip == 0)
	{
		host_ip = PGRget_ip_by_name(host_ptr->hostName);
	}
	srcHost_ip = get_srcHost_ip_from_tbl(header->from_host);
	if (srcHost_ip == 0)
	{
		srcHost_ip = PGRget_ip_by_name(header->from_host);
	}
	pid = ntohs(header->pid);
	ptr = Transaction_Tbl_Begin;
	while (ptr != NULL)
	{
		if ((ptr->useFlag == DATA_USE) &&
			(ptr->hostIP == host_ip) &&
			(ptr->port == host_ptr->port) &&
			(ptr->srcHostIP == srcHost_ip) &&
			(!strncmp(ptr->dbName,header->dbName,sizeof(ptr->dbName))) &&
			(!strncmp(ptr->userName,header->userName,sizeof(ptr->userName))) &&
			(ptr->pid == pid))
		{
			return ptr;
		}
		ptr = (TransactionTbl*)ptr->next;
	}
	return (TransactionTbl * )NULL;
}

static void
deleteTransactionTbl(HostTbl * host_ptr,ReplicateHeader * header)
{
	char * func ="deleteTransactionTbl()";

	TransactionTbl *ptr = (TransactionTbl*)NULL;
	TransactionTbl *last = (TransactionTbl*)NULL;
	TransactionTbl *next = (TransactionTbl*)NULL;

	ptr = getTransactionTbl(host_ptr,header);
	if (ptr != (TransactionTbl*)NULL)
	{
		last = (TransactionTbl*)ptr->last;
		next = (TransactionTbl*)ptr->next;
		if (ptr->conn != NULL)
		{
			PQfinish(ptr->conn);
		}
		if (last != (TransactionTbl*)NULL)
		{
			last->next = (char *)next;
		}
		else
		{
			Transaction_Tbl_Begin = next;
		}
		if (next != (TransactionTbl*)NULL)
		{
			next->last = (char *)last;
		}
		else
		{
			Transaction_Tbl_End = last;
		}
		free(ptr);
		ptr = NULL;
	}
	else
	{
#ifdef PRINT_DEBUG
		show_debug("%s: getTransactionTbl failed",func);
#endif
	}
}

static TransactionTbl *
setPGconnTbl(HostTbl * host_ptr, ReplicateHeader * header, int use_flag)
{
	char * func ="setPGconnTbl()";
	TransactionTbl * ptr = NULL;
	TransactionTbl work ;
	char port[8];
	char * hostName, *dbName, *userName;

	if ((host_ptr == NULL) || (header == NULL))
	{
		return (TransactionTbl *)NULL;
	}
	dbName = (char *)header->dbName;
	snprintf(port,sizeof(port),"%d", host_ptr->port);
	userName = (char *)header->userName;
	hostName = (char *)host_ptr->hostName;

	ptr = getPGconnTbl(host_ptr,header);
	if (ptr != NULL)
	{
		if (ptr->conn != NULL)
			PQfinish(ptr->conn);
		ptr->conn = pgr_createConn(hostName,port,dbName,userName);
		if (ptr->conn == NULL)
		{
			show_error("%s:Transaction is pooling but pgr_createConn failed",func);
			deletePGconnTbl(host_ptr, header);
			PGRset_host_status(host_ptr,DATA_ERR);
			ptr = NULL;
		}
		return ptr;
	}

	memset(&work,0,sizeof(work));
	work.hostIP = PGRget_ip_by_name(hostName);
	work.port = host_ptr->port;
	work.pid = (ntohs(header->pid) % Reserved_Connections);
	strncpy(work.dbName,header->dbName,sizeof(work.dbName));
	strncpy(work.userName,header->userName,sizeof(work.userName));
	work.conn = pgr_createConn(hostName,port,dbName,userName);
	if (work.conn == NULL)
	{
#ifdef PRINT_DEBUG
		show_debug("%s: %s@%s is not ready",func,port,hostName);
#endif
		return (TransactionTbl *)NULL;
	}
	work.useFlag = use_flag ;
	ptr = insertPGconnTbl(host_ptr,&work);
	if (ptr == (TransactionTbl *)NULL)
	{
		show_error("%s:insertPGconnTbl failed",func);
		return (TransactionTbl *)NULL;
	}
	return ptr;
}

static TransactionTbl *
insertPGconnTbl( HostTbl * host_ptr, TransactionTbl * datap)
{
	char * func = "insertPGconnTbl()";
	TransactionTbl * workp = NULL;

	if ((host_ptr == (HostTbl *)NULL) || (datap == (TransactionTbl*)NULL))
	{
		show_error("%s:host table or transaction table is NULL",func);
		return (TransactionTbl *)NULL;
	}
	workp = PGconn_Tbl_End;
	if (workp == (TransactionTbl *)NULL)
	{
		workp = (TransactionTbl*)malloc(sizeof(TransactionTbl));
		if (workp == (TransactionTbl*)NULL)
		{
			show_error("%s:malloc() failed. (%s)",func,strerror(errno));
			return (TransactionTbl *)NULL;
		}
		PGconn_Tbl_Begin = workp;
	}
	else
	{
		workp->next = (char *)malloc(sizeof(TransactionTbl));
		if ( workp->next  == (char*)NULL)
		{
			show_error("%s:malloc() failed. (%s)",func,strerror(errno));
			return (TransactionTbl *)NULL;
		}
		workp = (TransactionTbl*)workp->next;
	}
	memset(workp,0,sizeof(TransactionTbl));
	workp->next = NULL;
	workp->last = (char *)PGconn_Tbl_End;
	PGconn_Tbl_End = workp;
	workp->hostIP = datap->hostIP;
	workp->port = datap->port;
	workp->pid = datap->pid;
	strncpy(workp->dbName,datap->dbName,sizeof(workp->dbName));
	strncpy(workp->userName,datap->userName,sizeof(workp->userName));
	workp->conn = datap->conn;
	workp->useFlag = datap->useFlag;
	return workp;
}

static TransactionTbl *
getPGconnTbl( HostTbl * host_ptr, ReplicateHeader * header)
{
	TransactionTbl * ptr = NULL;
	unsigned int host_ip;
	unsigned short pid = 0;

	if (PGconn_Tbl_Begin == (TransactionTbl *) NULL)
	{
		return (TransactionTbl * )NULL;
	}
	if ((host_ptr == (HostTbl *)NULL) ||
		(header == (ReplicateHeader *)NULL))
	{
		return (TransactionTbl * )NULL;
	}
	host_ip = get_host_ip_from_tbl(host_ptr->hostName);
	if (host_ip == 0)
	{
		host_ip = PGRget_ip_by_name(host_ptr->hostName);
	}
	pid = (ntohs(header->pid) % Reserved_Connections);
	ptr = PGconn_Tbl_Begin;
	while (ptr != NULL)
	{
		if ( ((ptr->useFlag == DATA_USE)  || (ptr->useFlag == DATA_TEMP_USE)) &&
			(ptr->hostIP == host_ip) &&
			(ptr->port == host_ptr->port) &&
			(!strncmp(ptr->dbName,header->dbName,sizeof(ptr->dbName))) &&
			(!strncmp(ptr->userName,header->userName,sizeof(ptr->userName))) &&
			(ptr->pid == pid))
		{
			return ptr;
		}
		ptr = (TransactionTbl*)ptr->next;
	}
	return (TransactionTbl * )NULL;
}

static void
deletePGconnTbl(HostTbl * host_ptr,ReplicateHeader * header)
{
	TransactionTbl *ptr = (TransactionTbl*)NULL;
	TransactionTbl *last = (TransactionTbl*)NULL;
	TransactionTbl *next = (TransactionTbl*)NULL;

	ptr = getPGconnTbl(host_ptr,header);
	if (ptr != (TransactionTbl*)NULL)
	{
		last = (TransactionTbl*)ptr->last;
		next = (TransactionTbl*)ptr->next;
		if (ptr->conn != NULL)
		{
			PQfinish(ptr->conn);
		}
		if (last != (TransactionTbl*)NULL)
		{
			last->next = (char *)next;
		}
		else
		{
			PGconn_Tbl_Begin = next;
		}
		if (next != (TransactionTbl*)NULL)
		{
			next->last = (char *)last;
		}
		else
		{
			PGconn_Tbl_End = last;
		}
		free(ptr);
		ptr = NULL;
	}
}

static HostTbl *
deleteHostTbl(HostTbl * ptr)
{
	if (ptr != (HostTbl*)NULL)
	{
		memset(ptr,0,sizeof(HostTbl));
	}
	return ++ptr;
}

HostTbl *
PGRadd_HostTbl(HostTbl *conf_data, int useFlag)
{
	HostTbl * ptr = NULL;
	int cnt = 0;


	ptr = PGRget_HostTbl(conf_data->hostName, conf_data->port);
	if (ptr != (HostTbl*)NULL)
	{
		PGRset_host_status(ptr,useFlag);
		return ptr;
	}

	ptr = Host_Tbl_Begin;
	cnt = 1;
	while (ptr->useFlag != DATA_END)
	{
		if (ptr->useFlag == DATA_FREE)
		{
			break;
		}
		ptr ++;
		cnt ++;
	}
	if (cnt >= MAX_DB_SERVER)
	{
		return (HostTbl*)NULL;
	}
	if (ptr->useFlag == DATA_END)
	{
		(ptr + 1) -> useFlag = DATA_END;
	}
	memset(ptr,0,sizeof(HostTbl));
	ptr->hostNum = cnt;
	strncpy(ptr->hostName,conf_data->hostName,sizeof(ptr->hostName));
	ptr->port = conf_data->port;
	ptr->recoveryPort = conf_data->recoveryPort;
	ptr->lifecheckPort = conf_data->lifecheckPort;
	PGRset_host_status(ptr,useFlag);

	return ptr;
}

HostTbl *
PGRget_master(void)
{
	HostTbl * host_tbl = NULL;

	host_tbl = Host_Tbl_Begin;
	while(host_tbl->useFlag != DATA_END)
	{
		if (host_tbl->useFlag == DATA_USE)
		{
			return host_tbl;
		}
		host_tbl ++;
	}
	return (HostTbl *)NULL;
}

void
PGRset_recovery_status(int status)
{
	if (RecoverySemID <= 0)
		return;
	PGRsem_lock(RecoverySemID,SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		Recovery_Status_Inf->recovery_status = status;
	}
	PGRsem_unlock(RecoverySemID,SEM_NUM_OF_RECOVERY);
}

int
PGRget_recovery_status(void)
{
	int status = 0;

	if (RecoverySemID <= 0)
		return -1;
	PGRsem_lock(RecoverySemID,SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		status = Recovery_Status_Inf->recovery_status;
		PGRsem_unlock(RecoverySemID,SEM_NUM_OF_RECOVERY);
		return status;
	}
	PGRsem_unlock(RecoverySemID,SEM_NUM_OF_RECOVERY);
	return -1;
}

static void
begin_transaction_status(int recovery_status)
{
	if (RecoverySemID <= 0)
	{
		return ;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		if (recovery_status == RECOVERY_PREPARE_START)
		{
			Recovery_Status_Inf->transaction_count ++;
		}
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
}

static void
end_transaction_status(int recovery_status)
{
	if (RecoverySemID <= 0)
	{
		return ;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		if (recovery_status == RECOVERY_PREPARE_START)
		{
			Recovery_Status_Inf->transaction_count --;
		}
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
}

static void
set_transaction_status(int status)
{
	if (RecoverySemID <= 0)
	{
		return ;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		Recovery_Status_Inf->recovery_status = status;
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
}

int
PGRwait_transaction_count_clear(void)
{
	int cnt = 0;

	if (RecoverySemID <= 0)
	{
		return STATUS_ERROR;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		while(Recovery_Status_Inf->transaction_count > 0)
		{
			if (cnt > PGR_RECOVERY_RETRY_CNT)
			{
				PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
				return STATUS_ERROR;
			}
			usleep( PGR_RECOVERY_WAIT_MSEC );
			cnt ++;
		}
		PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
		return STATUS_OK;
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	return STATUS_ERROR;
}

int
PGRcheck_recovered_host(void)
{
	char * func = "PGRcheck_recovered_host()";
	HostTbl * ptr = NULL;
	int rtn = STATUS_OK;

	if (RecoverySemID <= 0 )
	{
		return STATUS_ERROR;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		if (Recovery_Status_Inf->useFlag != DATA_FREE)
		{
			ptr = PGRadd_HostTbl((HostTbl *)&(Recovery_Status_Inf->target_host),Recovery_Status_Inf->useFlag);
			if (ptr == (HostTbl *) NULL)
			{
				show_error("%s:PGRadd_HostTbl failed",func);
				rtn = STATUS_ERROR;
			}
			Recovery_Status_Inf->useFlag = DATA_FREE;
			memset((HostTbl *)&(Recovery_Status_Inf->target_host),0,sizeof(HostTbl));

		}
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	return rtn;
}

int
PGRset_recovered_host(HostTbl * target, int useFlag)
{
	if (SemID <= 0)
	{
		return -1;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		Recovery_Status_Inf->useFlag = useFlag;
		if (target != (HostTbl*)NULL)
		{
			memcpy((HostTbl *)&(Recovery_Status_Inf->target_host),target,sizeof(HostTbl));
			PGRset_host_status(target,useFlag);
		}
	}
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY);
	return 0;
}

static bool
is_master_in_recovery(char * host , int port)
{
	int status = 0;
	HostTbl * master = NULL;

	status = PGRget_recovery_status();
	if (status == RECOVERY_START)
	{
		master = PGRget_master();
		if (master == (HostTbl *)NULL)
		{
			return false;
		}
		if ((!strcmp(host,master->hostName)) &&
			(port == master->port))
		{
			return true;
		}
	}
	return false;
}

int
PGRinit_recovery(void)
{
	char * func = "PGRinit_recovery()";
	int size = 0;
	union semun sem_arg;
	int i = 0;

	if ((RecoverySemID = semget(IPC_PRIVATE,4,IPC_CREAT | IPC_EXCL | 0600)) < 0)
	{
		show_error("%s:semget() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	for ( i = 0 ; i < 4 ; i ++)
	{
		semctl(RecoverySemID, i, GETVAL, sem_arg);
		sem_arg.val = 1;
		semctl(RecoverySemID, i, SETVAL, sem_arg);
	}

	size = sizeof(RecoveryStatusInf);
	RecoveryShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (RecoveryShmid < 0)
	{
		show_error("%s:shmget() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	Recovery_Status_Inf = (RecoveryStatusInf *)shmat(RecoveryShmid,0,0);
	if (Recovery_Status_Inf == (RecoveryStatusInf *)-1)
	{
		show_error("%s:shmat() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(Recovery_Status_Inf,0,sizeof(RecoveryStatusInf));
	PGRset_recovery_status(RECOVERY_INIT);
	PGRset_recovered_host((HostTbl *)NULL, DATA_FREE);
	set_transaction_status(0);

	/*
	 * create message queue
	 */
	QueryLogMsgid = msgget (IPC_PRIVATE, 00666 | IPC_CREAT );
	if (QueryLogMsgid < 0)
	{
		show_error("%s,msgget() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}

	QueryLogAnsMsgid = msgget (IPC_PRIVATE, 00666 | IPC_CREAT );
	if (QueryLogAnsMsgid < 0)
	{
		show_error("%s:msgget() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}

	return STATUS_OK;
}

static void
clearHostTbl(void)
{
	HostTbl * ptr = NULL;

	if (Host_Tbl_Begin == NULL)
	{
		return;
	}
	/* normal socket close */
	ptr = Host_Tbl_Begin;
	while(ptr->useFlag != DATA_END)
	{
		ptr = deleteHostTbl(ptr);
	}	
}

void
PGRreplicate_exit(int sig)
{
	char fname[PGR_MESSAGE_BUFSIZE];
	int rtn = 0;

	Exit_Request = true;
	if (sig == SIGTERM)
	{
		if (Idle_Flag == BUSY_MODE)
		{
			return;
		}
	}

	signal(SIGCHLD,SIG_IGN);
	signal(sig,SIG_IGN);
	kill (0,sig);
	while (wait(NULL) > 0 )
	{
		fprintf(stderr,".");
	}
	fprintf(stderr,"\n ok. stop all child processes\n");

	/* recovery status clear */	
	PGRset_recovery_status(RECOVERY_INIT);

	/* normal socket close */
	clearHostTbl();

	if (Host_Tbl_Begin != (HostTbl *)NULL)
	{
		rtn = shmdt((char *)Host_Tbl_Begin);
		shmctl(HostTblShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Host_Tbl_Begin = NULL;
	}

	if (LoadBalanceTbl != (RecoveryTbl *)NULL)
	{
		rtn = shmdt((char *)LoadBalanceTbl);
		shmctl(LoadBalanceTblShmid,IPC_RMID,(struct shmid_ds *)NULL);
		LoadBalanceTbl = (RecoveryTbl *)NULL;
	}

	if (Cascade_Tbl != (ReplicateServerInfo *)NULL)
	{
		rtn = shmdt((char *)Cascade_Tbl);
		shmctl(CascadeTblShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Cascade_Tbl = NULL;
	}

	if (Cascade_Inf != (CascadeInf *)NULL)
	{
		rtn = shmdt((char *)Cascade_Inf);
		shmctl(CascadeInfShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Cascade_Inf = NULL;
	}

	if (Commit_Log_Tbl != (CommitLogInf *)NULL)
	{
		rtn = shmdt((char *)Commit_Log_Tbl);
		shmctl(CommitLogShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Commit_Log_Tbl = NULL;
	}

	if (Recovery_Status_Inf != (RecoveryStatusInf *)NULL)
	{
		rtn = shmdt((char *)Recovery_Status_Inf);
		shmctl(RecoveryShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Recovery_Status_Inf = NULL;
	}

	if (Lock_Wait_Tbl != (LockWaitInf *)NULL)
	{
		rtn = shmdt((char *)Lock_Wait_Tbl);
		shmctl(LockWaitTblShmid,IPC_RMID,(struct shmid_ds *)NULL);
		Lock_Wait_Tbl = NULL;
	}

	if (PGconnMsgid >= 0)
	{
		msgctl(PGconnMsgid, IPC_RMID,(struct msqid_ds *)NULL);
	}

	if (QueryLogMsgid >= 0)
	{
		msgctl(QueryLogMsgid,IPC_RMID,(struct msqid_ds *)NULL);
		QueryLogMsgid  = -1;
	}
	if (QueryLogAnsMsgid >= 0)
	{
		msgctl(QueryLogAnsMsgid,IPC_RMID,(struct msqid_ds *)NULL);
		QueryLogAnsMsgid = -1;
	}

	if (Com_Info.Log_Info.StatusFp != NULL)
	{
		fflush(Com_Info.Log_Info.StatusFp);
		fclose(Com_Info.Log_Info.StatusFp);
		Com_Info.Log_Info.StatusFp = NULL;
	}
	if (Com_Info.Log_Info.LogFp != NULL)
	{
		fflush(Com_Info.Log_Info.LogFp);
		fclose(Com_Info.Log_Info.LogFp);
		Com_Info.Log_Info.LogFp = NULL;
	}

	if (PGR_Result != NULL)
	{
		free(PGR_Result);
		PGR_Result = NULL;
	}
	if (PGR_Response_Inf != NULL)
	{
		free(PGR_Response_Inf);
		PGR_Response_Inf = NULL;
	}

	if (LoadBalanceTbl != NULL)
	{
		free(LoadBalanceTbl);
		LoadBalanceTbl = NULL;
	}

	if (PGR_Log_Header != NULL)
	{
		free(PGR_Log_Header);
		PGR_Log_Header = NULL;
	}

	if (PGR_Send_Query_ID != NULL)
	{
		free(PGR_Send_Query_ID);
		PGR_Send_Query_ID = NULL;
	}

	if (SemID > 0)
	{
		sem_quit(SemID);
		SemID = 0;
	}
	if (RecoverySemID > 0)
	{
		sem_quit(RecoverySemID);
		RecoverySemID = 0;
	}

	snprintf(fname, sizeof(fname), "%s/%s", PGR_Data_Path, PGREPLICATE_PID_FILE);
	unlink(fname);

	/* close socket between rlog process */
	if (Replicateion_Log->r_log_sock >= 0)
	{
		close(Replicateion_Log->r_log_sock);
		Replicateion_Log->r_log_sock = -1;
	}
	if (Replicateion_Log->RLog_Sock_Path != NULL)
	{
		unlink(Replicateion_Log->RLog_Sock_Path);
		free(Replicateion_Log->RLog_Sock_Path);
		Replicateion_Log->RLog_Sock_Path = NULL;
	}

	PGRsyn_quit();

	pthread_exit((void *) 0);
}

static int
send_cluster_status_to_load_balance(HostTbl * host_ptr,int status)
{
	RecoveryPacket packet;
	int rtn = 0;

	memset(&packet,0,sizeof(RecoveryPacket));
	packet.packet_no = htons(status);
	strncpy(packet.hostName,host_ptr->hostName,sizeof(packet.hostName));
	packet.port = htons(host_ptr->port);
	rtn = PGRsend_load_balance_packet(&packet);
	return rtn;
}

int
PGRset_host_status(HostTbl * host_ptr,int status)
{
	if (host_ptr == NULL)
	{
		return STATUS_ERROR;
	}
	if (host_ptr->useFlag != status)
	{
		host_ptr->useFlag = status;
		if (status == DATA_ERR )
		{
			send_cluster_status_to_load_balance(host_ptr,RECOVERY_ERROR_CONNECTION);
		}
		write_host_status_file(host_ptr);
	}
	return STATUS_OK;
}

static void
write_host_status_file(HostTbl * host_ptr)
{
	switch( host_ptr->useFlag)
	{
		case DATA_FREE:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"port(%d) host:%s free",
					host_ptr->port,
					host_ptr->hostName);
			break;
		case DATA_INIT:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"port(%d) host:%s initialize",
					host_ptr->port,
					host_ptr->hostName);
			break;
		case DATA_USE:
		case DATA_NEW:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"port(%d) host:%s start use",
					host_ptr->port,
					host_ptr->hostName);
			break;
		case DATA_ERR:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"port(%d) host:%s error",
					host_ptr->port,
					host_ptr->hostName);
			break;
		case DATA_END:
			PGRwrite_log_file(Com_Info.Log_Info.StatusFp,"port(%d) host:%s end",
					host_ptr->port,
					host_ptr->hostName);
			break;
	}
}

/*--------------------------------------------------
 * SYMBOL
 *     PGRsend_replicate_packet_to_server()
 * NOTES
 *     Send query data to the cluster DB and recieve result data. 
 * ARGS
 *     HostTbl * host_ptr: the record of cluster DB table (target)
 *     ReplicateHeader * header: header data
 *     char *query: query data 
 *     char * result: returned result data 
 * RETURN
 *     STATUS_OK: OK
 *     STATUS_ERROR: NG
 *     STATUS_LOCK_CONFLICT: Lock conflicted
 *---------------------------------------------------
 */
int
PGRsend_replicate_packet_to_server( HostTbl * host_ptr, ReplicateHeader * header, char *query , char * result,int type)
{
	char * func = "PGRsend_replicate_packet_to_server()";
	TransactionTbl * transaction_tbl = NULL;
	char *database = NULL;
	char port[8];
	char *userName = NULL;
	char * host = NULL;
	char * str = NULL;
	PGresult * res = (PGresult *)NULL;
	PGconn * conn = (PGconn *)NULL;
	bool sync_command_flg = false;
	int rtn = 0;
	int current_cluster = 0;
	int hostNum = 0;
	int query_size = 0;
	int status = STATUS_OK;

	if ((query == NULL) || (header == NULL))
	{
		show_error("%s:query is broken",func);
		return STATUS_ERROR;
	}
	query_size = ntohl(header->query_size);
	if (query_size < 0)
	{
		show_error("%s:query is broken",func);
		return STATUS_ERROR;
	}
	if (host_ptr == NULL)
	{
		show_error("%s:host table is empty",func);
		return STATUS_ERROR;
	}

	hostNum = host_ptr->hostNum;
	
	/* check query id */
	if ((*(PGR_Send_Query_ID + hostNum ) == ntohl(header->request_id)) &&
		( ntohl(header->request_id) != 0))
	{
		return STATUS_OK;
	}

	/*
	 * set up the connection
	 */
	database = (char *)header->dbName;
	snprintf(port,sizeof(port),"%d", host_ptr->port);
	userName = (char *)header->userName;
	host = host_ptr->hostName;
	if (PGR_Response_Inf != NULL)
	{
		current_cluster = PGR_Response_Inf->current_cluster;
	}
	/*
	 * When the query is transaction query...
	 */
	if (is_need_sync_time(header) == true)
	{
		sync_command_flg = true;
	}
	if ((header->cmdSts == CMD_STS_TRANSACTION ) ||
		(header->cmdSts == CMD_STS_SET_SESSION_AUTHORIZATION ))
	{
		if ((header->cmdSts == CMD_STS_TRANSACTION ) &&
			(header->cmdType != CMD_TYPE_BEGIN))
		{
			sync_command_flg = false;
		}
	}
	/*
	 * get the transaction table data
	 * it has the connection data with each cluster DB
	 */
	transaction_tbl = get_conn(host_ptr, header);
	if (transaction_tbl == NULL)
	{
		if (header->cmdSts != CMD_STS_NOTICE ) 
		{
			show_error("%s: could not get connection",func);
			PGRset_host_status(host_ptr,DATA_ERR);
		}
		return STATUS_ERROR;
	}
	conn = transaction_tbl->conn;
	if (conn == NULL)
	{
		show_error("%s:[%d@%s] may be down",func,host_ptr->port,host_ptr->hostName);
		if ( header->cmdSts != CMD_STS_NOTICE )
		{
			PGRset_host_status(host_ptr,DATA_ERR);
		}
		return STATUS_ERROR;
	}

	if (header->rlog > 0)
	{
		if (is_executed_query( conn, header) == true)
		{
			return STATUS_OK;
		}
	}
	else
	{
		send_sync_data(conn,header,sync_command_flg,current_cluster);
	}
	if ((header->cmdType == CMD_TYPE_COPY_DATA) ||
		(header->cmdType == CMD_TYPE_COPY_DATA_END))
	{
		/* copy data replication */
		rtn =PQputnbytes(conn, query,query_size);
		if (header->cmdType == CMD_TYPE_COPY_DATA_END)
		{
			rtn = PQendcopy(conn);
			if (rtn == 1)
			{
				PQfinish(transaction_tbl->conn);
				transaction_tbl->conn = (PGconn *)NULL;
				StartReplication[current_cluster] = true;
			}
		}
		*(PGR_Send_Query_ID + hostNum ) = ntohl(header->request_id);
		return STATUS_OK;
	}
	else if (header->cmdSts == CMD_STS_LARGE_OBJECT)
	{
		replicate_lo(conn,header->cmdType,(LOArgs *)query);
		return STATUS_OK;
	}
	else
	{
		status = replicate_query(res, conn, host_ptr,header, query, sync_command_flg, current_cluster);
		if ((status == STATUS_OK) && (res == NULL))
		{
			return STATUS_OK;
		}
	}

	if (res == NULL)
	{
		StartReplication[current_cluster] = true;
		return STATUS_ERROR;
	}
#ifdef PRINT_DEBUG
	show_debug("%s:PQexec send :%s",func,query);
#endif			
	str = PQcmdStatus(res);
	if ((str == NULL) || (*str == '\0'))
	{
		if ((result != NULL) && (res != NULL) && (res->errMsg != NULL))
		{
			snprintf(result,PGR_MESSAGE_BUFSIZE,"E%s",res->errMsg);
		}
		else
		{
			strcpy(result,"E");
		}
		StartReplication[current_cluster] = true;
	}
	else
	{
		if (!strncmp(str,PGR_LOCK_CONFLICT_NOTICE_CMD,strlen(PGR_LOCK_CONFLICT_NOTICE_CMD)))
		{
#ifdef PRINT_DEBUG
			show_debug("%s:LOCK CONFLICT from PQexec",func);
#endif			
			if (res != NULL)
				PQclear(res);
			return STATUS_LOCK_CONFLICT;
		}
		else if (!strncmp(str,PGR_DEADLOCK_DETECT_NOTICE_CMD,strlen(PGR_DEADLOCK_DETECT_NOTICE_CMD)))
		{
#ifdef PRINT_DEBUG
			show_debug("%s:DEADLOCK DETECTED from PQexec",func);
#endif			
			if (res != NULL)
				PQclear(res);
			return STATUS_DEADLOCK_DETECT;
		}
		snprintf(result,PGR_MESSAGE_BUFSIZE,"C%s",str);
	}

	if (res != NULL)
		PQclear(res);

	/* set send query id */
	*(PGR_Send_Query_ID + hostNum ) = ntohl(header->request_id);

	/*
	 * if the query is end transaction process...
	 */
	check_delete_transaction(host_ptr,header);
	return STATUS_OK;

}

static TransactionTbl *
get_conn( HostTbl * host_ptr, ReplicateHeader * header)
{
	TransactionTbl * transaction_tbl = NULL;
	/*
	 * get the transaction table data
	 * it has the connection data with each cluster db
	 */
	if ((header->cmdSts != CMD_STS_QUERY)  &&
		(header->cmdSts != CMD_STS_OTHER))
	{
		transaction_tbl = getTransactionTbl(host_ptr,header);
	}
	else
	{
		transaction_tbl = getPGconnTbl(host_ptr,header);
	}
	/*
	 * if the transaction process is new one, 
	 * create connection data and add the transaction table
	 */
	if (transaction_tbl == (TransactionTbl *)NULL)
	{
		if ((header->cmdSts != CMD_STS_QUERY)  &&
			(header->cmdSts != CMD_STS_OTHER))
		{
			transaction_tbl = setTransactionTbl(host_ptr, header);
		}
		else
		{
			transaction_tbl = setPGconnTbl(host_ptr, header, DATA_TEMP_USE);
			set_pgconn_queue(header);
		}
		if (transaction_tbl == (TransactionTbl *)NULL)
		{
			return (TransactionTbl*)NULL;
		}
	}
	return (transaction_tbl);
}

static TransactionTbl *
recreate_conn( HostTbl* host_ptr, ReplicateHeader* header)
{
	TransactionTbl * transaction_tbl = NULL;
	/*
	 * re-create connection data
	 */
	deletePGconnTbl(host_ptr, header);
	if ((header->cmdSts != CMD_STS_QUERY)  &&
		(header->cmdSts != CMD_STS_OTHER))
	{
		transaction_tbl = setTransactionTbl(host_ptr, header);
	}
	else
	{
		transaction_tbl = setPGconnTbl(host_ptr, header, DATA_TEMP_USE);
		set_pgconn_queue(header);
	}
	return transaction_tbl;
}

static int
send_sync_data(PGconn *conn, ReplicateHeader * header, bool sync_command_flg, int current_cluster)
{
	char * func = "send_sync_data()";
	static PGresult * res = (PGresult *)NULL;
	char sync_command[PGR_MESSAGE_BUFSIZE];

	/*
	 * execute query
	 */
	if ((( header->cmdSts != CMD_STS_NOTICE ) &&
		( header->cmdSts != CMD_STS_LARGE_OBJECT)) &&
		((sync_command_flg == true)           ||
		 (StartReplication[current_cluster] == true)))
	{
		snprintf(sync_command,sizeof(sync_command),
			"SELECT %s(%d,%u,%u,%u,%d) ",
			PGR_SYSTEM_COMMAND_FUNC,
			PGR_SET_CURRENT_TIME_FUNC_NO,
			(unsigned int)ntohl(header->tv.tv_sec),
			(unsigned int)ntohl(header->tv.tv_usec),
			(unsigned int)ntohl(PGR_Log_Header->replicate_id),
			PGR_Response_Inf->response_mode);
#ifdef PRINT_DEBUG
		show_debug("%s:sync_command(%s)",func,sync_command);
#endif			
		res = PQexec(conn, sync_command);
		if (res != NULL)
		{
			PQclear(res);
		}
	}
	res = NULL;
	return STATUS_OK;
}

static int
replicate_query(PGresult * res, PGconn * conn, HostTbl * host_ptr, ReplicateHeader * header, char *query, bool sync_command_flg, int current_cluster)
{
	char * func ="replicate_query()";
	TransactionTbl * transaction_tbl = NULL;
	int cnt = 0;

	while ((res = PQexec(conn, query)) == NULL)
	{
		if (header->cmdSts == CMD_STS_NOTICE )
		{
			StartReplication[current_cluster] = true;
			return STATUS_OK;
		}
		if (cnt > MAX_RETRY_TIMES)
		{
#ifdef PRINT_DEBUG
			show_debug("%s:PQexec failed",func);
#endif			
			if (header->cmdSts != CMD_STS_NOTICE )
			{
				PGRset_host_status(host_ptr,DATA_ERR);
			}
			StartReplication[current_cluster] = false;
			return STATUS_ERROR;
		}
		transaction_tbl = recreate_conn(host_ptr,header);
		if (transaction_tbl == NULL)
		{
			if (header->cmdSts != CMD_STS_NOTICE )
			{
				show_error("%s: could not get connection",func);
				PGRset_host_status(host_ptr,DATA_ERR);
			}
			StartReplication[current_cluster] = false;
			return STATUS_ERROR;
		}
		conn = transaction_tbl->conn;
		send_sync_data(conn,header,sync_command_flg,current_cluster);
		cnt ++;
	}
	return STATUS_OK;
}

static bool
is_executed_query_in_origin( ReplicateHeader *header )
{
	char *database = NULL;
	char port[8];
	char *userName = NULL;
	char * host = NULL;
	HostTbl * host_ptr = (HostTbl*)NULL;
	TransactionTbl * transaction_tbl = (TransactionTbl*)NULL;
	PGconn * conn = (PGconn *)NULL;
	bool result = false;

	if (Host_Tbl_Begin == NULL)
	{
		return STATUS_ERROR;
	}
	host_ptr = Host_Tbl_Begin;
	while(host_ptr->useFlag != DATA_END)
	{
		/*
		 * check the status of the cluster DB
		 */
		if (host_ptr->useFlag != DATA_USE)
		{
			host_ptr ++;
			continue;
		}
		if (PGRis_same_host(header->from_host,ntohs(header->port),host_ptr->hostName, host_ptr->port) == true)
		{
			break;
		}
		host_ptr ++;
	}
	if (host_ptr->useFlag == DATA_END)
	{
		return false;
	}
	/*
	 * set up the connection
	 */
	transaction_tbl = getTransactionTbl(host_ptr,header);
	if (transaction_tbl == (TransactionTbl *)NULL)
	{
		transaction_tbl = setTransactionTbl(host_ptr, header);
		if (transaction_tbl == (TransactionTbl *)NULL)
		{
			return false;
		}
	}
	else
	{
		if ((transaction_tbl->conn == (PGconn *)NULL) ||
			(transaction_tbl->conn->sock > 0))
		{
			database = (char *)header->dbName;
			snprintf(port,sizeof(port),"%d", host_ptr->port);
			userName = (char *)header->userName;
			host = host_ptr->hostName;
		 	transaction_tbl->conn = pgr_createConn(host,port,database,userName);
		}
	}
	conn = transaction_tbl->conn;
	if (conn == NULL)
	{
		return false;
	}

	result = is_executed_query( conn, header);
	deleteTransactionTbl(host_ptr,header);
	return result;
}

static bool
is_executed_query( PGconn *conn, ReplicateHeader * header)
{
	char * func = "is_executed_query()";
	static PGresult * res = (PGresult *)NULL;
	char sync_command[PGR_MESSAGE_BUFSIZE];
	char * str = NULL;

	snprintf(sync_command,sizeof(sync_command),
		"SELECT %s(%d,%u,%u,%u,%d) ",
		PGR_SYSTEM_COMMAND_FUNC,
		PGR_QUERY_CONFIRM_ANSWER_FUNC_NO,
		(unsigned int)ntohl(header->tv.tv_sec),
		(unsigned int)ntohl(header->tv.tv_usec),
		(unsigned int)ntohl(header->replicate_id),
		PGR_Response_Inf->response_mode);
#ifdef PRINT_DEBUG
	show_debug("%s:sync_command(%s)",func,sync_command);
#endif		
	res = PQexec(conn, sync_command);
	if (res != NULL)
	{
		str = PQcmdStatus(res);
#ifdef PRINT_DEBUG
show_debug("%s:PQcmdStatus return[%s]",func,str);
#endif		
		if ((str != NULL) &&
			(!strncmp(str,PGR_ALREADY_REPLICATED_NOTICE_CMD,strlen(PGR_ALREADY_REPLICATED_NOTICE_CMD))))
		{
			PQclear(res);
			return true;
		}
		PQclear(res);
	}
	return false;
}

static int
replicate_lo( PGconn * conn, char cmdType, LOArgs * query)
{
	int status = STATUS_OK;
	int mode = 0;
	Oid lobjId = 0;
	int fd = 0;
	char * buf = NULL;
	size_t len = 0;
	int offset = 0;
	int whence = 0;

	if ((conn == (PGconn *)NULL) || (query == (LOArgs *)NULL))
	{
		return STATUS_ERROR;
	}
	switch (cmdType)
	{
		case CMD_TYPE_LO_CREATE :
			mode = (int)ntohl(query->arg1);
			if (lo_creat(conn, mode) > 0)
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		case CMD_TYPE_LO_OPEN :
			lobjId = (Oid)ntohl(query->arg1);
			mode = (int)ntohl(query->arg2);
			if (lo_open(conn, lobjId, mode) > 0)
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		case CMD_TYPE_LO_WRITE :
			fd = (int)ntohl(query->arg1);
			len = (int)ntohl(query->arg2);
			buf = query->buf;
			if (lo_write(conn, fd, buf, len) == len )
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		case CMD_TYPE_LO_LSEEK :
			fd = (int)ntohl(query->arg1);
			offset = (int)ntohl(query->arg2);
			whence = (int)ntohl(query->arg3);
			if (lo_lseek(conn, fd, offset, whence) >= 0)
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		case CMD_TYPE_LO_CLOSE :
			fd = (int)ntohl(query->arg1);
			if (lo_close(conn, fd) == 0)
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		case CMD_TYPE_LO_UNLINK :
			lobjId = (Oid)ntohl(query->arg1);
			if (lo_unlink(conn,lobjId) >= 0)
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_ERROR;
			}
			break;
		default :
			break;
	}
	return status;
}

static int
check_delete_transaction (HostTbl * host_ptr, ReplicateHeader * header)
{
	char	   *database = NULL;

	if ((host_ptr == NULL) || (header == NULL))
	{
		return STATUS_ERROR;
	}
	database = (char *)header->dbName;
	if (header->cmdSts == CMD_STS_SET_SESSION_AUTHORIZATION )
	{
		if (header->cmdType == CMD_TYPE_SESSION_AUTHORIZATION_END )
		{
			deleteTransactionTbl(host_ptr,header);
		}
	}
	else
	{
		if (header->cmdSts == CMD_STS_TRANSACTION )
		{
			if ((header->cmdType == CMD_TYPE_COMMIT) ||
		 		(header->cmdType == CMD_TYPE_ROLLBACK))
			{
				/*
				 * logout the cluster DB and delete transaction table
				 */
				if (PGR_AutoCommit)
				{
					deleteTransactionTbl(host_ptr,header);
				}
			}
		}
	}
	delete_template(host_ptr, header);
	return STATUS_OK;
}

static void
check_transaction_status(ReplicateHeader * header,int recovery_status)
{
	if (header == (ReplicateHeader *)NULL)
	{
		return;
	}

	if (header->cmdSts == CMD_STS_TRANSACTION )
	{
		if (header->cmdType != CMD_TYPE_BEGIN )
		{
			begin_transaction_status(recovery_status);
		}
		else if ((header->cmdType != CMD_TYPE_COMMIT) &&
				 (header->cmdType != CMD_TYPE_ROLLBACK))
		{
			end_transaction_status(recovery_status);
		}
	}

}
/*
 * set query in queue 
 */
int
PGRset_queue(ReplicateHeader * header,char * query)
{
	char * func = "PGRset_queue()";

	int query_size = 0;

	if ((RecoverySemID <= 0) || (header == NULL))
	{
		return STATUS_ERROR;
	}
	query_size = ntohl(header->query_size);
	if (query_size < 0)
	{
		return STATUS_ERROR;
	}
	PGRsem_lock(RecoverySemID, SEM_NUM_OF_RECOVERY_QUEUE);
	/*
	 * open recovery queue file
	 */
	RecoveryQueue.queue_fp = PGRget_recovery_queue_file_for_write();
	if (RecoveryQueue.queue_fp == (FILE *)NULL)
	{
		return STATUS_ERROR;
	}
	fseek(RecoveryQueue.queue_fp,0,SEEK_END);
	if (fwrite(header,sizeof(ReplicateHeader),1,RecoveryQueue.queue_fp) < 1)
	{
		PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY_QUEUE);
		show_error("%s:recovery queue file header write error:(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	/*
	 * set query data
	 */
	if (query_size > 0)
	{
		if (fwrite(query,query_size,1,RecoveryQueue.queue_fp) < 1)
		{
			PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY_QUEUE);
			show_error("%s:recovery queue file query write error:(%s)",func,strerror(errno));
			return STATUS_ERROR;
		}
	}
	fflush(RecoveryQueue.queue_fp);
	PGRsem_unlock(RecoverySemID, SEM_NUM_OF_RECOVERY_QUEUE);

	return STATUS_OK;	
}

HostTbl *
PGRget_HostTbl(char * hostName, int port)
{
	HostTbl * ptr = NULL;
	int len = 0;

	if (Host_Tbl_Begin == NULL)
	{
		return NULL;
	}
	len = strlen(hostName);
	ptr = Host_Tbl_Begin;
	if (len > sizeof(ptr->hostName))
	{
		len = sizeof(ptr->hostName);
	}
	while(ptr->useFlag != DATA_END)
	{
		if ((! memcmp(ptr->hostName,hostName,len)) &&
			(ptr->port == port))
		{
			return ptr;
		}
		ptr ++;
	}
	return (HostTbl*)NULL;
}

static void
sem_quit(int semid)
{
	semctl(semid, 0, IPC_RMID);
}

void
PGRclear_transactions(void)
{
	TransactionTbl *t_ptr = (TransactionTbl*)NULL;
	TransactionTbl *t_last = (TransactionTbl*)NULL;

	t_ptr = Transaction_Tbl_Begin;
	while (t_ptr != (TransactionTbl*)NULL)
	{
		if (t_ptr->conn != NULL)
		{
			PQfinish(t_ptr->conn);
			t_ptr->conn = NULL;
		}
		t_last = t_ptr;
		t_ptr = (TransactionTbl*)t_ptr->next;
		free(t_last);
		t_last = NULL;
	}
}

void
PGRclear_connections(int use_flag)
{
	TransactionTbl *t_ptr = (TransactionTbl*)NULL;
	TransactionTbl *t_last = (TransactionTbl*)NULL;

	t_ptr = PGconn_Tbl_Begin;
	while (t_ptr != (TransactionTbl*)NULL)
	{
		if (t_ptr->useFlag != use_flag)
		{
			t_ptr = (TransactionTbl*)t_ptr->next;
			continue;
		}

		if (t_ptr->conn != NULL)
		{
			PQfinish(t_ptr->conn);
			t_ptr->conn = NULL;
		}
		t_last = t_ptr;
		t_ptr = (TransactionTbl*)t_ptr->next;
		free(t_last);
		t_last = NULL;
	}
}

static bool
is_need_sync_time(ReplicateHeader * header)
{
	bool rtn = false;

	if ((header->cmdSts == CMD_STS_QUERY ) &&
		((header->cmdType == CMD_TYPE_INSERT) || 
		 (header->cmdType == CMD_TYPE_UPDATE) || 
		 (header->cmdType == CMD_TYPE_DELETE) || 
		 (header->cmdType == CMD_TYPE_SET) || 
		 (header->cmdType == CMD_TYPE_EXECUTE)))
	{
		rtn = true;	
	}
	else
	{
		if ((header->cmdType == CMD_TYPE_COPY) ||
			(header->cmdType == CMD_TYPE_SELECT) ||
			(header->cmdType == CMD_TYPE_BEGIN))
		{
			rtn = true;
		}
		if ((header->cmdSts == CMD_STS_TRANSACTION ) &&
			(header->cmdType != CMD_TYPE_BEGIN))
		{
			rtn = false;
		}
	}
	return rtn;
}

static bool
is_need_wait_answer(ReplicateHeader * header)
{
	bool rtn = false;

	if ((header->cmdType == CMD_TYPE_COPY) ||
		(header->cmdType == CMD_TYPE_COPY_DATA) ||
		(header->cmdType == CMD_TYPE_COPY_DATA_END))
	{
		rtn = false;
	}
	else if ((header->cmdSts == CMD_STS_QUERY ) &&
		((header->cmdType == CMD_TYPE_INSERT) || 
		 (header->cmdType == CMD_TYPE_UPDATE) || 
		 (header->cmdType == CMD_TYPE_DELETE) || 
		 (header->cmdType == CMD_TYPE_EXECUTE)))
	{
		rtn = true;
	}
	else if ((header->cmdSts == CMD_STS_TRANSACTION ) ||
			(header->cmdSts == CMD_STS_SET_SESSION_AUTHORIZATION ) ||
			(header->cmdSts == CMD_STS_TEMP_TABLE ) ||
			(header->cmdType == CMD_TYPE_SELECT))
	{
		rtn = true;
	}

	return rtn;
}

static void
delete_template(HostTbl * ptr, ReplicateHeader * header)
{
	if ((ptr == (HostTbl *)NULL ) ||
		(header == (ReplicateHeader *)NULL) )
	{
		return;
	}

	if ((! strcmp(header->dbName,"template1")) ||
		(! strcmp(header->dbName,"template0")))
	{
		if ((header->cmdSts == CMD_STS_QUERY)  ||
			(header->cmdSts == CMD_STS_OTHER))
		{
			deletePGconnTbl(ptr,header);
		}
		else if (header->cmdSts == CMD_STS_NOTICE )
		{
			deleteTransactionTbl(ptr,header);
		}
	}
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    check_copy_command()
 * NOTES
 *    check the query which it is copy command or not 
 *    when the query is 'copy from', set 'stdin' after 'from' 
 * ARGS
 *    char * query: query strings(I)
 * RETURN
 *    copy command : changed copy command
 *    other command : NULL
 *--------------------------------------------------------------------
 */
static char *
check_copy_command(char * query)
{
	char * p;
	char * p1, *p2, *wp;
	char * buf;
	int size;

	if (query == NULL)
	{
		return NULL;
	}
	size = strlen(query) + strlen("  stdin  ");
	p = p1 = query;
	wp = strstr(p,"FROM");
	if (wp == NULL)
		wp = strstr(p,"from");
	
	if (wp != NULL)
	{
		p = wp + strlen("FROM");
		*p = '\0';
		p ++;
		while ((isspace(*p)) && (*p != '\0')) p++;
		while ((!isspace(*p)) && (*p != '\0')) p++;
		p2 = p;
		buf = malloc(size);
		if (buf == NULL)
		{
			return NULL;
		}
		snprintf(buf,size,"%s stdin %s",p1,p2);
		return buf;
	}
	return NULL;
}

static int
next_replication_id(void)
{
	char * func = "next_replication_id()";
	uint32_t replication_id = 0;

	if (Recovery_Status_Inf == (RecoveryStatusInf *)NULL)
	{
		show_error("%s: Recovery_Status_Inf is NULL",func);
		return -1;
	}
	replication_id = (Recovery_Status_Inf->replication_id & PGR_GET_DATA_FILTER);
	if (replication_id + 1 >= PGR_MAX_COUNTER)
	{
		if ((Recovery_Status_Inf->replication_id & PGR_GET_OVER_FLOW_FILTER) == 0)
		{
			Recovery_Status_Inf->replication_id = PGR_SET_OVER_FLOW;
		}
		else
		{
			Recovery_Status_Inf->replication_id = 0;
		}
		Recovery_Status_Inf->replication_id += PGR_MIN_COUNTER;
	}
	Recovery_Status_Inf->replication_id ++;
	return (Recovery_Status_Inf->replication_id);
}

int
PGRset_replication_id(uint32_t id)
{
	char * func = "set_replication_id()";
	uint32_t replication_id = 0;

	if (Recovery_Status_Inf == (RecoveryStatusInf *)NULL)
	{
		show_error("%s: Recovery_Status_Inf is NULL",func);
		return -1;
	}
	replication_id = (id & PGR_GET_DATA_FILTER);
	if (replication_id + 1 >= PGR_MAX_COUNTER)
	{
		if ((id & PGR_GET_OVER_FLOW_FILTER) == 0)
		{
			id = PGR_SET_OVER_FLOW;
		}
		else
		{
			id = 0;
		}
		id += PGR_MIN_COUNTER;
	}
	id ++;
	Recovery_Status_Inf->replication_id = id;
	return (Recovery_Status_Inf->replication_id);
}

int
PGRdo_replicate(int sock,ReplicateHeader *header, char * query)
{
	char * func = "PGRdo_replicate()";
	struct timeval tv;
	int status = STATUS_OK;
	int recovery_status = 0;
	bool need_sync_time = false;
	char * query_string = NULL;

	if ((header->cmdSts == CMD_STS_COPY) &&
		(header->cmdType == CMD_TYPE_COPY))
	{
		query_string = check_copy_command(query);
		if (query_string == NULL)
		{
			return LOOP_CONTINUE;
		}
	}
	else
	{
		query_string = query;
		if (header->cmdType == CMD_TYPE_SET)
		{
			if (is_autocommit_off(query_string) == true)
			{
				PGR_AutoCommit = false;
			}
			else if (is_autocommit_on(query_string) == true)
			{
				PGR_AutoCommit = true;
			}
		}
	}
	gettimeofday(&tv,NULL);
	header->tv.tv_sec = htonl(tv.tv_sec);
	header->tv.tv_usec = htonl(tv.tv_usec);
	need_sync_time = is_need_sync_time(header);
	/* save header for logging */
	if (need_sync_time == true)
	{
		if (PGR_Log_Header != NULL)
		{
			memcpy(PGR_Log_Header,header,sizeof(ReplicateHeader));
			if (header->rlog == 0)
			{
				PGR_Log_Header->replicate_id = htonl(next_replication_id());
			}
		}
	}
	/* check rlog */
	if (header->rlog == CONNECTION_SUSPENDED_TYPE )
	{
		if (PGRget_rlog_header(header) == STATUS_OK)
		{
			header->rlog = CONNECTION_SUSPENDED_TYPE;
		}
	}
	
	/* check recovery mode */
	recovery_status = PGRget_recovery_status();
	PGRcheck_recovered_host();
	check_transaction_status(header,recovery_status);
	if (PGRget_recovery_status() == RECOVERY_START)
	{
		PGRset_queue(header,query_string);
	}

	/* send replication packet */
	status = PGRreplicate_packet_send( header,query_string,sock);

	if ((header->cmdType == CMD_TYPE_COPY) &&
		(query_string != NULL))
	{
		free(query_string);
		query_string = NULL;
		query = NULL;
	}
	if (status == STATUS_ABORTED )
	{
#ifdef PRINT_DEBUG
		show_debug("%s:status is STATUS_ABORTED",func);
#endif			
		return LOOP_END;
	}
	if (status == STATUS_DEADLOCK_DETECT) 
	{
#ifdef PRINT_DEBUG
		show_debug("%s:status is STATUS_DEADLOCK_DETECT",func);
#endif			
		return LOOP_END;
	}
	return LOOP_CONTINUE;
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    PGRreturn_result()
 * NOTES
 *    Return result of execution 
 * ARGS
 *    int dest: socket of destination server (I)
 *    int wait: wait flag (I)
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *    NG: STATUS_LOCK_CONFLICT
 *    NG: STATUS_DEADLOCK_DETECT
 *--------------------------------------------------------------------
 */
int
PGRreturn_result(int dest, int wait)
{
	char * func = "PGRreturn_result()";
	fd_set	  wmask;
	struct timeval timeout;
	int rtn = 0;
	char * send_ptr = NULL;
	int send_size= 0;
	int buf_size = 0;
	int s = 0;
	int status = 0;
	int cnt = 0;
	int flag = 0;
	
	if (PGR_Result == NULL)
	{
		show_error("%s:PGR_Result is not initialize",func);
		return STATUS_ERROR;
	}
	if (dest < 0)
	{
		show_error("%s:destination socket is closed",func);
		return STATUS_ERROR;
	}
	send_ptr = PGR_Result;
	buf_size = PGR_MESSAGE_BUFSIZE;
	if (buf_size < 1)
	{
		buf_size = 1;
	}

	timeout.tv_sec = PGR_SEND_TIMEOUT;
	timeout.tv_usec = 0;

	/*
	 * Wait for something to happen.
	 */
#ifdef MSG_DONTWAIT
	flag |= MSG_DONTWAIT;
#endif
#ifdef MSG_NOSIGNAL
	flag |= MSG_NOSIGNAL;
#endif
	FD_ZERO(&wmask);
	FD_SET(dest,&wmask);
	rtn = select(dest+1, (fd_set *)NULL, &wmask, (fd_set *)NULL, &timeout);
	if (rtn && FD_ISSET(dest, &wmask))
	{
#ifdef PRINT_DEBUG
		show_debug("%s:PGRreturn_result[%s]",func,send_ptr);
#endif			
		for (;;)
		{
			s = send(dest,send_ptr + send_size,buf_size - send_size ,flag); 
			if (s < 0){
				show_error("%s:send error: %d(%s)",func,errno,strerror(errno));
				if (errno == EINTR)
				{
					usleep(PGR_SEND_WAIT_MSEC);
				}
#ifdef EPIPE
				if (errno == EPIPE)
				{
					memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
					return STATUS_ABORTED;
				}
#endif /* EPIPE */
#ifdef EHOSTUNREACH
				if (errno == EHOSTUNREACH)
				{
					memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
					return STATUS_ABORTED;
				}
#endif /* EHOSTUNREACH */
#ifdef ENOTSOCK
				if (errno == ENOTSOCK)
				{
					memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
					return STATUS_ABORTED;
				}
#endif /* ENOTSOCK */
				if (cnt < PGR_SEND_RETRY_CNT )
				{
					cnt ++;
					usleep(PGR_SEND_WAIT_MSEC);
					continue;
				}
				else
				{
					memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
					return STATUS_ERROR;
				}
			}
			if (s > 0)
			{
				send_size += s;
				if (send_size == buf_size)
				{
					status = STATUS_OK;
					if (wait == PGR_WAIT_ANSWER)
					{
#ifdef PRINT_DEBUG
						show_debug("%s:wait for answer",func);
#endif			
						status = read_answer(dest);
					}
#ifdef PRINT_DEBUG
					show_debug("%s:status of PGRreturn_result[%d]",func,status);
#endif			
					return status;
				}
			}
			usleep(PGR_SEND_WAIT_MSEC);
			if (flag != 0)
			{
				if (cnt < PGR_SEND_RETRY_CNT )
				{
					cnt ++;
					usleep(PGR_SEND_WAIT_MSEC);
					continue;
				}
				else
				{
					memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
					return STATUS_ERROR;
				}
			}
		}
	}
	memset(send_ptr, 0, PGR_MESSAGE_BUFSIZE);
	return STATUS_ERROR;
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    read_answer()
 * NOTES
 *    Receive answer packet
 * ARGS
 *    int dest: socket of destination server (I)
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *    NG: STATUS_LOCK_CONFLICT
 *    NG: STATUS_DEADLOCK_DETECT
 *--------------------------------------------------------------------
 */
static int
read_answer(int dest)
{
	char * func = "read_answer()";
	fd_set	  rmask;
	struct timeval timeout;
	int rtn;
	ReplicateHeader header;
	char * answer = NULL;
	int status = STATUS_ERROR;

	for(;;)
	{
		if (answer != NULL)
		{
			free(answer);
			answer = NULL;
		}
		timeout.tv_sec = PGR_RECV_TIMEOUT;
		timeout.tv_usec = 0;
		FD_ZERO(&rmask);
		FD_SET(dest,&rmask);
		rtn = select(dest+1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &timeout);
		if (rtn && FD_ISSET(dest, &rmask))
		{
			memset(&header,0,sizeof(ReplicateHeader));
			answer = PGRread_packet(dest,&header);
			if (answer == NULL)
			{
				status = STATUS_ERROR;
				break;
			}
			if ((header.cmdSts != CMD_STS_RESPONSE) && 
				(header.cmdSts != CMD_STS_NOTICE))
			{
				show_error("%s:none response packet received",func);
				free(answer);
				answer = NULL;
				status = STATUS_ERROR;
				break;
			}
#ifdef PRINT_DEBUG
			show_debug("answer[%s]",answer);
#endif			
			if (answer != NULL)
			{
				if (!strncmp(answer,PGR_QUERY_DONE_NOTICE_CMD,strlen(PGR_QUERY_DONE_NOTICE_CMD)))
				{
#ifdef PRINT_DEBUG
					show_debug("%s:QUERY DONE",func);
#endif			
					status = STATUS_OK;
				}
				else if (!strncmp(answer,PGR_QUERY_ABORTED_NOTICE_CMD,strlen(PGR_QUERY_ABORTED_NOTICE_CMD)))
				{
#ifdef PRINT_DEBUG
					show_debug("%s:QUERY ABORTED",func);
#endif			
					status = STATUS_ABORTED;
				}
				else if (!strncmp(answer,PGR_LOCK_CONFLICT_NOTICE_CMD,strlen(PGR_LOCK_CONFLICT_NOTICE_CMD)))
				{
#ifdef PRINT_DEBUG
					show_debug("%s:LOCK CONFLICT !!",func);
#endif			
					status = STATUS_LOCK_CONFLICT;
				}
				else if (!strncmp(answer,PGR_DEADLOCK_DETECT_NOTICE_CMD,strlen(PGR_DEADLOCK_DETECT_NOTICE_CMD)))
				{
#ifdef PRINT_DEBUG
					show_debug("%s:DEADLOCK DETECT !!",func);
#endif			
					status = STATUS_DEADLOCK_DETECT;
				}
				free(answer);
				answer = NULL;
			}
			return status;
		}
	}
	return status;
}

/*--------------------------------------------------
 * SYMBOL
 *     PGRreplicate_packet_send()
 * NOTES
 *     Send query to each cluster DB servers and return result.
 * ARGS 
 *     ReplicateHeader * header : packet header (I)
 *     char * query : query for replication (I)
 *     int dest : destination socket for return result (I)
 * RETURN
 *     OK : STATUS_OK
 *     NG : STATUS_ERROR
 *     DEADLOCK : STATUS_DEADLOCK_DETECT
 *---------------------------------------------------
 */
int
PGRreplicate_packet_send( ReplicateHeader * header, char * query,int dest)
{
	char *func = "PGRreplicate_packet_send()";
	HostTbl * host_ptr = (HostTbl*)NULL;
	HostTbl * source_host_ptr = (HostTbl*)NULL;
	int rtn = 0;
	int status = STATUS_OK;
	int sem_no = 0;
	int sem_id = 0;
	pthread_t thread[MAX_DB_SERVER];
	pthread_attr_t attr;
	int rc = 0;
	int t = 0;
	int t_cnt = 0;
	ThreadArgInf thread_arg[MAX_DB_SERVER];

	bool abort_test_flag;
	abort_test_flag = false;

#ifdef PRINT_DEBUG
	show_debug("%s:cmdSts=%c",func,header->cmdSts);
	show_debug("%s:cmdType=%c",func,header->cmdType);
	show_debug("%s:rlog=%d",func,header->rlog);
	show_debug("%s:request_id=%d",func,ntohl(header->request_id));
	show_debug("%s:replicate_id=%d",func,ntohl(header->replicate_id));
	show_debug("%s:port=%d",func,ntohs(header->port));
	show_debug("%s:pid=%d",func,ntohs(header->pid));
	show_debug("%s:from_host=%s",func,header->from_host);
	show_debug("%s:dbName=%s",func,header->dbName);
	show_debug("%s:userName=%s",func,header->userName);
	show_debug("%s:recieve sec=%u",func,ntohl(header->tv.tv_sec));
	show_debug("%s:recieve usec=%u",func,ntohl(header->tv.tv_usec));
	show_debug("%s:query_size=%d",func,ntohl(header->query_size));
	show_debug("%s:query=%s",func,query);
#endif			

	/* check rlog type */
	if (header->rlog > 0)
	{
		if (header->rlog == FROM_R_LOG_TYPE)
		{
			if (is_executed_query_in_origin(header) == false)
			{
#ifdef PRINT_DEBUG
				show_debug("this query is not yet done in source cluster db. so it wait for receive re-replicate request");
#endif
				/* wait re-replicate request */
				return STATUS_SKIP_REPLICATE;
			}
		}
	}
	/*
	 * loop while registrated cluster DB exist 
	 */
	if (Host_Tbl_Begin == NULL)
	{
		return STATUS_ERROR;
	}
	host_ptr = Host_Tbl_Begin;
	PGR_Response_Inf->current_cluster = 0;
	memset(PGR_Result,0,PGR_MESSAGE_BUFSIZE);

	sem_id = SemID;
	sem_no = 1;
#ifdef PRINT_DEBUG
	show_debug("sem_lock[%d]",sem_no);
#endif			
	PGRsem_lock(sem_id,sem_no);

	/* set replication log */
	if (is_need_use_rlog(header) == true)
	{
		PGRset_rlog(header,query);
	}

	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
	PGR_Response_Inf->current_cluster = 0;
	t_cnt = 0;
	while(host_ptr->useFlag != DATA_END)
	{
		/*
		 * check the status of the cluster DB
		 */
		if ((host_ptr->useFlag != DATA_USE) &&
			(host_ptr->useFlag != DATA_NEW))
		{
			host_ptr ++;
			continue;
		}
		/*
		 * skip loop during recover and the host name is master DB
		 */
		if (is_master_in_recovery(host_ptr->hostName, host_ptr->port) == true)
		{
			host_ptr ++;
			continue;
		}
#ifdef INSERT_ABORT_TEST
		if ((header->cmdSts == CMD_STS_QUERY) &&
			(header->cmdType == CMD_TYPE_INSERT))
		{
			if (t_cnt >= 1)
			{
				abort_test_flag = true;
				break;
			}
		}
#endif
#ifdef COMMIT_ABORT_TEST
		if ((header->cmdSts == CMD_STS_TRANSACTION) &&
			(header->cmdType == CMD_TYPE_COMMIT))
		{
			if (t_cnt >= 1)
			{
				abort_test_flag = true;
				break;
			}
		}
#endif
		/*
		 *  compare with the host name and the exceptional host name
		 */
		thread_arg[t_cnt].header = header;
		thread_arg[t_cnt].query = query;
		thread_arg[t_cnt].dest = dest;
		thread_arg[t_cnt].host_ptr = host_ptr;
		if (PGRis_same_host(header->from_host,ntohs(header->port),host_ptr->hostName, host_ptr->port) == true)
		{
			/* replication to source cluster db */
			source_host_ptr = host_ptr;
			if (header->rlog != FROM_R_LOG_TYPE )
			{
				rc = pthread_create(&thread[t_cnt], &attr, thread_send_source, (void*)&thread_arg[t_cnt]);
				t_cnt ++;
			}
			else
			{
#ifdef PRINT_DEBUG
				show_debug("%s: This simple query was suspended. Therefore this query is not re-replicated to source cluster db.",func);
#endif			
			}
		}
		else
		{
			/* replication to other cluster db */
			if ((header->rlog == CONNECTION_SUSPENDED_TYPE ) &&
				(header->cmdSts == CMD_STS_TRANSACTION) )
			{
#ifdef PRINT_DEBUG
				show_debug("%s: This transaction query was suspended. Therefore this query is not replicated to other cluster dbs.",func);
#endif			
			}
			else
			{
				rc = pthread_create(&thread[t_cnt], &attr, thread_send_cluster, (void*)&thread_arg[t_cnt]);
				t_cnt ++;
			}
		}
		if (rc)
		{
			show_error("pthread_create error");
		}
		/*
		 * send replication query to each cluster server
		 */

		host_ptr ++;
		PGR_Response_Inf->current_cluster ++;
		status = STATUS_OK;
	}	
	pthread_attr_destroy(&attr);
	for ( t = 0 ; t < t_cnt; t ++)
	{
		rc = pthread_join(thread[t], (void **)&status);
		pthread_detach(thread[t]);
	}

#ifdef INSERT_ABORT_TEST
	if (abort_test_flag)
	{
		PGRsem_unlock(sem_id,sem_no);
		PGRstop_pgreplicate(SIGINT);
	}
#endif
#ifdef COMMIT_ABORT_TEST
	if (abort_test_flag)
	{
		PGRsem_unlock(sem_id,sem_no);
		PGRstop_pgreplicate(SIGINT);
	}
#endif

	/* unset replication log */
	if (is_need_use_rlog(header) == true)
	{
		PGRunset_rlog(header,query);
	}

#ifdef PRINT_DEBUG
	show_debug("sem_unlock[%d]",sem_no);
#endif			
	PGRsem_unlock(sem_id,sem_no);
	if ((header->cmdType != CMD_TYPE_COPY)  &&
		(header->cmdType != CMD_TYPE_COPY_DATA) &&
		(header->cmdType != CMD_TYPE_COPY_DATA_END) &&
		(PGR_Response_Inf->response_mode == PGR_RELIABLE_MODE))
	{
		snprintf(PGR_Result,PGR_MESSAGE_BUFSIZE,"%d", PGR_RELIABLE_MODE_DONE_FUNC_NO);
		rtn = PGRreturn_result(dest,PGR_NOWAIT_ANSWER);
		if (source_host_ptr != NULL)
		{
			check_delete_transaction(source_host_ptr,header);
		}
	}
	return status;
}

static void *
thread_send_source(void * arg)
{
	char * func = "thread_send_source()";
	ThreadArgInf * thread_arg = NULL;
	ReplicateHeader * header = (ReplicateHeader*)NULL;
	char * query = NULL;
	int dest = 0;
	HostTbl * host_ptr = (HostTbl*)NULL;
	int status = STATUS_OK;

	if ((PGR_Result == NULL) || (arg == NULL))
	{
		show_error("%s:PGR_Result is not initialize",func);
		status = STATUS_ERROR;
		pthread_exit((void *) status);
	}
	thread_arg = (ThreadArgInf *)arg;
	header = thread_arg->header;
	query = thread_arg->query;
	dest = thread_arg->dest;
	host_ptr = thread_arg->host_ptr;

	if (header->rlog == FROM_R_LOG_TYPE )
	{
		/* It is not necessary to return rlog to source DB. */
#ifdef PRINT_DEBUG
	show_debug("%s: It is not necessary to return rlog to source DB",func);
#endif
		status = STATUS_OK;
		pthread_exit((void *) status);
	}

	memset (PGR_Result,0,PGR_MESSAGE_BUFSIZE);
	if (is_need_sync_time(header) == true)
	{
		snprintf(PGR_Result,PGR_MESSAGE_BUFSIZE,
			"%d,%u,%u,%u,%d", 
			PGR_SET_CURRENT_TIME_FUNC_NO,
			(unsigned int)ntohl(header->tv.tv_sec),
			(unsigned int)ntohl(header->tv.tv_usec),
			(unsigned int)ntohl(PGR_Log_Header->replicate_id),
			PGR_Response_Inf->response_mode);
	}
	/* execute query in the exceptional host */
	/* it is not use replication */
	if (is_need_wait_answer(header) == true)
	{
		status = PGRreturn_result(dest, PGR_WAIT_ANSWER);
	}
	else
	{
		status = PGRreturn_result(dest, PGR_NOWAIT_ANSWER);
	}
	if (status == STATUS_ERROR )
	{
		show_error("%s: %s[%d] should be down ",func,host_ptr->hostName,host_ptr->port);
		PGRset_host_status(host_ptr,DATA_ERR);
	}
	else if (status == STATUS_ABORTED)
	{
		snprintf(PGR_Result,PGR_MESSAGE_BUFSIZE,"%d", PGR_NOTICE_ABORT_FUNC_NO);
		status = PGRreturn_result(dest, PGR_NOWAIT_ANSWER);
		status = STATUS_ABORTED;
		pthread_exit((void *) status);
	}
	/* delete server table when query use template db */
	if (PGR_Response_Inf->response_mode != PGR_RELIABLE_MODE)
	{
		delete_template(host_ptr,header);
	}
	pthread_exit((void *) 0);
}

static void *
thread_send_cluster(void * arg)
{
	char * func = "thread_send_cluster()";
	ThreadArgInf * thread_arg = NULL;
	ReplicateHeader * header = (ReplicateHeader*)NULL;
	char * query = NULL;
	int dest = 0;
	HostTbl * host_ptr = (HostTbl*)NULL;
	int rtn = 0;
	int status = STATUS_OK;

	if ((PGR_Result == NULL) || (arg == NULL))
	{
		show_error("%s:PGR_Result is not initialize",func);
		status = STATUS_ERROR;
		pthread_exit((void *) status);
	}
	thread_arg = (ThreadArgInf *)arg;
	header = thread_arg->header;
	query = thread_arg->query;
	dest = thread_arg->dest;
	host_ptr = thread_arg->host_ptr;
	/*
	 * send replication query to each cluster server
	 */
	rtn = PGRreplicate_packet_send_each_server( host_ptr, false, header, query, dest);
	if (rtn == STATUS_ABORTED)
	{
		snprintf(PGR_Result,PGR_MESSAGE_BUFSIZE,"%d", PGR_NOTICE_ABORT_FUNC_NO);
		status = PGRreturn_result(dest, PGR_NOWAIT_ANSWER);
		status = STATUS_ABORTED;
		pthread_exit((void *) status);
	}
	/* delete server table when query use template db */
	delete_template(host_ptr,header);

	pthread_exit((void *) 0);
}

/*--------------------------------------------------
 * SYMBOL
 *     PGRreplicate_packet_send_each_server()
 * NOTES
 *     Send query to a cluster DB server and return result.
 * ARGS 
 *     HostTbl * ptr : cluster server info table (I)
 *     bool return_response : flag for return result(I)
 *     ReplicateHeader * header: header data (I)
 *     char * query : query data (I)
 *     int dest : socket of destination server(I)
 * RETURN
 *     OK : STATUS_OK
 *     NG : STATUS_ERROR
 *---------------------------------------------------
 */
int
PGRreplicate_packet_send_each_server( HostTbl * ptr, bool return_response, ReplicateHeader * header, char * query,int dest)
{
	char * func = "PGRreplicate_packet_send_each_server()";
	char * host;
	int rtn;

	host = ptr->hostName;
	/*
	 * send query to cluster DB
	 */
	if (PGR_Result == NULL)
	{
		show_error("%s:PGR_Result is not initialize",func);
		return STATUS_ERROR;
	}

	rtn = PGRsend_replicate_packet_to_server( ptr, header,query,PGR_Result, dest);

	return rtn;
}

/*--------------------------------------------------
 * SYMBOL
 *     PGRread_packet()
 * NOTES
 *     Read packet data and send the query to each cluster DB.
 *     The packet data has header data and query data.
 * ARGS 
 *     int sock : socket (I)
 *     ReplicateHeader *header : header data (O)
 * RETURN
 *     OK: pointer of read query
 *     NG: NULL
 *---------------------------------------------------
 */
char *
PGRread_packet(int sock, ReplicateHeader *header)
{
	char * func = "PGRread_packet()";
	int r =0;
	int cnt = 0;
	char * read_ptr = NULL;
	int read_size = 0;
	int header_size = 0;
	char * query = NULL;

	if (header == NULL)
	{
		return NULL;
	}
	memset(header,0,sizeof(ReplicateHeader));
	read_ptr = (char*)header;
	header_size = sizeof(ReplicateHeader);
	cnt = 0;
	for (;;){
		/*
		 * read header data
		 */
		r = recv(sock,read_ptr + read_size ,header_size - read_size, MSG_WAITALL);
		/*
		r = recv(sock,read_ptr + read_size ,header_size - read_size, 0);
		*/
		if (r < 0)
		{
			if (errno == EINTR)
			{
				usleep(PGR_RECV_WAIT_MSEC);
			}
#ifdef EPIPE
			if (errno == EPIPE)
			{
				show_error("%s:recv failed: (%s)",func,strerror(errno));
				return NULL;
			}
#endif /* EPIPE */
#ifdef EHOSTUNREACH
			if (errno == EHOSTUNREACH)
			{
				show_error("%s:recv failed: (%s)",func,strerror(errno));
				return NULL;
			}
#endif /* EHOSTUNREACH */
#ifdef ENOTSOCK
			if (errno == ENOTSOCK)
			{
				show_error("%s:recv failed: (%s)",func,strerror(errno));
				return NULL;
			}
#endif /* ENOTSOCK */
			if (cnt < PGR_RECV_RETRY_CNT )
			{
				cnt ++;
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
			else
			{
				show_error("%s:recv failed: (%s)",func,strerror(errno));
				return NULL;
			}
		}
		if (r > 0)
		{
			read_size += r;
			if ( read_size == header_size)
			{
				query = PGRread_query(sock,header);
				return query;
			}
		}
		/* if ((r == 0) && (read_size == 0)) */
		if (r == 0)
		{
			return NULL;
		}
		if (cnt < PGR_RECV_RETRY_CNT )
		{
			cnt ++;
			usleep(PGR_RECV_WAIT_MSEC);
			continue;
		}
		else
		{
			return NULL;
		}
	}
}

char *
PGRread_query(int sock, ReplicateHeader *header)
{
	char * func = "PGRread_query()";
	int r =0;
	int cnt = 0;
	char * read_ptr;
	int read_size = 0;
	int query_size = 0;
	char * query = NULL;

	query_size = ntohl(header->query_size);
	if (query_size < 0)
	{
		show_error("%s:receive size less than 0",func);
		return NULL;
	}
	query = malloc(query_size+4);
	if (query == NULL)
	{
		/*
		 * buffer allocation failed
		 */
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		return NULL;
	}
	memset(query,0,query_size+4);
	if (query_size == 0)
	{
		return query;
	}
	read_size = 0;
	cnt = 0;
	read_ptr = (char *)query;
	for (;;){
		/*
		 * read query data
		 */

		/*r = recv(sock,read_ptr + read_size ,query_size - read_size, MSG_WAITALL); */
		r = recv(sock,read_ptr + read_size ,query_size - read_size, 0); 
		if (r < 0)
		{
			if (errno == EINTR)
			{
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
#ifdef EPIPE
			if (errno == EPIPE)
			{
				free(query);
				query = NULL;
				return NULL;
			}
#endif /* EPIPE */
#ifdef EHOSTUNREACH
			if (errno == EHOSTUNREACH)
			{
				free(query);
				query = NULL;
				return NULL;
			}
#endif /* EHOSTUNREACH */
#ifdef ENOTSOCK
			if (errno == ENOTSOCK)
			{
				free(query);
				query = NULL;
				return NULL;
			}
#endif /* ENOTSOCK */
			if (cnt < PGR_RECV_RETRY_CNT )
			{
				cnt ++;
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
			else
			{
				free(query);
				query = NULL;
				return NULL;
			}
		}
		if (r > 0)
		{
			read_size += r;
			if ( read_size == query_size)
			{
				return query;
			}
		}
		if ((r == 0 ) && (read_size == 0))
		{
			free(query);
			query = NULL;
			return NULL;
		}
		if (cnt < PGR_RECV_RETRY_CNT )
		{
			cnt ++;
			usleep(PGR_RECV_WAIT_MSEC);
			continue;
		}
		else
		{
			free(query);
			query = NULL;
			return NULL;
		}
	}
	free(query);
	query = NULL;
	return NULL;
}

static bool
is_autocommit_off(char * query)
{
	int i;
	char buf[PGR_MESSAGE_BUFSIZE];
	char * p = NULL;

	if (query == NULL)
	{
		return false;
	}
	memset(buf,0,sizeof(buf));
	p = query;
	i = 0;
	while ( *p != '\0' )
	{
		buf[i++] = toupper(*p);
		p++;
		if (i >= (sizeof(buf) -2))
			break;
	}
	p = strstr(buf,"AUTOCOMMIT");
	if ( p == NULL)
	{
		return false;
	}
	p = strstr(buf,"OFF");
	if ( p == NULL )
	{
		return false;
	}
	return true;
}

static bool
is_autocommit_on(char * query)
{
	int i;
	char buf[PGR_MESSAGE_BUFSIZE];
	char * p = NULL;

	if (query == NULL)
	{
		return false;
	}
	memset(buf,0,sizeof(buf));
	p = query;
	i = 0;
	while ( *p != '\0' )
	{
		buf[i++] = toupper(*p);
		p++;
		if (i >= (sizeof(buf) -2))
			break;
	}
	p = strstr(buf,"AUTOCOMMIT");
	if ( p == NULL)
	{
		return false;
	}
	p = strstr(buf,"ON");
	if ( p == NULL )
	{
		return false;
	}
	return true;
}

static unsigned int 
get_host_ip_from_tbl(char * host)
{
	TransactionTbl * ptr = NULL;

	if (Transaction_Tbl_Begin == (TransactionTbl *) NULL)
	{
		return 0;
	}
	ptr = Transaction_Tbl_Begin;
	while (ptr != NULL)
	{
		if (!strncmp(ptr->host,host,sizeof(ptr->host)))
		{
			return ptr->hostIP;
		}
		ptr = (TransactionTbl*)ptr->next;
	}
	return 0;
}

static unsigned int 
get_srcHost_ip_from_tbl(char * srcHost)
{
	TransactionTbl * ptr = NULL;

	if (Transaction_Tbl_Begin == (TransactionTbl *) NULL)
	{
		return 0;
	}
	ptr = Transaction_Tbl_Begin;
	while (ptr != NULL)
	{
		if (!strncmp(ptr->srcHost,srcHost,sizeof(ptr->srcHost)))
		{
			return ptr->srcHostIP;
		}
		ptr = (TransactionTbl*)ptr->next;
	}
	return 0;
}

unsigned int
PGRget_next_query_id(void)
{
	if (PGR_Query_ID >= PGR_MAX_QUERY_ID)
	{
		PGR_Query_ID = 0;
	}
	PGR_Query_ID ++;
	return PGR_Query_ID;
}

/*
 * sync oid during cluster DB's 
 */
int
PGRsync_oid(ReplicateHeader *header)
{
	HostTbl * host_ptr = (HostTbl*)NULL;
	uint32_t max_oid = 0;
	uint32_t oid = 0;

	/* get current oid of all cluster db's */
	host_ptr = Host_Tbl_Begin;
	if (host_ptr == (HostTbl *)NULL)
	{
		return STATUS_ERROR;
	}
	while(host_ptr->useFlag != DATA_END)
	{
		/*
		 * check the status of the cluster DB
		 */
		if (host_ptr->useFlag != DATA_USE)
		{
			host_ptr ++;
			continue;
		}
		/*
		 * skip loop during recover and the host name is master DB
		 */
		if (is_master_in_recovery(host_ptr->hostName, host_ptr->port) == true)
		{
			host_ptr ++;
			continue;
		}
		oid = get_oid(host_ptr,header);
		if (max_oid < oid )
		{
			max_oid = oid;
		}
		host_ptr ++;
	}
	if (max_oid <= 0)
		return STATUS_ERROR;
	
	/* set oid in cluster db */
	host_ptr = Host_Tbl_Begin;
	while(host_ptr->useFlag != DATA_END)
	{
		/*
		 * check the status of the cluster DB
		 */
		if (host_ptr->useFlag != DATA_USE)
		{
			host_ptr ++;
			continue;
		}
		/*
		 * skip loop during recover and the host name is master DB
		 */
		if (is_master_in_recovery(host_ptr->hostName, host_ptr->port) == true)
		{
			host_ptr ++;
			continue;
		}
		set_oid(host_ptr,header,max_oid);
		host_ptr ++;
	}

	return STATUS_OK;
}

static int 
send_func(HostTbl * host_ptr,ReplicateHeader * header, char * func,char * result)
{
	char * f ="send_func()";
	char	   *database = NULL;
	char	   port[8];
	char	   *userName = NULL;
	char * host = NULL;
	char * str = NULL;
	TransactionTbl * transaction_tbl = (TransactionTbl *)NULL;
	PGresult * res = (PGresult *)NULL;
	PGconn * conn = (PGconn *)NULL;
	int rtn = 0;
	int current_cluster = 0;

	if ((host_ptr == (HostTbl *)NULL)		||
		(header == (ReplicateHeader *)NULL)	||
		(func == NULL)						||
		(result == NULL))
	{
		return STATUS_ERROR;
	}
	/*
	 * set up the connection
	 */
	database = (char *)header->dbName;
	snprintf(port,sizeof(port),"%d", host_ptr->port);
	userName = (char *)header->userName;
	host = host_ptr->hostName;
	if (PGR_Response_Inf != NULL)
	{
		current_cluster = PGR_Response_Inf->current_cluster;
	}

	/*
	 * get the transaction table data
	 * it has the connection data with each cluster DB
	 */
	transaction_tbl = getTransactionTbl(host_ptr,header);
	/*
	 * if the transaction process is new one, 
	 * create connection data and add the transaction table
	 */
	if (transaction_tbl == (TransactionTbl *)NULL)
	{
		transaction_tbl = setTransactionTbl(host_ptr, header);
		if (transaction_tbl == (TransactionTbl *)NULL)
		{
			show_error("%s:setTransactionTbl failed",f);
			if ( header->cmdSts != CMD_STS_NOTICE )
			{
				PGRset_host_status(host_ptr,DATA_ERR);
			}
			return STATUS_ERROR;
		}
		StartReplication[current_cluster] = true;
	}
	else
	{
		/*
		 * re-use the connection data
		 */
		if ((transaction_tbl->conn != (PGconn *)NULL) &&
			(transaction_tbl->conn->sock > 0))
		{
			StartReplication[current_cluster] = false;
		}
		else
		{
			PQfinish(transaction_tbl->conn);
		 	transaction_tbl->conn = pgr_createConn(host,port,database,userName);
			StartReplication[current_cluster] = true;
		}
	}
	conn = transaction_tbl->conn;

	if (conn == NULL)
	{
		show_error("%s:[%d@%s] may be down",f,host_ptr->port,host_ptr->hostName);
		if ( header->cmdSts != CMD_STS_NOTICE )
		{
			PGRset_host_status(host_ptr,DATA_ERR);
		}
		return STATUS_ERROR;
	}
	res = PQexec(conn, func);
	if (res == NULL)
	{
		StartReplication[current_cluster] = true;
		return STATUS_ERROR;
	}
	str = PQcmdStatus(res);
	if ((str == NULL) || (*str == '\0'))
	{
		rtn = STATUS_ERROR;
	}
	else
	{
		snprintf(result, PGR_MESSAGE_BUFSIZE, "%s",str);
		rtn = STATUS_OK;
	}
	if (res != NULL)
		PQclear(res);
	return rtn;	
}

static uint32_t
get_oid(HostTbl * host_ptr,ReplicateHeader * header)
{
	char * func = "get_oid()";
	char sync_command[PGR_MESSAGE_BUFSIZE];
	char result[PGR_MESSAGE_BUFSIZE];

	memset(result,0,sizeof(result));
	snprintf(sync_command,sizeof(sync_command),
		"SELECT %s(%d)",
		PGR_SYSTEM_COMMAND_FUNC, PGR_GET_OID_FUNC_NO);
#ifdef PRINT_DEBUG
	show_debug("%s:sync_command(%s)",func,sync_command);
#endif			

	if (send_func(host_ptr, header, sync_command, result) == STATUS_OK)
	{
		return (strtoul(result, NULL, 10));
	}
	return 0;
}

static int
set_oid(HostTbl * host_ptr,ReplicateHeader * header, uint32_t oid)
{
	char * func = "set_oid()";
	char sync_command[PGR_MESSAGE_BUFSIZE];
	char result[PGR_MESSAGE_BUFSIZE];

	memset(result,0,sizeof(result));
	snprintf(sync_command,sizeof(sync_command),
		"SELECT %s(%d,%u)",
		PGR_SYSTEM_COMMAND_FUNC, 
		PGR_SET_OID_FUNC_NO,
		oid);
#ifdef PRINT_DEBUG
	show_debug("%s:sync_command(%s)",func,sync_command);
#endif			
	return ( send_func(host_ptr, header, sync_command, result) );
}

void
PGRnotice_replication_server(char * hostName, unsigned short portNumber,unsigned short recoveryPortNumber, unsigned short lifecheckPortNumber, char * userName)
{
	char * func ="PGRnotice_replication_server()";
	ReplicateHeader  header;
	char query[PGR_MESSAGE_BUFSIZE];

	if (((hostName == NULL) || (*hostName == 0)) ||
		((userName == NULL) || (*userName == 0)) ||
		((portNumber == 0) || (recoveryPortNumber == 0)))
	{
#ifdef PRINT_DEBUG
		show_debug("%s: can not connect server[%s]",func,hostName);
#endif			
		return;
	}
	memset(&header,0,sizeof(ReplicateHeader));
	memset(query,0,sizeof(query));
	snprintf(query,sizeof(query)-1,"SELECT %s(%d,'%s',%d,%d,%d)",
			PGR_SYSTEM_COMMAND_FUNC,
			PGR_STARTUP_REPLICATION_SERVER_FUNC_NO,
			hostName,
			portNumber,
			recoveryPortNumber,
			lifecheckPortNumber);
	header.cmdSys = CMD_SYS_CALL;
	header.cmdSts = CMD_STS_NOTICE;
	header.query_size = htonl(strlen(query));
	header.query_id = htonl(PGRget_next_query_id());
	strncpy(header.from_host,hostName,sizeof(header.from_host));
	strncpy(header.userName,userName,sizeof(header.userName));
	strcpy(header.dbName,"template1");
	PGRreplicate_packet_send( &header, query, NOTICE_SYSTEM_CALL_TYPE );
}

static bool
is_need_use_rlog(ReplicateHeader * header)
{
	bool rtn = false;

	if ((Cascade_Inf->useFlag != DATA_USE) ||
		(PGR_Use_Replication_Log != true)  ||
		(header->rlog > 0))
	{
		return false;
	}
	if ((header->cmdSts == CMD_STS_QUERY ) &&
		((header->cmdType == CMD_TYPE_INSERT) || 
		 (header->cmdType == CMD_TYPE_UPDATE) || 
		 (header->cmdType == CMD_TYPE_DELETE) || 
		 (header->cmdType == CMD_TYPE_EXECUTE)))
	{
		rtn = true;	
	}
	else 
	{
		if ((header->cmdSts == CMD_STS_TRANSACTION ) &&
			(header->cmdType == CMD_TYPE_COMMIT))
		{
			rtn = true;
		}
	}
	return rtn;
}

/*
 * set db_name & user_name
 */
static int
set_pgconn_queue(ReplicateHeader * header)
{
	char * func = "set_pgconn_queue()";
	MsgData * msg_data = NULL;
	int size = 0;
	int rtn = 0;

	if ((PGconnMsgid < 0) || (header == NULL))
	{
		return STATUS_ERROR;
	}
	/*
	 * set db_login data
	 */
	size = sizeof(ReplicateHeader) + sizeof(MsgData);
	msg_data = (MsgData *) malloc(size+sizeof(long));
	if (msg_data == NULL)
	{
		show_error("%s:malloc() failed. reason: %s", func, strerror(errno));
		return STATUS_ERROR;
	}
	memset(msg_data,0,size+sizeof(long));
	msg_data->mtype = 1;
	memcpy(msg_data->mdata,header,sizeof(ReplicateHeader));

	/*
	 * send login data to queue
	 */
	rtn = msgsnd(PGconnMsgid, msg_data, sizeof(ReplicateHeader), IPC_NOWAIT);

	/*
	 * release memory
	 */
	free(msg_data);

	return STATUS_OK;	
}

/*
 * get db_name & user_name
 */
int
PGRget_pgconn_queue(void)
{
	char * func ="PGRget_pgconn_queue()";
	MsgData * msg_data = NULL;
	ReplicateHeader * header = NULL;
	HostTbl * host_ptr = (HostTbl*)NULL;
	TransactionTbl * ptr = NULL;
	int size = 0;

	if ((PGconnMsgid <= 0) || (Host_Tbl_Begin == NULL))
	{
		return STATUS_ERROR;
	}

	size = sizeof(MsgData) + sizeof(ReplicateHeader) ;
	msg_data = (MsgData*)malloc(size+sizeof(long));
	if (msg_data == (MsgData*)NULL)
	{
		show_error("%s:malloc() failed. (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(msg_data,0,size+sizeof(long));

	while (msgrcv( PGconnMsgid , msg_data, sizeof(ReplicateHeader) , 0, IPC_NOWAIT) != -1 )
	{
		header = (ReplicateHeader *)(msg_data->mdata);
		host_ptr = Host_Tbl_Begin;
		/* create new db login */
		while(host_ptr->useFlag != DATA_END)
		{
			/*
			 * check the status of the cluster DB
			 */
			if ((host_ptr->useFlag != DATA_USE) &&
				(host_ptr->useFlag != DATA_NEW))
			{
				host_ptr ++;
				continue;
			}
			if (host_ptr->useFlag == DATA_NEW)
			{
				deletePGconnTbl(host_ptr,header);
			}
			ptr = getPGconnTbl(host_ptr,header);
			if (ptr == NULL)
			{
				setPGconnTbl(host_ptr, header, DATA_USE);
				PGRset_host_status(host_ptr,DATA_USE);
			}
			host_ptr++;
		}
	}
	if (msg_data != NULL)
	{
		free(msg_data);
	}
	return STATUS_OK;
}
#endif /* USE_REPLICATION */
