/*
 * $Id: XY2LatLong.java,v 1.2 2002/03/17 10:52:42 nie Exp $
 */

package jp.jasminesoft.gcat.scalc;

/** 
 * 平面直角座標値から緯度経度値を求めるクラスです。
 *
 * @version $Revision: 1.2 $ $Date: 2002/03/17 10:52:42 $
 * @author  Yoshinori Nie
 */

public class XY2LatLong {
    /** 平面直角座標系の X 座標値 */
    private double x;

    /** 平面直角座標系の Y 座標値 */
    private double y;

    /** 平面直角座標系の系番号 */
    private int kei;

    /** 原点の緯度（ラジアン）*/
    private double gentenB;

    /** 原点の経度（度）*/
    private double gentenL;

    /** 求める緯度 */
    private Dms lat;

    /** 求める経度 */
    private Dms lng;

    /**
     * コンストラクタ
     *
     * @param kei 平面直角座標系の系番号
     */
    public XY2LatLong(int kei) {
	Coordinate coord = new Coordinate(kei);
	gentenB = coord.radianGB();
	gentenL = coord.angleGL();
    }

    /**
     * 平面直角座標系の系番号を返す。
     *
     * @return 系番号
     */
    public int getKei() { return kei; }

    /**
     * 平面直角座標系の X 値をセットする。
     *
     * @param x 平面直角座標系の X 値
     */
    public void setX(double x) {
	reset();
	this.x = x;
    }

    /**
     * 平面直角座標系の Y 値をセットする。
     *
     * @param y 平面直角座標系の Y 値
     */
    public void setY(double y) {
	reset();
	this.y = y;
    }

    /**
     * 現在、セットされている平面直角座標系の X 値を得る。
     *
     * @return 平面直角座標系の X 値
     */
    public double getX() { return x; }

    /**
     * 現在、セットされている平面直角座標系の Y 値を得る。
     *
     * @return 平面直角座標系の Y 値
     */
    public double getY() { return y; }

    /**
     * 緯度値を得る。
     *
     * @return 緯度
     */
    public double getLatitude() {
	if (lat == null)
	    calc();
	return lat.getValue();
    }

    /**
     * 緯度値を得る。
     *
     * @return 緯度(度分秒表記)
     */
    public String getDMSLatitude() {
	if (lat == null)
	    calc();
	return lat.getDMSValue();
    }

    /**
     * 経度値を得る。
     *
     * @return 経度
     */
    public double getLongitude() {
	if (lng == null)
	    calc();
	return lng.getValue();
    }

    /**
     * 経度値を得る。
     *
     * @return 経度(度分秒表記)
     */
    public String getDMSLongitude() {
	if (lat == null)
	    calc();
	return lng.getDMSValue();
    }

    /**
     * 算出結果のリセット
     */
    public void reset() {
	lng = null;
	lat = null;
    }

    /**
     * 座標変換 (平面直角座標 -> 緯度経度)
     */
    private void calc() {
        // Phiオブジェクト作成
	Phi p = new Phi(gentenB, x);
	double phi = p.getPhi();

        // LatLong オブジェクト作成
	LatLong llcnv = new LatLong(y, phi, gentenL);

	lat = new Dms(llcnv.getB());
	lng = new Dms(llcnv.getL());
    }

    /**
     * テストルーチン
     * ここで出力された値と、
     * http://vldb.gsi.go.jp/sokuchi/surveycalc/xy2blf.html
     * の結果を比較し、同一であれば大丈夫とみなした。
     */
    public static void main(String[] argv) {
	/*
	 * 平面直角座標 -> 緯度経度変換テスト
	 */
	int x = 0;
	int y = 0;
	if (argv.length >= 2) {
	    x = Integer.parseInt(argv[0]);
	    y = Integer.parseInt(argv[1]);
	}
	System.out.println("X:"+x+", Y:"+y);
	for (int i=1; i<=19; i++) {
	    XY2LatLong conv = new XY2LatLong(i);
	    conv.setX(x);
	    conv.setY(y); 
	    System.out.println("Kei:"+i+",lat:"+conv.getDMSLatitude()+"("+conv.getLatitude()+"),lng:"+conv.getDMSLongitude()+"("+conv.getLongitude()+")");
	}
    }
}
