// IMKit-Anthy: A Qtopia InputMethod interface for Anthy
// Copyright (C) 2002,2003  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_3 $
// $Id: engine_anthy.cpp,v 1.16 2004/03/16 15:46:33 yamaken Exp $

#include <ctype.h>
#include <string.h>
#include <qtextcodec.h>
#include <qpe/config.h>
#include "platform_qpe.h"
#include "engine_anthy.h"


#ifdef ANTHY_INPUT_MAP_INVALID
#error "ANTHY_INPUT_MAP_INVALID is predefined"
#else
#define ANTHY_INPUT_MAP_INVALID -0x10000
#endif

static BidirMap<int, InputMap>::pair_t n2a_inputmap_def[] = {
  {ANTHY_INPUT_MAP_ALPHABET,  IMKIT_INPUT_MAP_ALPHA},
  {ANTHY_INPUT_MAP_WALPHABET, IMKIT_INPUT_MAP_WALPHA},
  {ANTHY_INPUT_MAP_HIRAGANA,  IMKIT_INPUT_MAP_HIRAGANA},
  {ANTHY_INPUT_MAP_KATAKANA,  IMKIT_INPUT_MAP_KATAKANA}
};

static BidirMap<int, PreeditState>::pair_t n2a_preedit_state_def[] = {
  {ANTHY_INPUT_ST_CONV, IMKIT_PREEDIT_ST_CONV},
  {ANTHY_INPUT_ST_CSEG, IMKIT_PREEDIT_ST_CSEG},
  {ANTHY_INPUT_ST_EDIT, IMKIT_PREEDIT_ST_EDIT},
  {ANTHY_INPUT_ST_NONE, IMKIT_PREEDIT_ST_NONE}
};

BidirMap<int, InputMap>
AnthyEngine::n2a_inputmap(n2a_inputmap_def,
                          (sizeof(n2a_inputmap_def)
                           / BidirMap<int, InputMap>::pair_size),
                          ANTHY_INPUT_MAP_INVALID, IMKIT_INPUT_MAP_TERM);

BidirMap<int, PreeditState>
AnthyEngine::n2a_preedit_state(n2a_preedit_state_def,
                               (sizeof(n2a_preedit_state_def)
                                / BidirMap<int, PreeditState>::pair_size),
                               ANTHY_INPUT_ST_NONE, IMKIT_PREEDIT_ST_NONE);


//AnthySegment::AnthySegment(AnthyEngine *engine_init = 0,
//                           Segments::size_type index_init = 0)
//  : engine(engine_init), index(index_init)
//{
//}

AnthySegment::AnthySegment(struct anthy_input_segment *native_segment_init = 0)
  : native_segment(NULL)
{
  sync(native_segment_init);
}

AnthySegment::~AnthySegment(void) {
}

const QString &
AnthySegment::str(void) {
  return unicode_str;
}

const char *
AnthySegment::raw_str(void) const {
  const char *euc_str;

  euc_str = (native_segment) ? native_segment->str : NULL;

  return euc_str;
}

AnthySegment::Type
AnthySegment::type(void) const {
  int flag;
  
  flag = native_segment->flag;
  if (flag & ANTHY_INPUT_SF_CURSOR) {
    return ((raw_str()) ? IMKIT_SEG_FOCUS : IMKIT_SEG_CURSOR);
  } else if (flag & ANTHY_INPUT_SF_EDITTING) {
    return IMKIT_SEG_EDITING;
  } else if (flag & ANTHY_INPUT_SF_PENDING) {
    return IMKIT_SEG_PENDING;
  } else {
    return IMKIT_SEG_NORMAL;
  }
}

void
AnthySegment::sync(void) {
  const char *euc_str;
  const QTextCodec *codec;

  euc_str = (raw_str()) ? raw_str() : "";
  codec = AnthyEngine::codec();
  unicode_str = codec->toUnicode(euc_str);
}

void
AnthySegment::sync(struct anthy_input_segment *new_native_segment) {
  native_segment = new_native_segment;
  sync();
}

AnthyCandidate::
AnthyCandidate(struct anthy_input_segment *native_candidate_init = 0)
{
  sync(native_candidate_init);
}

SeparatorSegment::SeparatorSegment(void) : empty_str("") {
}

SeparatorSegment::~SeparatorSegment(void) {
}

const QString &
SeparatorSegment::str(void) {
  return empty_str;
}

SeparatorSegment::Type
SeparatorSegment::type(void) const {
  return IMKIT_SEG_SEPARATOR;
}

AnthyCandidate::~AnthyCandidate(void) {
}

const QString &
AnthyCandidate::str(void) {
  return unicode_str;
}

void
AnthyCandidate::sync(struct anthy_input_segment *new_native_candidate) {
  const char *euc_str;
  
  if (new_native_candidate && new_native_candidate->str) {
    euc_str = new_native_candidate->str;
  } else {
    euc_str = "";
  }
  unicode_str = AnthyEngine::codec()->toUnicode(euc_str);
}

const QTextCodec *
AnthyEngine::codec(void) {
  return QTextCodec::codecForName("eucJP");
}

AnthyEngine::AnthyEngine(void)
  : ai_conf(NULL), ai_ctx(NULL),
    preedit(NULL), segments(NULL), candidates(NULL),
    native_map_state(ANTHY_INPUT_MAP_HIRAGANA),
    prev_candidates_activity(false),
    _command_map(NULL), show_separator(true)
{
  Config conf("IMKit");
  int err;

  conf.setGroup("IMKit-Anthy");
  if (conf.hasKey("ShowSegmentSeparator")) {
    show_separator = conf.readEntry("ShowSegmentSeparator").toInt();
  }
  err = anthy_input_init();
  ai_conf = anthy_input_create_config();
  IMKitGlobal::ref_resource("anthy");
  IMKitGlobal::ref_resource("anthyinput");
  init_state();
  sync();
}

AnthyEngine::~AnthyEngine(void) {
  free_state();
  if (!IMKitGlobal::unref_resource("anthyinput") && ai_conf) {
    anthy_input_free_config(ai_conf);
    ai_conf = NULL;
    //anthy_conf_free();
  }
  if (!IMKitGlobal::unref_resource("anthy")) {
    anthy_quit();
  }
  imkit_delete_command_map(_command_map);
}

void
AnthyEngine::reset(void) {
  free_state();
  init_state();
  sync();
}

void
AnthyEngine::init_state(void) {
  ai_ctx = anthy_input_create_context(ai_conf);
  native_map_state = ANTHY_INPUT_MAP_HIRAGANA;
  init_segments();
  init_candidates();
}

void
AnthyEngine::free_state(void) {
  if (preedit) {
    anthy_input_free_preedit(preedit);
    preedit = NULL;
  }
  if (ai_ctx) {
    anthy_input_free_context(ai_ctx);
    ai_ctx = NULL;
  }
  imkit_delete_ref_container<Segments>(&segments);
  imkit_delete_ref_container<Candidates>(&candidates);
}

void
AnthyEngine::init_segments(void) {
  imkit_init_ref_container<Segments>(&segments);
}

void
AnthyEngine::init_candidates(void) {
  imkit_init_ref_container<Candidates>(&candidates);
}

const char *
AnthyEngine::name(void) const {
  return "Anthy";
}

const char *
AnthyEngine::language(void) const {
  return "ja";
}

InputMap
AnthyEngine::map_state(void) const {
  return n2a_inputmap.ordinary_map.lookup(native_map_state);
}

CommandMap *
AnthyEngine::command_map(void) {
  if (!_command_map) {
    _command_map = new CommandMap;
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, shrink_segment);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, enlarge_segment);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, prev_candidate);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, next_candidate);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, move_forward);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, move_back);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, beginning_of_line);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, end_of_line);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, cut);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, erase_prev);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, erase_next);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, commit);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, quit_one_level);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, select_map_alpha);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, select_map_walpha);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, select_map_hiragana);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, select_map_katakana);
    IMKIT_REGISTER_CMD0(*_command_map, AnthyEngine, this, input_space);
    IMKIT_REGISTER_CMD1(*_command_map, AnthyEngine, this, self_input_char);
  }

  return _command_map;
}

void
AnthyEngine::input_str(const QString &str) {
  //TODO: anthy_input_str()
  //TODO: Ϥդ
  sync();
}

void
AnthyEngine::input_char(QChar chr) {
  char ascii;

  //TODO: Ϥդ
  ascii = chr.latin1();
  if (isgraph(ascii)) {
    anthy_input_key(ai_ctx, ascii);
  }
  sync();
}

void
AnthyEngine::select_map(InputMap abstract_new_map) {
  int _native_map_state;
  int err;

  _native_map_state = n2a_inputmap.inverse_map.lookup(abstract_new_map);
  if (_native_map_state == ANTHY_INPUT_MAP_INVALID) return;

  native_map_state = _native_map_state;
  err = anthy_input_map_select(ai_ctx, native_map_state);
  emit map_changed(abstract_new_map);
}

void
AnthyEngine::select_candidate(Candidates::size_type which) {
  int err;

  err = anthy_input_select_candidate(ai_ctx, which);
  sync();
}

PreeditState
AnthyEngine::preedit_state(void) const {
  if (preedit) {
    return n2a_preedit_state.ordinary_map.lookup(preedit->state);
  } else {
    return IMKIT_PREEDIT_ST_NONE;
  }
}

void
AnthyEngine::sync(void) {
  if (preedit) {
    anthy_input_free_preedit(preedit);
  }
  preedit = anthy_input_get_preedit(ai_ctx);
  sync_segments();

  emit preedit_state_changed(preedit_state());
  emit map_changed(map_state());
  emit segments_changed(*segments);

  activate_candidates();

  if (preedit->commit) {
    committed_str = codec()->toUnicode(preedit->commit);
    emit committed(committed_str);
  }
}

void
AnthyEngine::sync_segments(void) {
  struct anthy_input_segment *segment;

  init_segments();
  
  for (segment = preedit->segment; segment; segment = segment->next) {
    segments->push_back(new AnthySegment(segment));
    if (show_separator && is_converting()) {
      segments->push_back(new SeparatorSegment);
    }
  }

  // remove trailing separator
  if (show_separator && is_converting() && 1 < segments->size()) {
    delete (*segments->rbegin());
    segments->pop_back();
  }
}

void
AnthyEngine::sync_candidates(void) {
  struct anthy_input_segment *current_segment, *candidate;
  int i, orig_index;

  init_candidates();

  if (!is_converting()) {
    return;
  }

  current_segment = preedit->cur_segment;
  orig_index = current_segment->cand_no;
  for (i = 0; i < current_segment->nr_cand; i++) {
    candidate = anthy_input_get_candidate(ai_ctx, i);
    candidates->push_back(new AnthyCandidate(candidate));
    anthy_input_free_segment(candidate);
  }

  //cand_noθ᤹
  candidate = anthy_input_get_candidate(ai_ctx, orig_index);
  anthy_input_free_segment(candidate);
}

void
AnthyEngine::activate_candidates(void) {
  bool activity;

  if (is_converting()) {
    struct anthy_input_segment *current_segment;
    int selected;
    bool reversely, has_activated;

    reversely = false;
    current_segment = preedit->cur_segment;
    selected = current_segment->cand_no;
    if (current_segment->flag & ANTHY_INPUT_SF_ENUM) {
      activity = true;
    } else if (current_segment->flag & ANTHY_INPUT_SF_ENUM_REVERSE) {
      activity = true;
      reversely = true;
    } else {
      activity = false;
    }
    
    has_activated = (!prev_candidates_activity && activity);
    if (has_activated) {
      sync_candidates();
      emit update_candidates(*candidates);
    }

    emit hilight_candidate(selected, reversely);
  } else {
    activity = false;
  }

  emit activation_hint_for_candidates(activity);
  prev_candidates_activity = activity;
}

bool
AnthyEngine::is_converting(void) {
  return (preedit->cur_segment && preedit->state == ANTHY_INPUT_ST_CONV);
}
