
require 'wx'
include Wx

# IDs for the controls and the menu commands
# menu items
SERVER_QUIT = 1000
SERVER_ABOUT = 1001

# id for sockets
SERVER_ID = 1002
SOCKET_ID = 1003

MAX_MSG_SIZE = 10000

class MyFrame < WxFrame
    def initialize
        super(nil, -1,"WxSocket demo: Server",
                             WxDefaultPosition, WxSize.new(300, 200))

        # Give the frame an icon
        SetIcon(WxIcon.new("mondrian.xpm"))

        # Make menus
        @m_menuFile = WxMenu.new
        @m_menuFile.Append(SERVER_ABOUT, "&About...\tCtrl-A", "Show about dialog")
        @m_menuFile.AppendSeparator()
        @m_menuFile.Append(SERVER_QUIT, "E&xit\tAlt-X", "Quit server")

        # Append menus to the menubar
        @m_menuBar = WxMenuBar.new
        @m_menuBar.Append(@m_menuFile, "&File")
        SetMenuBar(@m_menuBar)

        # Status bar
        CreateStatusBar(2)

        # Make a textctrl for logging
        @m_text  = WxTextCtrl.new(self, -1,
                               "Welcome to WxSocket demo: Server\n",
                               WxDefaultPosition, WxDefaultSize,
                               WxTE_MULTILINE | WxTE_READONLY)

        # Create the socket
        @m_server = WxSocketServer.new("localhost",3000)

        # We use Ok() here to see if the server is really listening
        if !@m_server.Ok()
            @m_text.AppendText("Could not listen at the specified port !\n\n")
            return nil
        else
            @m_text.AppendText("Server listening.\n\n")
        end

        # Setup the event handler and subscribe to connection events
        @m_server.SetEventHandler(self, SERVER_ID)
        @m_server.SetNotify(WxSOCKET_CONNECTION_FLAG)
        @m_server.Notify(TRUE)

        @m_busy = FALSE
        @m_numClients = 0
        UpdateStatusBar()

        EVT_MENU(self,SERVER_QUIT,  "OnQuit")
        EVT_MENU(self,SERVER_ABOUT, "OnAbout")
        EVT_SOCKET(self,SERVER_ID,  "OnServerEvent")
        EVT_SOCKET(self,SOCKET_ID,  "OnSocketEvent")

    end

    def OnQuit(event)
      # TRUE is to force the frame to close
      Close(TRUE)
    end

    def OnAbout(event)
      WxMessageBox("WxSocket demo: Server\n(c) 1999 Guillermo Rodriguez Garcia\n",
                   "About Server",
                   WxOK | WxICON_INFORMATION, self)
    end

    def Test1(sock)
      @m_text.AppendText("Test 1 begins\n")

      # Receive data from socket and send it back. We will first
      # get a byte with the buffer size, so we can specify the
      # exact size and use the WxSOCKET_WAITALL flag. Also, we
      # disabled input events so we won't have unwanted reentrance.
      # This way we can avoid the infamous WxSOCKET_BLOCK flag.

      sock.SetFlags(WxSOCKET_WAITALL)

      # Read the size
      len = "\0"
      sock.Read(len, 1)
      len = len.unpack("C")[0]
      buf = "\0" * len

      # Read the data
      sock.Read(buf, len)
      @m_text.AppendText("Got the data, sending it back\n")

      # Write it back
      sock.Write(buf, len)

      @m_text.AppendText("Test 1 ends\n\n")
    end

    def Test2(sock)

      buf = "\0" * MAX_MSG_SIZE

      @m_text.AppendText("Test 2 begins\n")

      # We don't need to set flags because ReadMsg and WriteMsg
      # are not affected by them anyway.

      # Read the message
      len = sock.ReadMsg(buf, MAX_MSG_SIZE).LastCount()
      s = sprintf("Client says: %s\n", buf)
      @m_text.AppendText(s)
      @m_text.AppendText("Sending the data back\n")

      # Write it back
      sock.WriteMsg(buf, len)

      @m_text.AppendText("Test 2 ends\n\n")

    end

    def Test3(sock)
      @m_text.AppendText("Test 3 begins\n")

      # This test is similar to the first one, but the len is
      # expressed in kbytes - self tests large data transfers.

      sock.SetFlags(WxSOCKET_WAITALL)

      # Read the size
      len = "\0"
      sock.Read(len, 1)
      len = len.unpack("C")[0]
      buf = "\0" * (len * 1024)

      # Read the data
      sock.Read(buf, len * 1024)
      @m_text.AppendText("Got the data, sending it back\n")

      # Write it back
      sock.Write(buf, len * 1024)

      @m_text.AppendText("Test 3 ends\n\n")
    end

    def OnServerEvent(event)

      s = "OnServerEvent: "

      case event.GetSocketEvent()
        when WxSOCKET_CONNECTION
            s << "WxSOCKET_CONNECTION\n"
        else
            s << "Unexpected event !\n"
      end

      @m_text.AppendText(s)

      # Accept new connection if there is one in the pending
      # connections queue, else exit. We use Accept(FALSE) for
      # non-blocking accept (although if we got here, there
      # should ALWAYS be a pending connection).

      sock = @m_server.Accept(FALSE)

      if sock
        @m_text.AppendText("New client connection accepted\n\n")
      else
        @m_text.AppendText("Error: couldn't accept a new connection\n\n")
        return nil
      end

      sock.SetEventHandler(self, SOCKET_ID)
      sock.SetNotify(WxSOCKET_INPUT_FLAG | WxSOCKET_LOST_FLAG)
      sock.Notify(TRUE)

      @m_numClients += 1
      UpdateStatusBar()
    end

    def OnSocketEvent(event)

      s = "OnSocketEvent: "
      sock = event.GetSocket()

      # First, print a message
      case event.GetSocketEvent()
        when WxSOCKET_INPUT
            s << "WxSOCKET_INPUT\n"
        when WxSOCKET_LOST
            s << "WxSOCKET_LOST\n"
        else
            s << "Unexpected event !\n"
      end

      @m_text.AppendText(s)

      # Now we process the event
      case event.GetSocketEvent()
        when WxSOCKET_INPUT
          # We disable input events, so that the test doesn't trigger
          # WxSocketEvent again.
          sock.SetNotify(WxSOCKET_LOST_FLAG)

          # Which test are we going to run?
          c = "\0"
          sock.Read(c, 1)

          case c[0]
            when 0xBE
                Test1(sock)
            when 0xCE
                Test2(sock)
            when 0xDE
                Test3(sock)
            else
                @m_text.AppendText("Unknown test id received from client\n\n")
          end

          # Enable input events again.
          sock.SetNotify(WxSOCKET_LOST_FLAG | WxSOCKET_INPUT_FLAG)

        when WxSOCKET_LOST

          @m_numClients -= 1

          # Destroy() should be used instead of delete wherever possible,
          # due to the fact that WxSocket uses 'delayed events' (see the
          # documentation for WxPostEvent) and we don't want an event to
          # arrive to the event handler (the frame, here) after the socket
          # has been deleted. Also, we might be doing some other thing with
          # the socket at the same time for example, we might be in the
          # middle of a test or something. Destroy() takes care of all
          # self for us.

          @m_text.AppendText("Deleting socket.\n\n")
          sock.Destroy()
      end

      UpdateStatusBar()
    end

    # convenience functions
    def UpdateStatusBar()
      s = sprintf("%d clients connected", @m_numClients)
      SetStatusText(s, 1)
    end

end

class MyApp < WxApp
    def OnInit
        # Create the main application window
        frame = MyFrame.new

        # Show it and tell the application that it's our main window
        frame.Show(TRUE)
        SetTopWindow(frame)        
    end
end

a = MyApp.new
a.MainLoop()

