
require 'wx'
include Wx

MENU_FILE_OPEN = 100
MENU_FILE_SAVE = 101
MENU_FILE_QUIT = 102
MENU_EDIT_GOTO = 103
MENU_OPTION_BACKGROUND = 104
MENU_OPTION_FONT = 105
MENU_OPTION_DIRECTORY = 106
MENU_INFO_ABOUT = 107


class RangeValidator < WxValidator

  def initialize(value,min=nil,max=nil)
    super()
    if(min==nil && max==nil)
      Copy(value)
    else
      @m_min = min
      @m_max = max
      @m_value = value
    end
  end

  def m_min
    @m_min
  end
  def m_max
    @m_max
  end
  def m_value
    @m_value
  end

  def Clone
    return RangeValidator.new(self)
  end

  def Copy(val)
    @m_min   = val.m_min
    @m_max   = val.m_max
    @m_value = val.m_value
    return true
  end

  def Validate(parent)
    win = GetWindow()
    return false if !win
    control = win
    result = false

    currentValue = control.GetValue

    if !currentValue.ToLong(value)
      message = sprintf(message,"%s is not a valid number", currentValue.c_str())
    else
      if value < m_min || value > m_max
          message = sprintf(message,"%ld is not between %ld and %ld", value, m_min, m_max)
      else
          result = true
      end
    end

    if !result
      WxMessageBox(message)
      control.SetFocus
    end

    return result
  end

  def TransferToWindow

    win = GetWindow()
    return false if !win
    return false if !@m_value
    return false if !win.IsKindOf(CLASSINFO(WxTextCtrl))

    control = win
    control.SetValue(@m_value)

    return true
  end

  def TransferFromWindow
    win = GetWindow()
    return false if !win
    return false if !@m_value

    return false if !win.IsKindOf(CLASSINFO(WxTextCtrl))

    control = win
    m_value = control.GetValue

    return true
  end
end

class GotoLineDialog < WxDialog
  def initialize(parent)
    super(parent, -1, "Goto Line...", WxDefaultPosition,
               WxSize.new(250, 120), WxDEFAULT_DIALOG_STYLE)
    @m_lineNumber = "1"
    SetAutoLayout(TRUE)

    layout = WxLayoutConstraints.new

    # Label is centered on the dialog and on top of the dialog.
    @m_pLabel = WxStaticText.new(self, -1, "")
    layout.top.SameAs(self, WxTop, 10)
    layout.centreX.SameAs(self, WxCentreX)
    layout.width.AsIs
    layout.height.AsIs
    @m_pLabel.SetConstraints(layout)

    # LineNumberCtrl is centered on the dialog and below Label.
    @m_pLineNumberCtrl = WxTextCtrl.new(self, -1)
    layout = WxLayoutConstraints.new
    layout.top.Below(@m_pLabel, 10)
    layout.centreX.SameAs(self, WxCentreX)
    layout.width.Absolute(70)
    layout.height.AsIs
    @m_pLineNumberCtrl.SetConstraints(layout)

    # cancelButton is put at the right and bottom of the dialog.
    cancelButton = WxButton.new(self, WxID_CANCEL, "Cancel")
    layout = WxLayoutConstraints.new
    layout.bottom.SameAs(self, WxBottom, 10)
    layout.right.SameAs(self, WxRight, 10)
    layout.width.AsIs
    layout.height.AsIs
    cancelButton.SetConstraints(layout)

    # okButton is put at the left of the cancelButton.
    okButton = WxButton.new(self, WxID_OK, "Ok")
    layout = WxLayoutConstraints.new
    layout.bottom.SameAs(cancelButton, WxBottom)
    layout.right.LeftOf(cancelButton, 10)
    layout.width.SameAs(cancelButton, WxWidth)
    layout.height.AsIs
    okButton.SetConstraints(layout)

    EVT_INIT_DIALOG(self,"OnInitDialog")
  end

  def OnInitDialog(event)
    label = @m_lineNumber
    label = sprintf(label,"Enter a line between 1 and %ld", @m_maxLineNumber)
    @m_pLabel.SetLabel(label)
    Layout()
    rangeValidator = RangeValidator .new(@m_lineNumber, 1, @m_maxLineNumber)
    @m_pLineNumberCtrl.SetValidator(rangeValidator)
    TransferDataToWindow()
    @m_pLineNumberCtrl.SetFocus
  end

  def SetMaxLineNumber(maxLineNumber)
    @m_maxLineNumber = maxLineNumber
  end

  def GetLineNumber
    # Return 0 when toLong fails.
    return @m_lineNumber.to_i
  end
end

class AboutDialog < WxDialog
  def initialize(parent)
    super(parent, -1, "About Simple Text Editor", WxDefaultPosition,
               WxSize.new(200, 200), WxDEFAULT_DIALOG_STYLE)
    SetAutoLayout(TRUE)

    layout = WxLayoutConstraints.new

    layout.top.SameAs(self, WxTop, 10)
    layout.centreX.SameAs(self, WxCentreX)
    layout.width.AsIs
    layout.height.AsIs
    @m_pInfoText = WxStaticText.new(self, -1, "", WxPoint.new(-1, -1), WxDefaultSize, WxALIGN_CENTER)
    @m_pInfoText.SetConstraints(layout)

    layout = WxLayoutConstraints.new
    layout.top.Below(@m_pInfoText, 10)
    layout.centreX.SameAs(self, WxCentreX)
    layout.width.PercentOf(self, WxWidth, 80)
    layout.height.AsIs

    @m_pOkButton = WxButton.new(self, WxID_OK, "Ok", WxPoint.new(-1, -1))
    @m_pOkButton.SetConstraints(layout)

    Layout()
  end

  def SetText(text)
    @m_pInfoText.SetLabel(text)
    # Call layout, because the static text could be resized.
    Layout()
  end
end

class TextFrame < WxFrame
  def initialize(title, xpos, ypos, width, height)
    super(nil,-1, title, WxPoint.new(xpos, ypos), WxSize.new(width, height))
    @m_pTextCtrl = WxTextCtrl.new(self, -1, "Type some text...",
	WxDefaultPosition, WxDefaultSize, WxTE_MULTILINE)

    @m_pMenuBar = WxMenuBar.new

    # File Menu
    @m_pFileMenu = WxMenu.new
    @m_pFileMenu.Append(MENU_FILE_OPEN, "&Open", "Opens an existing file")
    @m_pFileMenu.Append(MENU_FILE_SAVE, "&Save", "Save the content")
    @m_pFileMenu.AppendSeparator
    @m_pFileMenu.Append(MENU_FILE_QUIT, "&Quit", "Quit the application")
    @m_pMenuBar.Append(@m_pFileMenu, "&File")

    # Edit Menu
    @m_pEditMenu = WxMenu.new
    @m_pEditMenu.Append(MENU_EDIT_GOTO, "&Goto Line", "Jump to a line")
    @m_pMenuBar.Append(@m_pEditMenu, "&Edit")

    # Option Menu
    @m_pOptionMenu = WxMenu.new
    @m_pOptionMenu.Append(MENU_OPTION_BACKGROUND, "&Background", "Sets the background colour of the editor")
    @m_pOptionMenu.Append(MENU_OPTION_FONT, "&Font", "Sets the font of the editor")
    @m_pOptionMenu.Append(MENU_OPTION_DIRECTORY, "&Directory", "Sets the working directory")
    @m_pMenuBar.Append(@m_pOptionMenu, "&Options")

    # About menu
    @m_pInfoMenu = WxMenu.new
    @m_pInfoMenu.Append(MENU_INFO_ABOUT, "&About", "Shows information about the application")
    @m_pMenuBar.Append(@m_pInfoMenu, "&Info")

    SetMenuBar(@m_pMenuBar)

    # Statusbar
    CreateStatusBar(3)
    SetStatusText("Ready", 0)

    EVT_MENU(self,MENU_FILE_OPEN, "OnMenuFileOpen")
    EVT_MENU(self,MENU_FILE_SAVE, "OnMenuFileSave")
    EVT_MENU(self,MENU_FILE_QUIT, "OnMenuFileQuit")
    EVT_MENU(self,MENU_EDIT_GOTO, "OnMenuEditGoto")
    EVT_MENU(self,MENU_INFO_ABOUT, "OnMenuInfoAbout")
    EVT_MENU(self,MENU_OPTION_BACKGROUND, "OnMenuOptionBackgroundColor")
    EVT_MENU(self,MENU_OPTION_FONT, "OnMenuOptionFont")
    EVT_MENU(self,MENU_OPTION_DIRECTORY, "OnMenuOptionDirectory")
    EVT_CLOSE(self,"OnClose")
  end

  def OnMenuFileOpen(event)
    dlg = WxFileDialog.new(self, "Open a text file",
		"", "", "All files(*.*)|*.*|Text Files(*.txt)|*.txt",
		WxOPEN, WxDefaultPosition)
    if dlg.ShowModal == WxID_OK
  	@m_pTextCtrl.LoadFile(dlg.GetFilename)
	SetStatusText(dlg.GetFilename, 0)
    end
    dlg.Destroy
  end

  def OnMenuFileOpen(event)
    dlg = WxFileDialog.new(self, "Open a text file",
		"", "", "All files(*.*)|*.*|Text Files(*.txt)|*.txt",
		WxOPEN, WxDefaultPosition)
    if dlg.ShowModal == WxID_OK
	@m_pTextCtrl.LoadFile(dlg.GetFilename)
	SetStatusText(dlg.GetFilename, 0)
    end
    dlg.Destroy()
  end

  def OnMenuFileSave(event)
    dlg = WxFileDialog.new(self, "Save a text file",
		"", "", "All files(*.*)|*.*|Text Files(*.txt)|*.txt",
		WxSAVE, WxDefaultPosition)
    if dlg.ShowModal == WxID_OK
	@m_pTextCtrl.SaveFile(dlg.GetPath)
	SetStatusText(dlg.GetFilename, 0)
    end
    dlg.Destroy
    @m_pTextCtrl.DiscardEdits
  end

  def OnMenuFileQuit(event)
    Close(FALSE)
  end

  def OnMenuInfoAbout(event)
    dlg = AboutDialog.new(self)
    dlg.SetText("(c) 2001 S.A.W. Franky Braem\nSimple Text Editor\n")
    dlg.ShowModal
    dlg.Destroy
  end

  def OnMenuEditGoto(event)
    dlg = GotoLineDialog.new(self)
    dlg.SetMaxLineNumber(@m_pTextCtrl.GetNumberOfLines)
    if dlg.ShowModal == WxID_OK
	# Convert the linenumber to a position and set self as the new insertion point.
	p dlg.GetLineNumber
	@m_pTextCtrl.SetInsertionPoint(@m_pTextCtrl.XYToPosition(0, dlg.GetLineNumber - 1))
    end
    dlg.Destroy
  end

  def OnMenuOptionBackgroundColor(event)
    colourData = WxColourData.new
    colour = @m_pTextCtrl.GetBackgroundColour
    colourData.SetColour(colour)
    colourData.SetChooseFull(true)
    dlg = WxColourDialog.new(self, colourData)
    if dlg.ShowModal == WxID_OK
	colourData = dlg.GetColourData
	@m_pTextCtrl.SetBackgroundColour(colourData.GetColour)
	@m_pTextCtrl.Refresh
    end
    dlg.Destroy
  end

  def OnMenuOptionFont(event)

    fontData = WxFontData.new
    font = @m_pTextCtrl.GetFont
    fontData.SetInitialFont(font)
    colour = @m_pTextCtrl.GetForegroundColour
    fontData.SetColour(colour)
    fontData.SetShowHelp(true)

    dlg = WxFontDialog.new(self, fontData)
    if dlg.ShowModal == WxID_OK
	fontData = dlg.GetFontData
	font = fontData.GetChosenFont
	@m_pTextCtrl.SetFont(font)
	@m_pTextCtrl.SetForegroundColour(fontData.GetColour)
	@m_pTextCtrl.Refresh
    end
    dlg.Destroy
  end

  def OnMenuOptionDirectory(event)
    dlg = WxDirDialog.new(self, "Select a Working directory", WxGetCwd())
    if dlg.ShowModal == WxID_OK
	WxSetWorkingDirectory(dlg.GetPath)
    end
    dlg.Destroy
  end

  def OnClose(event)

    destroy = true
    if event.CanVeto
	if @m_pTextCtrl.IsModified
	  dlg = WxMessageDialog.new(self, "Text is changed!\nAre you sure you want to exit?",
		"Text changed!!!", WxYES_NO | WxNO_DEFAULT)
	  result = dlg.ShowModal
	  if result == WxID_NO
		event.Veto
		destroy = false
	  end
	end
    end
    if destroy
	Destroy()
    end
  end
end


class WxRbApp < WxApp
  def OnInit
    frame = TextFrame.new("Simple Text Editor", 100, 100, 400, 300)
    frame.Show(TRUE)
    SetTopWindow(frame)
  end

end

a = WxRbApp.new
a.MainLoop

