
require 'wx'
include Wx

# menu items
Caret_Quit = 1
Caret_About = 2
Caret_SetBlinkTime = 3
Caret_Move = 4

# controls start here (the numbers are, of course, arbitrary)
Caret_Text = 1000


# MyCanvas is a canvas on which you can type
class MyCanvas < WxScrolledWindow

    def initialize(parent)
        super(parent, -1,
                WxDefaultPosition, WxDefaultSize,
                WxSUNKEN_BORDER )

        SetBackgroundColour(WxWHITE)

        @m_font = WxFont.new(12, WxFONTFAMILY_TELETYPE,
                        WxFONTSTYLE_NORMAL, WxFONTWEIGHT_NORMAL)

        @m_xCaret = @m_yCaret =
        @m_xChars = @m_yChars = 0

        @m_xMargin = @m_yMargin = 5
        @m_text = nil

        CreateCaret()

        EVT_PAINT(self,"OnPaint")
        EVT_SIZE(self,"OnSize")
        EVT_CHAR(self,"OnChar")
    end

    def [](x,y)
        @m_text[x + @m_xChars * y,1]
    end

    def []=(x,y,z)
        @m_text[x + @m_xChars * y,1] = z
    end

    # caret movement
    def Home()
        @m_xCaret = 0
    end

    def End()
        @m_xCaret = @m_xChars - 1
    end

    def FirstLine()
        @m_yCaret = 0
    end

    def LastLine()
        @m_yCaret = @m_yChars - 1
    end

    def PrevChar()
        if @m_xCaret == 0
          End()
          PrevLine()
        else
          @m_xCaret -= 1
        end
    end

    def NextChar()
        @m_xCaret += 1
        if @m_xCaret == @m_xChars
            Home()
            NextLine()
        end
    end

    def PrevLine()
        if @m_yCaret == 0
            LastLine()
        else
            @m_yCaret -= 1
        end
    end

    def NextLine()
        @m_yCaret += 1
        if @m_yCaret == @m_yChars
            FirstLine()
        end
    end

    def CreateCaret()
        dc = WxClientDC.new(self)
        dc.SetFont(@m_font)
        @m_heightChar = dc.GetCharHeight()
        @m_widthChar = dc.GetCharWidth()

        caret = WxCaret.new(self, @m_widthChar, @m_heightChar)
        SetCaret(caret)

        caret.Move(@m_xMargin, @m_yMargin)
        caret.Show()
    end

    def MoveCaret(x,y)
        @m_xCaret = x
        @m_yCaret = y

        DoMoveCaret()
    end

    def DoMoveCaret()
        WxLogStatus("Caret is at (%d, %d)", @m_xCaret, @m_yCaret)

        GetCaret().Move(@m_xMargin + @m_xCaret * @m_widthChar,
                         @m_yMargin + @m_yCaret * @m_heightChar)
    end

    def OnSize(event)
        @m_xChars = (event.GetSize().x - 2*@m_xMargin) / @m_widthChar
        @m_yChars = (event.GetSize().y - 2*@m_yMargin) / @m_heightChar
        if @m_xChars == 0
            @m_xChars = 1
        end
        if @m_yChars == 0
            @m_yChars = 1
        end

        @m_text = " " * @m_xChars * @m_yChars

        frame = WxFrame(GetParent())
        if frame && frame.GetStatusBar()
            msg = sprintf("Panel size is (%d, %d)", @m_xChars, @m_yChars)
            frame.SetStatusText(msg, 1)
        end
        event.Skip()
    end

    def OnPaint(event)
        m_caret = GetCaret()
        if m_caret
            m_caret.Hide()
        end

        dc = WxPaintDC.new(self)
        PrepareDC(dc)
        dc.Clear()

        dc.SetFont(@m_font)

        for y in 0 ... @m_yChars
            line = @m_text[@m_xChars * y,@m_xChars]
            dc.DrawText( line, @m_xMargin, @m_yMargin + y * @m_heightChar )
        end

        dc.free

        if m_caret
            m_caret.Show()
        end

    end

    def OnChar(event)
        case event.KeyCode()
            when WXK_LEFT
                PrevChar()
            when WXK_RIGHT
                NextChar()
            when WXK_UP
                PrevLine()
            when WXK_DOWN
                NextLine()
            when WXK_HOME
                Home()
            when WXK_END
                End()
            when WXK_RETURN
                Home()
                NextLine()
            else
                if !event.AltDown()
                    ch = event.KeyCode()
                    self[@m_xCaret, @m_yCaret] = ch.chr

                    m_caret = GetCaret()
                    if m_caret
                        m_caret.Hide()
                    end

                    dc = WxClientDC.new(self)
                    dc.SetFont(@m_font)
                    dc.SetBackgroundMode(WxSOLID) # overwrite old value
                    dc.DrawText(ch.chr, @m_xMargin + @m_xCaret * @m_widthChar,
                                    @m_yMargin + @m_yCaret * @m_heightChar )
                    dc.free

                    NextChar()
                    if m_caret
                        m_caret.Show()
                    end
                else
                    event.Skip()
                end
        end
        DoMoveCaret()
    end

end

class MyFrame < WxFrame
    def initialize(title,pos,size)
        super(nil, -1, title, pos, size)
        # set the frame icon
        SetIcon(WxIcon.new("mondrian.xpm",WxBITMAP_TYPE_XPM))

        # create a menu bar
        menuFile = WxMenu.new

        menuFile.Append(Caret_SetBlinkTime, "&Blink time...\tCtrl-B")
        menuFile.Append(Caret_Move, "&Move caret\tCtrl-C")
        menuFile.AppendSeparator()
        menuFile.Append(Caret_About, "&About...\tCtrl-A", "Show about dialog")
        menuFile.AppendSeparator()
        menuFile.Append(Caret_Quit, "E&xit\tAlt-X", "Quit self program")

        # now append the freshly created menu to the menu bar...
        menuBar = WxMenuBar.new
        menuBar.Append(menuFile, "&File")

        # ... and attach self menu bar to the frame
        SetMenuBar(menuBar)

        @m_canvas = MyCanvas.new(self)

        # create a status bar just for fun (by default with 1 pane only)
        CreateStatusBar(2)
        SetStatusText("Welcome to WxWindows!")

        EVT_MENU(self,Caret_Quit,  "OnQuit")
        EVT_MENU(self,Caret_About, "OnAbout")
        EVT_MENU(self,Caret_SetBlinkTime, "OnSetBlinkTime")
        EVT_MENU(self,Caret_Move, "OnCaretMove")

    end

    def OnQuit(event)
        # TRUE is to force the frame to close
        Close(TRUE)
    end

    def OnAbout(event)
        WxMessageBox("The caret WxWindows sample.\n 1999 Vadim Zeitlin",
                 "About Caret", WxOK | WxICON_INFORMATION, self)
    end

    def OnCaretMove(event)
        @m_canvas.MoveCaret(10, 10)
    end

    def OnSetBlinkTime(event)

        blinkTime = WxGetNumberFromUser(
                      "The caret blink time is the time between two blinks",
                      "Time in milliseconds:",
                      "WxCaret sample",
                      WxCaret::GetBlinkTime(), 0, 10000,
                      self)
        if blinkTime != -1
            WxCaret::SetBlinkTime(blinkTime)
            @m_canvas.CreateCaret()
            WxLogStatus(self,"Blink time set to %d milliseconds.", blinkTime)
        end
    end
end

class WxRbApp < WxApp
    def OnInit()    
  	    frame = MyFrame.new("Caret WxWindows sample", WxPoint.new(50, 50), WxSize.new(450, 340))
	    frame.Show(TRUE)
    end
end

a = WxRbApp.new
a.MainLoop()

