/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.fn.RecodeFn;
import woolpack.utils.Utils;
import woolpack.validator.ValidatorContext;
import woolpack.validator.ValidatorUtils;

public class ValidatorBuilderTest extends TestCase {

	public void testGetCheckExpressionTrue() {
		final List<String> list = new ArrayList<String>();
		final List<ValidatorContext> validatorList = new ArrayList<ValidatorContext>();
		
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(
				FnUtils.recode(FnUtils.fix(true), validatorList, null));
		
		final Fn<EEContext, Void> expression = validatorBuilder.getCheckExpression(
				new RecodeFn<EEContext, Void>(null, "true", list),
				new RecodeFn<EEContext, Void>(null, "false", list));
		
		final EEContext context = new EEContext();
		context.setId("id0");
		context.setInput(Utils.map("key0", Arrays.asList("value0", "value1")));
		expression.exec(context);

		assertEquals(Arrays.asList("true"), list);

		assertEquals("id0", validatorList.get(0).getId());
		assertEquals(
				Utils.map("key0", Arrays.asList("value0", "value1")),
				validatorList.get(0).getInputMap());
	}

	public void testGetCheckExpressionFalse() {
		final List<String> list = new ArrayList<String>();
		final List<ValidatorContext> validatorList = new ArrayList<ValidatorContext>();
		
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(
				FnUtils.recode(ValidatorUtils.message("message0"), validatorList, null));
		
		final Fn<EEContext, Void> expression = validatorBuilder.getCheckExpression(
				new RecodeFn<EEContext, Void>(null, "true", list),
				new RecodeFn<EEContext, Void>(null, "false", list));
		
		final EEContext context = new EEContext();
		context.setId("id0");
		context.setInput(Utils.map("key0", Arrays.asList("value0", "value1")));
		expression.exec(context);

		assertEquals(Arrays.asList("false"), list);

		assertEquals("id0", validatorList.get(0).getId());
		assertEquals(
				Utils.map("key0", Arrays.asList("value0", "value1")),
				validatorList.get(0).getInputMap());

		assertEquals("id0", context.getId());
		assertEquals(Arrays.asList("message0"),
				ValidatorBuilder.DEFALT_MESSAGES_EL.getValue(context));
	}

	public void testGetReplaceExpressionTrue() {
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(
				FnUtils.fix(true), ValidatorBuilder.DEFALT_MESSAGES_EL, true);
		final Fn<EEContext, Void> expression = validatorBuilder.getReplaceExpression();
		final EEContext context = new EEContext();
		context.setInput(new HashMap());
		context.setConfig(Utils.map("messageKey0", (Object) "messageValue0"));
		{
			final List<?> list = Utils.list("messageKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.exec(context);
			assertEquals(Arrays.asList("messageValue0"), list);
		}
		{
			final List<?> list = Utils.list("messageKey1");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.exec(context);
			assertEquals(Arrays.asList("messageKey1"), list);
		}
	}

	public void testGetMessageExpressionFalse() {
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(
				FnUtils.fix(true));
		final Fn<EEContext, Void> expression = validatorBuilder.getReplaceExpression();
		final EEContext context = new EEContext();
		context.setConfig(Utils.map("messageKey0", (Object) "messageValue0"));
		{
			final List<?> list = Utils.list("messageKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.exec(context);
			assertEquals(Arrays.asList("messageValue0"), list);
		}
		{
			final List<?> list = Utils.list("messageKey0").list("messageKey1");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			try {
				expression.exec(context);
				fail();
			} catch (final IllegalStateException e) {
				assertEquals("not found.", e.getMessage());
			}
			assertEquals(2, list.size());
			assertEquals(Arrays.asList("messageValue0", "messageKey1"), list);
		}
	}
}
