/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import woolpack.bool.And;
import woolpack.bool.BooleanOperator;
import woolpack.fn.Fn;

/**
 * プロパティ名で委譲先を分岐する{@link Fn}です。
 * プロパティ名に対応する値が存在しない場合も委譲するため、
 * ブラウザから送信されない項目を必須を検証する際に使用することができます。
 * 
 * @author nakamura
 * 
 */
public class NameBranch implements Fn<ValidatorContext, Boolean> {
	private BooleanOperator<ValidatorContext> operator;
	private Map<String, ? extends Fn<? super ValidatorContext, Boolean>> map;

	/**
	 * @param operator 値検証一覧の呼び出し方。
	 * @param map プロパティ名と委譲先の対応表。
	 */
	public NameBranch(
			final BooleanOperator<ValidatorContext> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean>> map) {
		this.operator = operator;
		this.map = map;
	}

	/**
	 * {@link And}を使用します。
	 * @param map プロパティ名と委譲先の対応表。
	 */
	public NameBranch(
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean>> map) {
		this(new And<ValidatorContext>(), map);
	}

	protected Collection<String> keySet(final ValidatorContext context) {
		return map.keySet();
	}

	public Boolean exec(final ValidatorContext context) {
		return operator.exec(context,
			new Iterable<Fn<ValidatorContext, Boolean>>() {
				public Iterator<Fn<ValidatorContext, Boolean>> iterator() {
					return new Iterator<Fn<ValidatorContext, Boolean>>() {
						private final Iterator<String> it = keySet(context).iterator();

						public boolean hasNext() {
							return it.hasNext();
						}

						public Fn<ValidatorContext, Boolean> next() {
							return new Fn<ValidatorContext, Boolean>() {
								public Boolean exec(
										final ValidatorContext context) {
									final String key = it.next();
									final Fn<? super ValidatorContext, Boolean> e = map.get(key);
									if (e != null) {
										final String baseKey = context.getKey();
										try {
											context.setKey(key);
											return e.exec(context);
										} finally {
											context.setKey(baseKey);
										}
									} else {
										return true;
									}
								}
							};
						}

						public void remove() {
							throw new UnsupportedOperationException();
						}
					};
				}
			}
		);
	}

	public Map<String, ? extends Fn<? super ValidatorContext, Boolean>> getMap() {
		return map;
	}
	public void setMap(final Map<String, ? extends Fn<? super ValidatorContext, Boolean>> map) {
		this.map = map;
	}
	public BooleanOperator<ValidatorContext> getOperator() {
		return operator;
	}
	public void setOperator(final BooleanOperator<ValidatorContext> operator) {
		this.operator = operator;
	}
}
