/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.idnode;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.bool.BoolUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.xml.NodeFindable;
import woolpack.xml.NodeFindableChildNodesImpl;
import woolpack.xml.XmlUtils;

/**
 * HTML の内部フレームを流し込む{@link Fn}です。
 * 各フレームをマージする際にターゲットの HTML HEAD タグを残します。
 * Struts の Tiles プラグインのようにレイアウトを制御するために使用します。
 * 
 * @author nakamura
 * 
 */
public class IFrameInserter<C extends IdNodeContext> implements Fn<C, Void> {
	
	private static final NodeFindable IFRAME = new NodeFindableChildNodesImpl(
	FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("IFRAME")), false);

	private static final NodeFindable BODY = new NodeFindableChildNodesImpl(
	FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("BODY")), false);

	private static final NodeFindable HEAD = new NodeFindableChildNodesImpl(
	FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("HEAD")), false);
	
	private String frameId;
	private String targetName;
	private Fn<? super C, Void> nodeMaker;
	private Fn<String, String> idConverter;

	/**
	 * @param frameId 親 HTML の id。
	 * @param targetName {@link IdNodeContext#getId()}で生成された DOM ノードを流し込む"//iframe[\@name]"の値。
	 * @param nodeMaker ノードを作成する委譲先。
	 * @param idConverter "//iframe[[\@src]"のidに変換するための変換器。
	 * 
	 */
	public IFrameInserter(
			final String frameId,
			final String targetName,
			final Fn<? super C, Void> nodeMaker,
			final Fn<String, String> idConverter) {
		this.frameId = frameId;
		this.targetName = targetName;
		this.nodeMaker = nodeMaker;
		this.idConverter = idConverter;
	}
	
	private Node getNode(final String id, final C c) {
		final String baseId = c.getId();
		final Node baseNode = c.getNode();
		try {
			c.setId(id);
			nodeMaker.exec(c);
			return  c.getNode();
		} finally {
			c.setId(baseId);
			c.setNode(baseNode);
		}
	}
	
	private static void importNode(final Node node0, final Node node1) {
		node0.getParentNode().insertBefore(
				XmlUtils.getDocumentNode(node0).importNode(node1, true),
				node0);
	}

	public Void exec(final C c) {
		final Node node0 = getNode(frameId, c);
		final NodeList node0IframeList = IFRAME.evaluateList(node0);
		for(int i = 0; i < node0IframeList.getLength(); i++) {
			final Element node0Iframe = (Element) node0IframeList.item(i);
			final boolean targetFlag = targetName.equals(node0Iframe.getAttribute("name"));
			final Node node1 = getNode(
					targetFlag ? c.getId() : idConverter.exec(node0Iframe.getAttribute("src")),		
					c);
			{
				final Node node1Body = BODY.evaluateOne(node1);
				Node tmp1 = node1Body.getFirstChild();
				while (tmp1 != null) {
					importNode(node0Iframe, tmp1);
					tmp1 = tmp1.getNextSibling();
				}
				XmlUtils.removeThis(node0Iframe);
			}
			if(targetFlag) {
				final Node node0Head = HEAD.evaluateOne(node0);
				importNode(node0Head, HEAD.evaluateOne(node1));
				XmlUtils.removeThis(node0Head);
			}
		}
		c.setNode(node0);
		return null;
	}

	public String getFrameId() {
		return frameId;
	}
	public void setFrameId(final String frameId) {
		this.frameId = frameId;
	}
	public Fn<String, String> getIdConverter() {
		return idConverter;
	}
	public void setIdConverter(final Fn<String, String> idConverter) {
		this.idConverter = idConverter;
	}
	public Fn<? super C, Void> getNodeMaker() {
		return nodeMaker;
	}
	public void setNodeMaker(final Fn<? super C, Void> nodeMaker) {
		this.nodeMaker = nodeMaker;
	}
	public String getTargetName() {
		return targetName;
	}
	public void setTargetName(final String targetName) {
		this.targetName = targetName;
	}
}
