#include "FileUtils.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <stdlib.h>
#include "dirchar.h"

#if !defined(WIN32)
#include <dirent.h>
#include <unistd.h>
#include <sys/errno.h>
#include <arpa/inet.h>
#include <sys/param.h>
#else
#include "boost/shared_array.hpp"
#endif//WIN32

// FileUtilsLite includes only what is needed by the prefPane
// so it was split into a seperate file
#include "FileUtilsLite.cpp"

#if !defined(_PREFPANE_)

#if defined(_CARBON_)
//#include <TextCommon.h>
static bool gEncodingInited = false;
static TextToUnicodeInfo gShiftJISToUTF8ConversionInfo;
static UnicodeToTextInfo gUTF8ToShiftJISConversionInfo;
static TextToUnicodeInfo gMacRomanToUTF8ConversionInfo;
static UnicodeToTextInfo gUTF8ToMacRomanConversionInfo;
#endif // _CARBON_

#include "HLServer.h"
#include "HLProtocol.h"
#include <algorithm>

#if defined(_SENDFILE_) && !defined(_FREEBSD_)
    #include <sys/sendfile.h>
#else
    #define COPY_BUFSIZE		0xF000
#endif

static void addExtensionFromType(string &ioFilePath, u_int32_t inType);
static bool pathExists(const string &inPath, struct stat *outStat);

static FileTypeExtensionMap gFileTypeExtMap;
static FileTypeCreatorMap gFileTypeCreatorMap;


//-----------------------------------------------------------------------//
////2003/07/21 added by ortana. replace all strings of target.
//-----------------------------------------------------------------------//
template<typename E, typename T, typename A>
std::basic_string<E,T,A>
FileUtils::replace_all_target(
  const std::basic_string<E,T,A>& source,   // change
  const std::basic_string<E,T,A>& pattern,  // pattern
  const std::basic_string<E,T,A>& placement // placement
  ) {
  std::basic_string<E,T,A> result(source);
  for ( std::string::size_type pos = 0 ;
        std::string::npos != (pos = result.find(pattern,pos));
        pos += placement.size() )
    result.replace(pos, pattern.size(), placement);
  return result;
}

#if defined(WIN32)

	//-----------------------------------------------------------------------//
	//2003/07/22 added by ortana.
	//-----------------------------------------------------------------------//
	void setNullChar( char * buff )
	{
		char *temp = (char*)memchr( buff , '\0' , MAX_PATH );
	  temp++;
		*temp = '\0';
	}

	//bool FileUtils::copyFile(const string fromPath , const string toPath)
	//{
	//	char from_Path[MAX_PATH];
	//	char to_Path[MAX_PATH];

	//	//getAbsolutPath( fromPath , from_Path );
	//	getAbsolutPath( toPath	 , to_Path );

	//	strcpy( from_Path , fromPath.c_str() );
	//	setNullChar( from_Path );
	//	setNullChar( to_Path );

	//SHFILEOPSTRUCT fileOp = {
	//	NULL, // hwnd
	//	FO_COPY, // wFunc
	//	from_Path, // pFrom
	//	to_Path, // pTo
	//	FOF_NOCONFIRMATION | FOF_NOERRORUI | FOF_SILENT, // fFlags
	//	FALSE, // fAnyOperationsAborted 
	//	NULL, // hNameMappings,
	//	NULL //lpszProgressTitle
	//};

	//return SHFileOperation( &fileOp ) == 0;
	//}


	//-----------------------------------------------------------------------//
	////2003/07/22 added by orta.
	//-----------------------------------------------------------------------//
	void FileUtils::getAbsolutPath(const string &inPath, char * buff )
	{
		if( !PathIsRelative( inPath.c_str() ) )
		{
			strcpy( buff , inPath.c_str() );
			return;
		}

		GetCurrentDirectory(MAX_PATH,buff);

		string path( inPath );
		string pattern("/");
		string placement("\\");
		path = replace_all_target( path , pattern , placement );

		sprintf( buff , "%s\\%s" , buff , path.c_str());
	}


		//-----------------------------------------------------------------------//
	////2003/07/22 added by orta.
	//-----------------------------------------------------------------------//
bool FileUtils::deletePath(const string &inPath)
{
	//2003/07/21 SHFileOperation need the absolute path.
	//modified by ortana.

	char from[MAX_PATH];
	
	getAbsolutPath( inPath , from );

	//file name must be terminated by a single NULL character. 
	//An additional NULL character must be appended to the end of
	//the final name to indicate the end of pFrom. 
	setNullChar( from );

	SHFILEOPSTRUCT fileOp = {
		NULL, // hwnd
		FO_DELETE, // wFunc
		from, // pFrom
		NULL, // pTo
		FOF_NOCONFIRMATION | FOF_NOERRORUI | FOF_SILENT | FOF_ALLOWUNDO, // fFlags
		FALSE, // fAnyOperationsAborted 
		NULL, // hNameMappings,
		NULL //lpszProgressTitle
	};

	return SHFileOperation( &fileOp ) == 0;
}



//-----------------------------------------------------------------------//
////2003/07/21 added by ortana. resolve *.lnk files
//-----------------------------------------------------------------------//
HRESULT FileUtils::getShortcutLink(const LPSTR pszLink, LPSTR pszFilePath, 
																	 int cchMaxPath, WIN32_FIND_DATA *wfd)
{
	HRESULT hres;
	IShellLink *psl;

	hres = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
		IID_IShellLink, (void **)&psl);
	if (SUCCEEDED(hres))
	{
		IPersistFile *ppf;


		hres = psl->QueryInterface(IID_IPersistFile, (void **)&ppf);
		if (SUCCEEDED(hres))
		{
			WORD wsz[MAX_PATH];

			MultiByteToWideChar(CP_ACP, 0, pszLink, -1, wsz, MAX_PATH);

			hres = ppf->Load(wsz, STGM_READ);

			if (SUCCEEDED(hres))
			{
				hres = psl->GetPath(pszFilePath, cchMaxPath, wfd, SLGP_UNCPRIORITY);
			}

			ppf->Release();
		}
		psl->Release();
	}
	return hres;
}

//-----------------------------------------------------------------------//
////2003/07/21 added by ortana. resolve *.lnk files. 
//-----------------------------------------------------------------------//
BOOL FileUtils::resolve_Link(LPSTR sourcePath ,const LPSTR linkPath)
{
	WIN32_FIND_DATA wfd;
	if ( SUCCEEDED(CoInitialize(NULL)) ) {
		FileUtils::getShortcutLink(
			linkPath,
			sourcePath,
			MAX_PATH,
			&wfd);
		CoUninitialize();
		return TRUE;
	}
	return FALSE;
}

//-----------------------------------------------------------------------//
//2003/07/22 SaveShellLink. added by ortana.
//-----------------------------------------------------------------------//
BOOL SaveShellLink( IShellLink *psl, const WCHAR *wcLinkName, BOOL bRemember )
{
	HRESULT       hResult;
	IPersistFile *ppf;

	if ( S_OK != psl->QueryInterface( IID_IPersistFile, (void **)&ppf ) )
		return FALSE;

	hResult = ppf->Save( wcLinkName, bRemember );
	ppf->Release();
	return S_OK == hResult;
}//SaveShellLink

//-----------------------------------------------------------------------//
// Make a Shortcut file.
//     LinkFile ccc name of shortcut file(*.lnk)
//   TargetFile ccc target of link
//    Arguments ccc Arguments of command line
//      WorkDir ccc work directory
//      ShowCmd ccc window type on start up.
//
//2003/07/22 added by ortana.
//-----------------------------------------------------------------------//
BOOL FileUtils::CreateShellLink( const char *LinkFile, const char *TargetFile, const char *Arguments, const char *WorkDir, int ShowCmd )
{
	IShellLink *psl;
	WCHAR       wcLinkFile[MAX_PATH+1];

	CoInitialize( NULL );
	if ( S_OK == CoCreateInstance( CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, IID_IShellLink, (void **)&psl ) )
	{
		psl->SetPath( TargetFile );                           
		if ( Arguments ) psl->SetArguments( Arguments );      
		if ( WorkDir   ) psl->SetWorkingDirectory( WorkDir ); 
		psl->SetShowCmd( ShowCmd );                           

		MultiByteToWideChar( CP_ACP, 0, LinkFile, -1, wcLinkFile, MAX_PATH ); 
		if ( SaveShellLink( psl, wcLinkFile, TRUE ) ) 
		{
			psl->Release();
			CoUninitialize();
			return TRUE;
		}
		psl->Release();
	}
	CoUninitialize();
	return FALSE; 
}//CreateShellLink


//-----------------------------------------------------------------------//
//added by ortana.
//-----------------------------------------------------------------------//
void longToByteArray(char * array , long l)
{
	*(array)	 =  ((l >> 24) & 0xff);
	*(++array) = ((l >> 16) & 0xff);
	*(++array) = ((l >> 8 ) & 0xff);
	*(++array) =   (l  & 0xff);
}

//-----------------------------------------------------------------------//
//added by ortana.
//-----------------------------------------------------------------------//
int GetDaycountOfMonth(int y, int m)
{
	int magic[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

	if ((y % 4 == 0) && ((y % 100 != 0) || (y % 400 == 0))) {
		/* when Leap year */
		++magic[1];
	}

	return magic[m - 1];
}


//-----------------------------------------------------------------------//
//get file time. added by ortana.
//-----------------------------------------------------------------------//
BOOL FileUtils::getFileTime(const string filePath , char *createdTime , char *modifiedTime)
{
	//HANDLE hFile;

	//hFile = CreateFile(filePath.c_str(),    // filename
	//	0,                       
	//	FILE_SHARE_READ | FILE_SHARE_WRITE,
	//	0,                       
	//	OPEN_EXISTING,           
	//	FILE_ATTRIBUTE_NORMAL,  
	//	0                       
	//	);                     



	  WIN32_FIND_DATAA ffd;

    HANDLE  hObject = FindFirstFile(filePath.c_str(), &ffd);

    if (hObject == INVALID_HANDLE_VALUE)
    {
			return FALSE;
		}
		FindClose( hObject );

	FILETIME ftCreate,ftWrite,ftAccess;
	FILETIME sftCreate , sftWrite;

//	GetFileTime(hFile,&ftCreate,&ftAccess,&ftWrite);
	ftCreate = ffd.ftCreationTime;
	ftWrite = ffd.ftLastWriteTime;

	FileTimeToLocalFileTime( &ftCreate ,  &sftCreate );
	FileTimeToLocalFileTime( &ftWrite ,		&sftWrite );

	SYSTEMTIME stCreateTime,stModifiedTime;
	FileTimeToSystemTime(&sftCreate , &stCreateTime);
	FileTimeToSystemTime(&sftWrite  , &stModifiedTime);

	createdTime[0]	= (stCreateTime.wYear >> 8) & 0xff;
	modifiedTime[0]	= (stModifiedTime.wYear >> 8) & 0xff;
	createdTime[1]	= stCreateTime.wYear & 0xff;
	modifiedTime[1] = stModifiedTime.wYear & 0xff;
	createdTime[2]	= modifiedTime[2] = createdTime[3] = modifiedTime[3] = 0;

	long crdays = 0;
	long mddays = 0;
	for(int i = 1 ; i < stCreateTime.wMonth ; i++ )
	{
		crdays += GetDaycountOfMonth( stCreateTime.wYear , i );
	}
	for(int i = 1 ; i < stModifiedTime.wMonth ; i++ )
	{
		mddays += GetDaycountOfMonth( stModifiedTime.wYear , i );
	}

	long crnSeconds = ((crdays + stCreateTime.wDay -1 ) * 24 * 60 * 60) + 
		(stCreateTime.wHour * 60 * 60 ) + 
		(stCreateTime.wMinute * 60) +
		(stCreateTime.wSecond);
	long mdnSencods = ((mddays + stModifiedTime.wDay - 1 ) * 24 * 60 * 60) + 
		(stModifiedTime.wHour * 60 * 60 ) + 
		(stModifiedTime.wMinute * 60) +
		(stModifiedTime.wSecond);
	longToByteArray( &createdTime[4] , crnSeconds );
	longToByteArray( &modifiedTime[4] ,mdnSencods );

	return TRUE;
}

//-----------------------------------------------------------------------//
//
// check that file or directory is exist.
//
// pram
//      lpszFileName
//          filePath
//          
//      bDirectory
//          TRUE ; search a directoryBFALSE ; search a file
//
// return
//      if a item is exist,return TRUE.
//     
////2003/07/22 added by ortana.
//-----------------------------------------------------------------------//
const BOOL FileUtils::IsExistObject(const LPCTSTR lpszFileName,const BOOL bDirectory)
{
    WIN32_FIND_DATAA ffd;

    HANDLE  hFindFile = FindFirstFile(lpszFileName, &ffd);

    if (hFindFile != INVALID_HANDLE_VALUE)
    {
        FindClose(hFindFile);

        if ( (bDirectory) &&
            (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) )
            return TRUE;

        if ( !(bDirectory) &&
            !(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) )
            return TRUE;
    }
    return FALSE;
}

#endif//WIN32

bool pathExists(const string &inPath, struct stat *outStat)
{
#ifdef WIN32
	string winPath(inPath);
// windows doesn't like having a trailing '/' in directory paths
	if (winPath[winPath.length() - 1] == '/')
		winPath[winPath.length() - 1] = NULL;
	if (lstat(winPath.c_str(), outStat) == 0)
#else
	if (lstat(inPath.c_str(), outStat) == 0)
#endif
	{
		if (S_ISLNK(outStat->st_mode))
		{
			// this is a link
			if (stat(inPath.c_str(), outStat) != 0)
				// error resolving link
				return false;
		}
	}
	else
	{
		DEBUG_CALL(printf("lstat error: %s for path: %s\n", strerror(errno), inPath.c_str()));
		return false;
	}
	
	return true;
}

void addExtensionFromType(string &ioFilePath, u_int32_t inType)
{
	FileTypeExtensionMap::iterator iter = gFileTypeExtMap.find(inType);
	if (iter != gFileTypeExtMap.end())
	{
		ioFilePath.append(".");
		ioFilePath.append(iter->second);
	}
	else
	{
		DEBUG_CALL(printf("no extension for type: 0x%X\n", inType));
	}
}

void FileUtils::Init()
{
#if defined(_CARBON_)
    if (!gEncodingInited)
    {
        TextEncoding utf8TextEncoding = CreateTextEncoding(kTextEncodingUnicodeDefault,
            kTextEncodingDefaultVariant/*kUnicodeNoSubset*/, kUnicodeUTF8Format);
        UnicodeMapping mappingInfo;
        mappingInfo.unicodeEncoding = utf8TextEncoding;
        mappingInfo.otherEncoding = kTextEncodingShiftJIS; /*kTextEncodingMacRoman;*/
        mappingInfo.mappingVersion = kUnicodeUseLatestMapping;
        
        OSStatus status = CreateTextToUnicodeInfo(&mappingInfo, &gShiftJISToUTF8ConversionInfo);
        if (status == noErr)
        {
            mappingInfo.unicodeEncoding = utf8TextEncoding;
            mappingInfo.otherEncoding = kTextEncodingShiftJIS;
            mappingInfo.mappingVersion = kUnicodeUseLatestMapping;
            
            status = CreateUnicodeToTextInfo(&mappingInfo, &gUTF8ToShiftJISConversionInfo);
            if (status == noErr)
            {
                mappingInfo.unicodeEncoding = utf8TextEncoding;
                mappingInfo.otherEncoding = kTextEncodingMacRoman;
                mappingInfo.mappingVersion = kUnicodeUseLatestMapping;
                
                status = CreateTextToUnicodeInfo(&mappingInfo, &gMacRomanToUTF8ConversionInfo);
                if (status == noErr)
                {
                    mappingInfo.unicodeEncoding = utf8TextEncoding;
                    mappingInfo.otherEncoding = kTextEncodingMacRoman;
                    mappingInfo.mappingVersion = kUnicodeUseLatestMapping;
                    
                    status = CreateUnicodeToTextInfo(&mappingInfo, &gUTF8ToMacRomanConversionInfo);
                    if (status == noErr)
                        gEncodingInited = true;
                    else
                    {
                        DEBUG_CALL(printf("error creating conversion info[4]: %d\n", (int)status));
                    }
                }
                else
                {
                    DEBUG_CALL(printf("error creating conversion info[3]: %d\n", (int)status));
                }
            }
            else
            {
                DEBUG_CALL(printf("error creating conversion info[2]: %d\n", (int)status));
            }
        }
        else
        {
            DEBUG_CALL(printf("error creating conversion info[1]: %d\n", (int)status));
        }
    }
#endif // _CARBON_
}

void FileUtils::fixLineEndings(string &ioString)
{
    string::size_type pos = 0;
    while ((pos = ioString.find('\n', pos)) != string::npos)
    {
        if (pos != 0 && ioString[pos - 1] == '\r')
            // DOS line endings need to remove \n from \r\n combo
            ioString.erase(pos, 1);
        else
            // unix line endings, just replace \n with \r
            ioString.replace(pos, 1, 1, '\r');
        pos++;
    }
}

void FileUtils::removeExtension(string &ioFilePath)
{
	string::size_type extPos = ioFilePath.rfind('.');
	
	if (extPos != string::npos)
	{
		ioFilePath = ioFilePath.substr(0, extPos);
	}
}

void FileUtils::createTypeMaps()
{
    // DON'T MODIFY THE ORDER OF THIS, IT IS IMPORTANT!
    // see look up code
	gFileTypeExtMap.insert(make_pair(FILE_TRANSFER_TYPE, (char *)"hpf"));
	gFileTypeExtMap.insert(make_pair(JPEG_TYPE, (char *)"jpg"));
	gFileTypeExtMap.insert(make_pair(JPEG_TYPE, (char *)"jpeg"));
	gFileTypeExtMap.insert(make_pair(MPEG_TYPE, (char *)"mpg"));
	gFileTypeExtMap.insert(make_pair(MPEG_TYPE, (char *)"mpeg"));
	gFileTypeExtMap.insert(make_pair(MOOV_TYPE, (char *)"mov"));
	gFileTypeExtMap.insert(make_pair(GIF_TYPE, (char *)"gif"));
	gFileTypeExtMap.insert(make_pair(PNG_TYPE, (char *)"png"));
	gFileTypeExtMap.insert(make_pair(PICT_TYPE, (char *)"pict"));
	gFileTypeExtMap.insert(make_pair(MP3_3_TYPE, (char *)"mp3"));
	gFileTypeExtMap.insert(make_pair(MP3_3_TYPE, (char *)"mp2"));
	gFileTypeExtMap.insert(make_pair(WAV_2_TYPE, (char *)"wav"));
	gFileTypeExtMap.insert(make_pair(WAV_1_TYPE, (char *)"wav"));
	gFileTypeExtMap.insert(make_pair(AVI_TYPE, (char *)"avi"));
	gFileTypeExtMap.insert(make_pair(SIT4_TYPE, (char *)"sit"));
	gFileTypeExtMap.insert(make_pair(SIT5_TYPE, (char *)"sit"));
	gFileTypeExtMap.insert(make_pair(GZIP_TYPE, (char *)"gz"));
	gFileTypeExtMap.insert(make_pair(GZIP_TYPE, (char *)"tgz"));
	gFileTypeExtMap.insert(make_pair(ZIP_TYPE, (char *)"zip"));
	gFileTypeExtMap.insert(make_pair(RTF_TYPE, (char *)"rtf"));
	gFileTypeExtMap.insert(make_pair(PDF_TYPE, (char *)"pdf"));
	gFileTypeExtMap.insert(make_pair(PSD_TYPE, (char *)"psd"));
	
	gFileTypeCreatorMap.insert(make_pair(FILE_TRANSFER_TYPE, HOTLINE_CREATOR)); // hotline client
	gFileTypeCreatorMap.insert(make_pair(JPEG_TYPE, QUICKTIME_CREATOR)); // quicktime
	gFileTypeCreatorMap.insert(make_pair(GIF_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(PNG_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(PICT_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(MPEG_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(MOOV_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(MP3_1_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(MP3_2_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(MP3_3_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(WAV_1_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(WAV_2_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(AVI_TYPE, QUICKTIME_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(SIT5_TYPE, STUFFIT_CREATOR)); // stuffit expander
	gFileTypeCreatorMap.insert(make_pair(SIT4_TYPE, STUFFIT_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(GZIP_TYPE, STUFFIT_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(ZIP_TYPE, STUFFIT_CREATOR));
	gFileTypeCreatorMap.insert(make_pair(RTF_TYPE, TEXTEDIT_CREATOR)); // text edit
	gFileTypeCreatorMap.insert(make_pair(PDF_TYPE, ACROBAT_CREATOR)); // adobe acrobat
	gFileTypeCreatorMap.insert(make_pair(PSD_TYPE, PHOTOSHOP_CREATOR)); // photoshop
}

bool FileUtils::PathVectorToRealPath(const string &inRootPath,
    const PathVector &inPathVector, string &outRealPath)
{
    PathVector::const_iterator iter = inPathVector.begin();
#ifndef WIN32
    char realPathBuf[MAXPATHLEN];
#endif // WIN32   
	outRealPath.assign(inRootPath);
        
    while (iter != inPathVector.end())
    {
        {
            string pathComponent = (*iter);
            FileUtils::NetworkToHomeString(pathComponent);
					
						//2003/07/21 added by ortana.
#if defined(WIN32)
						//resolve shortcut file
						if( pathComponent.find(".lnk") != string.npos )
						{
							char srcPath[MAX_PATH];
							string lnkPath( outRealPath + pathComponent );
							if( FileUtils::resolve_Link( srcPath , (LPSTR)lnkPath.data() ) )
							{
								outRealPath = srcPath;
							}
							else
							{
								//TODO : when occur err.
							}
						}
						else
						{
#endif//WIN32
            outRealPath.append(pathComponent);
#if defined(WIN32)
						}
#endif//WIN32
        }
        
    #if _CARBON_
        resolveAlias(outRealPath);
    #endif // _CARBON_
        
        struct stat sb;
        if (lstat(outRealPath.c_str(), &sb) == 0)
        {
#ifndef WIN32
            if (S_ISLNK(sb.st_mode) && stat(outRealPath.c_str(), &sb) == 0)
            {
                // devin says FSResolveAlias resolves symlinks too,
                // so calling realpath here is most likely not needed
                if (realpath(outRealPath.c_str(), realPathBuf) == NULL)
                {
                    DEBUG_CALL(printf("realpath error: %s\n", strerror(errno)); fflush(stdout));
                    outRealPath.erase();
                    return false;
                }
                outRealPath.assign(realPathBuf);
            }
#endif
            
            if (S_ISDIR(sb.st_mode) && outRealPath[outRealPath.length() - 1] != DIRCHAR)
                outRealPath += DIRCHAR;
        }
        else
        {
            outRealPath.erase();
            return false;
        }
        
        iter++;
    }
    return true;
}

bool FileUtils::movePath(const string &inFromPath, const string &inToPath, const string &inFileName)
{
	string toPath(inToPath);
    string fromPath(inFromPath);
	struct stat sb;
    
    fromPath.append(inFileName);
	if (stat(fromPath.c_str(), &sb) == 0)
	{
		dev_t diff_device = sb.st_dev;
#ifdef WIN32
// windows doesn't like a trailing '/' on directory paths
		string winToPath(toPath);
		if (winToPath[winToPath.length() - 1] == '/')
			winToPath[winToPath.length() - 1] = NULL;
		if (stat(winToPath.c_str(), &sb) == 0)
#else
		if (stat(toPath.c_str(), &sb) == 0)
#endif
		{
			diff_device = diff_device != (dev_t)sb.st_dev;
			toPath.append(inFileName);
			
			if ((diff_device ? copy_and_unlink(fromPath, toPath)
				: rename(fromPath.c_str(), toPath.c_str())) == 0)
			{
				return true;
			}
		}
	}
	return false;
}

u_int32_t FileUtils::creatorForType(const u_int32_t inType)
{
	if (inType && inType != UNKNOWN_TYPE)
	{
		FileTypeCreatorMap::iterator iter = gFileTypeCreatorMap.find(inType);
		if (iter != gFileTypeCreatorMap.end())
		{
			return iter->second;
		}
		
		DEBUG_CALL(printf("no creator for type: 0x%X\n", inType));
	}
	return UNKNOWN_CREATOR;
}

u_int32_t FileUtils::fileTypeFromName(const string &inFileName)
{
	string fileExt;
	if (getFileExtension(inFileName, fileExt))
	{
		FileTypeExtensionMap::iterator iter = gFileTypeExtMap.begin();
		while (iter != gFileTypeExtMap.end())
		{
			if (noCaseCompare(iter->second, fileExt))
			{
				return iter->first;
			}
			iter++;
		}
	}
	
	return UNKNOWN_TYPE;
}

void FileUtils::postProcessUpload(const string &inFilePath, u_int32_t inType)
{
	string filePath = inFilePath;
	string fileExt;
	
	removeExtension(filePath);
	if (!getFileExtension(filePath, fileExt))
	{
		addExtensionFromType(filePath, inType);
	}
	
	if (rename(inFilePath.c_str(), filePath.c_str()) != 0)
	{
		// error renaming file!
		DEBUG_CALL(printf("error renaming upload file: %s\n", strerror(errno)); fflush(stdout));
	}
}

bool FileUtils::checkUploadPathAndResumeSize(const string &inPath,
	const string &inName, string &outFullPath, u_int32_t &outLocalDataForkSize)
{
	struct stat sb;
	outLocalDataForkSize = 0;
	// first check out the upload dir to make sure it is a dir
	if (pathExists(inPath, &sb))
	{
		if (S_ISDIR(sb.st_mode))
		{
			outFullPath.assign(inPath);
			outFullPath.append(inName);
			// don't allow upload if the filename already exists
			if (pathExists(outFullPath, &sb))
			{
				return false;
			}
			
			outFullPath.append(".hpf");
			// upload path seems valid (uploading to a directory)
			if (pathExists(outFullPath, &sb))
			{
				if (!S_ISDIR(sb.st_mode))
				{
					outLocalDataForkSize = sb.st_size;
					return true;
				}
				else
				{
					DEBUG_CALL(printf("file path is really a dir\n"));
					return false;
				}
			}
			
			return true;
		}
	}
	else
	{
		DEBUG_CALL(printf("u/l path doesn't exist: %s\n", inPath.c_str()));
	}
	
	return false;
}

bool FileUtils::getFileExtension(const string &inPath, string &outExtension)
{
	string::size_type extPos = inPath.rfind('.');
	
	if (++extPos != string::npos)
	{
		outExtension = inPath.substr(extPos);
		return true;
	}
	else
		outExtension.erase();
	
	return false;
}

void FileUtils::formatSize(u_int64_t inBytes, string &outSizeString)
{
	char sizeBuf[64];
	
    if ((inBytes / 1048576) >= 1024)
        snprintf(sizeBuf, 63, "%.1fGb", ((inBytes / 1048576.0) / 1024.0));
	else if ((inBytes / 1048576) >= 1)
		snprintf(sizeBuf, 63, "%.1fMb", inBytes / 1048576.0);
	else if ((inBytes / 1024) >= 1)
		snprintf(sizeBuf, 63, "%.1fKb", inBytes / 1024.0);
	else if (inBytes > 1)
		snprintf(sizeBuf, 63, "%u bytes", (u_int32_t)inBytes);
    else
        snprintf(sizeBuf, 63, "%u byte", (u_int32_t)inBytes);
	
	outSizeString.assign(sizeBuf);
}

void FileUtils::formatTime(unsigned int inSeconds, string &outTimeString)
{
	char timeBuf[64];
	
	timeBuf[0] = 0;
	outTimeString.erase();
	if ((inSeconds / 86400) > 1)
	{
		snprintf(timeBuf, 63, "%u days", inSeconds / 86400);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 86400) * 86400);
	}
    else if ((inSeconds / 86400) == 1)
    {
        snprintf(timeBuf, 63, "%u day", inSeconds / 86400);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 86400) * 86400);
    }
	
	if ((inSeconds / 3600) > 1)
	{
        if (outTimeString.length())
            outTimeString.append(" ");
		snprintf(timeBuf, 63, "%u hrs", inSeconds / 3600);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 3600) * 3600);
	}
    else if ((inSeconds / 3600) == 1)
	{
        if (outTimeString.length())
            outTimeString.append(" ");
		snprintf(timeBuf, 63, "%u hr", inSeconds / 3600);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 3600) * 3600);
	}
	
	if ((inSeconds / 60) > 1)
	{
        if (outTimeString.length())
            outTimeString.append(" ");
		snprintf(timeBuf, 63, "%u mins", inSeconds / 60);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 60) * 60);
	}
    else if ((inSeconds / 60) == 1)
	{
        if (outTimeString.length())
            outTimeString.append(" ");
		snprintf(timeBuf, 63, "%u min", inSeconds / 60);
		outTimeString.append(timeBuf);
		inSeconds -= ((inSeconds / 60) * 60);
	}
	
	if (inSeconds || outTimeString.empty())
	{
        if (outTimeString.length())
            outTimeString.append(" ");
        if (inSeconds == 1)
        {
            snprintf(timeBuf, 63, "%u sec", inSeconds);
        }
        else
        {
            snprintf(timeBuf, 63, "%u secs", inSeconds);
        }
		outTimeString.append(timeBuf);
	}
}

int FileUtils::copy_and_unlink (const string &inFrom, const string &inTo)
{
	int rfd, wfd, err = 0;
	
	rfd = open(inFrom.c_str(), O_RDONLY | O_BINARY, 0);
	
	if (rfd < 0)
		return rfd;
	
	wfd = open(inTo.c_str(), O_WRONLY | O_BINARY | O_CREAT | O_TRUNC, gServer->Config().filePermissions);
	
	if (wfd < 0)
	{
		close(rfd);
		return wfd;
	}

#if defined(_SENDFILE_) && !defined(_FREEBSD_)
    
    // freebsd sendfile() requires a socket to send to
    // linux lets me send from a file descriptor to a file descriptor
    // there are probably other cases, sendfile() is NOT portable
    
    off_t startOffset = 0;
    off_t fileSize = lseek(rfd, 0, SEEK_END);
    if (fileSize != -1)
    {
        if (lseek(rfd, 0, SEEK_SET) != -1)
        {
            ssize_t sentSize = sendfile(wfd, rfd, &startOffset, (size_t)fileSize);
            if (sentSize == -1)
            {
                err = errno;
            }
            else if (sentSize != fileSize)
            {
                err = -1;
            }
        }
        else
            err = errno;
    }
    else
        err = errno;

#else
    
    int r;
    char *buf = (char *)malloc(COPY_BUFSIZE);
    
	while (1)
	{
		r = read(rfd, buf, COPY_BUFSIZE);
		if (r == 0)
		{
			err = 0;
			break;
		}
		if (r < 0)
		{
			err = errno;
			break;
		}
		if (write(wfd, buf, r) != r)
		{
			err = errno;
			break;
		}
	}
	free(buf);
    
#endif // defined(_SENDFILE_)

	close(rfd);
	close(wfd);

	if (err)
	{
		errno = err;
		return -1;
	}
	else
		return unlink(inFrom.c_str());
}

bool FileUtils::noCaseCompare(const string &s1, const string &s2)
{
    if (s1.size() == s2.size())
	{
        string t1 = s1;
        string t2 = s2;

        transform(t1.begin(), t1.end(), t1.begin(), (int(*)(int))tolower);
        transform(t2.begin(), t2.end(), t2.begin(), (int(*)(int))tolower);
        
        return (t1 == t2);
    }
    return false;
}

bool FileUtils::noCaseStrStr(const string &s1, const string &s2)
{
    string t1 = s1;
    string t2 = s2;

    transform(t1.begin(), t1.end(), t1.begin(), (int(*)(int))tolower);
    transform(t2.begin(), t2.end(), t2.begin(), (int(*)(int))tolower);
    
    return (strstr(t1.c_str(), t2.c_str()) != NULL);
}

bool FileUtils::canUserAccessPath(const HLUser &inUser, const PathVector &inUploadPathVector,
    bool &outIsDropBoxesDir, bool &outIsUsersDropBox)
{
    outIsUsersDropBox = false;
    outIsDropBoxesDir = false;
	if (!inUploadPathVector.empty())
	{
		PathVector::const_iterator iter = inUploadPathVector.begin();
		while (iter != inUploadPathVector.end())
		{
            if (outIsDropBoxesDir)
			{
                outIsDropBoxesDir = false;
				if ((*iter) == inUser.Login())
                {
                    outIsUsersDropBox = true;
                    return true;
                }
                else if (inUser.Access().modify_users)
                {
                    return true;
                }
				else
					return false;
			}
			else if (noCaseStrStr((*iter), string("drop boxes")))
			{
				outIsDropBoxesDir = true;
			}
            else if (noCaseStrStr((*iter), string("drop box")))
            {
                if (inUser.Access().view_drop_boxes)
                    return true;
                else
                    return false;
            }
			iter++;
		}
	}
	return true;
}

// i don't think i need this function
// i am going to change everything to use canUserAccessPath()
/*
bool FileUtils::isDropBoxPath(const PathVector &inPath)
{
	if (!inPath.empty())
	{
		PathVector::const_iterator iter = inPath.begin();
		while (iter != inPath.end())
		{
			if (noCaseCompare((*iter), string("DROP BOXES")))
			{
				return true;
			}
			iter++;
		}
	}
	
	return false;
}
*/

bool FileUtils::isValidUploadPath(const PathVector &inUploadPathVector)
{
	if (!inUploadPathVector.empty())
	{
		PathVector::const_iterator iter = inUploadPathVector.begin();
        while (iter != inUploadPathVector.end())
        {
            if (noCaseStrStr((*iter), string("drop boxes")))
            {
                iter++;
                if (iter != inUploadPathVector.end())
                    return true;
                else
                    return false;
            }
            else if (noCaseStrStr((*iter), "drop box"))
            {
                return true;
            }
            else if (noCaseStrStr((*iter), string("upload")))
            {
                // allow users to upload into any dir inside an upload dir
                return true;
            }
            iter++;
        }
	}
	
	return false;
}

bool FileUtils::addFileInfoToPacket(const string &inPath, const string &inFileName, HLPacket &ioPacket)
{
	string fullPath(inPath);
	string fileName = inFileName;
	// now append the fileName, after the path has been resolved
	FileUtils::NetworkToHomeString(fileName);
	fullPath.append(fileName);
	



#if defined(_CARBON_)
	bool gotRef;
	FSRef ref;
	
	DEBUG_CALL(printf("CARBON file get info: %s\n", fullPath.c_str()));
	
	gotRef = PathToFSRef(fullPath, &ref);
	if (!gotRef)
	{
		fullPath.append(".hpf");
		gotRef = PathToFSRef(fullPath, &ref);
	}
	
	if (gotRef)
	{
		FSCatalogInfo catInfo;
		if (FSGetCatalogInfo(&ref,
			kFSCatInfoNodeFlags | kFSCatInfoCreateDate | kFSCatInfoContentMod |
			kFSCatInfoFinderInfo | kFSCatInfoValence | kFSCatInfoDataSizes,
			// not supporting resource forks
			//kFSCatInfoFinderInfo | kFSCatInfoValence | kFSCatInfoDataSizes | kFSCatInfoRsrcSizes,
			&catInfo, NULL, NULL, NULL) == noErr)
		{
			FInfo *finderInfo = (FInfo *)catInfo.finderInfo;
			
			// to get the comment i'll need to call PBDTGetCommentSync
			// i seem to remember some issue with comments
			// from OS 9 not being present in OS X. i'm not sure i even
			// want to support comments.
			// taskPacket.AddStringObject(HTLS_DATA_FILE_COMMENT, );
			ioPacket.AddStringObject(HTLS_DATA_FILE_NAME, inFileName);
            if (catInfo.nodeFlags & kFSNodeIsDirectoryMask)
			{
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_ICON, DIR_TYPE);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_TYPE, DIR_TYPE);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_CREATOR, 0);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_NFILES, catInfo.valence);
			}
			else
			{
				// if (finderInfo->fdFlags & kIsAlias)
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_ICON, finderInfo->fdType);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_TYPE, finderInfo->fdType);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_CREATOR, finderInfo->fdCreator);
				ioPacket.AddUInt32Object(HTLS_DATA_FILE_SIZE, (UInt32)catInfo.dataLogicalSize);
				// not supporting resource forks
				//taskPacket.AddUInt32Object(HTLS_DATA_FILE_SIZE,
				//	(UInt32)catInfo.dataLogicalSize + catInfo.rsrcLogicalSize);
			}
			
			// i don't think i'm going to support the dates because the format
			// of the dates in FSCatalogInfo is different than what Hinks was using
			//taskPacket.AddObject(HTLS_DATA_FILE_DATE_CREATE, 8, (char *)&catInfo.createDate);
			//taskPacket.AddObject(HTLS_DATA_FILE_DATE_MODIFY, 8, (char *)&catInfo.contentModDate);
			
			return true;
		}
	}
#else
	struct stat sb;
	bool noError = false;
	
	DEBUG_CALL(printf("POSIX file get info: %s\n", fullPath.c_str()));
	

//#ifndef WIN32

	if (lstat(fullPath.c_str(), &sb) == 0)
	{
		noError = true;//.....stupid.
	}

//#endif//!WIN32
	
	if (!noError)
	{
		fullPath.append(".hpf");
		if (lstat(fullPath.c_str(), &sb) == 0)
		{
			noError = true;
		}
	}
	
	if (noError)
	{
		if (S_ISLNK(sb.st_mode))
		{
			// this is a link
			if (stat(fullPath.c_str(), &sb) == 0)
			{
				// resolved link
			}
			else
			{
				// error resolving link?
				noError = false;
			}
		}
	}
	
	if (noError)
	{
		ioPacket.AddStringObject(HTLS_DATA_FILE_NAME, inFileName);
		if (S_ISDIR(sb.st_mode))
		{

//#ifndef WIN32
			ioPacket.AddUInt32Object(HTLS_DATA_FILE_ICON, DIR_TYPE);
            ioPacket.AddUInt32Object(HTLS_DATA_FILE_TYPE, DIR_TYPE);
            ioPacket.AddUInt32Object(HTLS_DATA_FILE_CREATOR, 0);
			//taskPacket.AddUInt32Object(HTLS_DATA_FILE_NFILES, catInfo.valence);

//#endif//!WIN32
		}
		else
		{
            u_int32_t fileType = fileTypeFromName(fullPath);
			ioPacket.AddUInt32Object(HTLS_DATA_FILE_ICON, fileType);
            ioPacket.AddUInt32Object(HTLS_DATA_FILE_TYPE, fileType);
            ioPacket.AddUInt32Object(HTLS_DATA_FILE_CREATOR, creatorForType(fileType));
			ioPacket.AddUInt32Object(HTLS_DATA_FILE_SIZE, sb.st_size);
		}

			//2003/07/22 added by ortana.
#ifdef WIN32
			char creted_buff[8];
			char modified_buff[8];
			FileUtils::getFileTime( fullPath , creted_buff , modified_buff );
			ioPacket.AddObject(HTLS_DATA_FILE_DATE_CREATE, 8, creted_buff );
			ioPacket.AddObject(HTLS_DATA_FILE_DATE_MODIFY, 8, modified_buff);
#endif//WIN32

		return true;
	}
#endif
	
	return false;
}

		//-----------------------------------------------------------------------//
		////2003/07/22 added by ortana.
		//-----------------------------------------------------------------------//
int FileUtils::countAllFile(const char * dirPath )
{
	int count = 0;
	WIN32_FIND_DATA FindFileData;
	HANDLE hFind = FindFirstFile(dirPath, &FindFileData);
	if(hFind != INVALID_HANDLE_VALUE){
		do{
			DEBUG_CALL(cout << FindFileData.cFileName << endl);  // t@C̕\
			if( FindFileData.cFileName[0] != '.' &&
				!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) &&
				!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) )
			{
				count++;
			}
		} while(FindNextFile(hFind, &FindFileData));
		FindClose(hFind);
		return count;
	}
	else{
		return 0;
	}
	return 0;
}

void FileUtils::scanPathToPacket(const string &inPath, HLPacket &ioPacket)
{
	char hdrDataBuf[SIZEOF_HL_FILELIST_HDR + MAXPATHLEN];
	struct hl_filelist_hdr *fileHdr = (struct hl_filelist_hdr *)hdrDataBuf;

#ifdef WIN32
	WIN32_FIND_DATAA data;
	string winPath(inPath);
	for (string::iterator itr(winPath.begin()); itr != winPath.end(); itr++)
	{
		if (*itr == '/')
			*itr = '\\';
	}
	winPath += "*";
	DEBUG_CALL(printf("win32 path: %s\n", winPath.c_str()));

	HANDLE hFind = FindFirstFileA(winPath.c_str(), &data);
	if (hFind == INVALID_HANDLE_VALUE)
	{
		DEBUG_CALL(printf("error listing files\n"));
		return;
	}

	do//------------do while----------//2003/07/22 modefied by ortana.
	{
		bool isLink = false;
		bool isValid = true;
		WIN32_FIND_DATAA old_data;

		string fileName(data.cFileName);
		char sourcePath[MAX_PATH];
		fflush(stdout);

		if (fileName[0] == '.' ||
			(data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN))
		{
			continue;
		}

		string filePath(inPath);
		filePath.append(fileName);

		//resolve shortcut file
		if( filePath.find(".lnk") != string.npos )
		{
			isLink = true;

			FileUtils::resolve_Link( sourcePath , (LPSTR)filePath.data());
			old_data = data;
			HANDLE handle = FindFirstFileA( sourcePath  , &data );
			FindClose(handle);
		}
		else
		{
			strcpy( sourcePath , filePath.c_str() );
		}

		//if a object is drive or directory
		if( GetDriveType( sourcePath ) > 1 )
		{
			data.dwFileAttributes = 17;
			sprintf( sourcePath , "%s*" , sourcePath );
		}
		else if(data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			sprintf( sourcePath , "%s\\*" , sourcePath );
		}

		fileHdr->ftype = 0;
		fileHdr->fcreator = 0;
		fileHdr->fsize = 0;
		fileHdr->fcount = 0;

		//if (data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
		//	isValid = false;

		//if (isValid)
		//{

		if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			fileHdr->ftype = htonl(DIR_TYPE);
			fileHdr->fsize = htonl(/*subItemCount*/countAllFile(sourcePath));//count files in the directory. modified by ortana.
			fileHdr->fcount = htonl(0);
		}
		else//file
		{
			// don't convert byte order yet
			// because we have to use it for lookup
			fileHdr->ftype = fileTypeFromName(fileName);
			fileHdr->fcreator = htonl(creatorForType(fileHdr->ftype));

			if( isLink )
			{
				if( !FileUtils::IsExistObject( sourcePath , TRUE ) )
				{
					fileHdr->ftype = htonl(ALIAS_TYPE);
				}
			}
			else
			{
				fileHdr->ftype = htonl(fileHdr->ftype);
			}
			//fileHdr->ftype = htonl(fileHdr->ftype);
			fileHdr->fsize = htonl((data.nFileSizeHigh * MAXDWORD) + data.nFileSizeLow);
		}



		//if (fileHdr->ftype == htonl(FILE_TRANSFER_TYPE))
		//{
		// have to remove the .hpf from the file name
		//removeExtension(fileName);
		//}

		HomeToNetworkString(fileName);

		fileHdr->fnlen = htonl((u_int32_t)fileName.length());
		memcpy(fileHdr->fname, fileName.c_str(), fileName.length());

		ioPacket.AddObject(HTLS_DATA_FILE_LIST,
			SIZEOF_HL_FILELIST_HDR + fileName.length(), (char *)fileHdr);

		//			}
		//		}

		if( isLink )
		{
			data = old_data;
		}


	} while (FindNextFileA(hFind, &data));//------------do while----------//

	FindClose(hFind);

#else//end function of WIN32
	DIR *dir;
	struct dirent *de;
	struct stat sb;
	
	dir = opendir(inPath.c_str());
	if (dir == NULL)
	{
		return;
	}
	
	while ((de = readdir(dir)) != NULL)
	{
		bool isLink = false;
        bool isValid = true;
		string fileName(de->d_name);
		if (fileName[0] != '.')
		{
			string filePath(inPath);
			filePath.append(fileName);
			
            fileHdr->ftype = 0;
			fileHdr->fcreator = 0;
			fileHdr->fsize = 0;
			fileHdr->fcount = 0;

#if defined(_CARBON_)
            resolveAlias(filePath);
            FSRef theRef;
            if (PathToFSRef(filePath, &theRef))
            {
                FSCatalogInfo catInfo;
                if (FSGetCatalogInfo(&theRef, kFSCatInfoFinderInfo,
                    &catInfo, NULL, NULL, NULL) == noErr)
                {
                    FInfo *finderInfo = (FInfo *)catInfo.finderInfo;
                    if (finderInfo->fdFlags & kIsInvisible)
                    {
                        isValid = false;
                    }
                }
                else
                    isValid = false;
            }
            else
			{
                //isValid = false;
				//fileHdr->ftype = htonl(ALIAS_TYPE);
			}
#endif // _CARBON_
            if (isValid)
            {
                if (lstat(filePath.c_str(), &sb) != 0)
    
                {
                    fileHdr->ftype = htonl(ALIAS_TYPE);
                }
                else
                {
                    if (S_ISLNK(sb.st_mode))
                    {
                        isLink = true;
                        if (stat(filePath.c_str(), &sb) != 0)
                        {
                            // unresolved link
                            fileHdr->ftype = htonl(ALIAS_TYPE);
                        }
                    }
                    
                    if (S_ISDIR(sb.st_mode))
                    {
                        fileHdr->ftype = htonl(DIR_TYPE);
                        // count sub items
                        DIR *subDir = opendir(filePath.c_str());
                        u_int32_t subItemCount = 0;
                        if (subDir)
                        {
                            struct dirent *subDe;
                            while ((subDe = readdir(subDir)) != NULL)
                            {
                                string subFileName(subDe->d_name);
                                if (subFileName[0] != '.')
                                    subItemCount++;
                            }
                            closedir(subDir);
                        }
                        fileHdr->fsize = htonl(subItemCount);
                        fileHdr->fcount = htonl(subItemCount);
                    }
                    else if (fileHdr->ftype != htonl(ALIAS_TYPE))
                    {
                        // don't convert byte order yet
                        // because we have to use it for lookup
                        fileHdr->ftype = fileTypeFromName(fileName);
                        fileHdr->fcreator = htonl(creatorForType(fileHdr->ftype));
                        fileHdr->ftype = htonl(fileHdr->ftype);
                        fileHdr->fsize = htonl(sb.st_size);
                    }
                }
                
                if (fileHdr->ftype == htonl(FILE_TRANSFER_TYPE))
                {
                    // have to remove the .hpf from the file name
                    removeExtension(fileName);
                }
                
                HomeToNetworkString(fileName);

                fileHdr->fnlen = htonl((u_int32_t)fileName.length());
                memcpy(fileHdr->fname, fileName.c_str(), fileName.length());
                                
                ioPacket.AddObject(HTLS_DATA_FILE_LIST,
                    SIZEOF_HL_FILELIST_HDR + fileName.length(), (char *)fileHdr);
            }
		}
	}
	closedir(dir);
#endif // WIN32
}

void FileUtils::HomeToNetworkString(string &ioString)
{
#if defined(_CARBON_)
    if (gEncodingInited)
    {
        unsigned int bufSize = ioString.size() * 2;
        char *convBuf = (char *)malloc(bufSize);
        ByteCount inputRead, outputLen = 0;
        OSStatus status = -1;
		
		if (gServer->Config().textEncoding == "MacRoman")
		{
			status = ConvertFromUnicodeToText(gUTF8ToMacRomanConversionInfo,
                ioString.size(), (const UniChar *)ioString.c_str(),
                kUnicodeLooseMappingsMask, /* flags */
                0, NULL, /* input offsets */
                0, NULL, /* output offsets */
                bufSize, &inputRead, &outputLen, convBuf);
		}
		else if (gServer->Config().textEncoding == "Shift-JIS")
		{
			status = ConvertFromUnicodeToText(gUTF8ToShiftJISConversionInfo,
				ioString.size(), (const UniChar *)ioString.c_str(),
				kUnicodeLooseMappingsMask, /* flags */
				0, NULL, /* input offsets */
				0, NULL, /* output offsets */
				bufSize, &inputRead, &outputLen, convBuf);
		}
		
        if (status == noErr)
            ioString.assign(convBuf, outputLen);
        else
        {
			DEBUG_CALL(printf("error converting uft8 to text(%d): %s\n",
				(int)status, ioString.c_str()));
        }
        free(convBuf);
    }
#else
	(void)ioString;
#endif // _CARBON_
}

void FileUtils::NetworkToHomeString(string &ioString)
{
#if defined(_CARBON_)
    if (gEncodingInited)
    {
        unsigned int bufSize = ioString.size() * 2;
        char *convBuf = (char *)malloc(bufSize);
        ByteCount inputRead, outputLen = 0;
		OSStatus status = -1;
		if (gServer->Config().textEncoding == "MacRoman")
		{
			status = ConvertFromTextToUnicode(gMacRomanToUTF8ConversionInfo,
				ioString.size(), ioString.c_str(),
				kUnicodeLooseMappingsMask, /* flags */
				0, NULL, /* input offsets */
				0, NULL, /* output offsets */
				bufSize, &inputRead, &outputLen, (UniChar *)convBuf);
		}
		else if (gServer->Config().textEncoding == "Shift-JIS")
		{
			status = ConvertFromTextToUnicode(gShiftJISToUTF8ConversionInfo,
				ioString.size(), ioString.c_str(),
				kUnicodeLooseMappingsMask, /* flags */
				0, NULL, /* input offsets */
				0, NULL, /* output offsets */
				bufSize, &inputRead, &outputLen, (UniChar *)convBuf);
		}
		
		if (status == noErr)
			ioString.assign(convBuf, outputLen);
		else
		{
			DEBUG_CALL(printf("error converting text to uft8(%d): %s\n",
				(int)status, ioString.c_str()));
		}
        free(convBuf);
    }
#else
	(void)ioString;
#endif // _CARBON_
}

void FileUtils::splitString(const string &inString, const string &inDelimiter, vector<string> &outStrings)
{
	string::size_type delPos = 0;
	string::size_type startPos = 0;
	while ((delPos = inString.find(inDelimiter, startPos)) != string::npos)
	{
		if (delPos - startPos > 0)
		{
			string subStr = inString.substr(startPos, delPos - startPos);
			outStrings.push_back(subStr);
		}
		startPos = delPos + inDelimiter.length();
	}
	
	// catch the straggler
	// this will get whatever is after the last delimiter until the end of the string
	if (startPos != 0 && startPos < inString.length())
		outStrings.push_back(inString.substr(startPos));
	
	// don't let them leave empty handed
	// if the string doesn't contain the delimiter then we return the string
	if (outStrings.empty())
		outStrings.push_back(inString);
}

#endif // !defined(_PREFPANE_)

