# -*- coding: utf-8 -*-
# = Wikibot library
# 
# An implementation of bot that automate work on Wikipedia or other MediaWiki
# sites.
# 
# AutumnSnow derived a part of this code from RWikiBot 1.0.10 on January 18, 2008.
# RWikiBot was written by Edwin Sidney Roger.
# 
# == Copyright notice
# 
# Copyright (c) 2009 AutumnSnow
# Copyright (c) 2007-2008 Edwin Sidney Roger
# 
# == License
# 
# You can redistribute it and/or modify it under either the terms of the GNU
# General Public License, version 2 (see the file COPYING.txt).
# 

require 'uri'
require 'net/http'
require 'rexml/document'
require 'rexml/element'
require 'rexml/attribute'

# ボットの実装です。
# 設定ファイル <tt>config.xml</tt> に記述された内容にしたがって動作します。
class WikiBot
	
	USER_AGENT = 'Ruby wiki bot'
	
	def initialize(config)
		@config = config
	end
	
	# Login
	#
	# This is the method that will allow the bot to log in to the wiki. Its not always necessary, but bots need to log in to save changes or retrieve watchlists. 
	#
	# No variables are accepted and the username/password are stored in config.xml.
	def login
		
		post_me = {'lgname'=>@config.user_name,'lgpassword'=>@config.password}
		
		result = make_request('login', post_me)
		
		# Now we need to changed some @config stuff, specifically that we're logged in and the variables of that
		# This will also change the make_request, but I'll comment there
		login_element = result.get_elements('/api/login').first
		result = login_element.attribute('result').to_s
		if result.eql?('Success') then
			# All lg variables are directly from API and stored in config that way
			@config.logged_in = true
			@config.user_id = login_element.attribute('lguserid').to_s
			@config.lgtoken = login_element.attribute('lgtoken').to_s
			@config.cookie_prefix = login_element.attribute('cookieprefix').to_s
			@config.session_id = login_element.attribute('sessionid').to_s
			$stderr.puts 'Bot logged in.'
			return true
		else
			error_message = 'Login failure, result: '
			if result.eql?('NotExists') then
				error_message += 'Account not exists'
			elsif result.eql?('WrongPass') then
				error_message += 'Wrong password'
			else
				error_message += result
			end
			raise MediaWikiApiError, error_message
			#return false
		end
		
	end
	
	# Logout
	#
	def logout
		
		login_result = make_request('logout', {})
		@config.logged_in = false
		$stderr.puts 'Bot logged out.'
		
	end
	
	def current_revision_text(title)
		
		# Make the request
		result = make_request('query', {'titles' => title, 'prop' => 'revisions', 'rvprop' => 'content'})
		
		revision_elements = result.get_elements('//rev')
		
		if revision_elements.size == 0 then
			raise MediaWikiApiError,
				'Query failure, ' + result.to_s
		end
		
		# Result processing
		return revision_elements.first.text
		
	end
	
	def obtain_edit_token(title)
	
		# Make the request
		result = make_request('query', {'titles' => title, 'prop' => 'info|revisions', 'intoken' => 'edit'})
		
		# Result processing
		@config.edit_token = result.get_elements('//page').first.attribute('edittoken').to_s
		@config.timestamp = result.get_elements('//rev').first.attribute('timestamp').to_s
		return nil
		
	end
	
	def edit(title, text, summary)
		
		post_this = {'title' => title, 'text' => text, 'summary' => summary,
				'token' => @config.edit_token, 'edittime' => Time.now.strftime("%Y%m%d%H%M%S")}
		if @config.timestamp != nil then
			post_this.store('basetimestamp', @config.timestamp)
		end
		
		if @config.mark_edit_as_bot then
			post_this.store('bot', '1')
		end
		
		# Make the request
		result = make_request('edit', post_this)
		error_elements = result.get_elements('//error')
		if error_elements.size > 0 then
			error_element = error_elements.first
			raise MediaWikiApiError,
				'Edit failure, code=' + error_element.attribute('code').to_s +
				', info="' + error_element.attribute('info').to_s + '"'
		end
		result_attribute_value = result.get_elements('//edit').first.attribute('result').to_s
		if not result_attribute_value.eql?('Success') then
			error_element = error_elements.first
			raise MediaWikiApiError,
				'Edit failure, ' + result.to_s
		end
		
		# Result processing
		return nil
		
	end
	
	private 
	
	# Make Request is a method that actually handles making the request to the API. Since the API is somewhat standardized, this method is able to accept the action and a hash of variables, and it handles all the fun things MediaWiki likes to be weird over, like cookies and limits and actions. Its very solid, but I didn't want it public because it also does some post processing, and that's not very OO. 
	def make_request (action, post_this)
		
		
		#Housekeeping. We need to add format and action to the request hash
		post_this['format'] = 'xml'
		post_this['action'] = action
		if @config.logged_in then
			post_this['lgusername'] = @config.user_name
			post_this['lgtoken'] = @config.lgtoken
			post_this['lguserid'] = @config.user_id
			cookie_user_id_field_name = 'UserID'
			cookie_user_name_field_name = 'UserName'
			cookie_token_field_name = 'Token'
			cookie_session_field_name = '_session'
			cookie_prefixed_user_id_field_name =
				@config.cookie_prefix + cookie_user_id_field_name
			cookie_prefixed_user_name_field_name =
				@config.cookie_prefix + cookie_user_name_field_name
			cookie_prefixed_token_field_name =
				@config.cookie_prefix + cookie_token_field_name
			cookie_prefixed_session_field_name =
				@config.cookie_prefix + cookie_session_field_name
			if @config.cookie.index(cookie_prefixed_user_id_field_name) == nil then
				@config.cookie +=
					'; ' + cookie_prefixed_user_id_field_name + '=' + @config.user_id
			end
			if @config.cookie.index(cookie_prefixed_user_name_field_name) == nil then
				@config.cookie +=
					'; ' + cookie_prefixed_user_name_field_name + '=' + @config.user_name
			end
			if @config.cookie.index(cookie_prefixed_token_field_name) == nil then
				@config.cookie +=
					'; ' + cookie_prefixed_token_field_name + '=' + @config.lgtoken
			end
			if @config.cookie.index(cookie_prefixed_session_field_name) == nil then
				@config.cookie +=
					'; ' + cookie_prefixed_session_field_name + '=' + @config.session_id
			end
		end
		#Send the actual request
		# This is working agian. There was a bug in Ruby 1.8.4
		url = URI.parse(@config.target_wiki_uri + @config.wiki_path + 'api.php')
		req = Net::HTTP::Post.new(url.path)
		if (@config.cookie != nil) then
			req.add_field('Cookie', @config.cookie)
		end
		req.add_field('User-Agent', USER_AGENT)
		req.set_form_data(post_this)
		res = Net::HTTP.new(url.host, url.port).start {|http| http.request(req) }
		case res
		when Net::HTTPSuccess, Net::HTTPRedirection
			#OK
		else
			res.error!
		end
		cookie = res['set-cookie']
		if cookie != nil then
			@config.cookie = cookie
		end
		result = REXML::Document.new(REXML::Source.new(res.body)).root
		
		#Return the response
		return result
		
	end
	
end

class MediaWikiApiError < StandardError

end

# 設定ファイル <tt>config.xml</tt> を表現するクラスです。
class Config
	
	attr_reader :working_directory
	attr_reader :console_encoding
	attr_reader :file_browser, :text_viewer
	attr_reader :user_name, :password
	attr_reader :target_language, :target_wiki_uri, :target_statistics_page
	attr_reader :wiki_path
	attr_reader :edit_target_page, :edit_header
	attr_reader :mark_edit_as_bot
	attr_accessor :user_id, :logged_in, :lgtoken, :cookie, :cookie_prefix, :session_id
	attr_accessor :edit_token, :timestamp
	
	def initialize(working_directory = '.')
		@working_directory = working_directory
		config_element = REXML::Document.new(REXML::Source.new(open('config.xml').read())).root
		file_browser_elements = config_element.get_elements('fileBrowser')
		@console_encoding = config_element.get_elements('consoleEncoding').first.text
		if file_browser_elements.size >= 1 then
			@file_browser = file_browser_elements.first.text
		else
			@file_browser = nil
		end
		text_viewer_elements = config_element.get_elements('textViewer')
		if text_viewer_elements.size >= 1 then
			@text_viewer = text_viewer_elements.first.text
		else
			@text_viewer = nil
		end
		@user_name = config_element.get_elements('userName').first.text
		@password = config_element.get_elements('password').first.text
		@target_language = config_element.get_elements('targetLanguage').first.text
		@target_wiki_uri = config_element.get_elements('targetWikiUri').first.text
		@target_statistics_page = config_element.get_elements('targetStatisticsPage').first.text
		@wiki_path = config_element.get_elements('wikiPath').first.text
		edit_target_element = config_element.get_elements('editTargetPage').first
		edit_header_element = config_element.get_elements('editHeader').first
		@edit_header = ''
		if edit_target_element == nil then
			@edit_target_page = @target_statistics_page
		else
			@edit_target_page = edit_target_element.text
		end
		if edit_header_element != nil then
			@edit_header = edit_header_element.text
		end
		@mark_edit_as_bot = config_element.get_elements('markEditAsBot').first.text.downcase.eql?('true')
		@logged_in = false
	end
end

