# -*- encoding: utf-8 -*-
# = Mediawiki languages statistics library
# 
# このライブラリに含まれるクラス群は、ウィキペディアにおける
# http://ja.wikipedia.org/wiki/Wikipedia:全言語版の統計
# のページの更新を支援します。
# このライブラリは、ライブラリ利用者に次の3つのクラスを提供します。
# 
# * Taker ウィキペディアの各言語版のサイトから統計情報を取得して更新用ウィキテキストを出力します。
# * Updater Taker が作成し出力した更新用ウィキテキストで、全言語版の統計ページを更新します。
# * LanguageResourcePrinter 現在の統計ページをもとにしてリソースファイルのlanguage要素を
#   出力します。
# 
# == Copyright notice
# 
# Copyright (c) 2009 AutumnSnow
# 
# == License
# 
# You can redistribute it and/or modify it under either the terms of the GNU
# General Public License, version 2 (see the file COPYING.txt).
# 

require 'uri'
require 'net/http'
require 'rexml/document'
require 'rexml/element'
require 'rexml/attribute'
require 'monitor'
require 'fileutils'
require 'iconv'

require 'wikibot'

Net::HTTP.version_1_2

# ウィキペディアの各言語版のサイトから統計情報を取得して更新用ウィキテキストを出力します。
# ウィキテキストの出力に際しては、リソースファイル <tt>resource.xml</tt> を参照します。
# このクラスのオブジェクトの処理実行後に Taker#status を参照すると、
# 実行の終了ステータスが得られます。
# 実行の終了ステータスは、次のように設定されます。
# * 0: 正常に終了したとき。
# * 1: 実行中に異常があったとき。Special:SiteMatrixに照らして、
#   リソースファイル <tt>resource.xml</tt> に不足している言語 (language要素) があった場合など
#   (この場合は、不足している言語については不備のある更新用ウィキテキストが出力されます) 。
class Taker
	
	attr_reader :status
	
	def execute
		
		@status = 0
		time_started = Time.now.utc
		working_directory_format = DateTimeFormat.new('${year}${month}${day}T${hour}${minute}${second}')
		working_directory = working_directory_format.format(time_started)
		Dir.mkdir(working_directory)
		logger = Logger.new([$stderr, File.new(working_directory + '/log.txt', 'w')])
		date_time_format = DateTimeFormat.new()
		logger.log '統計情報の取得と更新用ウィキテキストの作成'
		logger.log '開始しました。 ' + date_time_format.format(time_started)
		config = Config.new(working_directory)
		logger.log('作業ディレクトリ: ' + config.working_directory)
		current_page_file_name = config.working_directory + '/current-page.txt'
		resource = Resource.new(logger)
		statisticsFetcher = StatisticsFetcher.new(config, resource, logger)
		statisticsFetcher.execute()
		@status = statisticsFetcher.status
		
		bot = WikiBot.new(config)
		bot.login
		current_revision_text = bot.current_revision_text(config.target_statistics_page)
		bot.logout
		current_page_file = File.new(current_page_file_name, 'w')
		current_page_file.print(current_revision_text)
		current_page_file.close()
		LanguageEntry.minimum_number_of_articles_for_calculating_depth =
			resource.minimum_number_of_articles_for_calculating_depth
		new_statistics_page =
			NewStatisticsPage.new(statisticsFetcher, current_revision_text, resource, logger)
		new_statistics_page.number_of_languages_in_site_matrix =
			statisticsFetcher.number_of_languages_in_site_matrix()
		new_statistics_page.print
		FileUtils.copy(Dir.glob(config.working_directory + '/*'), '.')
		FileUtils.copy(Resource::FILE_NAME, config.working_directory)
		if (config.file_browser != nil) then
			launch(config.file_browser, config.working_directory, 'ファイルブラウザの起動に失敗しました: ', logger)
		end
		if (config.text_viewer != nil) then
			launch(config.text_viewer, new_statistics_page.file_name, 'テキストビューワの起動に失敗しました: ', logger)
		end
		time_ended = Time.now.utc
		date_time_format.set_date_time(time_ended)
		logger.log '終了しました。 ' + date_time_format.format
		logger.log '要した時間: ' + (time_ended - time_started).to_s + ' 秒'
	end
	
	def launch(command, argument, error_message, logger)
		if fork_is_available? then
			pid = fork do
				begin
					exec(command, argument)
				rescue SystemCallError => e
					logger.log error_message + command + ', ' + e.to_s
				end
			end
			Process.detach(pid)
		else
			exec_result = system('start', command, argument)
			if not exec_result then
				logger.log error_message + command
			end
		end
	end
	
	def fork_is_available?
		fork_unavailables = ['mingw', 'mswin', 'java']
		fork_unavailables.each do |unavailable_keyword|
			if RUBY_PLATFORM.index(unavailable_keyword) != nil then
				return false
			end
		end
		return true
	end
	
end

# 現在の統計ページをもとにしてリソースファイル <tt>resource.xml</tt> のlanguage要素を
# ファイルに出力します。
# 出力ファイルの名前は、<tt>resource-languages.xml</tt> です。
class LanguageResourcePrinter
	
	def execute()
		time_started = Time.now.utc
		date_time_format = DateTimeFormat.new()
		logger = Logger.new([$stderr, File.new('log.txt', 'w')])
		logger.log '現在の統計ページをもとにリソースファイルのlanguage要素を出力'
		logger.log '開始しました。 ' + date_time_format.format(time_started)
		config = Config.new()
		current_page_file_name = 'current-page.txt'
		resource = Resource.new(logger)
		bot = WikiBot.new(config)
		bot.login
		current_revision_text = bot.current_revision_text(config.target_statistics_page)
		bot.logout
		output_file_name = 'resource-languages.xml'
		logger.log '出力ファイル: ' + output_file_name
		output = File.new(output_file_name, 'w')
#		output = $stdout
		WikiTable.new(current_revision_text, resource, logger).rows.each do |row|
			output.puts('<language><code>' + row.code + '</code>' +
					'<description>' + row.description + '</description>' +
					'<articleOfWiki>' + row.article_of_wiki + '</articleOfWiki>' +
					'<englishDescription>' + row.english_description + '</englishDescription>' +
					'<score>' + row.score + '</score></language>')
			output.flush
		end
		output.close
		time_ended = Time.now.utc
		logger.log '終了しました。 ' + date_time_format.format(time_ended)
		logger.log '要した時間: ' + (time_ended - time_started).to_s + ' 秒'
	end
	
end

class StatisticsFetcher
	
	attr_reader :status
	attr_reader :config,
			:target_wiki_url,
			:fetch_start_time, :fetch_end_time
	attr_reader :language_entries, :number_of_languages_in_site_matrix
	
	STATISTICS_INFO_FILE_NAME = 'statistics-info.xml'
	STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME = 'workingDirectory'
	STATISTICS_INFO_FETCH_START_DATE_TIME_ELEMENT_NAME = 'fetchStartDateTime'
	STATISTICS_INFO_FETCH_END_DATE_TIME_ELEMENT_NAME = 'fetchEndDateTime'
	STATISTICS_INFO_NUMBER_OF_LANGUAGES_IN_SITE_MATRIX_ELEMENT_NAME = 'numberOfLanguagesInSiteMatrix'
	STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME = 'updateSummary'
	
	def initialize(config, resource, logger)
		@config = config
		@resource = resource
		@logger = logger
		@target_wiki_url = config.target_wiki_url
		@language_entries = Hash.new
		@status = 0
	end
	
	def fetch_uri(uri_string, limit = 10 )
		raise ArgumentError, 'http redirect too deep' if limit == 0
		response = Net::HTTP.get_response(URI.parse(uri_string))
		case response
		when Net::HTTPSuccess     then response
		when Net::HTTPRedirection then fetch_uri(response['location'], limit - 1)
		else
			response.error!
		end
	end
	
	def string_to_integer(string)
		if string.strip.length == 0 then
			return nil
		else
			return string.to_i
		end
	end
	
	def execute()
		@resource.target_wiki_main_page =
			REXML::Document.new(fetch_uri(@target_wiki_url).body).get_elements('//h1').first.text
		response = fetch_uri(@target_wiki_url + @config.wiki_path + 'index.php?title=Special:SiteMatrix&action=raw')
		site_matrix_file_name = @config.working_directory + '/site-matrix.xml'
		file = File.new(site_matrix_file_name, 'w')
		file.print(response.body)
		file.close
		file = File.new(site_matrix_file_name, 'r')
		language_elements = REXML::Document.new(file).root.get_elements(
			'matrix/language[count(site[' + LanguageOfSiteMatrix::CODE_PREDICATE + ']) = 1][@name != ""]')
		@logger.log('現在あるウィキペディアの言語コードの数 (Special:SiteMatrix) : ' + language_elements.size.to_s)
		threads = []
		lock = Monitor.new
		@logger.log('リソースファイルとSpecial:SiteMatrixの照合を開始します。')
		languages_of_site_matrix = Hash.new
		language_elements.each do |element|
			language = LanguageOfSiteMatrix.new(element)
			languages_of_site_matrix.store(language.code, language)
			if not @resource.languages.has_key?(element.attribute('code').to_s) then
				@status = 1
				@logger.log('次の言語版の記述がリソースファイルに存在しません: ' +
					language.code + ', ' + language.name + ', ' + language.wiki_url)
			end
		end
		@resource.languages.values.each do |language|
			if not languages_of_site_matrix.has_key?(language.code) then
				@logger.log('次のリソースファイルの記述がSpecial:SiteMatrixに存在しません: ' +
					language.code + ', ' + language.description + ', ' + language.english_description)
			end
		end
		@logger.log('リソースファイルとSpecial:SiteMatrixの照合を終了しました。')
		@fetch_start_time = Time.now.utc
		date_time_format = DateTimeFormat.new
		@logger.log('統計情報の取得を開始しました。 ' + date_time_format.format(@fetch_start_time))
		languages_of_site_matrix.values.each do |language|
			threads << Thread.new(language) do |language_thread_local|
				language_code = language_thread_local.code
				language_name = language_thread_local.name
				wiki_url = language_thread_local.wiki_url
				language_file_name = @config.working_directory + '/' + language_code + '.txt'
				language_entry = LanguageEntry.new(language_code,
						language_name, wiki_url, language_file_name)
				lock.synchronize {
					@language_entries.store(language_code, language_entry)
				}
				failed = true
				20.times do
					begin
						response = fetch_uri(wiki_url +
							'/w/index.php?title=Special:Statistics&action=raw')
						if response.body.length > 0 then
							response.body.split(';').each do |attribute|
								attribute_splitted = attribute.split('=')
								attribute_name = attribute_splitted.first
								attribute_value = attribute_splitted[1]
								case attribute_name
								when 'total'
									language_entry.number_of_pages =
										string_to_integer(attribute_value)
								when 'good'
									language_entry.number_of_articles =
										string_to_integer(attribute_value)
								when 'edits'
									language_entry.number_of_edits =
										string_to_integer(attribute_value)
								when 'users'
									language_entry.number_of_users =
										string_to_integer(attribute_value)
								when 'activeusers'
									language_entry.number_of_active_users =
										string_to_integer(attribute_value)
								when 'admins'
									language_entry.number_of_administrators =
										string_to_integer(attribute_value)
								when 'images'
									language_entry.number_of_files =
										string_to_integer(attribute_value)
								else
								end
							end
							failed = false
							break
						end
						#puts 'Fetched SiteMatrix from ' + language_url
					rescue Errno::ETIMEDOUT => exception
						#puts 'Connection timed out (to be retried): ' + language_url
					rescue Errno::ECONNREFUSED => exception
						#puts 'Connection refused (to be retried): ' + language_url
					rescue Net::HTTPFatalError => exception
					rescue EOFError => exception
					end
				end
				if failed then
					@logger.log exception.to_s + ', ' + language_url
					@status = 1
				end
			end
		end
		threads.each do |thread|
			thread.join
		end
		@number_of_languages_in_site_matrix = language_elements.size
		@fetch_end_time = Time.now.utc
		@logger.log('統計情報の取得を終了しました。 ' + date_time_format.format(@fetch_end_time))
		@logger.log('統計情報の取得に要した時間: ' + (@fetch_end_time - @fetch_start_time).to_s + ' 秒')
		file = File.new(@config.working_directory + '/' + STATISTICS_INFO_FILE_NAME, 'w')
		file.puts('<?xml version="1.0" encoding="UTF-8"?>')
		file.puts('<statisticsInfo>')
		file.puts('  <' + STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME + '>' +
				@config.working_directory +
				'</' + STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME + '>')
		file.puts('  <' + STATISTICS_INFO_FETCH_START_DATE_TIME_ELEMENT_NAME + '>' +
				date_time_format.format(@fetch_start_time) +
				'</' + STATISTICS_INFO_FETCH_START_DATE_TIME_ELEMENT_NAME + '>')
		file.puts('  <' + STATISTICS_INFO_FETCH_END_DATE_TIME_ELEMENT_NAME + '>' +
				date_time_format.format(@fetch_end_time) +
				'</' + STATISTICS_INFO_FETCH_END_DATE_TIME_ELEMENT_NAME + '>')
		file.puts('  <' + STATISTICS_INFO_NUMBER_OF_LANGUAGES_IN_SITE_MATRIX_ELEMENT_NAME + '>' +
				@number_of_languages_in_site_matrix.to_s +
				'</' + STATISTICS_INFO_NUMBER_OF_LANGUAGES_IN_SITE_MATRIX_ELEMENT_NAME + '>')
		date_time_format.set_date_time(@fetch_end_time)
		update_summary = @resource.update_summary.
								sub('${year}', date_time_format.year).
								sub('${month}', date_time_format.month.to_i.to_s).
								sub('${day}', date_time_format.day.to_i.to_s).
								sub('${hour}', date_time_format.hour).
								sub('${minute}', date_time_format.minute).
								sub('${timezone}', date_time_format.timezone)
		file.puts('  <' + STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME + '>' +
				update_summary +
				'</' + STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME + '>')

		file.puts('</statisticsInfo>')
		file.close
	end

end

class NewStatisticsPage
	
	attr_reader :file_name
	attr_reader :current_statistics_page, :language_entries
	attr_reader :header, :tables, :table_header
	attr_writer :number_of_languages_in_site_matrix
	
	FILE_NAME = 'new-page.txt'
	
	def initialize(statistics_fetcher, current_statistics_page, resource, logger)
		@statistics_fetcher = statistics_fetcher
		
		@resource = resource
		@logger = logger
		@language_entries = statistics_fetcher.language_entries
		@file_name = @statistics_fetcher.config.working_directory + '/' + FILE_NAME
		@config = statistics_fetcher.config
		@as_of = statistics_fetcher.fetch_end_time
		@header = ''
		@table_header = ''
		@language_missed = []
		@language_new = language_entries.values
		parse_current_page(current_statistics_page)
	end
	
	def parse_current_page(current_statistics_page)
		wiki_table = WikiTable.new(current_statistics_page, @resource, @logger)
		rows = wiki_table.rows
		rows.each do |row|
			language_entry = @language_entries.fetch(row.code, nil)
			if language_entry == nil then
				@language_missed.push(language_code)
				next
			end
			@language_new.delete(language_entry)
		end
		@logger.log('新規開設された言語版:')
		@language_new.each do |language_entry|
			@logger.log(' * ' + language_entry.code +
				', ' + language_entry.name +
				', ' + language_entry.wiki_url)
		end
		if @language_new.size == 0 then
			@logger.log(' なし')
		end
		@logger.log('閉鎖されているとみられる言語版:')
		@language_missed.each do |language_code|
			message = ' * ' + language_code
			language_of_resource = @resource.languages.fetch(language_code, nil)
			if (language_of_resource != nil) then
				message += ', ' + language_of_resource.description +
				', ' + language_of_resource.english_description
			end
			@logger.log(message)
		end
		if @language_missed.size == 0 then
			@logger.log(' なし')
		end
	end
	
	def total_integer_value(integer, field_name, language_entry, error_messages)
		if integer == nil then
			return 0
		end
		if integer < 0 then
			error_message = language_entry.description + ' (' + language_entry.code + ') から取得した' + field_name
			error_messages.push(error_message)
			return 0
		end
		return integer
	end
	
	def print()
		total_number_of_articles = 0
		total_number_of_pages = 0
		total_number_of_edits = 0
		total_number_of_administrators = 0
		total_number_of_users = 0
		total_number_of_active_users = 0
		total_number_of_files = 0
		number_of_languages = @language_entries.values.size
		sorted_language_entries = @language_entries.values.sort.reverse
		file = File.new(@file_name, 'w')
		date_time_format = DateTimeFormat.new
		date_time_format.set_date_time(@as_of)
		file.puts @resource.page_header.
						sub('${year}', date_time_format.year).
						sub('${month}', date_time_format.month.to_i.to_s).
						sub('${day}', date_time_format.day.to_i.to_s).
						sub('${hour}', date_time_format.hour).
						sub('${minute}', date_time_format.minute).
						sub('${number.of.languages}', number_of_languages.to_s)
		table_infos = @resource.table_Infos.clone()
		number_format = NumberFormat.new
		rank = 0
		previous_number_of_articles = 0
		same_rank = 0
		is_printing = false
		error_messages = Array.new
		sorted_language_entries.each do |language_entry|
			table_infos.each do |table_info|
				if language_entry.number_of_articles.to_i >= table_info.or_more then
					if table_info.header_is_not_printed then
						if is_printing then
							file.puts '|}'
							file.puts
						else
							is_printing = true
						end
						file.puts table_info.label
						file.puts @resource.table_header.rstrip
						table_info.header_is_not_printed = false
					end
					break
				end
			end
			if @resource.languages.has_key?(language_entry.code) then
				language_of_resource = @resource.languages.fetch(language_entry.code)
				language_entry.description = language_of_resource.description
				language_entry.article_of_wiki = language_of_resource.article_of_wiki
				language_entry.english_description = language_of_resource.english_description
				language_entry.score = language_of_resource.score
			else
				language_entry.description = ''
				language_entry.article_of_wiki = nil
				language_entry.english_description = ''
				language_entry.score = ''
			end
			if rank == 0 then
				rank += 1
				same_rank = 1
			elsif language_entry.number_of_articles.to_i < previous_number_of_articles.to_i then
				rank += same_rank
				same_rank = 1
			else
				same_rank += 1
			end
			previous_number_of_articles = language_entry.number_of_articles.to_i
			file.puts '|-'
			file.puts '| ' + @resource.table_column_attributes[0] + rank.to_s
			description_text = language_entry.description
			if language_entry.article_of_wiki != nil then
				description_text += ' - [[' + language_entry.article_of_wiki +
								'|' + @resource.caption_of_article_of_wiki + ']]'
			end
			file.puts '| ' + @resource.table_column_attributes[1] + description_text
			file.puts '| ' + @resource.table_column_attributes[2] + language_entry.english_description
			if language_entry.code.eql?(@config.target_language) then
				wiki_main_page_link = @resource.target_wiki_main_page
			else
				wiki_main_page_link = ':' + language_entry.code + ':'
			end
			file.puts '| ' + @resource.table_column_attributes[3] +
				'[[' + wiki_main_page_link + '|' + language_entry.code + ']]'
			file.puts '| ' + @resource.table_column_attributes[4] +
				'[{{fullurl:' + language_entry.code + ':Special:Statistics|action=raw}} \'\'\'' +
				number_format.format(language_entry.number_of_articles) +'\'\'\']'
			file.puts '| ' + @resource.table_column_attributes[5] +
				number_format.format(language_entry.number_of_pages)
			file.puts '| ' + @resource.table_column_attributes[6] +
				'[[:' + language_entry.code + ':Special:Statistics|' + 
				number_format.format(language_entry.number_of_edits) + ']]'
			file.puts '| ' + @resource.table_column_attributes[7] +
				'[[:' + language_entry.code + ':Special:Listadmins|' + 
				number_format.format(language_entry.number_of_administrators) + ']]'
			file.puts '| ' + @resource.table_column_attributes[8] +
				'[[:' + language_entry.code + ':Special:Listusers|' + 
				number_format.format(language_entry.number_of_users) + ']]'
			file.puts '| ' + @resource.table_column_attributes[9] +
				'[[:' + language_entry.code + ':Special:Statistics|' + 
				number_format.format(language_entry.number_of_active_users) + ']]'
			file.puts '| ' + @resource.table_column_attributes[10] +
				'[[:' + language_entry.code + ':Special:ListFiles|' + 
				number_format.format(language_entry.number_of_files) + ']]'
			file.puts '| ' + @resource.table_column_attributes[11] +
				(language_entry.number_of_articles.to_i >=
					LanguageEntry.minimum_number_of_articles_for_calculating_depth ?
						language_entry.depth.to_s : ' - ')
			file.puts '| ' + @resource.table_column_attributes[12] + language_entry.score
			total_number_of_articles +=
				total_integer_value(language_entry.number_of_articles,
					'純記事数', language_entry, error_messages)
			total_number_of_pages +=
				total_integer_value(language_entry.number_of_pages,
					'総項目数', language_entry, error_messages)
			total_number_of_edits +=
				total_integer_value(language_entry.number_of_edits,
					'編集数', language_entry, error_messages)
			total_number_of_administrators +=
				total_integer_value(language_entry.number_of_administrators,
					'管理者数', language_entry, error_messages)
			total_number_of_users +=
				total_integer_value(language_entry.number_of_users,
					'登録者数', language_entry, error_messages)
			total_number_of_active_users +=
				total_integer_value(language_entry.number_of_active_users,
					'活動中の登録者数', language_entry, error_messages)
			total_number_of_files +=
				total_integer_value(language_entry.number_of_files,
					'ファイル数', language_entry, error_messages)
		end
		file.puts '|}'
		file.puts
		file.puts @resource.summary_table_header
		file.puts '|-'
		file.puts '| \'\'\'' + number_of_languages.to_s + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_articles) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_pages) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_edits) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_administrators) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_users) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_active_users) + '\'\'\''
		file.puts '| \'\'\'' + number_format.format(total_number_of_files) + '\'\'\''
		file.puts '|}'
		if error_messages.size > 0 then
			file.puts
			message = '一部のウィキペディアの統計情報で、マイナスの値が返されてきたものが'
			file.puts '(注記) ' + message + 'あった。'
			@logger.log(message + 'ありました。')
			message = '詳細は次のとおり。'
			file.puts message
			@logger.log(message)
			error_messages.each do |error_message|
				message = '* ' + error_message
				file.puts message
				@logger.log(message)
			end
		end
		file.puts @resource.footer
		file.flush
		file.close
	end
end

# Taker が作成し出力した更新用ウィキテキストで、全言語版の統計ページを更新します。
class Updater
	
	def initialize(config = nil)
		@config = config
	end
	
	STATISTICS_INFO_FILE_NAME = StatisticsFetcher::STATISTICS_INFO_FILE_NAME
	STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME =
		StatisticsFetcher::STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME
	STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME =
		StatisticsFetcher::STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME
	
	def execute
		time_started = Time.now.utc
		statistics_info_element = REXML::Document.new(File.new(STATISTICS_INFO_FILE_NAME), 'r').root
		if @config == nil then
			@config = Config.new(statistics_info_element.
					get_elements('//' + STATISTICS_INFO_WORKING_DIRECTORY_ELEMENT_NAME).first.text)
		end
		log_file_name = @config.working_directory + '/log.txt'
		logger = Logger.new([$stderr, File.new(log_file_name, 'a')])
		logger.log '統計ページの更新'
		date_time_format = DateTimeFormat.new()
		logger.log '開始しました。 ' + date_time_format.format(time_started)
		resource = Resource.new(logger)
		StatisticsFetcher::STATISTICS_INFO_FILE_NAME
		text = ''
		text += @config.edit_header
		File.new(NewStatisticsPage::FILE_NAME, 'r').readlines.each do |line|
			text += line
		end
		bot = WikiBot.new(@config)
		bot.login
		bot.obtain_edit_token(@config.edit_target_page)
		update_summary = statistics_info_element.
					get_elements('//' + STATISTICS_INFO_UPDATE_SUMMARY_ELEMENT_NAME).first.text
		logger.log '更新します。 - ' + @config.edit_target_page
		bot.edit(@config.edit_target_page, text, update_summary)
#		bot.edit(@config.edit_target_page, @config.edit_header, '砂場ならし')
		logger.log '更新しました。 - ' + update_summary
		bot.logout
		time_ended = Time.now.utc
		logger.log '終了しました。 ' + date_time_format.format(time_ended)
		logger.log '要した時間: ' + (time_ended - time_started).to_s + ' 秒'
		FileUtils.copy(log_file_name, '.')
	end
	
end

class WikiTable
	
	attr_reader :rows
	
	def initialize(current_statistics_page, resource, logger)
		header_is_not_ended = true
		tables_is_not_ended = true
		summary_table_header_is_not_ended = true
		footer_is_not_ended = true
		number_of_parsed_tables = 0
		tables = ''
		current_statistics_page.split(/\n/).each do |line|
			if header_is_not_ended then
				if line.index(Regexp.new(resource.table_Infos.first.label)) != nil then
					header_is_not_ended = false
				else
					next
				end
			end
			if tables_is_not_ended then
				tables += line + "\n"
				if line.index(/^\|\}/) != nil then
					number_of_parsed_tables += 1
					if number_of_parsed_tables >= resource.table_Infos.size() then
						tables = tables.lstrip()
						tables_is_not_ended = false
					end
				end
				next
			end
		end
		sequence_of_wiki_row = ''
		tables.split(/\n/).each do |line|
			if /\|.*/.match(line) != nil then
				sequence_of_wiki_row += (line + "\n")
			end
		end
		wiki_rows = sequence_of_wiki_row.split(/^\|-\s*\n/)
		@rows = Array.new
		wiki_rows.each do |wiki_row|
			wiki_fields = wiki_row.split("\n")
			match_data = /\[\[\:?(.+)\:?\|(.+)\]\]/.match(wiki_fields[3])
			if match_data == nil then
				next
			end
			code = match_data[2]
			field1 = field_value(wiki_fields[1])
			splitted_description = field1.split(/\s+-\s+/)
			if splitted_description.size == 2 then
				description = splitted_description[0]
				article_of_wiki = splitted_description[1].sub(/\[\[/, '').sub(/\|.+\]\]/, '')
			elsif splitted_description.size == 1 then
				description = splitted_description[0]
				article_of_wiki = ''
			else
				logger.log('Language description parse error: ' + field1)
				exit(1)
			end
			english_description = field_value(wiki_fields[2])
			score = field_value(wiki_fields[12])
			row = WikiTableRow.new(code, description, article_of_wiki, english_description, score)
			@rows.push(row)
		end
	end
	
	def field_value(field)
		regexp = /.+["']\s+\|\s+(.+)/
		match_data = regexp.match(field)
		result = ''
		if match_data != nil then
			result = match_data[1]
		else
			result = field.sub(/\|\s*/, '')
		end
		return result
	end
	
end

class WikiTableRow
	
	attr_reader :code, :description, :article_of_wiki, :english_description, :score
	
	def initialize(code, description, article_of_wiki, english_description, score)
		@code = code
		@description = description
		@english_description = english_description
		@article_of_wiki = article_of_wiki
		@score = score
	end
	
end

class Logger
	
	def initialize(error_outputs)
		@error_outputs = error_outputs
		@console_outputs = [$stdout, $stderr, STDOUT, STDERR]
	end
	
	def log(message)
		@error_outputs.each do |error_output|
			output_message = message
			if @console_outputs.include?(error_output) then
				output_message = Iconv.conv(Config.new.console_encoding, 'UTF-8', message)
			end
			error_output.puts output_message
			error_output.flush
		end
	end

end

class TableInfo
	
	attr_reader :label, :or_more
	attr_accessor :header_is_not_printed
	
	def initialize(label, or_more)
		@label = label
		@or_more = or_more
		@header_is_not_printed = true
	end
	
end

# リソースファイル <tt>resource.xml</tt> の内容を表現するクラスです。
class Resource
	
	attr_accessor :target_wiki_main_page
	attr_reader :update_summary
	attr_reader :caption_of_article_of_wiki
	attr_reader :target_language, :target_wiki_url, :target_statistics_page
	attr_reader :minimum_number_of_articles_for_calculating_depth
	attr_reader :table_Infos
	attr_reader :page_header, :table_header, :table_column_attributes
	attr_reader :summary_table_header, :footer
	attr_reader :languages
	
	FILE_NAME = 'resource.xml'
	
	def initialize(logger)
		resource_element = REXML::Document.new(File.new(FILE_NAME), 'r').root
		@update_summary = resource_element.get_elements('updateSummary').first.text
		@caption_of_article_of_wiki = resource_element.get_elements('captionOfarticleOfWiki').first.text
		@minimum_number_of_articles_for_calculating_depth =
			resource_element.get_elements('minimumNumberOfArticlesForCalculatingDepth').first.text.to_i
		@table_Infos = []
		resource_element.get_elements('tableInfos/tableLabel').each do |label_element|
			@table_Infos.push(
				TableInfo.new(label_element.text,
				label_element.attribute('orMore').to_s.to_i))
		end
		@page_header = resource_element.get_elements('pageHeader').first.text
		@table_header = resource_element.get_elements('tableHeader').first.text
		@table_column_attributes = []
		attributes_element = resource_element.get_elements('tableColumnAttributes').first
		@table_column_attributes.push(attributes_element.get_elements('rank').first.text)
		@table_column_attributes.push(attributes_element.get_elements('description').first.text)
		@table_column_attributes.push(attributes_element.get_elements('englishDescription').first.text)
		@table_column_attributes.push(attributes_element.get_elements('link').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfArticles').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfPages').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfEdits').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfAdministrators').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfUsers').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfActiveUsers').first.text)
		@table_column_attributes.push(attributes_element.get_elements('numberOfFiles').first.text)
		@table_column_attributes.push(attributes_element.get_elements('depth').first.text)
		@table_column_attributes.push(attributes_element.get_elements('score').first.text)
		@table_column_attributes.size.times do |i|
			if @table_column_attributes[i] == nil then
				@table_column_attributes[i] = ''
			end
		end
		@summary_table_header = resource_element.get_elements('summaryTableHeader').first.text
		@footer = resource_element.get_elements('footer').first.text
		@languages = Hash.new
		resource_element.get_elements('languages').first.get_elements('language').each do |language_element|
			language_element = LanguageOfResource.new(language_element)
			if (@languages.has_key?(language_element.code)) then
				logger.log('リソースファイル中の言語コード "' + language_element.code + '" が重複しています。中断します。')
				exit(1)
			end
			@languages.store(language_element.code, language_element)
		end
	end
end

class LanguageOfResource
	
	attr_reader :code, :description, :article_of_wiki, :english_description, :score
	
	def initialize(element)
		@code = element.get_elements('code').first.text
		@description = element.get_elements('description').first.text
		@english_description = element.get_elements('englishDescription').first.text
		@score = element.get_elements('score').first.text
		article_title_element = element.get_elements('articleOfWiki').first
		if article_title_element == nil then
			@article_of_wiki = nil
		else
			@article_of_wiki = article_title_element.text
		end
	end
	
end

class LanguageEntry
	
	include Comparable
	
	attr_reader :code, :name, :wiki_url, :file_name
	attr_accessor :description, :article_of_wiki, :english_description
	attr_accessor :number_of_articles, :number_of_pages, :number_of_edits
	attr_accessor :number_of_administrators, :number_of_users, :number_of_active_users
	attr_accessor :number_of_files, :score
	attr_reader :minimum_number_of_articles_for_calculating_depth
	
	@@minimum_number_of_articles_for_calculating_depth = nil
	
	def LanguageEntry.minimum_number_of_articles_for_calculating_depth
		return @@minimum_number_of_articles_for_calculating_depth
	end
	
	def LanguageEntry.minimum_number_of_articles_for_calculating_depth=(minimum_number)
		@@minimum_number_of_articles_for_calculating_depth = minimum_number
	end
	
	def initialize(code, name, wiki_url, file_name)
		@code = code
		@name = name
		@wiki_url = wiki_url
		@file_name = file_name
		@description = name.capitalize
		@english_description = nil
		@score = nil
	end
	
	def <=>(other)
		# 大小比較 (ソート) の基準は、純記事数、総項目数、総編集数
		if @number_of_articles.to_i != other.number_of_articles.to_i then
			return @number_of_articles.to_i <=> other.number_of_articles.to_i
		elsif @number_of_pages.to_i != other.number_of_pages.to_i then
			return @number_of_pages.to_i <=> other.number_of_pages.to_i
		else
			return @number_of_edits.to_i <=> other.number_of_edits.to_i
		end
	end
	
	def depth()
		if @number_of_articles < @@minimum_number_of_articles_for_calculating_depth then
			return nil
		end
		# (edits / pages) * (((pages - articles) / articles) ** 2)
		return (((number_of_edits.to_f) / (number_of_pages.to_f)) *
				(((number_of_pages - number_of_articles).to_f / number_of_articles.to_f) ** 2)).to_i
	end
	
end

class LanguageOfSiteMatrix
	
	attr_reader :code, :name, :wiki_url
	
	CODE_PREDICATE = '@code="wiki"'
		
	def initialize(element)
		@code = element.attribute('code').to_s
		@name = element.attribute('name').to_s
		@wiki_url = element.get_elements(
					'site[' + CODE_PREDICATE + ']').first.attribute('url').to_s
	end
	
end

class DateTimeFormat
	
	attr_reader :date_time, :year, :month, :day, :hour, :minute, :second, :timezone
	
	def initialize(pattern = '${year}-${month}-${day} ${hour}:${minute}:${second} (${timezone})')
		@pattern = pattern
		@regexp = Regexp.new(@pattern.
						gsub('${', '(${').
						gsub('}', '})').
						sub('${year}', '\d{1,4}').
						sub('${month}', '\d{1,2}').
						sub('${day}', '\d{1,2}').
						sub('${hour}', '\d{2}').
						sub('${minute}', '\d{2}').
						sub('${second}', '\d{2}').
						sub('${timezone}', '.*'))
	end
	
	def set_date_time(date_time)
		@date_time = date_time
		@year = @date_time.year.to_s
		length = 2
		@month = @date_time.month.to_s.rjust(length, '0')
		@day = @date_time.day.to_s.rjust(length, '0')
		@hour = @date_time.hour.to_s.rjust(length, '0')
		@minute = @date_time.min.to_s.rjust(length, '0')
		@second = @date_time.sec.to_s.rjust(length, '0')
		@timezone = @date_time.zone
	end

	def format(date_time = @date_time)
		set_date_time(date_time)
		@pattern.
			sub('${year}', @year).
			sub('${month}', @month).
			sub('${day}', @day).
			sub('${hour}', @hour).
			sub('${minute}', @minute).
			sub('${second}', @second).
			sub('${timezone}', @timezone)
	end
	
	def parse(string)
		match_data = string.match(@regexp)
		set_date_time(Time.gm(
					match_data[1].to_i,
					match_data[2].to_i,
					match_data[3].to_i,
					match_data[4].to_i,
					match_data[5].to_i,
					match_data[6].to_i))
	end
	
end

class NumberFormat
	
	def initialize
		@regexp = /(\d)(?=(?:\d\d\d)+(?!\d))/
	end
	
	def format(number)
		if number == nil then
			return '-'
		end
		if number < 0 then
			return '0'
		end
		return number.to_s.gsub(@regexp, '\1,')
	end
	
end

