/*
  VA
  Satofumi KAMIMURA
  $Id$
*/

#include "serial_device.h"
#include "ringBuffer.h"

#if defined _MBCS || defined WIN32
#define WINDOWS
#else
#define LINUX
#endif

#include <stdio.h>
#ifndef WINDOWS
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <termios.h>
#else
#include <windows.h>
#endif


enum {
  C_FALSE = 0, C_TRUE = 1,
  RING_SIZE_SHIFT = 11,
};


#ifdef LINUX
static int fd = -1;
static struct termios sio;
static struct pollfd nfds;
#else
static HANDLE hComm;
#endif
static ringBuffer_t Ring;
static unsigned char Buffer[1 << RING_SIZE_SHIFT];


static void flush(void) {
#ifdef LINUX
  if (serial_is_connect()) {
    tcflush(fd, TCIOFLUSH);
    clearRingBuffer(&Ring);
  }
#endif
}


int serial_set_baudrate(long baudrate) {
#ifdef LINUX
  long baudrate_value;
#else
  DCB dcb;
#endif
  if (!serial_is_connect()) {
    return NO_CONNECT;
  }

#ifdef LINUX  
  // {[[g̕ύX
  switch (baudrate) {

  case 9600:
    baudrate_value = B9600;
    break;
    
  case 19200:
    baudrate_value = B19200;
    break;

  case 38400:
    baudrate_value = B38400;
    break;
    
  case 57600:
    baudrate_value = B57600;
    break;
    
  case 115200:
    baudrate_value = B115200;
    break;
    
  default:
    return BAUDRATE_ADJUST_ERROR;
    break;
  }

  cfsetispeed(&sio, baudrate_value);
  cfsetospeed(&sio, baudrate_value);
  tcsetattr(fd, TCSANOW, &sio);
#else
  
  GetCommState(hComm, &dcb);
  dcb.BaudRate = baudrate;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.fParity = FALSE;
  dcb.StopBits = ONESTOPBIT;
  SetCommState(hComm, &dcb);
#endif
  
  return 0;
}


int serial_open(const char *devName, long baudrate) {
  int ret_value;
#ifdef WINDOWS
  char comPort[16];
#endif
  
  initRingBuffer(&Ring, Buffer, RING_SIZE_SHIFT);

  if (serial_is_connect()) {
    serial_close();
  }

#ifdef LINUX
  fd = open(devName, O_RDWR | O_SYNC | O_NOCTTY);
  if (fd < 0) {
    perror(devName);
    return DEVICE_OPEN_ERROR;
  }

  sio.c_iflag = IGNPAR;
  sio.c_oflag = 0;
  sio.c_cflag = CS8 | CREAD | CLOCAL;
  sio.c_lflag = 0;
  
  sio.c_cc[VMIN] = 0;
  sio.c_cc[VTIME] = 0;

  tcsetattr(fd, TCSANOW, &sio);

  /* ^CAEgݒ */
  nfds.fd = fd;
  nfds.events = POLLIN | POLLPRI | POLLERR | POLLHUP | POLLNVAL;
  nfds.revents = 0;
  
#else
  sprintf(comPort, "\\\\.\\%s", devName);
  hComm = CreateFile(comPort, GENERIC_READ | GENERIC_WRITE, 0, NULL,
		     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hComm == INVALID_HANDLE_VALUE) {
    LPVOID lpMsg;
    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
		  FORMAT_MESSAGE_FROM_SYSTEM |
		  FORMAT_MESSAGE_IGNORE_INSERTS,
		  NULL, GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		  (LPTSTR)&lpMsg, 0, NULL);
    printf("port(%s) open failed: %s\n", devName, (char *)lpMsg);
    LocalFree(lpMsg);
    return DEVICE_OPEN_ERROR;
  }
#endif

  /* {[[g̐ݒ */
  ret_value = serial_set_baudrate(baudrate);
  if (ret_value < 0) {
    return ret_value;
  }

  flush();

  return 0;
}


void serial_close(void) {
#ifdef LINUX
  if (fd > 0) {
    close(fd);
    fd = -1;
  }
#else
  if (hComm != INVALID_HANDLE_VALUE) {
    CloseHandle(hComm);
  }
#endif
}


int serial_is_connect(void) {
#ifdef LINUX
  return (fd > 0) ? C_TRUE : C_FALSE;
#else
  return (hComm != INVALID_HANDLE_VALUE) ? C_TRUE : C_FALSE;
#endif
}


static void check(int size, int timeout) {
  int require_size;
  int filled;
  unsigned char buffer[BUFSIZ];

#ifdef LINUX
  int n;
#else
  DWORD n;
  int read_size;
#endif

  if (!serial_is_connect()) {
    return;
  }

  require_size = (size < 0) ? getRingFreeSize(&Ring) : size;
  require_size = (require_size > BUFSIZ) ? BUFSIZ : require_size;
  filled = getRingFillSize(&Ring);

#ifdef LINUX
  while (filled < require_size) {
    if (poll(&nfds, 1, timeout) == 0) {
      break;                    // timeout
    }
    n = read(fd, buffer, require_size);
    if (n < 0) {
      return;
    }
    putRingBufferStr(&Ring, buffer, n);
    filled += n;
  }
#else
  if (filled >= require_size) {
    return;
  }

  read_size = require_size;
  if (timeout > 0) {
    COMMTIMEOUTS pcto;
    GetCommTimeouts(hComm, &pcto);
    pcto.ReadIntervalTimeout = timeout;
    pcto.ReadTotalTimeoutConstant = SCI_TIMEOUT;
    pcto.ReadTotalTimeoutMultiplier = read_size;
    SetCommTimeouts(hComm, &pcto);
    
  } else {
    DWORD dwErrors;
    COMSTAT ComStat;
    ClearCommError(hComm, &dwErrors, &ComStat);
    read_size = ((int)ComStat.cbInQue > require_size)
      ? require_size : ComStat.cbInQue;
  }
  ReadFile(hComm, buffer, read_size, &n, NULL);
  putRingBufferStr(&Ring, buffer, n);
#endif
}

int serial_recv(unsigned char *data, int size, int timeout) {
  int len;
  
  if (!serial_is_connect()) {
    return NO_CONNECT;
  }
  
  check(size, timeout);
  len = (getRingFillSize(&Ring) > size) ? size : getRingFillSize(&Ring);
  if (len == 0) {
    return 0;
  }
  getRingBufferStr(&Ring, data, len);
  
  return len;
}


int serial_send(const unsigned char *data, int length) {
#ifdef WINDOWS
  DWORD n;
#endif
  if (!serial_is_connect()) {
    return NO_CONNECT;
  }
#ifdef LINUX
  return write(fd, data, length);
#else
  WriteFile(hComm, data, length, &n, NULL);
  return n;
#endif
}


void serial_flush(void) {
#ifdef LINUX
  if (serial_is_connect()) {
    tcflush(fd, TCIOFLUSH);
    clearRingBuffer(&Ring);
  }
#endif
}
