/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "aviextra.h"


static void avi_extra_class_init (AviExtraClass *klass);
static void avi_extra_init       (AviExtra      *extra);
#if GTK_CHECK_VERSION(2,2,0)
static void avi_extra_dispose    (GObject       *object);
#endif /* GTK_CHECK_VERSION(2,2,0) */


static GtkVBoxClass *parent_class = NULL;


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
avi_extra_get_type (void)
{
  static GType extra_type = 0;

  if (!extra_type)
    {
      const static GTypeInfo extra_info =
      {
        sizeof (AviExtraClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)avi_extra_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (AviExtra),
        0,                  /* n_preallocs */
        (GInstanceInitFunc)avi_extra_init,
      };

      extra_type = g_type_register_static (GTK_TYPE_VBOX, "AviExtra",
                                                            &extra_info, 0);
    }

  return extra_type;
}


static void
avi_extra_class_init (AviExtraClass *klass)
{
#if GTK_CHECK_VERSION(2,2,0)
  GObjectClass *object_class;

  object_class = (GObjectClass *)klass;
#endif /* GTK_CHECK_VERSION(2,2,0) */

  parent_class = g_type_class_peek_parent (klass);
#if GTK_CHECK_VERSION(2,2,0)
  object_class->dispose = avi_extra_dispose;
#endif /* GTK_CHECK_VERSION(2,2,0) */
}


static void
avi_extra_init (AviExtra *extra)
{
  extra->avi_edit = NULL;
  extra->video = 0;
  extra->audio = 0;
#if GTK_CHECK_VERSION(2,2,0)
  extra->combo = NULL;
#endif /* GTK_CHECK_VERSION(2,2,0) */
  extra->frame = NULL;
  extra->notebook = NULL;
  extra->spin_start = NULL;
  extra->spin_end = NULL;
  extra->radio_avi = NULL;
  extra->radio_gsr = NULL;
  extra->radio_bitmap = NULL;
#if GTK_CHECK_VERSION(2,2,0)
  extra->radio_pixbuf = NULL;
#endif /* GTK_CHECK_VERSION(2,2,0) */
  extra->radio_wave = NULL;
  extra->radio_scenario = NULL;
#if GTK_CHECK_VERSION(2,2,0)
  extra->hbox_current = NULL;
  extra->hbox_ico = NULL;
  extra->hbox_jpeg = NULL;
  extra->hbox_png = NULL;
  extra->check_ico_depth = NULL;
  extra->check_ico_cursor = NULL;
  extra->radio_ico_depth16 = NULL;
  extra->radio_ico_depth24 = NULL;
  extra->radio_ico_depth32 = NULL;
  extra->label_ico_x = NULL;
  extra->label_ico_y = NULL;
  extra->spin_ico_x = NULL;
  extra->spin_ico_y = NULL;
  extra->check_jpeg = NULL;
  extra->spin_jpeg = NULL;
  extra->check_png = NULL;
  extra->entry_png = NULL;
  extra->spin_png = NULL;
#endif /* not GTK_CHECK_VERSION(2,2,0) */
}


#if GTK_CHECK_VERSION(2,2,0)
static void
avi_extra_dispose (GObject *object)
{
  AviExtra *extra = AVI_EXTRA (object);

  if (extra->hbox_ico)
    gtk_widget_destroy (extra->hbox_ico);
  if (extra->hbox_jpeg)
    gtk_widget_destroy (extra->hbox_jpeg);
  if (extra->hbox_png)
    gtk_widget_destroy (extra->hbox_png);

  if (G_OBJECT_CLASS (parent_class)->dispose)
    G_OBJECT_CLASS (parent_class)->dispose (object);
}
#endif /* GTK_CHECK_VERSION(2,2,0) */


/******************************************************************************
*                                                                             *
* ja:ダイアログ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
avi_extra_new (void)
{
  return GTK_WIDGET (g_object_new (AVI_TYPE_EXTRA, NULL));
}


/*  ja:AVI編集ハンドルを取得する
    extra,ウィジェット
      RET,AVI編集ハンドル,NULL:設定なし                                     */
AviEdit **
avi_extra_get_handle (AviExtra *extra)
{
  return extra ? extra->avi_edit : NULL;
}


/******************************************************************************
*                                                                             *
*                                                                             *
*                                                                             *
******************************************************************************/
struct _AviExtraStream
{
  AviEdit *avi_edit;                        /* ja:AVI編集構造体 */
  GtkWidget *button;                        /* ja:設定 */
  GtkWidget *check_null;                    /* ja:ヌルフレーム */
  GtkWidget *check_recomp;                  /* ja:再圧縮 */
  GtkWidget *combo;                         /* en:Codec */
  GtkWidget *label_quality, *spin_quality;  /* ja:品質 */
  GtkWidget *label_key, *spin_key;          /* ja:キーフレーム */
  GtkWidget *label_rate, *spin_rate;        /* ja:データレート */
};
typedef struct _AviExtraStream AviExtraStream;


/******************************************************************************
*                                                                             *
******************************************************************************/
/* ja:ラジオボタン(AVI)が押された */
static void
avi_extra_toggled_avi (GtkWidget *widget,
                       AviExtra  *extra)
{
  if (extra->frame)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->frame), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), TRUE);
  if (GTK_IS_NOTEBOOK (extra->notebook))
    {
      gint i;
      GList *glist;

      glist = gtk_container_get_children (GTK_CONTAINER (extra->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        TRUE);
      g_list_free (glist);
    }
}


/* ja:ラジオボタン(ビットマップ)が押された */
static void
avi_extra_toggled_bitmap (GtkWidget *widget,
                          AviExtra  *extra)
{
  gtk_widget_set_sensitive (GTK_WIDGET (extra->frame), TRUE);
#if GTK_CHECK_VERSION(2,2,0)
  if (extra->combo)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->combo), FALSE);
  if (extra->hbox_current)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->hbox_current), FALSE);
#endif /* GTK_CHECK_VERSION(2,2,0) */
  if (extra->video > 1)
    {
      gint i;
      GList *glist;

      gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), TRUE);
      glist = gtk_container_get_children (GTK_CONTAINER (extra->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        FALSE);
      g_list_free (glist);
    }
  else
    {
      gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), FALSE);
    }
}


#if GTK_CHECK_VERSION(2,2,0)
/* ja:ラジオボタン(GdkPixbuf)が押された */
static void
avi_extra_toggled_pixbuf (GtkWidget *widget,
                          AviExtra  *extra)
{
  gtk_widget_set_sensitive (GTK_WIDGET (extra->frame), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (extra->combo), TRUE);
  if (extra->hbox_ico)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->hbox_ico), TRUE);
  if (extra->hbox_jpeg)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->hbox_jpeg), TRUE);
  if (extra->hbox_png)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->hbox_png), TRUE);
  if (extra->video > 1)
    {
      gint i;
      GList *glist;

      gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), TRUE);
      glist = gtk_container_get_children (GTK_CONTAINER (extra->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        FALSE);
      g_list_free (glist);
    }
  else
    {
      gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), FALSE);
    }
}
#endif /* GTK_CHECK_VERSION(2,2,0) */


/* ja:ラジオボタン(WAVE)が押された */
static void
avi_extra_toggled_wave (GtkWidget *widget,
                        AviExtra  *extra)
{
  if (extra->frame)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->frame), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), TRUE);
  if (GTK_IS_NOTEBOOK (extra->notebook))
    {
      gint i;
      GList *glist;

      glist = gtk_container_get_children (GTK_CONTAINER (extra->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          GtkWidget *child;
          AviExtraStream *avi_stream;

          child = g_list_nth_data (glist, i);
          avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
          gtk_widget_set_sensitive (GTK_WIDGET (child),
                    avi_edit_type (avi_stream->avi_edit) == AVI_TYPE_AUDIO);
        }
      g_list_free (glist);
    }
  else
    {
      gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), TRUE);
    }
}


/* ja:ラジオボタン(シナリオオブジェクト)が押された */
static void
avi_extra_toggled_scenario (GtkWidget *widget,
                            AviExtra  *extra)
{
  if (extra->frame)
    gtk_widget_set_sensitive (GTK_WIDGET (extra->frame), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (extra->notebook), FALSE);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/* ja:ラジオボタン(ビデオストリーム)が押された */
static void
avi_extra_toggled_video (GtkWidget *widget,
                         AviExtra  *extra)
{
  gint max;
  AviEdit *avi_edit;

  avi_edit = g_object_get_data (G_OBJECT (widget), "user_data");
  max = avi_edit_length (avi_edit) - 1;
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (extra->spin_start), 0, max);
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (extra->spin_end), 0, max);
#if GTK_CHECK_VERSION(2,2,0)
  if (extra->spin_ico_x)
    gtk_spin_button_set_range (GTK_SPIN_BUTTON (extra->spin_ico_x),
                                        0, avi_edit_get_width (avi_edit) - 1);
  if (extra->spin_ico_y)
    gtk_spin_button_set_range (GTK_SPIN_BUTTON (extra->spin_ico_y),
                                        0, avi_edit_get_height (avi_edit) - 1);
#endif /* GTK_CHECK_VERSION(2,2,0) */
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/* ja:コンボボックスの内容が変わったとき(ビデオ) */
static void
avi_extra_changed_video (GtkWidget      *widget,
                         AviExtraStream *avi_stream)
{
  gint quality, key_frame;
#if GTK_CHECK_VERSION(2,4,0)
  GtkTreeIter iter;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  const gchar *text;
  gint i;
  GList *glist;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  IcmObject *icm_object = NULL;

#if GTK_CHECK_VERSION(2,4,0)
  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (avi_stream->combo), &iter))
    gtk_tree_model_get
        (gtk_combo_box_get_model (GTK_COMBO_BOX (avi_stream->combo)), &iter,
                                                        1, &icm_object, -1);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  text = gtk_entry_get_text (GTK_ENTRY (widget));
  glist = gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      const gchar *label;

      label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                    (GTK_BIN (g_list_nth_data (glist, i)))));
      if (g_strcmp (label, text) == 0)
        {
          icm_object = g_object_get_data
                        (G_OBJECT (g_list_nth_data (glist, i)), "user_data");
          break;
        }
    }
  g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_object_set_data (G_OBJECT (avi_stream->combo), "user_data", icm_object);

  quality = icm_object ? icm_compress_get_default_quality (icm_object) : -1;
  key_frame = icm_object ? icm_compress_get_default_key_frame (icm_object) : 0;
  /* ja:スピンボタン */
  if (avi_stream->spin_quality)
    {
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label_quality),
                                                                quality >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin_quality),
                                                                quality >= 0);
      if (quality >= 0)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (avi_stream->spin_quality),
                                                                    quality);
    }
  if (avi_stream->spin_key)
    {
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label_key),
                                                                key_frame > 0);
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin_key),
                                                                key_frame > 0);
      if (key_frame > 0)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (avi_stream->spin_key),
                                                                    key_frame);
    }
  if (avi_stream->spin_rate)
    {
      if (icm_object && (icm_get_flags (icm_object) & ICM_FLAG_CRUNCH))
        {
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label_rate), TRUE);
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin_rate), TRUE);
        }
      else
        {
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label_rate),
                                                                        FALSE);
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin_rate), FALSE);
        }
    }
  /* ja:チェックボタン */
  if (avi_stream->check_recomp)
    gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->check_recomp),
                                                        icm_object != NULL);
  /* ja:ボタン */
  if (avi_stream->button)
    gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->button),
                        icm_object && icm_is_dialog_configure (icm_object));
}


/* ja:設定ボタンが押された(ビデオ) */
static void
avi_extra_clicked_config_video (GtkWidget *widget,
                                GtkWidget *combo)
{
  icm_dialog_configure (g_object_get_data (G_OBJECT (combo), "user_data"),
                                            gtk_widget_get_toplevel (widget));
}


/* ja:コンボボックスの内容が変わったとき(オーディオ) */
static void
avi_extra_changed_audio (GtkWidget      *widget,
                         AviExtraStream *avi_stream)
{
#if GTK_CHECK_VERSION(2,4,0)
  GtkTreeIter iter;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  const gchar *text;
  gint i;
  GList *glist;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  WaveFormatEx *wfx = NULL;

#if GTK_CHECK_VERSION(2,4,0)
  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (avi_stream->combo), &iter))
    gtk_tree_model_get
        (gtk_combo_box_get_model (GTK_COMBO_BOX (avi_stream->combo)), &iter,
                                                                2, &wfx, -1);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  text = gtk_entry_get_text (GTK_ENTRY (widget));
  glist = gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      const gchar *label;

      label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                    (GTK_BIN (g_list_nth_data (glist, i)))));
      if (g_strcmp (label, text) == 0)
        {
          wfx = g_object_get_data
                        (G_OBJECT (g_list_nth_data (glist, i)), "user_data");
          break;
        }
    }
  g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_object_set_data (G_OBJECT (avi_stream->combo), "user_data", wfx);
  /* ja:チェックボタン */
  if (avi_stream->check_recomp)
    gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->check_recomp),
                                                                wfx != NULL);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/* ja:ページの破棄 */
static void
avi_extra_destroy_vbox (GtkWidget      *widget,
                        AviExtraStream *avi_stream)
{
  g_free (avi_stream);
}


#if GTK_CHECK_VERSION(2,4,0)
/* ja:コンボボックスの破棄(ビデオ) */
static void
avi_extra_destroy_combo_video (GtkWidget *widget,
                               gpointer   user_data)
{
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (widget));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    {
      IcmObject *icm_object;

      gtk_tree_model_get (model, &iter, 1, &icm_object, -1);
      icm_close (icm_object);
    }
}


/* ja:コンボボックスの破棄(オーディオ) */
static void
avi_extra_destroy_combo_audio (GtkWidget *widget,
                               gpointer   user_data)
{
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (widget));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    {
      WaveFormatEx *wfx;

      gtk_tree_model_get (model, &iter, 2, &wfx, -1);
      g_free (wfx);
    }
}
#else /* not GTK_CHECK_VERSION(2,4,0) */
/* ja:コンボボックスの破棄(ビデオ) */
static void
avi_extra_destroy_list_video (GtkWidget *widget,
                              gpointer   user_data)
{
  gint i;
  GList *glist;

  glist = gtk_container_get_children (GTK_CONTAINER (widget));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    icm_close (g_object_get_data (G_OBJECT (g_list_nth_data (glist, i)),
                                                                "user_data"));
  g_list_free (glist);
}


/* ja:コンボボックスの破棄(オーディオ) */
static void
avi_extra_destroy_list_audio (GtkWidget *widget,
                              gpointer   user_data)
{
  gint i;
  GList *glist;

  glist = gtk_container_get_children (GTK_CONTAINER (widget));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    g_free (g_object_get_data (G_OBJECT (g_list_nth_data (glist, i)),
                                                                "user_data"));
  g_list_free (glist);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


/******************************************************************************
*                                                                             *
******************************************************************************/
#if GTK_CHECK_VERSION(2,2,0)
/* ja:コンボボックスの内容が変わったとき(GdkPixbuf) */
static void
avi_extra_changed_pixbuf (GtkWidget *widget,
                          AviExtra  *extra)
{
# if GTK_CHECK_VERSION(2,4,0)
  GtkTreeIter iter;
# else /* not GTK_CHECK_VERSION(2,4,0) */
  const gchar *text;
  gint i;
  GList *glist;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  GdkPixbufFormat *format = NULL;
  GtkWidget *vbox;

# if GTK_CHECK_VERSION(2,4,0)
  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (extra->combo), &iter))
    gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (extra->combo)),
                                                        &iter, 1, &format, -1);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  text = gtk_entry_get_text (GTK_ENTRY (widget));
  glist = gtk_container_get_children
                            (GTK_CONTAINER (GTK_COMBO (extra->combo)->list));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      const gchar *label;

      label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                    (GTK_BIN (g_list_nth_data (glist, i)))));
      if (g_strcmp (label, text) == 0)
        {
          format = g_object_get_data
                        (G_OBJECT (g_list_nth_data (glist, i)), "user_data");
          break;
        }
    }
  g_list_free (glist);
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  vbox = gtk_widget_get_parent (gtk_widget_get_parent (extra->combo));
  if (extra->hbox_current)
    {
      gtk_container_remove (GTK_CONTAINER (vbox), extra->hbox_current);
      extra->hbox_current = NULL;
    }
  if (format)
    {
      gchar *name;

      name = gdk_pixbuf_format_get_name (format);
      if (extra->hbox_ico && g_strcmp (name, "ico") == 0)
        extra->hbox_current = extra->hbox_ico;
      else if (extra->hbox_jpeg && g_strcmp (name, "jpeg") == 0)
        extra->hbox_current = extra->hbox_jpeg;
      else if (extra->hbox_png && g_strcmp (name, "png") == 0)
        extra->hbox_current = extra->hbox_png;
      if (extra->hbox_current)
        {
          g_object_ref (extra->hbox_current);
          gtk_box_pack_start (GTK_BOX (vbox),
                                        extra->hbox_current, FALSE, FALSE, 0);
        }
      g_free (name);
    }
}


/* ja:チェックボタン(ico,色)が押された */
static void
avi_extra_toggled_ico_depth (GtkWidget *widget,
                             AviExtra  *extra)
{
  gboolean sensitive;

  sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
  gtk_widget_set_sensitive (extra->radio_ico_depth16, sensitive);
  gtk_widget_set_sensitive (extra->radio_ico_depth24, sensitive);
  gtk_widget_set_sensitive (extra->radio_ico_depth32, sensitive);
}


/* ja:チェックボタン(ico,カーソル)が押された */
static void
avi_extra_toggled_ico_cursor (GtkWidget *widget,
                              AviExtra  *extra)
{
  gboolean sensitive;

  sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
  gtk_widget_set_sensitive (extra->label_ico_x, sensitive);
  gtk_widget_set_sensitive (extra->label_ico_y, sensitive);
  gtk_widget_set_sensitive (extra->spin_ico_x, sensitive);
  gtk_widget_set_sensitive (extra->spin_ico_y, sensitive);
}


/* ja:チェックボタン(jpeg)が押された */
static void
avi_extra_toggled_jpeg (GtkWidget *widget,
                        AviExtra  *extra)
{
  gtk_widget_set_sensitive (extra->spin_jpeg,
                    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)));
}


/* ja:チェックボタン(png)が押された */
static void
avi_extra_toggled_png (GtkWidget *widget,
                       AviExtra  *extra)
{
  gtk_widget_set_sensitive (extra->spin_png,
                    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)));
}
#endif /* GTK_CHECK_VERSION(2,2,0) */


/******************************************************************************
*                                                                             *
******************************************************************************/
static gint
avi_extra_set_handle_sort (GdkPixbufFormat *format_a,
                           GdkPixbufFormat *format_b)
{
  gchar *desc_a, *desc_b;
  gint result;

  desc_a = gdk_pixbuf_format_get_description (format_a);
  desc_b = gdk_pixbuf_format_get_description (format_b);
  result = g_strcmp (desc_a, desc_b);
  g_free (desc_a);
  g_free (desc_b);
  return result;
}


/*  ja:AVI編集ハンドルを設定する
       extra,ウィジェット
    avi_edit,AVI編集ハンドル,NULL:設定なし                                  */
void
avi_extra_set_handle (AviExtra  *extra,
                      AviEdit  **avi_edit)
{
  gint i, streams, video = 0, audio = 0, leng_handler, leng_tag;
  GList *glist;
  GSList *gslist;           /* ja:ラジオボタン */
  GList *glist_handler;     /* ja:登録されている4文字コード */
  GList *glist_tag;         /* ja:登録されているタグ */
  GSList *gslist_v = NULL;  /* ja:ラジオボタン */
  GSList *gslist_a = NULL;  /* ja:ラジオボタン */
  GtkWidget *label0;
  GtkWidget *frame, *hbox0;

  if (!extra)
    return;
  glist = gtk_container_get_children (GTK_CONTAINER (extra));
  if (glist)
    {
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_container_remove (GTK_CONTAINER (extra),
                                    GTK_WIDGET (g_list_nth_data (glist, i)));
      g_list_free (glist);
    }
#if GTK_CHECK_VERSION(2,2,0)
  if (extra->hbox_ico)
    gtk_widget_destroy (extra->hbox_ico);
  if (extra->hbox_jpeg)
    gtk_widget_destroy (extra->hbox_jpeg);
  if (extra->hbox_png)
    gtk_widget_destroy (extra->hbox_png);
#endif /* GTK_CHECK_VERSION(2,2,0) */
  extra->avi_edit = avi_edit;
  extra->video = 0;
  extra->audio = 0;
#if GTK_CHECK_VERSION(2,2,0)
  extra->combo = NULL;
#endif /* GTK_CHECK_VERSION(2,2,0) */
  extra->frame = NULL;
  extra->notebook = NULL;
  extra->spin_start = NULL;
  extra->spin_end = NULL;
  extra->radio_avi = NULL;
  extra->radio_gsr = NULL;
  extra->radio_bitmap = NULL;
#if GTK_CHECK_VERSION(2,2,0)
  extra->radio_pixbuf = NULL;
#endif /* GTK_CHECK_VERSION(2,2,0) */
  extra->radio_wave = NULL;
  extra->radio_scenario = NULL;
#if GTK_CHECK_VERSION(2,2,0)
  extra->hbox_current = NULL;
  extra->hbox_ico = NULL;
  extra->hbox_jpeg = NULL;
  extra->hbox_png = NULL;
  extra->check_ico_depth = NULL;
  extra->check_ico_cursor = NULL;
  extra->radio_ico_depth16 = NULL;
  extra->radio_ico_depth24 = NULL;
  extra->radio_ico_depth32 = NULL;
  extra->label_ico_x = NULL;
  extra->label_ico_y = NULL;
  extra->spin_ico_x = NULL;
  extra->spin_ico_y = NULL;
  extra->check_jpeg = NULL;
  extra->spin_jpeg = NULL;
  extra->check_png = NULL;
  extra->entry_png = NULL;
  extra->spin_png = NULL;
#endif /* not GTK_CHECK_VERSION(2,2,0) */
  if (!avi_edit)
    return;
  for (streams = 0; avi_edit[streams]; streams++)
    switch (avi_edit_type (avi_edit [streams]))
      {
        case AVI_TYPE_VIDEO: video++; break;
        case AVI_TYPE_AUDIO: audio++; break;
        default: return;
      }
  if (streams <= 0)
    return;
  extra->video = video;
  extra->audio = audio;
  /* ja:ストリーム */
  if (streams > 1)
    {
      /* ja:ノートブック */
      extra->notebook = gtk_notebook_new ();
      gtk_notebook_set_scrollable (GTK_NOTEBOOK (extra->notebook), TRUE);
    }
  else
    {
      /* ja:フレーム */
      extra->notebook = gtk_frame_new ("");
    }
  video = audio = 0;
  glist_handler = icm_get_handler_list ();
  leng_handler = g_list_length (glist_handler);
  glist_tag = acm_get_tag_list ();
  leng_tag = g_list_length (glist_tag);
  for (i = 0; i < streams; i++)
    {
      gchar *text;
      GtkWidget *vbox;
      AviExtraStream *avi_stream;

      avi_stream = g_malloc0 (sizeof (AviExtraStream));
      avi_stream->avi_edit = avi_edit[i];
      vbox = gtk_vbox_new (FALSE, SPACING);
      gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
      g_object_set_data (G_OBJECT (vbox), "user_data", avi_stream);
      g_signal_connect (G_OBJECT (vbox), "destroy",
                            G_CALLBACK (avi_extra_destroy_vbox), avi_stream);
      if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
        { /* ja:ビデオストリーム */
          gint j;
          GList *glist_object;      /* ja:IcmObjectのリスト */
          GList *glist_name;        /* ja:名前のリスト */

          /* ja:チェックボタン */
          avi_stream->check_null = gtk_check_button_new_with_mnemonic
                                                            (_("_Null Frame"));
          gtk_toggle_button_set_active
                            (GTK_TOGGLE_BUTTON (avi_stream->check_null), TRUE);
          /* ja:未圧縮 */
          glist_object = g_list_append (NULL, NULL);
          glist_name = g_list_append (NULL, _("Uncompress"));
          /* ja:各4文字コードについて */
          for (j = 0; j < leng_handler; j++)
            {
              IcmObject *icm_object;

              icm_object = icm_open
                        (GPOINTER_TO_UINT (g_list_nth_data (glist_handler, j)),
                                                            ICM_MODE_COMPRESS);
              if (!icm_object)
                continue;
              if (!icm_compress_query (icm_object, avi_edit[i]->bmih))
                {
                  icm_close (icm_object);
                  continue;
                }
              /* ja:スピンボタン */
              if (!avi_stream->spin_quality
                        && icm_compress_get_default_quality (icm_object) >= 0)
                avi_stream->spin_quality = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (-1, -1, 10000, 1, 5, 0)),
                                 0, 0);
              if (!avi_stream->spin_key
                        && icm_compress_get_default_key_frame (icm_object) > 0)
                avi_stream->spin_key = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (0, 0, G_MAXINT, 1, 5, 0)),
                                 0, 0);
              if (!avi_stream->spin_rate
                            && (icm_get_flags (icm_object) & ICM_FLAG_CRUNCH))
                avi_stream->spin_rate = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (0, 0, G_MAXINT, 1, 5, 0)),
                                 0, 0);
              /* ja:ボタン */
              if (!avi_stream->button && icm_is_dialog_configure (icm_object))
                avi_stream->button = gtk_button_new_from_stock
                                                    (GTK_STOCK_PREFERENCES);
              glist_object = g_list_append (glist_object, icm_object);
              glist_name = g_list_append (glist_name,
                                        (gpointer)icm_get_name (icm_object));
            }
          if (g_list_length (glist_name) > 1)
            {
#if GTK_CHECK_VERSION(2,4,0)
              GtkListStore *store;
              GtkCellRenderer *renderer;
#endif /* GTK_CHECK_VERSION(2,4,0) */

              /* ja:チェックボタン */
              avi_stream->check_recomp = gtk_check_button_new_with_mnemonic
                                                            (_("_Recompress"));
              gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON (avi_stream->check_recomp), FALSE);
              gtk_widget_set_sensitive (avi_stream->check_recomp, FALSE);
              /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
              store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_POINTER);
              for (j = g_list_length (glist_name) - 1; j >= 0; j--)
                {
                  GtkTreeIter iter;

                  gtk_list_store_prepend (store, &iter);
                  gtk_list_store_set (store, &iter,
                                    0, g_list_nth_data (glist_name, j),
                                    1, g_list_nth_data (glist_object, j), -1);
                }
              avi_stream->combo = gtk_combo_box_new_with_model
                                                    (GTK_TREE_MODEL (store));
              g_object_unref (store);
              renderer = gtk_cell_renderer_text_new ();
              gtk_cell_layout_pack_start
                        (GTK_CELL_LAYOUT (avi_stream->combo), renderer, FALSE);
              gtk_cell_layout_set_attributes
                                (GTK_CELL_LAYOUT (avi_stream->combo), renderer,
                                                            "text", 0, NULL);
              gtk_combo_box_set_active (GTK_COMBO_BOX (avi_stream->combo), 0);
              g_signal_connect (G_OBJECT (avi_stream->combo), "changed",
                            G_CALLBACK (avi_extra_changed_video), avi_stream);
              g_signal_connect (G_OBJECT (avi_stream->combo), "destroy",
                            G_CALLBACK (avi_extra_destroy_combo_video), NULL);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              avi_stream->combo = gtk_combo_new ();
                  gtk_combo_set_popdown_strings (GTK_COMBO (avi_stream->combo),
                                                                glist_name);
              glist = gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
              for (j = g_list_length (glist) - 1; j >= 0; j--)
                    g_object_set_data (G_OBJECT (g_list_nth_data (glist, j)),
                            "user_data", g_list_nth_data (glist_object, j));
              g_list_free (glist);
              gtk_combo_set_value_in_list (GTK_COMBO (avi_stream->combo),
                                                                TRUE, FALSE);
              gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO
                                        (avi_stream->combo)->entry), FALSE);
              gtk_entry_set_text (GTK_ENTRY (GTK_COMBO
                            (avi_stream->combo)->entry),
                            (const gchar *)g_list_nth_data (glist_name, 0));
              g_signal_connect (G_OBJECT (GTK_COMBO
                            (avi_stream->combo)->entry), "changed",
                            G_CALLBACK (avi_extra_changed_video), avi_stream);
              g_signal_connect (G_OBJECT (GTK_COMBO
                            (avi_stream->combo)->list), "destroy",
                            G_CALLBACK (avi_extra_destroy_list_video), NULL);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
              /* ja:ラベル */
              label0 = gtk_label_new_with_mnemonic (_("_Compressor"));
#if GTK_CHECK_VERSION(2,4,0)
              gtk_label_set_mnemonic_widget (GTK_LABEL (label0),
                                                            avi_stream->combo);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              gtk_label_set_mnemonic_widget (GTK_LABEL (label0),
                                        GTK_COMBO (avi_stream->combo)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
            }
          else
            {
              label0 = gtk_label_new ((const gchar *)
                                              g_list_nth_data (glist_name, 0));
            }
          g_list_free (glist_object);
          g_list_free (glist_name);
          if (avi_stream->spin_quality)
            {
              gtk_spin_button_set_numeric
                            (GTK_SPIN_BUTTON (avi_stream->spin_quality), TRUE);
              avi_stream->label_quality = gtk_label_new_with_mnemonic
                                                            (_("_Quality"));
              gtk_label_set_mnemonic_widget
                                        (GTK_LABEL (avi_stream->label_quality),
                                                    avi_stream->spin_quality);
              gtk_widget_set_sensitive (avi_stream->spin_quality, FALSE);
              gtk_widget_set_sensitive (avi_stream->label_quality, FALSE);
            }
          if (avi_stream->spin_key)
            {
              gtk_spin_button_set_numeric
                                (GTK_SPIN_BUTTON (avi_stream->spin_key), TRUE);
              avi_stream->label_key = gtk_label_new_with_mnemonic
                                                            (_("_Key Frame"));
              gtk_label_set_mnemonic_widget
                    (GTK_LABEL (avi_stream->label_key), avi_stream->spin_key);
              gtk_widget_set_sensitive (avi_stream->spin_key, FALSE);
              gtk_widget_set_sensitive (avi_stream->label_key, FALSE);
            }
          if (avi_stream->spin_rate)
            {
              gtk_spin_button_set_numeric
                            (GTK_SPIN_BUTTON (avi_stream->spin_rate), TRUE);
              avi_stream->label_rate = gtk_label_new_with_mnemonic
                                                            (_("_Data Rate"));
              gtk_label_set_mnemonic_widget
                                        (GTK_LABEL (avi_stream->label_rate),
                                                        avi_stream->spin_rate);
              gtk_widget_set_sensitive (avi_stream->spin_rate, FALSE);
              gtk_widget_set_sensitive (avi_stream->label_rate, FALSE);
            }
          if (avi_stream->button)
            {
              g_signal_connect (G_OBJECT (avi_stream->button), "clicked",
                                G_CALLBACK (avi_extra_clicked_config_video),
                                                            avi_stream->combo);
              gtk_widget_set_sensitive (avi_stream->button, FALSE);
            }
          /* ja:フレームとボックス */
          if (avi_stream->combo)
            {
              hbox0 = gtk_hbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0), label0, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                            avi_stream->combo, TRUE, TRUE, 0);
              gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          else
            {
              gtk_box_pack_start (GTK_BOX (vbox), label0, FALSE, FALSE, 0);
            }
          if (avi_stream->spin_quality)
            {
              hbox0 = gtk_hbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                avi_stream->label_quality, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                    avi_stream->spin_quality, TRUE, TRUE, 0);
              gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          if (avi_stream->spin_key)
            {
              hbox0 = gtk_hbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                    avi_stream->label_key, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->spin_key, TRUE, TRUE, 0);
              gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          if (avi_stream->spin_rate)
            {
              hbox0 = gtk_hbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                    avi_stream->label_rate, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->spin_rate, TRUE, TRUE, 0);
              gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          gtk_box_pack_end (GTK_BOX (vbox),
                                    avi_stream->check_null, FALSE, FALSE, 0);
          if (avi_stream->check_recomp && avi_stream->button)
            {
              hbox0 = gtk_hbox_new (FALSE, SPACING);
              if (avi_stream->check_recomp)
                gtk_box_pack_start (GTK_BOX (hbox0),
                                    avi_stream->check_recomp, TRUE, TRUE, 0);
              if (avi_stream->button)
                gtk_box_pack_end (GTK_BOX (hbox0),
                                        avi_stream->button, FALSE, FALSE, 0);
              gtk_box_pack_end (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          video++;
        }
      else
        { /* ja:オーディオストリーム */
          gint j;
          GList *glist_wfx;         /* ja:WaveFormatEx構造体のリスト */
          GList *glist_name;        /* ja:名前のリスト */
#if GTK_CHECK_VERSION(2,4,0)
          gchar **ary;
#endif /* GTK_CHECK_VERSION(2,4,0) */

          /* ja:未圧縮 */
          glist_wfx = g_list_append (NULL, NULL);
#if GTK_CHECK_VERSION(2,4,0)
          ary = g_malloc (sizeof (gchar *) * 2);
          ary[0] = g_strdup (g_strdup (_("PCM")));
          ary[1] = NULL;
          glist_name = g_list_append (NULL, ary);
#else /* not GTK_CHECK_VERSION(2,4,0) */
          glist_name = g_list_append (NULL, g_strdup (_("PCM")));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
          /* ja:各タグについて */
          for (j = 0; j < leng_tag; j++)
            {
              gsize leng;
              gchar *name;
              gint index = 0;
              AcmObject *acm_object;
              WaveFormatEx *wfx;

              acm_object = acm_open ((guint16)GPOINTER_TO_UINT
                                            (g_list_nth_data (glist_tag, j)));
              if (!acm_object)
                continue;
              leng = acm_compress_get_format_size (acm_object,
                                                            avi_edit[i]->wfx);
              wfx = g_malloc (leng);
              while (acm_compress_enum_format (acm_object,
                                            avi_edit[i]->wfx, wfx, index++)
                            && (name = acm_get_format_name (acm_object, wfx)))
                {
#if GTK_CHECK_VERSION(2,4,0)
                  ary = g_malloc (sizeof (gchar *) * 3);
                  ary[0] = g_strdup (acm_get_name (acm_object));
                  ary[1] = name;
                  ary[2] = NULL;
                  glist_name = g_list_append (glist_name, ary);
#else /* not GTK_CHECK_VERSION(2,4,0) */
                  gchar *tmp;

                  tmp = g_strconcat (acm_get_name (acm_object), " ",
                                                                name, NULL);
                  g_free (name);
                  glist_name = g_list_append (glist_name, tmp);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
                  glist_wfx = g_list_append (glist_wfx, g_memdup (wfx, leng));
                }
              g_free (wfx);
              acm_close (acm_object);
            }
          if (g_list_length (glist_name) > 1)
            {
#if GTK_CHECK_VERSION(2,4,0)
              GtkListStore *store;
              GtkCellRenderer *renderer;
#endif /* GTK_CHECK_VERSION(2,4,0) */

              /* ja:チェックボタン */
              avi_stream->check_recomp = gtk_check_button_new_with_mnemonic
                                                            (_("_Recompress"));
              gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON (avi_stream->check_recomp), TRUE);
              gtk_widget_set_sensitive (avi_stream->check_recomp, FALSE);
              /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
              store = gtk_list_store_new (3,
                                G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);
              for (j = g_list_length (glist_name) - 1; j >= 0; j--)
                {
                  gchar **ary;
                  GtkTreeIter iter;

                  ary = g_list_nth_data (glist_name, j);
                  gtk_list_store_prepend (store, &iter);
                  gtk_list_store_set (store, &iter,
                                        0, ary[0], 1, ary[1],
                                        2, g_list_nth_data (glist_wfx, j), -1);
                }
              avi_stream->combo = gtk_combo_box_new_with_model
                                                    (GTK_TREE_MODEL (store));
              g_object_unref (store);
              renderer = gtk_cell_renderer_text_new ();
              gtk_cell_layout_pack_start
                        (GTK_CELL_LAYOUT (avi_stream->combo), renderer, FALSE);
              gtk_cell_layout_set_attributes
                                (GTK_CELL_LAYOUT (avi_stream->combo), renderer,
                                                            "text", 0, NULL);
              renderer = gtk_cell_renderer_text_new ();
              gtk_cell_layout_pack_start
                        (GTK_CELL_LAYOUT (avi_stream->combo), renderer, FALSE);
              gtk_cell_layout_set_attributes
                                (GTK_CELL_LAYOUT (avi_stream->combo), renderer,
                                                            "text", 1, NULL);
              gtk_combo_box_set_active (GTK_COMBO_BOX (avi_stream->combo), 0);
              g_signal_connect (G_OBJECT (avi_stream->combo), "changed",
                            G_CALLBACK (avi_extra_changed_audio), avi_stream);
              g_signal_connect (G_OBJECT (avi_stream->combo), "destroy",
                            G_CALLBACK (avi_extra_destroy_combo_audio), NULL);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              avi_stream->combo = gtk_combo_new ();
              gtk_combo_set_popdown_strings (GTK_COMBO (avi_stream->combo),
                                                                glist_name);
              glist = gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
              for (j = g_list_length (glist) - 1; j >= 0; j--)
                    g_object_set_data (G_OBJECT (g_list_nth_data (glist, j)),
                                "user_data", g_list_nth_data (glist_wfx, j));
              g_list_free (glist);
              gtk_combo_set_value_in_list (GTK_COMBO (avi_stream->combo),
                                                                TRUE, FALSE);
              gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO
                                        (avi_stream->combo)->entry), FALSE);
              gtk_entry_set_text (GTK_ENTRY (GTK_COMBO
                            (avi_stream->combo)->entry),
                            (const gchar *)g_list_nth_data (glist_name, 0));
              g_signal_connect (G_OBJECT (GTK_COMBO
                            (avi_stream->combo)->entry), "changed",
                            G_CALLBACK (avi_extra_changed_audio), avi_stream);
              g_signal_connect (G_OBJECT (GTK_COMBO
                            (avi_stream->combo)->list), "destroy",
                            G_CALLBACK (avi_extra_destroy_list_audio), NULL);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
              /* ja:ラベル */
              label0 = gtk_label_new_with_mnemonic (_("_Compressor"));
#if GTK_CHECK_VERSION(2,4,0)
              gtk_label_set_mnemonic_widget (GTK_LABEL (label0),
                                                            avi_stream->combo);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              gtk_label_set_mnemonic_widget (GTK_LABEL (label0),
                                        GTK_COMBO (avi_stream->combo)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
            }
          else
            {
#if GTK_CHECK_VERSION(2,4,0)
              gchar *tmp;

              ary = g_list_nth_data (glist_name, 0);
              tmp = g_strconcat (ary[0], " ", ary[1], NULL);
              label0 = gtk_label_new (tmp);
              g_free (tmp);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              label0 = gtk_label_new ((const gchar *)
                                              g_list_nth_data (glist_name, 0));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
            }
          for (j = g_list_length (glist_name) - 1; j >= 0; j--)
#if GTK_CHECK_VERSION(2,4,0)
            g_strfreev (g_list_nth_data (glist_name, j));
#else /* not GTK_CHECK_VERSION(2,4,0) */
            g_free (g_list_nth_data (glist_name, j));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
          g_list_free (glist_wfx);
          g_list_free (glist_name);
          /* ja:フレームとボックス */
          if (avi_stream->combo)
            {
              hbox0 = gtk_hbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0), label0, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (hbox0),
                                            avi_stream->combo, TRUE, TRUE, 0);
              gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
            }
          else
            {
              gtk_box_pack_start (GTK_BOX (vbox), label0, FALSE, FALSE, 0);
            }
          if (avi_stream->check_recomp)
            gtk_box_pack_end (GTK_BOX (vbox),
                                    avi_stream->check_recomp, FALSE, FALSE, 0);
          audio++;
        }
       text = avi_edit_get_name (avi_edit[i])
                                ? g_strdup (avi_edit_get_name (avi_edit[i]))
                                : avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                                        ? g_strdup_printf ("Video #%d", video)
                                        : g_strdup_printf ("Audio #%d", audio);
      if (GTK_IS_NOTEBOOK (extra->notebook))
        {
          GtkWidget *radio;

          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                                                        && extra->video > 1)
            {
              radio = gtk_radio_button_new_with_label (gslist_v, text);
              g_object_set_data (G_OBJECT (radio), "user_data", avi_edit[i]);
              g_signal_connect (G_OBJECT (radio), "toggled",
                                G_CALLBACK (avi_extra_toggled_video), extra);
              gslist_v = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
            }
          else if (avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO
                                                        && extra->audio > 1)
            {
              radio = gtk_radio_button_new_with_label (gslist_a, text);
              gslist_a = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
            }
          else
            {
              radio = NULL;
            }
          gtk_notebook_append_page (GTK_NOTEBOOK (extra->notebook),
                                vbox, radio ? radio : gtk_label_new (text));
        }
      else
        {
          gtk_container_add (GTK_CONTAINER (extra->notebook), vbox);
          gtk_frame_set_label (GTK_FRAME (extra->notebook), text);
        }
      g_free (text);
    }
  g_list_free (glist_handler);
  g_list_free (glist_tag);
  /* en:AVI,GtkShot Raw */
  if (extra->video > 0 || extra->audio > 1)
    {
      /* ja:ラジオボタン */
      extra->radio_avi = gtk_radio_button_new_with_mnemonic (NULL, _("_AVI"));
      g_signal_connect (G_OBJECT (extra->radio_avi), "toggled",
                                    G_CALLBACK (avi_extra_toggled_avi), extra);
      gslist = gtk_radio_button_get_group
                                        (GTK_RADIO_BUTTON (extra->radio_avi));
      extra->radio_gsr = gtk_radio_button_new_with_mnemonic
                                                (gslist, _("_GtkShot Raw"));
      g_signal_connect (G_OBJECT (extra->radio_gsr), "toggled",
                                    G_CALLBACK (avi_extra_toggled_avi), extra);
      gslist = gtk_radio_button_get_group
                                        (GTK_RADIO_BUTTON (extra->radio_gsr));
    }
  else
    {
      gslist = NULL;
    }
  /* ja:ビットマップ,GdkPixbuf */
  if (extra->video > 0)
    {
      GtkWidget *label1, *hbox1;
#if GTK_CHECK_VERSION(2,2,0)
      gint j;
      GSList *gslist_formats, *gslist_sorted = NULL;
      GtkWidget *vbox;
#endif /* GTK_CHECK_VERSION(2,2,0) */

      /* ja:ラジオボタン */
      extra->radio_bitmap = gtk_radio_button_new_with_mnemonic
                                                        (gslist, _("_Bitmap"));
      g_signal_connect (G_OBJECT (extra->radio_bitmap), "toggled",
                                G_CALLBACK (avi_extra_toggled_bitmap), extra);
      gslist = gtk_radio_button_get_group
                                    (GTK_RADIO_BUTTON (extra->radio_bitmap));
      /* ja:スピンボタン */
      for (i = 0; i < streams; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
          break;
      extra->spin_start = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (0,
                            0, avi_edit_length (avi_edit[i]) - 1, 1, 5, 0)),
             0, 0);
      extra->spin_end = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (avi_edit_length (avi_edit[i]) - 1,
                            0, avi_edit_length (avi_edit[i]) - 1, 1, 5, 0)),
             0, 0);
      gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (extra->spin_start), TRUE);
      gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (extra->spin_end), TRUE);
      /* ja:ラベル */
      label0 = gtk_label_new_with_mnemonic (_("_Start"));
      label1 = gtk_label_new_with_mnemonic (_("_End"));
      gtk_label_set_mnemonic_widget (GTK_LABEL (label0), extra->spin_start);
      gtk_label_set_mnemonic_widget (GTK_LABEL (label1), extra->spin_end);
      /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,2,0)
      gslist_formats = gdk_pixbuf_get_formats ();
      for (j = 0; j < g_slist_length (gslist_formats); j++)
        {
          GdkPixbufFormat *format;

          format = g_slist_nth_data (gslist_formats, j);
          if (gdk_pixbuf_format_is_writable (format))
            gslist_sorted = g_slist_insert_sorted (gslist_sorted, format,
                                    (GCompareFunc)avi_extra_set_handle_sort);
        }
      g_slist_free (gslist_formats);
      if (g_slist_length (gslist_sorted) > 0)
        {
# if GTK_CHECK_VERSION(2,4,0)
          GtkCellRenderer *renderer;
          GtkListStore *store;

          store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_POINTER);
# elif GTK_CHECK_VERSION(2,2,0)
          GList *glist = NULL;

# endif /* GTK_CHECK_VERSION(2,2,0) */
          for (j = 0; j < g_slist_length (gslist_sorted); j++)
            {
              gchar *desc, *name;
              GdkPixbufFormat *format;
# if GTK_CHECK_VERSION(2,4,0)
              GtkTreeIter iter;
# endif /* GTK_CHECK_VERSION(2,4,0) */

              format = g_slist_nth_data (gslist_sorted, j);
              desc = gdk_pixbuf_format_get_description (format);
              name = gdk_pixbuf_format_get_name (format);
              if (g_strcmp (name, "ico") == 0)
                {
                  /* ja:色 */
                  extra->check_ico_depth = gtk_check_button_new_with_mnemonic
                                                                (_("_Color"));
                  gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON (extra->check_ico_depth), FALSE);
                  g_signal_connect (G_OBJECT (extra->check_ico_depth),
                            "toggled",
                            G_CALLBACK (avi_extra_toggled_ico_depth), extra);
                  extra->radio_ico_depth16 = gtk_radio_button_new_with_mnemonic
                                                        (NULL, _("_16 bits"));
                  extra->radio_ico_depth24
                            = gtk_radio_button_new_with_mnemonic_from_widget
                                (GTK_RADIO_BUTTON (extra->radio_ico_depth16),
                                                            _("_Full Color"));
                  extra->radio_ico_depth32
                            = gtk_radio_button_new_with_mnemonic_from_widget
                                (GTK_RADIO_BUTTON (extra->radio_ico_depth24),
                                                                _("_32 bits"));
                  gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON (extra->radio_ico_depth24), TRUE);
                  /* ja:カーソル */
                  extra->check_ico_cursor = gtk_check_button_new_with_mnemonic
                                                                (_("_Cursor"));
                  gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON (extra->check_ico_cursor), FALSE);
                  g_signal_connect (G_OBJECT (extra->check_ico_cursor),
                            "toggled",
                            G_CALLBACK (avi_extra_toggled_ico_cursor), extra);
                  extra->spin_ico_x = gtk_spin_button_new (GTK_ADJUSTMENT
                        (gtk_adjustment_new (0,
                         0, avi_edit_get_width (avi_edit[i]) - 1, 1, 5, 0)),
                         0, 0);
                  extra->spin_ico_y = gtk_spin_button_new (GTK_ADJUSTMENT
                        (gtk_adjustment_new (0,
                         0, avi_edit_get_height (avi_edit[i]) - 1, 1, 5, 0)),
                         0, 0);
                  extra->label_ico_x = gtk_label_new_with_mnemonic
                                                            (_("_X Hot Spot"));
                  extra->label_ico_y = gtk_label_new_with_mnemonic
                                                            (_("_Y Hot Spot"));
                  gtk_label_set_mnemonic_widget
                        (GTK_LABEL (extra->label_ico_x), extra->spin_ico_x);
                  gtk_label_set_mnemonic_widget
                        (GTK_LABEL (extra->label_ico_y), extra->spin_ico_y);
                  /* ja:フレームとボックス */
                  extra->hbox_ico = gtk_hbox_new (FALSE, SPACING);
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->check_ico_depth, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->radio_ico_depth16, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->radio_ico_depth24, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->radio_ico_depth32, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_ico),
                                    hbox0, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_ico),
                                    extra->check_ico_cursor, FALSE, FALSE, 0);
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->label_ico_x, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->spin_ico_x, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_ico),
                                    hbox0, TRUE, TRUE, 0);
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->label_ico_y, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                    extra->spin_ico_y, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_ico),
                                    hbox0, TRUE, TRUE, 0);
                  gtk_widget_set_sensitive
                                (GTK_WIDGET (extra->radio_ico_depth16), FALSE);
                  gtk_widget_set_sensitive
                                (GTK_WIDGET (extra->radio_ico_depth24), FALSE);
                  gtk_widget_set_sensitive
                                (GTK_WIDGET (extra->radio_ico_depth32), FALSE);
                  gtk_widget_set_sensitive
                                    (GTK_WIDGET (extra->label_ico_x), FALSE);
                  gtk_widget_set_sensitive
                                    (GTK_WIDGET (extra->spin_ico_x), FALSE);
                  gtk_widget_set_sensitive
                                    (GTK_WIDGET (extra->label_ico_y), FALSE);
                  gtk_widget_set_sensitive
                                    (GTK_WIDGET (extra->spin_ico_y), FALSE);
                  gtk_widget_show_all (extra->hbox_ico);
                  if (j <= 0)
                    extra->hbox_current = extra->hbox_ico;
                }
              else if (g_strcmp (name, "jpeg") == 0)
                {
                  extra->check_jpeg = gtk_check_button_new_with_mnemonic
                                                            (_("_Quality"));
                  gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (extra->check_jpeg), FALSE);
                  g_signal_connect (G_OBJECT (extra->check_jpeg), "toggled",
                                G_CALLBACK (avi_extra_toggled_jpeg), extra);
                  extra->spin_jpeg = gtk_spin_button_new (GTK_ADJUSTMENT
                            (gtk_adjustment_new (100, 0, 100, 1, 5, 0)), 0, 0);
                  /* ja:フレームとボックス */
                  extra->hbox_jpeg = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_jpeg),
                                        extra->check_jpeg, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_jpeg),
                                        extra->spin_jpeg, TRUE, TRUE, 0);
                  gtk_widget_set_sensitive
                                        (GTK_WIDGET (extra->spin_jpeg), FALSE);
                  gtk_widget_show_all (GTK_WIDGET (extra->hbox_jpeg));
                  if (j <= 0)
                    extra->hbox_current = extra->hbox_jpeg;
                }
              else if (g_strcmp (name, "png") == 0)
                {
                  extra->check_png = gtk_check_button_new_with_mnemonic
                                                        (_("_Compress Level"));
                  gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (extra->check_png), FALSE);
                  g_signal_connect (G_OBJECT (extra->check_png), "toggled",
                                    G_CALLBACK (avi_extra_toggled_png), extra);
                  extra->spin_png = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (9, 0, 9, 1, 1, 0)), 0, 0);
                  extra->entry_png = gtk_entry_new ();
                  gtk_entry_set_max_length (GTK_ENTRY (extra->entry_png), 79);
                  /* ja:フレームとボックス */
                  extra->hbox_png = gtk_hbox_new (FALSE, SPACING);
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                            extra->check_png, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                            extra->spin_png, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_png),
                                            hbox0, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (extra->hbox_png),
                                            extra->entry_png, TRUE, TRUE, 0);
                  gtk_widget_set_sensitive
                                        (GTK_WIDGET (extra->spin_png), FALSE);
                  gtk_widget_show_all (GTK_WIDGET (extra->hbox_png));
                  if (j <= 0)
                    extra->hbox_current = extra->hbox_png;
                }
# if GTK_CHECK_VERSION(2,4,0)
              gtk_list_store_append (store, &iter);
              gtk_list_store_set (store, &iter, 0, desc, 1, format, -1);
              g_free (desc);
              g_free (name);
            }
          extra->combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL (store));
          g_object_unref (store);
          renderer = gtk_cell_renderer_text_new ();
          gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (extra->combo),
                                                            renderer, FALSE);
          gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (extra->combo),
                                                    renderer, "text", 0, NULL);
          gtk_combo_box_set_active (GTK_COMBO_BOX (extra->combo), 0);
          if (extra->hbox_ico || extra->hbox_jpeg || extra->hbox_png)
            g_signal_connect (G_OBJECT (extra->combo), "changed",
                                G_CALLBACK (avi_extra_changed_pixbuf), extra);
# else /* not GTK_CHECK_VERSION(2,4,0) */
              glist = g_list_append (glist, desc);
              g_free (name);
            }
          extra->combo = gtk_combo_new ();
          gtk_combo_set_popdown_strings (GTK_COMBO (extra->combo), glist);
          for (j = g_list_length (glist) - 1; j >= 0; j--)
            g_free (g_list_nth_data (glist, j));
          g_list_free (glist);
          glist = gtk_container_get_children
                            (GTK_CONTAINER (GTK_COMBO (extra->combo)->list));
          for (j = g_list_length (glist) - 1; j >= 0; j--)
            g_object_set_data (G_OBJECT (g_list_nth_data (glist, j)),
                            "user_data", g_slist_nth_data (gslist_sorted, j));
          g_list_free (glist);
          gtk_combo_set_value_in_list (GTK_COMBO (extra->combo), TRUE, FALSE);
          gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO
                                                (extra->combo)->entry), FALSE);
          gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (extra->combo)->entry),
                            (const gchar *)g_list_nth_data (glist_name, 0));
          if (extra->hbox_ico || extra->hbox_jpeg || extra->hbox_png)
            g_signal_connect (G_OBJECT (GTK_COMBO (extra->combo)->entry),
                    "changed", G_CALLBACK (avi_extra_changed_pixbuf), extra);
# endif /* not GTK_CHECK_VERSION(2,4,0) */
          if (extra->hbox_current)
            gtk_widget_set_sensitive (GTK_WIDGET (extra->hbox_current), FALSE);
          gtk_widget_set_sensitive (GTK_WIDGET (extra->combo), FALSE);
        }
      g_slist_free (gslist_sorted);
      /* ja:ラジオボタン */
      if (extra->combo)
        {
          extra->radio_pixbuf = gtk_radio_button_new_with_mnemonic
                                                    (gslist, _("_GdkPixbuf"));
          g_signal_connect (G_OBJECT (extra->radio_pixbuf), "toggled",
                                G_CALLBACK (avi_extra_toggled_pixbuf), extra);
          gslist = gtk_radio_button_get_group
                                    (GTK_RADIO_BUTTON (extra->radio_pixbuf));
        }
      /* ja:フレームとボックス */
      if (extra->hbox_ico || extra->hbox_jpeg || extra->hbox_png)
        {
          vbox = gtk_vbox_new (FALSE, SPACING);
          gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
        }
      else
        {
          vbox = NULL;
        }
#endif /* GTK_CHECK_VERSION(2,2,0) */
      hbox0 = gtk_hbox_new (FALSE, SPACING);
#if GTK_CHECK_VERSION(2,2,0)
      if (!vbox)
#endif /* GTK_CHECK_VERSION(2,2,0) */
        gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), label0, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), extra->spin_start, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), extra->spin_end, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
      if (extra->combo)
        gtk_box_pack_start (GTK_BOX (hbox0), extra->combo, FALSE, FALSE, 0);
#if GTK_CHECK_VERSION(2,2,0)
      if (vbox)
        gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
      if (extra->hbox_current)
        {
          g_object_ref (extra->hbox_current);
          gtk_box_pack_start (GTK_BOX (vbox),
                                        extra->hbox_current, FALSE, FALSE, 0);
      }
#endif /* GTK_CHECK_VERSION(2,2,0) */
      extra->frame = gtk_frame_new (_("Range and Format"));
#if GTK_CHECK_VERSION(2,2,0)
      gtk_container_add (GTK_CONTAINER (extra->frame), vbox ? vbox : hbox0);
#else /* not GTK_CHECK_VERSION(2,2,0) */
      gtk_container_add (GTK_CONTAINER (extra->frame), hbox0);
#endif /* not GTK_CHECK_VERSION(2,2,0) */
      gtk_widget_set_sensitive (extra->frame, FALSE);
    }
  /* en:WAVE */
  if (extra->audio > 0)
    {
      /* ja:ラジオボタン */
      extra->radio_wave = gtk_radio_button_new_with_mnemonic
                                                        (gslist, _("_WAVE"));
      g_signal_connect (G_OBJECT (extra->radio_wave), "toggled",
                                G_CALLBACK (avi_extra_toggled_wave), extra);
      gslist = gtk_radio_button_get_group
                                        (GTK_RADIO_BUTTON (extra->radio_wave));
    }
  /* ja:シナリオオブジェクト */
  /* ja:ラジオボタン */
  extra->radio_scenario = gtk_radio_button_new_with_mnemonic
                                            (gslist, _("Scenario _Object"));
  g_signal_connect (G_OBJECT (extra->radio_scenario), "toggled",
                            G_CALLBACK (avi_extra_toggled_scenario), extra);
  /* ja:ラジオボタン */
  gtk_toggle_button_toggled (GTK_TOGGLE_BUTTON (extra->radio_avi
                                ? extra->radio_avi : extra->radio_bitmap
                                ? extra->radio_bitmap : extra->radio_wave));

  /* ja:フレームとボックス */
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
  if (extra->radio_avi)
    gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_avi, FALSE, FALSE, 0);
  if (extra->radio_gsr)
    gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_gsr, FALSE, FALSE, 0);
  if (extra->radio_bitmap)
    gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_bitmap, FALSE, FALSE, 0);
#if GTK_CHECK_VERSION(2,2,0)
  if (extra->radio_pixbuf)
    gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_pixbuf, FALSE, FALSE, 0);
#endif /* GTK_CHECK_VERSION(2,2,0) */
  if (extra->radio_wave)
    gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_wave, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), extra->radio_scenario, FALSE, FALSE, 0);
  frame = gtk_frame_new (_("Format"));
  gtk_container_add (GTK_CONTAINER (frame), hbox0);

  gtk_box_pack_start (GTK_BOX (extra), frame, FALSE, FALSE, 0);
  gtk_widget_show_all (frame);
  if (extra->frame)
    {
      gtk_box_pack_start (GTK_BOX (extra), extra->frame, FALSE, FALSE, 0);
      gtk_widget_show_all (extra->frame);
    }
  gtk_box_pack_start (GTK_BOX (extra), extra->notebook, FALSE, FALSE, 0);
  gtk_widget_show_all (extra->notebook);
}


/*  ja:保存用のオプションを取得する
    extra,ウィジェット
      RET,オプション,NULL:エラー                                            */
AviSave *
avi_extra_get_option (AviExtra *extra)
{
  gint streams;
  GList *glist;
  AviExtraStream *avi_stream;
  AviSave *avi_save;

  if (!extra)
    return NULL;
  avi_save = g_malloc0 (sizeof (AviSave));
  if (GTK_IS_NOTEBOOK (extra->notebook))
    {
      glist = gtk_container_get_children (GTK_CONTAINER (extra->notebook));
      streams = g_list_length (glist);
    }
  else
    {
      glist = g_list_append (NULL,
                                gtk_bin_get_child (GTK_BIN (extra->notebook)));
      streams = 1;
    }
  if (extra->radio_avi
    && (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_avi))
    || gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_gsr))))
    {
      gint i;

      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_avi)))
        {
          avi_save->type = AVI_FILE_AVI;
          avi_save->extension = g_strdup ("avi");
        }
      else
        {
          avi_save->type = AVI_FILE_GSR;
          avi_save->extension = g_strdup ("gsr");
        }
      avi_save->avi_opt = g_malloc0 ((streams + 1) * sizeof (AviOptions));
      for (i = 0; i < streams; i++)
        {
          GtkWidget *child;
          WaveFormatEx *wfx;

          child = g_list_nth_data (glist, i);
          avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
          avi_save->avi_opt[i].type = avi_edit_type (avi_stream->avi_edit);
          if (avi_save->avi_opt[i].type == AVI_TYPE_VIDEO)
            { /* ja:ビデオストリーム */
              IcmObject *icm_object;

              avi_save->avi_opt[i].nullframe = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (avi_stream->check_null));
              if (avi_stream->combo && (icm_object = g_object_get_data
                                (G_OBJECT (avi_stream->combo), "user_data")))
                {
                  avi_save->avi_opt[i].handler = icm_get_handler (icm_object);
                  avi_save->avi_opt[i].quality = avi_stream->spin_quality
                        && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin_quality)
                            ? gtk_spin_button_get_value_as_int
                            (GTK_SPIN_BUTTON (avi_stream->spin_quality)) : -1;
                  avi_save->avi_opt[i].key_frame = avi_stream->spin_key
                            && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin_key)
                                ? gtk_spin_button_get_value_as_int
                                (GTK_SPIN_BUTTON (avi_stream->spin_key)) : 0;
                  avi_save->avi_opt[i].data_rate = avi_stream->spin_rate
                            && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin_rate)
                                ? gtk_spin_button_get_value_as_int
                                (GTK_SPIN_BUTTON (avi_stream->spin_rate)) : 0;
                  avi_save->avi_opt[i].recompress = avi_stream->check_recomp
                        && GTK_WIDGET_IS_SENSITIVE (avi_stream->check_recomp)
                                && gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (avi_stream->check_recomp));
                  avi_save->avi_opt[i].param_size
                                            = icm_get_state_size (icm_object);
                  if (avi_save->avi_opt[i].param_size > 0)
                    {
                      avi_save->avi_opt[i].param
                                = g_malloc (avi_save->avi_opt[i].param_size);
                      icm_get_state (icm_object, avi_save->avi_opt[i].param,
                                            avi_save->avi_opt[i].param_size);
                    }
                  avi_save->avi_opt[i].bmih_size = icm_compress_get_format_size
                                    (icm_object, avi_stream->avi_edit->bmih);
                  avi_save->avi_opt[i].bmih
                                = g_malloc (avi_save->avi_opt[i].bmih_size);
                  if (!icm_compress_get_format (icm_object,
                        avi_stream->avi_edit->bmih, avi_save->avi_opt[i].bmih))
                    {
                      g_free (avi_save->avi_opt[i].bmih);
                      avi_save->avi_opt[i].bmih = NULL;
                      avi_save->avi_opt[i].bmih_size = 0;
                    }
                }
            }
          else if (avi_stream->combo && (wfx = g_object_get_data
                                (G_OBJECT (avi_stream->combo), "user_data")))
            { /* ja:オーディオストリーム */
              avi_save->avi_opt[i].recompress = !avi_stream->check_recomp
                        || !GTK_WIDGET_IS_SENSITIVE (avi_stream->check_recomp)
                        || gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (avi_stream->check_recomp));
              avi_save->avi_opt[i].wfx
                                    = g_memdup (wfx, wf_header_bytes (wfx));
            }
        }
    }
  else if ((extra->radio_bitmap
    && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_bitmap)))
#if GTK_CHECK_VERSION(2,2,0)
    || (extra->radio_pixbuf
    && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_pixbuf)))
#endif /* GTK_CHECK_VERSION(2,2,0) */
                                                                            )
    {
#if GTK_CHECK_VERSION(2,2,0)
      avi_save->type = gtk_toggle_button_get_active
                                    (GTK_TOGGLE_BUTTON (extra->radio_bitmap))
                                        ? AVI_FILE_BITMAP : AVI_FILE_PIXBUF;
#else /* not GTK_CHECK_VERSION(2,2,0) */
      avi_save->type = AVI_FILE_BITMAP;
#endif /* not GTK_CHECK_VERSION(2,2,0) */
      avi_save->start = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_start));
      avi_save->end = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_end));
      if (avi_save->start > avi_save->end)
        {
          gint value;

          value = avi_save->start;
          avi_save->start = avi_save->end;
          avi_save->end = value;
        }
      if (GTK_IS_NOTEBOOK (extra->notebook))
        {
          gint i;

          for (i = 0; i < streams; i++)
            {
              GtkWidget *child, *radio;

              child = gtk_notebook_get_nth_page
                                        (GTK_NOTEBOOK (extra->notebook), i);
              radio = gtk_notebook_get_tab_label
                                    (GTK_NOTEBOOK (extra->notebook), child);
              avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
              if (avi_edit_type (avi_stream->avi_edit) == AVI_TYPE_VIDEO
                && (!GTK_IS_RADIO_BUTTON (radio)
                || gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio))))
                break;
            }
          avi_save->stream = i;
        }
#if GTK_CHECK_VERSION(2,2,0)
      if (avi_save->type == AVI_FILE_PIXBUF)
        {
# if GTK_CHECK_VERSION(2,4,0)
          GtkTreeIter iter;
# else /* not GTK_CHECK_VERSION(2,4,0) */
          const gchar *text;
          gint i;
          GList *glist_combo;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
          GdkPixbufFormat *format = NULL;

# if GTK_CHECK_VERSION(2,4,0)
          if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (extra->combo),
                                                                        &iter))
            gtk_tree_model_get (gtk_combo_box_get_model
                        (GTK_COMBO_BOX (extra->combo)), &iter, 1, &format, -1);
# else /* not GTK_CHECK_VERSION(2,4,0) */
          text = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (extra->combo)->entry));
          glist_combo = gtk_container_get_children
                            (GTK_CONTAINER (GTK_COMBO (extra->combo)->list));
          for (i = g_list_length (glist_combo) - 1; i >= 0; i--)
            {
              const gchar *label;

              label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                (GTK_BIN (g_list_nth_data (glist_combo, i)))));
              if (g_strcmp (label, text) == 0)
                {
                  format = g_object_get_data (G_OBJECT (g_list_nth_data
                                            (glist_combo, i)), "user_data");
                  break;
                }
            }
          g_list_free (glist_combo);
# endif /* not GTK_CHECK_VERSION(2,4,0) */
          if (format)
            {
              gchar **ext;
              gint i;

              avi_save->name = gdk_pixbuf_format_get_name (format);
              ext = gdk_pixbuf_format_get_extensions (format);
              if (ext && ext[0])
                avi_save->extension = g_strdup (ext[0]);
              g_strfreev (ext);
              i = 0;
              avi_save->keys = g_malloc (sizeof (gchar *) * 4);
              avi_save->values = g_malloc (sizeof (gchar *) * 4);
              if (extra->hbox_ico && g_strcmp (avi_save->name, "ico") == 0)
                {
                  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                                    (extra->check_ico_depth)))
                    {
                      avi_save->keys[i] = g_strdup ("depth");
                      avi_save->values[i] = g_strdup
                            (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                    (extra->radio_ico_depth16)) ? "16"
                            : gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                    (extra->radio_ico_depth24)) ? "24" : "32");
                      i++;
                    }
                  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                                    (extra->check_ico_cursor)))
                    {
                      avi_save->keys[i] = g_strdup ("x_hot");
                      avi_save->values[i] = g_strdup_printf ("%d",
                                        gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_ico_x)));
                      i++;
                      avi_save->keys[i] = g_strdup ("y_hot");
                      avi_save->values[i] = g_strdup_printf ("%d",
                                        gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_ico_y)));
                      i++;
                    }
                }
              else if (extra->hbox_jpeg
                                    && g_strcmp (avi_save->name, "jpeg") == 0)
                {
                  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                                        (extra->check_jpeg)))
                    {
                      avi_save->keys[i] = g_strdup ("quality");
                      avi_save->values[i] = g_strdup_printf ("%d",
                                        gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_jpeg)));
                      i++;
                    }
                }
              else if (extra->hbox_png
                                    && g_strcmp (avi_save->name, "png") == 0)
                {
# if GTK_CHECK_VERSION(2,4,0)
                  const gchar *text;
# endif /* GTK_CHECK_VERSION(2,4,0) */

                  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                                        (extra->check_png)))
                    {
                      avi_save->keys[i] = g_strdup ("compression");
                      avi_save->values[i] = g_strdup_printf ("%d",
                                        gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (extra->spin_png)));
                      i++;
                    }
                  text = gtk_entry_get_text (GTK_ENTRY (extra->entry_png));
                  if (text && g_strlen (text) > 0)
                    {
                      avi_save->keys[i] = g_strdup_printf ("tEXt::%s", text);
                      avi_save->values[i] = g_strdup ("");
                      i++;
                    }
                }
              if (i <= 0)
                {
                  g_free (avi_save->keys);
                  g_free (avi_save->values);
                  avi_save->keys = NULL;
                  avi_save->values = NULL;
                }
              else
                {
                  avi_save->keys[i] = NULL;
                  avi_save->values[i] = NULL;
                }
            }
        }
      else
        {
#endif /* GTK_CHECK_VERSION(2,2,0) */
          avi_save->extension = g_strdup ("bmp");
#if GTK_CHECK_VERSION(2,2,0)
        }
#endif /* GTK_CHECK_VERSION(2,2,0) */
    }
  else if (extra->radio_wave
    && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (extra->radio_wave)))
    {
      WaveFormatEx *wfx;

      avi_save->type = AVI_FILE_WAVE;
      avi_save->extension = g_strdup ("wav");
      avi_save->avi_opt = g_malloc0 (sizeof (AviOptions) * 2);
      avi_save->avi_opt[0].type = AVI_TYPE_AUDIO;
      avi_stream = NULL;
      if (GTK_IS_NOTEBOOK (extra->notebook))
        {
          gint i;

          for (i = 0; i < streams; i++)
            {
              GtkWidget *child, *radio;

              child = gtk_notebook_get_nth_page
                                        (GTK_NOTEBOOK (extra->notebook), i);
              radio = gtk_notebook_get_tab_label
                                    (GTK_NOTEBOOK (extra->notebook), child);
              avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
              if (avi_edit_type (avi_stream->avi_edit) == AVI_TYPE_AUDIO
                && (!GTK_IS_RADIO_BUTTON (radio)
                || gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio))))
                break;
            }
          avi_save->stream = i;
        }
      else
        {
          GtkWidget *child;

          child = g_list_nth_data (glist, 0);
          avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
        }
      if (avi_stream->combo && (wfx = g_object_get_data
                                (G_OBJECT (avi_stream->combo), "user_data")))
        avi_save->avi_opt[0].wfx = g_memdup (wfx, wf_header_bytes (wfx));
      avi_save->avi_opt[0].recompress = !avi_stream->check_recomp
                        || !GTK_WIDGET_IS_SENSITIVE (avi_stream->check_recomp)
                        || gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (avi_stream->check_recomp));
    }
  else if (extra->radio_scenario && gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (extra->radio_scenario)))
    {
      avi_save->type = AVI_FILE_SCENARIO;
      avi_save->extension = g_strdup ("scn");
    }
  else
    {
      g_free (avi_save);
      avi_save = NULL;
    }
  g_list_free (glist);
  return avi_save;
}


/*  ja:ビットマップ保存の範囲を設定する
    extra,ウィジェット
    start,最初のフレーム
      end,最後のフレーム
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
avi_extra_set_range (AviExtra   *extra,
                     const gint  start,
                     const gint  end)
{
  if (!extra || !extra->spin_start || !extra->spin_end)
    return FALSE;
  gtk_spin_button_set_value
                    (GTK_SPIN_BUTTON (extra->spin_start), MIN (start, end));
  gtk_spin_button_set_value
                    (GTK_SPIN_BUTTON (extra->spin_end), MAX (start, end));
  return TRUE;
}
