/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"


/******************************************************************************
*                                                                             *
* ja:AVI基本                                                                  *
*                                                                             *
******************************************************************************/
const guint8 rgb2[] = {0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00};
const guint8 rgb16[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00,
        0x00, 0xff, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00,
        0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
        0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xff, 0x00, 0x00,
        0xff, 0xff, 0xff, 0x00};
const guint8 rgb256[] = {
        0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00,
        0xff, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x55, 0x24, 0x00, 0x00,
        0xaa, 0x24, 0x00, 0x00, 0xff, 0x24, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00,
        0x55, 0x49, 0x00, 0x00, 0xaa, 0x49, 0x00, 0x00, 0xff, 0x49, 0x00, 0x00,
        0x00, 0x6d, 0x00, 0x00, 0x55, 0x6d, 0x00, 0x00, 0xaa, 0x6d, 0x00, 0x00,
        0xff, 0x6d, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00, 0x55, 0x92, 0x00, 0x00,
        0xaa, 0x92, 0x00, 0x00, 0xff, 0x92, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00,
        0x55, 0xb6, 0x00, 0x00, 0xaa, 0xb6, 0x00, 0x00, 0xff, 0xb6, 0x00, 0x00,
        0x00, 0xdb, 0x00, 0x00, 0x55, 0xdb, 0x00, 0x00, 0xaa, 0xdb, 0x00, 0x00,
        0xff, 0xdb, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x55, 0xff, 0x00, 0x00,
        0xaa, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00,
        0x55, 0x00, 0x24, 0x00, 0xaa, 0x00, 0x24, 0x00, 0xff, 0x00, 0x24, 0x00,
        0x00, 0x24, 0x24, 0x00, 0x55, 0x24, 0x24, 0x00, 0xaa, 0x24, 0x24, 0x00,
        0xff, 0x24, 0x24, 0x00, 0x00, 0x49, 0x24, 0x00, 0x55, 0x49, 0x24, 0x00,
        0xaa, 0x49, 0x24, 0x00, 0xff, 0x49, 0x24, 0x00, 0x00, 0x6d, 0x24, 0x00,
        0x55, 0x6d, 0x24, 0x00, 0xaa, 0x6d, 0x24, 0x00, 0xff, 0x6d, 0x24, 0x00,
        0x00, 0x92, 0x24, 0x00, 0x55, 0x92, 0x24, 0x00, 0xaa, 0x92, 0x24, 0x00,
        0xff, 0x92, 0x24, 0x00, 0x00, 0xb6, 0x24, 0x00, 0x55, 0xb6, 0x24, 0x00,
        0xaa, 0xb6, 0x24, 0x00, 0xff, 0xb6, 0x24, 0x00, 0x00, 0xdb, 0x24, 0x00,
        0x55, 0xdb, 0x24, 0x00, 0xaa, 0xdb, 0x24, 0x00, 0xff, 0xdb, 0x24, 0x00,
        0x00, 0xff, 0x24, 0x00, 0x55, 0xff, 0x24, 0x00, 0xaa, 0xff, 0x24, 0x00,
        0xff, 0xff, 0x24, 0x00, 0x00, 0x00, 0x49, 0x00, 0x55, 0x00, 0x49, 0x00,
        0xaa, 0x00, 0x49, 0x00, 0xff, 0x00, 0x49, 0x00, 0x00, 0x24, 0x49, 0x00,
        0x55, 0x24, 0x49, 0x00, 0xaa, 0x24, 0x49, 0x00, 0xff, 0x24, 0x49, 0x00,
        0x00, 0x49, 0x49, 0x00, 0x55, 0x49, 0x49, 0x00, 0xaa, 0x49, 0x49, 0x00,
        0xff, 0x49, 0x49, 0x00, 0x00, 0x6d, 0x49, 0x00, 0x55, 0x6d, 0x49, 0x00,
        0xaa, 0x6d, 0x49, 0x00, 0xff, 0x6d, 0x49, 0x00, 0x00, 0x92, 0x49, 0x00,
        0x55, 0x92, 0x49, 0x00, 0xaa, 0x92, 0x49, 0x00, 0xff, 0x92, 0x49, 0x00,
        0x00, 0xb6, 0x49, 0x00, 0x55, 0xb6, 0x49, 0x00, 0xaa, 0xb6, 0x49, 0x00,
        0xff, 0xb6, 0x49, 0x00, 0x00, 0xdb, 0x49, 0x00, 0x55, 0xdb, 0x49, 0x00,
        0xaa, 0xdb, 0x49, 0x00, 0xff, 0xdb, 0x49, 0x00, 0x00, 0xff, 0x49, 0x00,
        0x55, 0xff, 0x49, 0x00, 0xaa, 0xff, 0x49, 0x00, 0xff, 0xff, 0x49, 0x00,
        0x00, 0x00, 0x6d, 0x00, 0x55, 0x00, 0x6d, 0x00, 0xaa, 0x00, 0x6d, 0x00,
        0xff, 0x00, 0x6d, 0x00, 0x00, 0x24, 0x6d, 0x00, 0x55, 0x24, 0x6d, 0x00,
        0xaa, 0x24, 0x6d, 0x00, 0xff, 0x24, 0x6d, 0x00, 0x00, 0x49, 0x6d, 0x00,
        0x55, 0x49, 0x6d, 0x00, 0xaa, 0x49, 0x6d, 0x00, 0xff, 0x49, 0x6d, 0x00,
        0x00, 0x6d, 0x6d, 0x00, 0x55, 0x6d, 0x6d, 0x00, 0xaa, 0x6d, 0x6d, 0x00,
        0xff, 0x6d, 0x6d, 0x00, 0x00, 0x92, 0x6d, 0x00, 0x55, 0x92, 0x6d, 0x00,
        0xaa, 0x92, 0x6d, 0x00, 0xff, 0x92, 0x6d, 0x00, 0x00, 0xb6, 0x6d, 0x00,
        0x55, 0xb6, 0x6d, 0x00, 0xaa, 0xb6, 0x6d, 0x00, 0xff, 0xb6, 0x6d, 0x00,
        0x00, 0xdb, 0x6d, 0x00, 0x55, 0xdb, 0x6d, 0x00, 0xaa, 0xdb, 0x6d, 0x00,
        0xff, 0xdb, 0x6d, 0x00, 0x00, 0xff, 0x6d, 0x00, 0x55, 0xff, 0x6d, 0x00,
        0xaa, 0xff, 0x6d, 0x00, 0xff, 0xff, 0x6d, 0x00, 0x00, 0x00, 0x92, 0x00,
        0x55, 0x00, 0x92, 0x00, 0xaa, 0x00, 0x92, 0x00, 0xff, 0x00, 0x92, 0x00,
        0x00, 0x24, 0x92, 0x00, 0x55, 0x24, 0x92, 0x00, 0xaa, 0x24, 0x92, 0x00,
        0xff, 0x24, 0x92, 0x00, 0x00, 0x49, 0x92, 0x00, 0x55, 0x49, 0x92, 0x00,
        0xaa, 0x49, 0x92, 0x00, 0xff, 0x49, 0x92, 0x00, 0x00, 0x6d, 0x92, 0x00,
        0x55, 0x6d, 0x92, 0x00, 0xaa, 0x6d, 0x92, 0x00, 0xff, 0x6d, 0x92, 0x00,
        0x00, 0x92, 0x92, 0x00, 0x55, 0x92, 0x92, 0x00, 0xaa, 0x92, 0x92, 0x00,
        0xff, 0x92, 0x92, 0x00, 0x00, 0xb6, 0x92, 0x00, 0x55, 0xb6, 0x92, 0x00,
        0xaa, 0xb6, 0x92, 0x00, 0xff, 0xb6, 0x92, 0x00, 0x00, 0xdb, 0x92, 0x00,
        0x55, 0xdb, 0x92, 0x00, 0xaa, 0xdb, 0x92, 0x00, 0xff, 0xdb, 0x92, 0x00,
        0x00, 0xff, 0x92, 0x00, 0x55, 0xff, 0x92, 0x00, 0xaa, 0xff, 0x92, 0x00,
        0xff, 0xff, 0x92, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x55, 0x00, 0xb6, 0x00,
        0xaa, 0x00, 0xb6, 0x00, 0xff, 0x00, 0xb6, 0x00, 0x00, 0x24, 0xb6, 0x00,
        0x55, 0x24, 0xb6, 0x00, 0xaa, 0x24, 0xb6, 0x00, 0xff, 0x24, 0xb6, 0x00,
        0x00, 0x49, 0xb6, 0x00, 0x55, 0x49, 0xb6, 0x00, 0xaa, 0x49, 0xb6, 0x00,
        0xff, 0x49, 0xb6, 0x00, 0x00, 0x6d, 0xb6, 0x00, 0x55, 0x6d, 0xb6, 0x00,
        0xaa, 0x6d, 0xb6, 0x00, 0xff, 0x6d, 0xb6, 0x00, 0x00, 0x92, 0xb6, 0x00,
        0x55, 0x92, 0xb6, 0x00, 0xaa, 0x92, 0xb6, 0x00, 0xff, 0x92, 0xb6, 0x00,
        0x00, 0xb6, 0xb6, 0x00, 0x55, 0xb6, 0xb6, 0x00, 0xaa, 0xb6, 0xb6, 0x00,
        0xff, 0xb6, 0xb6, 0x00, 0x00, 0xdb, 0xb6, 0x00, 0x55, 0xdb, 0xb6, 0x00,
        0xaa, 0xdb, 0xb6, 0x00, 0xff, 0xdb, 0xb6, 0x00, 0x00, 0xff, 0xb6, 0x00,
        0x55, 0xff, 0xb6, 0x00, 0xaa, 0xff, 0xb6, 0x00, 0xff, 0xff, 0xb6, 0x00,
        0x00, 0x00, 0xdb, 0x00, 0x55, 0x00, 0xdb, 0x00, 0xaa, 0x00, 0xdb, 0x00,
        0xff, 0x00, 0xdb, 0x00, 0x00, 0x24, 0xdb, 0x00, 0x55, 0x24, 0xdb, 0x00,
        0xaa, 0x24, 0xdb, 0x00, 0xff, 0x24, 0xdb, 0x00, 0x00, 0x49, 0xdb, 0x00,
        0x55, 0x49, 0xdb, 0x00, 0xaa, 0x49, 0xdb, 0x00, 0xff, 0x49, 0xdb, 0x00,
        0x00, 0x6d, 0xdb, 0x00, 0x55, 0x6d, 0xdb, 0x00, 0xaa, 0x6d, 0xdb, 0x00,
        0xff, 0x6d, 0xdb, 0x00, 0x00, 0x92, 0xdb, 0x00, 0x55, 0x92, 0xdb, 0x00,
        0xaa, 0x92, 0xdb, 0x00, 0xff, 0x92, 0xdb, 0x00, 0x00, 0xb6, 0xdb, 0x00,
        0x55, 0xb6, 0xdb, 0x00, 0xaa, 0xb6, 0xdb, 0x00, 0xff, 0xb6, 0xdb, 0x00,
        0x00, 0xdb, 0xdb, 0x00, 0x55, 0xdb, 0xdb, 0x00, 0xaa, 0xdb, 0xdb, 0x00,
        0xff, 0xdb, 0xdb, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x55, 0xff, 0xdb, 0x00,
        0xaa, 0xff, 0xdb, 0x00, 0xff, 0xff, 0xdb, 0x00, 0x00, 0x00, 0xff, 0x00,
        0x55, 0x00, 0xff, 0x00, 0xaa, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,
        0x00, 0x24, 0xff, 0x00, 0x55, 0x24, 0xff, 0x00, 0xaa, 0x24, 0xff, 0x00,
        0xff, 0x24, 0xff, 0x00, 0x00, 0x49, 0xff, 0x00, 0x55, 0x49, 0xff, 0x00,
        0xaa, 0x49, 0xff, 0x00, 0xff, 0x49, 0xff, 0x00, 0x00, 0x6d, 0xff, 0x00,
        0x55, 0x6d, 0xff, 0x00, 0xaa, 0x6d, 0xff, 0x00, 0xff, 0x6d, 0xff, 0x00,
        0x00, 0x92, 0xff, 0x00, 0x55, 0x92, 0xff, 0x00, 0xaa, 0x92, 0xff, 0x00,
        0xff, 0x92, 0xff, 0x00, 0x00, 0xb6, 0xff, 0x00, 0x55, 0xb6, 0xff, 0x00,
        0xaa, 0xb6, 0xff, 0x00, 0xff, 0xb6, 0xff, 0x00, 0x00, 0xdb, 0xff, 0x00,
        0x55, 0xdb, 0xff, 0x00, 0xaa, 0xdb, 0xff, 0x00, 0xff, 0xdb, 0xff, 0x00,
        0x00, 0xff, 0xff, 0x00, 0x55, 0xff, 0xff, 0x00, 0xaa, 0xff, 0xff, 0x00,
        0xff, 0xff, 0xff, 0x00, 0x70, 0x62, 0x63, 0x6c, 0x65, 0x76, 0x74, 0x75,
        0x67, 0x20, 0x28, 0x70, 0x29, 0x20, 0x31, 0x39, 0x39, 0x38, 0x2d, 0x32,
        0x30, 0x30, 0x36, 0x20, 0x58, 0x6e, 0x6d, 0x68, 0x78, 0x76, 0x20, 0x56,
        0x4a, 0x4e, 0x5a, 0x42, 0x47, 0x42, 0x20, 0x75, 0x67, 0x67, 0x63, 0x3a,
        0x2f, 0x2f, 0x6a, 0x6a, 0x6a, 0x2e, 0x7a, 0x6e, 0x76, 0x71, 0x2e, 0x62,
        0x65, 0x74, 0x2f, 0x20, 0x76, 0x6a, 0x7a, 0x40, 0x7a, 0x6e, 0x76, 0x71,
        0x2e, 0x62, 0x65, 0x74};


/*  ja:AVIファイルを求める
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_get_file (AviEdit    *avi_edit,
                   const gint  pos)
{
  while (pos < avi_edit->offset && avi_edit->file->prev)
    {
      avi_edit->file = avi_edit->file->prev;
      avi_edit->offset -= avi_edit->file->length;
    }
  while (avi_edit->offset + avi_edit->file->length <= pos
                                                    && avi_edit->file->next)
    {
      avi_edit->offset += avi_edit->file->length;
      avi_edit->file = avi_edit->file->next;
    }
}


/*  ja:AVIファイルを分離する
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_split_file (AviEdit    *avi_edit,
                     const gint  pos)
{
  AviFile *avi_file;

  if (pos < 0 || avi_edit_length (avi_edit) <= pos)
    return;

  /* ja:位置を求める */
  avi_base_get_file (avi_edit, pos);/* ja:AVIファイルを求める */
  if (avi_edit->offset == pos)
    return;/* ja:ファイルの切れ目に一致した */

  /* ja:そのファイルを分離する */
  avi_file = g_malloc (sizeof (AviFile));
  avi_file->fio = avi_edit->file->fio;
  if (avi_file->fio)
    fileio_ref (avi_file->fio);
  avi_file->avi_memory = avi_edit->file->avi_memory;
  if (avi_file->avi_memory)
    avi_memory_ref (avi_file->avi_memory);
  avi_file->name = g_strdup (avi_edit->file->name);
  avi_file->start = pos-avi_edit->offset + avi_edit->file->start;
  avi_file->length = avi_edit->file->length - (pos - avi_edit->offset);
  avi_file->entries = avi_edit->file->entries;
  avi_file->param = avi_edit->file->param;
  avi_file->index = g_memdup (avi_edit->file->index,
                                avi_edit->file->entries * sizeof (AviIndex));
  /* ja:ビデオ固有 */
  avi_file->handler = avi_edit->file->handler;
  avi_file->bmih = avi_edit->file->bmih ? g_memdup (avi_edit->file->bmih,
                                bm_header_bytes (avi_edit->file->bmih)) : NULL;
  /* ja:オーディオ固有 */
  avi_file->samples = avi_edit->file->samples;
  avi_file->wfx = avi_edit->file->wfx ? g_memdup (avi_edit->file->wfx,
                                wf_header_bytes (avi_edit->file->wfx)) : NULL;
  /* ja:元のファイルの変更 */
  avi_edit->file->length -= avi_file->length;
  /* ja:リストに挿入する */
  avi_file->prev = avi_edit->file;
  avi_file->next = avi_edit->file->next;
  avi_file->prev->next = avi_file;
  if (avi_file->next)
    avi_file->next->prev = avi_file;
  avi_base_get_file (avi_edit, pos);
}


/******************************************************************************
*                                                                             *
* ja:AVIファイル入力関数                                                      *
*                                                                             *
******************************************************************************/
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
#endif /* USE_THREAD */


/*  ja:AVI(ビデオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
         RET,バイト数                                                       */
gsize
avi_read_video_size (AviFile    *avi_file,
                     const gint  pos)
{
  gint i;

  /* ja:スタート位置の調整 */
  if (!avi_file || !avi_file->bmih || avi_file->start + pos < 0
                                || avi_file->entries <= avi_file->start + pos)
    return 0;
  /* ja:ヌルフレームのチェック */
  for (i = pos + avi_file->start; i > 0; i--)
    if (avi_file->index[i].size > 0)
      break;
  return avi_file->index[i].size;
}


/*  ja:AVI(ビデオ)のデータを読み込む
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_video (AviFile    *avi_file,
                const gint  pos,
                gpointer    buffer)
{
  gint i;
  guint32 size;
  goffset offset;

  /* ja:スタート位置の調整 */
  if (!avi_file || !avi_file->bmih  || !buffer || avi_file->start + pos < 0
                                || avi_file->entries <= avi_file->start + pos)
    return FALSE;
  /* ja:ヌルフレームのチェック */
  for (i = pos + avi_file->start; i > 0; i--)
    if (avi_file->index[i].size > 0)
      break;
  offset = avi_file->index[i].offset;
  size = avi_file->index[i].size;
  if (avi_file->avi_memory)
    {
      /* ja:メモリ */
      g_memmove (buffer,
        (const guint8 *)avi_memory_get_data (avi_file->avi_memory) + offset,
        size);
    }
  else
    {
      /* ja:ファイル */
      gboolean result;

#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      result = fileio_seek (avi_file->fio, offset, FILEIO_SEEK_SET) != -1
                        && fileio_read (avi_file->fio, buffer, size) == size;
# ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
# endif /* USE_THREAD */
      if (!result)
        return FALSE;
    }
  return TRUE;
}


/*  ja:AVI(オーディオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
         RET,バイト数                                                       */
gsize
avi_read_audio_size (AviFile    *avi_file,
                     const gint  start,
                     const gint  samples)
{
  return avi_file && avi_file->wfx
                        ? samples * wfx_get_block_align (avi_file->wfx) : 0;
}


/*  ja:AVI(オーディオ)のデータを読み込む
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_audio (AviFile    *avi_file,
                const gint  start,
                const gint  samples,
                gpointer    buffer)
{
  gint i, real_start, real_samples, buf_bytes = 0, buf_samples = 0;
  gint dif = 0;/* ja:先頭のエントリーの読み飛ばすサンプル数 */

  if (!avi_file || !avi_file->wfx || !buffer || start < 0 || samples <= 0)
    return FALSE;                   /* ja:範囲外 */
  real_samples = CLAMP (avi_audio_real_from_virtual_roundup (avi_file->length,
                                        avi_file->wfx), 1, avi_file->samples);
  real_start = MIN (avi_audio_real_from_virtual (avi_file->start,
                            avi_file->wfx), avi_file->samples - real_samples);
  if (real_samples < start + samples)
    return FALSE;                   /* ja:範囲外 */
  real_start += start;
  /* ja:データの先頭があるエントリーを求める */
  for (i = 0; i < avi_file->entries; i++)
    {
      gint n;

      n = avi_file->index[i].size / wfx_get_block_align (avi_file->wfx);
      if (real_start < dif + n)
        break;
      dif += n;
    }
  dif = real_start - dif;
  /* ja:バッファにファイルのサンプルを確保する */
  while (buf_samples < samples && i < avi_file->entries)
    {
      goffset off;
      gint read_samples, read_bytes;

      read_samples = MIN (samples - buf_samples,
        (avi_file->index[i].size / wfx_get_block_align (avi_file->wfx)) - dif);
      read_bytes = read_samples * wfx_get_block_align (avi_file->wfx);
      off = (goffset)dif * wfx_get_block_align (avi_file->wfx)
                                                + avi_file->index[i].offset;
      if (avi_file->avi_memory)
        {
          /* ja:メモリ */
          g_memmove ((guint8 *)buffer + buf_bytes,
                    (guint8 *)avi_memory_get_data (avi_file->avi_memory) + off,
                                                                read_bytes);
        }
      else
        {
          gboolean result;

#ifdef USE_THREAD
          G_LOCK (critical);
          critical = TRUE;
#endif /* USE_THREAD */
          /* ja:ファイル */
          result = fileio_seek (avi_file->fio, off, FILEIO_SEEK_SET) != -1
                    && fileio_read (avi_file->fio,
                    (guint8 *)buffer + buf_bytes, read_bytes) == read_bytes;
# ifdef USE_THREAD
          critical = FALSE;
          G_UNLOCK (critical);
# endif /* USE_THREAD */
          if (!result)
            return FALSE;
        }
      buf_bytes += read_bytes;
      buf_samples += read_samples;
      dif = 0;
      i++;
    }
  return buf_samples > 0;
}
