/*
    Video maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "change.h"
#include "size.h"
#include "thread.h"


/******************************************************************************
*                                                                             *
* ja:フレーム数変更関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:フレームを変更する(独立スレッド)
    chgfrm,フレーム数変更構造体
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
change_frames (ChangeFrames *chgfrm)
{
  gint i;
  VmaidWindow *vmaid;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  id = g_thread_self ();
#endif /* USE_THREAD */
  thread_insert (id, THREAD_MODE_CHANGE, NULL);

  vmaid = g_malloc0 (sizeof (VmaidWindow));
  for (i = 0; i < chgfrm->frames; i++)
    {
      AviEdit *avi_edit_tmp;

      if (!thread_idling (id, i * 100 / chgfrm->frames))
        {
          avi_edit_close (vmaid->avi_edit[0]);
          vmaid->avi_edit[0] = NULL;
          break;
        }
      avi_edit_tmp = avi_edit_copy (chgfrm->avi_edit,
                avi_edit_length (chgfrm->avi_edit) * i / chgfrm->frames, 1);
      if (!avi_edit_tmp)
        {
          avi_edit_close (vmaid->avi_edit[0]);
          vmaid->avi_edit[0] = NULL;
          while (thread_idling (id, -1));
          break;
        }
      if (vmaid->avi_edit[0])
        {
          if (!avi_edit_paste (vmaid->avi_edit[0], i, avi_edit_tmp))
            {
              avi_edit_close (avi_edit_tmp);
              avi_edit_close (vmaid->avi_edit[0]);
              vmaid->avi_edit[0] = NULL;
              while (thread_idling (id, -1));
              break;
            }
        }
      else
        {
          vmaid->avi_edit[0] = avi_edit_tmp;
        }
    }
  if (!avi_edit_close (chgfrm->avi_edit))
    {
      avi_edit_close (vmaid->avi_edit[0]);
      vmaid->avi_edit[0] = NULL;
      while (thread_idling (id, -1));
    }
  g_free (chgfrm);
  if (vmaid->avi_edit[0])
    {
      vmaid->selfrm = default_selfrm;
      vmaid->select.stream = -1;
      vmaid->select.frame = -1;
      size_set_scale (default_view, &vmaid->width, &vmaid->height,
                                    avi_edit_get_width (vmaid->avi_edit[0]),
                                    avi_edit_get_height (vmaid->avi_edit[0]));
      vmaid->rate = avi_edit_get_rate (vmaid->avi_edit[0]);
      vmaid->scale = avi_edit_get_scale (vmaid->avi_edit[0]);
      thread_window (vmaid, NULL);
    }
  else
    {
      g_free (vmaid);
      vmaid = NULL;
    }
  thread_delete (id);
  return vmaid != NULL;
}


/******************************************************************************
*                                                                             *
* ja:フレーム数変更ダイアログ関数群                                           *
*                                                                             *
******************************************************************************/
/* ja:エントリーでリターンが押された */
static void
change_dialog_activate (GtkWidget *widget,
                        GtkWidget *dialog)
{
  gint value;
  ChangeConfig *chgcfg;

  chgcfg = g_object_get_data (G_OBJECT (dialog), "user_data");
  if (misc_str_to_val (&value,
            gtk_entry_get_text (GTK_ENTRY (widget)), 10, FALSE) && value > 0)
    gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}


/* ja:エントリーが変更された */
static void
change_dialog_changed (GtkWidget *widget,
                       GtkWidget *dialog)
{
  gint value;

  if (misc_str_to_val (&value,
                        gtk_entry_get_text (GTK_ENTRY (widget)), 10, FALSE))
    {
      GtkWidget *label;
      ChangeConfig *chgcfg;

      label = g_object_get_data (G_OBJECT (widget), "user_data");
      chgcfg = g_object_get_data (G_OBJECT (dialog), "user_data");
      if (label)
        {
          gchar *text;
          gint t;

          t = (glonglong)value * chgcfg->scale * 1000 / chgcfg->rate;
          text = g_strdup_printf ("%02d:%02d:%02d",
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
          gtk_label_set_text (GTK_LABEL (label), text);
          g_free (text);
        }
      gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK,
                                                                    value > 0);
    }
  else
    {
      gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK,
                                                                        FALSE);
    }
}


/* ja:ボタンが押された */
static void
change_dialog_clicked (GtkWidget    *widget,
                       ChangeConfig *chgcfg)
{
  gchar *text;

  text = misc_str_from_val ((gint)MAX ((glonglong)chgcfg->times * chgcfg->rate
                        / ((glonglong)chgcfg->scale * 1000), 1), 10, 0, FALSE);
  gtk_entry_set_text
        (GTK_ENTRY (g_object_get_data (G_OBJECT (widget), "user_data")), text);
  g_free (text);
}


/*  ja:フレーム数変更ダイアログ表示
    chgcfg,フレーム数変更構造体
       RET.TRUE:OK,FALSE:Cancel                                             */
gboolean
change_dialog (ChangeConfig *chgcfg)
{
  gboolean result;
  gchar *text;
  GtkWidget *dialog, *button, *entry, *label, *vbox;

  /* ja:メインウインドウ */
  dialog = gtk_dialog_new_with_buttons
                            (_("Change Frames"),
                            GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
#ifdef G_OS_WIN32
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
#endif /* not G_OS_WIN32 */
                            NULL);
  g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
  g_object_set_data (G_OBJECT (dialog), "user_data", chgcfg);
  /* ja:ラベル */
  label = chgcfg->rate > 0 && chgcfg->scale > 0 ? gtk_label_new (NULL) : NULL;
  /* ja:エントリー */
  entry = gtk_entry_new ();
  g_object_set_data (G_OBJECT (entry), "user_data", label);
  g_signal_connect (G_OBJECT (entry), "changed",
                                G_CALLBACK (change_dialog_changed), dialog);
  text = misc_str_from_val (chgcfg->frames, 10, 0, FALSE);
  gtk_entry_set_text (GTK_ENTRY (entry), text);
  g_free (text);
  g_signal_connect (G_OBJECT (entry), "activate",
                                G_CALLBACK (change_dialog_activate), dialog);
  gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);
  /* ja:ボタン */
  if (chgcfg->times > 0)
    {
      button = gtk_button_new_with_mnemonic (_("Justify audio stream"));
      g_signal_connect (G_OBJECT (button), "clicked",
                                G_CALLBACK (change_dialog_clicked), chgcfg);
      g_object_set_data (G_OBJECT (button), "user_data", entry);
    }
  else
    {
      button = NULL;
    }
  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  if (button)
    {
      GtkWidget *hbox;

      hbox = gtk_hbox_new (FALSE, SPACING);
      gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
      gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
    }
  else
    {
      gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
    }
  if (label)
    gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);

  /* ja:表示 */
  gtk_widget_grab_focus (entry);
  gtk_widget_show_all (vbox);

  result = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK;
  if (result)
    misc_str_to_val (&chgcfg->frames,
                            gtk_entry_get_text (GTK_ENTRY (entry)), 10, FALSE);
  gtk_widget_destroy (dialog);

  return result;
}
