/*
    Video maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "file.h"
#include "size.h"
#include "thread.h"
#include "misc/fileio.h"


/******************************************************************************
*                                                                             *
* ja:ファイル入力関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:AVIファイルを開く
     avi,AVI編集構造体
    file,ファイル名                                                         */
static void
file_open_avi (AviEdit     *avi_edit[],
               const gchar *file)
{
  AviEdit **avi_edit_tmp;

  avi_edit[0] = avi_edit[1] = NULL;
  avi_edit_tmp = avi_edit_open (file);
  if (avi_edit_tmp)
    {
      gint i;

      for (i = 0; avi_edit_tmp[i]; i++)
        switch (avi_edit_type (avi_edit_tmp[i]))
          {
            case AVI_TYPE_VIDEO:
              if (!avi_edit[0])
                avi_edit[0] = avi_edit_tmp[i];
              else
                avi_edit_close (avi_edit_tmp[i]);
              break;
            case AVI_TYPE_AUDIO:
              if (!avi_edit[1])
                avi_edit[1] = avi_edit_tmp[i];
              else
                avi_edit_close (avi_edit_tmp[i]);
              break;
            default:
              avi_edit_close (avi_edit_tmp[i]);
          }
      g_free (avi_edit_tmp);
    }
}


/*  ja:ファイルを開く(独立スレッド)
    file_open,Open情報
     RET,ウインドウ情報                                                     */
VmaidWindow *
file_open_edit (FileOpen *file_open)
{
  gint i;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */
  VmaidWindow *vmaid;

#ifdef USE_THREAD
  id = g_thread_self ();
#endif /* USE_THREAD */
  thread_insert (id, THREAD_MODE_OPEN, file_open ? file_open->file : NULL);

  vmaid = g_malloc0 (sizeof (VmaidWindow));
  vmaid->selfrm = default_selfrm;
  vmaid->select.stream = -1;
  vmaid->select.frame = -1;
  if (file_open && file_open->file)
    {
      gchar *format = NULL;
      gint j, value, max = 0;

      file_open_avi (vmaid->avi_edit, file_open->file);
      if ((vmaid->avi_edit[0] || vmaid->avi_edit[1]) && file_open->merge)
        {
          gchar *name;

          /* ja:連番 */
          name = g_path_get_basename (file_open->file);
          for (i = g_strlen (name) - 1; i >= 0; i--)
            if ('0' <= name[i] && name[i] <= '9')
              break;
          if (i >= 0)
            {
              gchar *dir, *head, *foot;

              dir = g_path_get_dirname (file_open->file);
              head = g_strdup (name);
              foot = name + i + 1;
              for (j = i - 1; j >= 0; j--)
                if (head[j] < '0' || '9' < head[j])
                  break;
              misc_str_to_val (&value, head + j + 1, 10, FALSE);
              head[j + 1] = '\0';
              format = g_strdup_printf ("%s/%s%%0%dd%s",
                                                    dir, head, i - j, foot);
              g_free (dir);
              g_free (head);
              max = 1;
              while (j < i)
                {
                  max *= 10;
                  j++;
                }
            }
          g_free (name);
        }
      if (format)
        {
          gboolean result = TRUE;
          gchar *file;
          AviEdit *avi_edit_old[2], *avi_edit_tmp[2];

          /* ja:連番 */
          for (i = value - 1; i >= 0; i--)
            {
              /* ja:連番前 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_edit_dup (vmaid->avi_edit[j]);
                    if (!avi_edit_paste (vmaid->avi_edit[j],
                                                        0, avi_edit_tmp[j]))
                      result = FALSE;
                    }
                  else
                    {
                      avi_edit_old[j] = NULL;
                      vmaid->avi_edit[j] = avi_edit_tmp[j];
                    }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_edit_close (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_edit_close (vmaid->avi_edit[j]);
                      vmaid->avi_edit[j] = avi_edit_old[j];
                    }
                  break;
                }
            }
          for (i = value + 1, result = TRUE; i < max && result; i++)
            {
              /* ja:連番後 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_edit_dup (vmaid->avi_edit[j]);
                    if (!avi_edit_paste (vmaid->avi_edit[j],
                        avi_edit_length (vmaid->avi_edit[j]), avi_edit_tmp[j]))
                      result = FALSE;
                  }
                else
                  {
                    avi_edit_old[j] = NULL;
                    vmaid->avi_edit[j] = avi_edit_tmp[j];
                  }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_edit_close (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_edit_close (vmaid->avi_edit[j]);
                      vmaid->avi_edit[j] = avi_edit_old[j];
                    }
                }
            }
        }
    }

  if (!thread_idling (id, 100))
    {
      g_free (file_open->file);
      g_free (file_open);
      for (i = 0; i < 2; i++)
        if (vmaid->avi_edit[i])
          avi_edit_close (vmaid->avi_edit[i]);
      g_free (vmaid);
      thread_delete (id);
      return NULL;
    }

  if (vmaid->avi_edit[0])
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height,
                                    avi_edit_get_width (vmaid->avi_edit[0]),
                                    avi_edit_get_height (vmaid->avi_edit[0]));
      vmaid->rate = avi_edit_get_rate (vmaid->avi_edit[0]);
      vmaid->scale = avi_edit_get_scale (vmaid->avi_edit[0]);
    }
  else
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height, -1, -1);
      vmaid->rate = default_rate;
      vmaid->scale = 1;
      if (vmaid->avi_edit[1])
        vmaid->cursor.stream = 1;
    }

  thread_window (vmaid, file_open ? file_open->file : NULL);
  /* ja:新規作成ではなく、ファイルが存在する時には履歴に加える */
  if (file_open && g_file_test (file_open->file, G_FILE_TEST_EXISTS))
    thread_history (file_open->file);

  if (file_open)
    {
      g_free (file_open->file);
      g_free (file_open);
    }
  thread_delete (id);

  return vmaid;
}


/******************************************************************************
*                                                                             *
* ja:ファイル出力関数群                                                       *
*                                                                             *
******************************************************************************/
static gboolean
file_save_edit_callback (gint percent, gpointer user_data)
{
#ifdef USE_THREAD
  return thread_idling (g_thread_self (), percent);
#else /* not USE_THREAD */
  return thread_idling (THREAD_ID_STANDARD, percent);
#endif /* not USE_THREAD */
}


/*  ja:AVIファイルを保存する(独立スレッド)
    file_save,Save情報
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
file_save_edit (FileSave *file_save)
{
  gboolean result;
  gint i;
#ifdef USE_THREAD
  GThread *id;
  id = g_thread_self ();
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */

  thread_insert (id, THREAD_MODE_SAVE, file_save->file);

  result = avi_save_with_options (file_save->avi_edit, file_save->file,
                        file_save->avi_save, file_save_edit_callback, NULL);
  g_free (file_save->file);
  for (i = 0; i < 2; i++)
    if (file_save->avi_edit[i])
      avi_edit_close (file_save->avi_edit[i]);
  avi_save_free (file_save->avi_save);
  g_free (file_save);

  if (!result)
    while (thread_idling (id, -1));

  thread_delete (id);
  return result;
}
