/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "icm.h"
#include "chunk.h"
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
# include <tchar.h>
# include <windows.h>
# include <vfw.h>
# ifdef G_OS_WIN32
#  include <gdk/gdkwin32.h>
# endif /* G_OS_WIN32 */
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


struct _IcmObject
{
  gint counter;
  guint32 handler;
  gint mode;
  gint flags;
  gchar *name;
  BitmapInfoHeader *bmih_in;
  BitmapInfoHeader *bmih_out;
  gsize size;
  gint frame;
  gint key_point;
  gint key_frame;
  gint quality;
  gpointer prev;
  gsize prev_size;
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  HDRVR hDrvr;
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
};


/******************************************************************************
*                                                                             *
* ja:ICM初期化                                                                *
*                                                                             *
******************************************************************************/
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static GHashTable *ghash_handler = NULL;


static guint32
icm_handler_tolower (const guint32 key)
{
  gint i;
  guint32 a = 0;

  for (i = 24; i >= 0; i -= 8)
    {
      a <<= 8;
      a |= (guchar)g_ascii_tolower ((gchar)(key >> i));
    }
  return a;
}


/*static guint32
icm_handler_toupper (const guint32 key)
{
  gint i;
  guint32 a = 0;

  for (i = 24; i >= 0; i -= 8)
    {
      a <<= 8;
      a |= (guchar)g_ascii_toupper ((gchar)(key >> i));
    }
  return a;
}*/


static guint
icm_hash_handler (gconstpointer key)
{
  return icm_handler_tolower (GPOINTER_TO_UINT (key));
}


static gboolean
icm_equal_handler (gconstpointer a,
                   gconstpointer b)
{
  return icm_handler_tolower (GPOINTER_TO_UINT (a))
                                == icm_handler_tolower (GPOINTER_TO_UINT (b));
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ICMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_init (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  DWORD dwIndex, dwName, dwType, dwValue;
  HKEY hKey;
  TCHAR szName[MAX_PATH], szValue[MAX_PATH];

  if (ghash_handler)
    return FALSE;
  ghash_handler = g_hash_table_new (icm_hash_handler, icm_equal_handler);
  if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
        _T("Software\\Microsoft\\Windows NT\\CurrentVersion\\Drivers32"), 0,
                                    KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS)
    return FALSE;
  for (dwIndex = 0; dwName = MAX_PATH * sizeof (TCHAR),
                    dwValue = MAX_PATH * sizeof (TCHAR),
            RegEnumValue (hKey, dwIndex, szName, &dwName, NULL,
            &dwType, (LPBYTE)szValue, &dwValue) == ERROR_SUCCESS; dwIndex++)
    if (dwType == REG_SZ
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[0]))) == 'V'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[1]))) == 'I'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[2]))) == 'D'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[3]))) == 'C'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[4]))) == '.')
      {
        gchar *key;
# ifndef UNICODE
        gchar *utf8str;
# endif /* not UNICODE */
        gunichar2 *value;
        guint32 handler;

# ifdef UNICODE
        key = g_utf16_to_utf8 (szName + 5, -1, NULL, NULL, NULL);
        value = g_memdup (szName, MAX_PATH * sizeof (gunichar2));
# else /* not UNICODE */
        key = g_locale_to_utf8 (szName + 5, -1, NULL, NULL, NULL);
        utf8str = g_locale_to_utf8 (szName, -1, NULL, NULL, NULL);
        value = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
        g_free (utf8str);
# endif /* not UNICODE */
        handler = icm_handler_tolower (icm_handler_from_string (key));
        g_free (key);
        if (key && value && handler != 0
                                        && !g_hash_table_lookup (ghash_handler,
                                                GUINT_TO_POINTER (handler)))
          g_hash_table_insert (ghash_handler,
                                            GUINT_TO_POINTER (handler), value);
        else
          g_free (value);
      }
  return RegCloseKey (hKey) == ERROR_SUCCESS;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static void
icm_exit_callback (gpointer key,
                   gpointer value,
                   gpointer ghash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable *)ghash, key,
                                                    &orig_key, &orig_value))
    g_free (orig_value);
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ICMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_exit (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (ghash_handler)
    {
      g_hash_table_foreach (ghash_handler, icm_exit_callback, ghash_handler);
      g_hash_table_destroy (ghash_handler);
      ghash_handler = NULL;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:ICM4文字コード処理                                                       *
*                                                                             *
******************************************************************************/
/*  ja:4文字コード→文字列
    handler,4文字コード
        RET,文字列                                                          */
gchar *
icm_handler_to_string (const guint32 handler)
{
  gchar c, *text = NULL, *temp0, *temp1;
  gint i;

  for (i = 24; i >= 0; i -= 8)
    {
      c = (handler >> i) & 0xff;
      temp0 = g_strdup_printf (g_ascii_isalnum (c) ? "%c" : "%%%02x",
                                                                    (guint8)c);
      temp1 = g_strconcat (temp0, text, NULL);
      g_free (temp0);
      g_free (text);
      text = temp1;
    }
  return text;
}


/*  ja:4文字コード←文字列
    text,文字列
     RET,4文字コード                                                        */
guint32
icm_handler_from_string (const gchar *text)
{
  gint i, j = 0, k;
  guint32 handler = 0;

  for (i = 0; i < 4; i++)
    {
      handler >>= 8;
      if (text[j] == '%')
        {
          j++;
          for (k = 28; k >= 24; k -= 4)
            {
              if ('0' <= text[j] && text[j] <= '9')
                handler |= (text[j] - '0') << k;
              else if ('A' <= text[j] && text[j] <= 'A')
                handler |= (text[j] - 'A') << k;
              else if ('a' <= text[j] && text[j] <= 'f')
                handler |= (text[j] - 'a')<<k;
              else
                break;
              j++;
            }
          if (k >= 24)
            return 0;
        }
      else if (text[j] != '\0')
        {
          handler |= text[j++] << 24;
        }
      else
        {
          return 0;
        }
    }
  return handler;
}


/******************************************************************************
*                                                                             *
* ja:ICM基本                                                                  *
*                                                                             *
******************************************************************************/
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static gint
icm_get_handler_list_compare (gconstpointer key0,
                              gconstpointer key1)
{
  gchar *str0, *str1;
  gint result;

  str0 = icm_handler_to_string (GPOINTER_TO_UINT (key0));
  str1 = icm_handler_to_string (GPOINTER_TO_UINT (key1));
  result = g_strcmp (str0, str1);
  g_free (str0);
  g_free (str1);
  return result;
}


static void
icm_get_handler_list_callback (gpointer   key,
                               gpointer   value,
                               GList    **glist)
{
  *glist = g_list_insert_sorted (*glist, key, icm_get_handler_list_compare);
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:登録されている4文字コードを取得する
    RET,4文字コードのリスト,NULL:登録なし                                   */
GList *
icm_get_handler_list (void)
{
  GList *glist = NULL;

#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (ghash_handler)
    g_hash_table_foreach (ghash_handler,
                                (GHFunc)icm_get_handler_list_callback, &glist);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return glist;
}


/*  ja:Codecを開く
    handler,4文字コード
       mode,モード
        RET,IcmObject,NULL:エラー                                           */
IcmObject *
icm_open (const guint32 handler,
          const gint    mode)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gunichar2 *name;
  IcmObject *icm_object = NULL;

  name = g_hash_table_lookup (ghash_handler, GUINT_TO_POINTER (handler));
  if (name)
    {
      HDRVR hDrvr;
      ICOPEN icopen;

      g_memset (&icopen, 0, sizeof (ICOPEN));
      icopen.dwSize = sizeof (ICOPEN);
      icopen.fccType = ICTYPE_VIDEO;
      icopen.fccHandler = handler;
      switch (mode)
        {
          case ICM_MODE_COMPRESS:
            icopen.dwFlags = ICMODE_COMPRESS;
            break;
          case ICM_MODE_DECOMPRESS:
            icopen.dwFlags = ICMODE_DECOMPRESS;
            break;
          case ICM_MODE_FASTDECOMPRESS:
            icopen.dwFlags = ICMODE_FASTDECOMPRESS;
            break;
          case ICM_MODE_QUERY:
            icopen.dwFlags = ICMODE_QUERY;
            break;
          case ICM_MODE_FASTCOMPRESS:
            icopen.dwFlags = ICMODE_FASTCOMPRESS;
            break;
          case ICM_MODE_DRAW:
            icopen.dwFlags = ICMODE_DRAW;
        }
      hDrvr = OpenDriver (name, NULL, GPOINTER_TO_INT (&icopen));
      if (hDrvr)
        {
          ICINFO icinfo;

          icm_object = g_malloc0 (sizeof (IcmObject));
          icm_object->counter = 1;
          icm_object->handler = handler;
          icm_object->mode = mode;
          icm_object->hDrvr = hDrvr;
          g_memset (&icinfo, 0, sizeof (ICINFO));
          icinfo.dwSize = sizeof (ICINFO);
          SendDriverMessage (icm_object->hDrvr, ICM_GETINFO,
                                GPOINTER_TO_INT (&icinfo), sizeof (ICINFO));
          icm_object->name = g_utf16_to_utf8 (icinfo.szDescription, -1,
                                                            NULL, NULL, NULL);
          if (!icm_object->name || icm_object->name[0] == '\0')
            {
              g_free (icm_object->name);
              icm_object->name = g_utf16_to_utf8 (icinfo.szName, -1,
                                                            NULL, NULL, NULL);
            }
          if (!icm_object->name || icm_object->name[0] == '\0')
            {
              g_free (icm_object->name);
              icm_object->name = g_utf16_to_utf8 (icinfo.szDriver, -1,
                                                            NULL, NULL, NULL);
            }
          if (!icm_object->name || icm_object->name[0] == '\0')
            {
              g_free (icm_object->name);
              icm_object->name = icm_handler_to_string (handler);
            }
          if (icinfo.dwFlags & VIDCF_QUALITY)
            icm_object->flags = ICM_FLAG_QUALITY;
          if (icinfo.dwFlags & VIDCF_CRUNCH)
            icm_object->flags |= ICM_FLAG_CRUNCH;
          if (icinfo.dwFlags & VIDCF_TEMPORAL)
            icm_object->flags |= ICM_FLAG_TEMPORAL;
          if (icinfo.dwFlags & VIDCF_COMPRESSFRAMES)
            icm_object->flags |= ICM_FLAG_COMPRESSFRAMES;
          if (icinfo.dwFlags & VIDCF_DRAW)
            icm_object->flags |= ICM_FLAG_DRAW;
          if (icinfo.dwFlags & VIDCF_FASTTEMPORALC)
            icm_object->flags |= ICM_FLAG_FASTTEMPORALC;
          if (icinfo.dwFlags & VIDCF_FASTTEMPORALD)
            icm_object->flags |= ICM_FLAG_FASTTEMPORALD;
        }
    }
  return icm_object;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return NULL;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Codecの参照数を増やす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_ref (IcmObject *icm_object)
{
  return icm_object ? icm_object->counter++, TRUE : FALSE;
}


/*  ja:Codecの参照数を減らす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_unref (IcmObject *icm_object)
{
  if (icm_object && --(icm_object->counter) <= 0)
    {
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
      CloseDriver (icm_object->hDrvr, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
      g_free (icm_object->name);
      g_free (icm_object->bmih_in);
      g_free (icm_object->bmih_out);
      g_free (icm_object->prev);
      g_free (icm_object);
    }
  return TRUE;
}


/*  ja:Codecを複製する
    icm_object,IcmObject
           RET,IcmObject,NULL:エラー                                        */
IcmObject *
icm_dup (IcmObject *icm_object)
{
  return icm_object ? icm_open (icm_object->handler, icm_object->mode) : NULL;
}


/******************************************************************************
*                                                                             *
* ja:ICM情報                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:Codecの4文字コードを取得する
    icm_object,IcmObject
           RET,4文字コード,0:エラー                                         */
guint32
icm_get_handler (IcmObject *icm_object)
{
  return icm_object ? icm_object->handler : 0;
}


/*  ja:Codecの名前を取得する
    icm_object,IcmObject
           RET,名前,NULL:エラー                                             */
const gchar *
icm_get_name (IcmObject *icm_object)
{
  return icm_object ? icm_object->name : NULL;
}


/*  ja:Codecのフラグを取得する
    icm_object,IcmObject
           RET,フラグ                                                       */
gint
icm_get_flags (IcmObject *icm_object)
{
  return icm_object ? icm_object->flags : 0;
}


/*  ja:Codecの属性を取得する
    icm_object,IcmObject
           RET,ICM_ATTR_NATIVE,ICM_ATTR_WIN32                               */
gint
icm_get_attribute (IcmObject *icm_object)
{
  return ICM_ATTR_WIN32;
}


/******************************************************************************
*                                                                             *
* ja:ICMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_about (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr
        && SendDriverMessage (icm_object->hDrvr, ICM_ABOUT, -1, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Aboutダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_about (IcmObject *icm_object,
                  GtkWidget *widget)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr)
    SendDriverMessage (icm_object->hDrvr, ICM_ABOUT,
# if defined (G_OS_WIN32)
        widget ? GPOINTER_TO_INT (GDK_WINDOW_HWND (widget->window)) : 0, 0);
# elif defined (USE_W32LDR)
                                                GPOINTER_TO_INT (widget), 0);
# endif /* defined (USE_W32LDR) */
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_configure (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr
    && SendDriverMessage (icm_object->hDrvr, ICM_CONFIGURE, -1, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:設定ダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_configure (IcmObject *icm_object,
                      GtkWidget *widget)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr)
    SendDriverMessage (icm_object->hDrvr, ICM_CONFIGURE,
# if defined (G_OS_WIN32)
        widget ? GPOINTER_TO_INT (GDK_WINDOW_HWND (widget->window)) : 0, 0);
# elif defined (USE_W32LDR)
                                                GPOINTER_TO_INT (widget), 0);
# endif /* defined (USE_W32LDR) */
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:ドライバ設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_preference (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr
            && SendDriverMessage (icm_object->hDrvr, DRV_QUERYCONFIGURE, 0, 0);
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:ドライバ設定ダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_preference (IcmObject *icm_object,
                       GtkWidget *widget)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr)
    SendDriverMessage (icm_object->hDrvr, DRV_CONFIGURE,
# if defined (G_OS_WIN32)
        widget ? GPOINTER_TO_INT (GDK_WINDOW_HWND (widget->window)) : 0, 0);
# elif defined (USE_W32LDR)
                                                GPOINTER_TO_INT (widget), 0);
# endif /* defined (USE_W32LDR) */
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/******************************************************************************
*                                                                             *
* ja:ICMパラメータ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:パラメータのサイズを取得する
    icm_object,IcmObject
           RET,サイズ                                                       */
gsize
icm_get_state_size (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr
            ? SendDriverMessage (icm_object->hDrvr, ICM_GETSTATE, 0, 0) : 0;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:パラメータを取得する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_get_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr && param && param_size > 0
                        && SendDriverMessage (icm_object->hDrvr, ICM_GETSTATE,
                            GPOINTER_TO_INT (param), param_size) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:パラメータを設定する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_set_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr && param && param_size > 0)
    {
      SendDriverMessage (icm_object->hDrvr, ICM_SETSTATE,
                                        GPOINTER_TO_INT (param), param_size);
      return TRUE;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
}


/******************************************************************************
*                                                                             *
* en:ICM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_compress_query (IcmObject              *icm_object,
                    const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr && bmih
                    && SendDriverMessage (icm_object->hDrvr,
                    ICM_COMPRESS_QUERY, GPOINTER_TO_INT (bmih), 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮のデフォルトの周期を取得する
    icm_object,IcmObject
           RET,デフォルト(0:未対応)                                         */
gint
icm_compress_get_default_key_frame (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr
                        && (icm_get_flags (icm_object) & ICM_FLAG_TEMPORAL))
    {
      DWORD dwKeyFrame;

      if (SendDriverMessage (icm_object->hDrvr, ICM_GETDEFAULTKEYFRAMERATE,
                                GPOINTER_TO_INT (&dwKeyFrame), 0) == ICERR_OK)
        return dwKeyFrame;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
}


/*  ja:圧縮のデフォルトの品質を取得する
    icm_object,IcmObject
           RET,デフォルト(-1:未対応)                                        */
gint
icm_compress_get_default_quality (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)

  if (icm_object && icm_object->hDrvr
                            && (icm_get_flags (icm_object) & ICM_FLAG_QUALITY))
    {
      DWORD dwQuality;

      if (SendDriverMessage (icm_object->hDrvr, ICM_GETDEFAULTQUALITY,
                                GPOINTER_TO_INT (&dwQuality), 0) == ICERR_OK)
        return dwQuality;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return -1;
}


/*  ja:圧縮後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_format_size (IcmObject              *icm_object,
                              const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr || !bmih)
    return 0;
  return SendDriverMessage (icm_object->hDrvr,
                        ICM_COMPRESS_GET_FORMAT, GPOINTER_TO_INT (bmih), 0);
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_get_format (IcmObject              *icm_object,
                         const BitmapInfoHeader *bmih_in,
                         BitmapInfoHeader       *bmih_out)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr && bmih_in && bmih_out
        && SendDriverMessage (icm_object->hDrvr, ICM_COMPRESS_GET_FORMAT,
            GPOINTER_TO_INT (bmih_in), GPOINTER_TO_INT (bmih_out)) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮後のデータの最大サイズを取得する
    icm_object,IcmObject
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_size (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr
                                && icm_object->bmih_in && icm_object->bmih_out
                ? SendDriverMessage (icm_object->hDrvr, ICM_COMPRESS_GET_SIZE,
                                GPOINTER_TO_INT (icm_object->bmih_in),
                                GPOINTER_TO_INT (icm_object->bmih_out)) : 0;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮を設定する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
     bmih_size,圧縮されたビットマップのヘッタのサイズ
     key_frame,周期(1:すべてキーフレーム,2～:周期,0:デフォルト)
       quality,品質0～10000,-1:デフォルト
     data_rate,1秒間のサイズ(0:無効)
          rate,レート
         scale,スケール
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_frames_info (IcmObject              *icm_object,
                          const BitmapInfoHeader *bmih_in,
                          const BitmapInfoHeader *bmih_out,
                          const gsize             bmih_size,
                          const gint              key_frame,
                          const gint              quality,
                          const gsize             data_rate,
                          const guint32           rate,
                          const guint32           scale)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr)
    return FALSE;
  if (bmih_in)
    {
      g_free (icm_object->bmih_in);
      icm_object->bmih_in = g_memdup (bmih_in, bm_header_bytes (bmih_in));
    }
  if (!icm_object->bmih_in)
    return FALSE;
  if (bmih_out && bmih_size > 0)
    {
      g_free (icm_object->bmih_out);
      icm_object->bmih_out = g_memdup (bmih_out, bmih_size);
    }
  else if (!icm_object->bmih_out)
    {
      icm_object->bmih_out
            = g_malloc (icm_compress_get_format_size (icm_object, bmih_in));
      if (!icm_compress_get_format (icm_object, bmih_in, icm_object->bmih_out))
        {
          g_free (icm_object->bmih_out);
          icm_object->bmih_out = NULL;
          return FALSE;
        }
    }
  if (0 <= quality && quality <= 10000)
    icm_object->quality = quality;
  else if (quality == -1)
    icm_object->quality = icm_compress_get_default_quality (icm_object);
  if (icm_object->quality < 0 && 10000 < icm_object->quality)
    icm_object->quality = 0;
  icm_object->key_frame = key_frame >= 1 ? key_frame
                    : MAX (icm_compress_get_default_key_frame (icm_object), 1);
  if (icm_get_flags (icm_object) & ICM_FLAG_COMPRESSFRAMES)
    {
      ICCOMPRESSFRAMES iccompressframes;

      g_memset (&iccompressframes, 0, sizeof (ICCOMPRESSFRAMES));
      iccompressframes.lpbiOutput = (LPBITMAPINFOHEADER)icm_object->bmih_out;
      iccompressframes.lpbiInput = (LPBITMAPINFOHEADER)icm_object->bmih_in;
      iccompressframes.lQuality = icm_object->quality;
      iccompressframes.lDataRate = data_rate > 0 ? data_rate : G_MAXINT;
      iccompressframes.lKeyRate = icm_object->key_frame;
      iccompressframes.dwRate = MAX (rate, 1);
      iccompressframes.dwScale = MAX (scale, 1);
      SendDriverMessage (icm_object->hDrvr, ICM_COMPRESS_FRAMES_INFO,
            GPOINTER_TO_INT (&iccompressframes), sizeof (ICCOMPRESSFRAMES));
    }
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮を開始する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_begin (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr
                                || !icm_object->bmih_in || icm_object->prev)
    return FALSE;
  if (!icm_object->bmih_out)
    {
      icm_object->bmih_out = g_malloc (icm_compress_get_format_size
                                            (icm_object, icm_object->bmih_in));
      if (!icm_compress_get_format (icm_object,
                                    icm_object->bmih_in, icm_object->bmih_out))
        {
          g_free (icm_object->bmih_out);
          icm_object->bmih_out = NULL;
          return FALSE;
        }
    }
  if (SendDriverMessage (icm_object->hDrvr, ICM_COMPRESS_BEGIN,
                            GPOINTER_TO_INT (icm_object->bmih_in),
                            GPOINTER_TO_INT (icm_object->bmih_out)) == ICERR_OK
                && (icm_object->size = icm_compress_get_size (icm_object)) > 0)
    {
      icm_object->frame = 0;
      icm_object->key_point = 0;
      icm_object->prev_size = bm_image_bytes (icm_object->bmih_in);
      icm_object->prev = g_malloc (icm_object->prev_size);
      return TRUE;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
}


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_end (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr || !icm_object->prev)
    return FALSE;
  g_free (icm_object->prev);
  icm_object->prev = NULL;
  return SendDriverMessage (icm_object->hDrvr,
                                        ICM_COMPRESS_END, 0, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮する
    icm_object,IcmObject
     key_frame,TRUE:キーフレーム,FALSE:非キーフレーム
         input,圧縮するデータ
        output,圧縮されたデータ
           RET,フレームのサイズ,0:エラー                                    */
gsize
icm_compress (IcmObject      *icm_object,
              gboolean       *key_frame,
              const gpointer  input,
              gpointer        output)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  DWORD dwFlags,dwID;
  ICCOMPRESS iccompress;

  if (!icm_object || !icm_object->hDrvr || !input || !output
                            || !icm_object->bmih_in || !icm_object->bmih_out
                            || !icm_object->prev || icm_object->prev_size <= 0)
    return 0;
  iccompress.dwFlags = icm_object->key_point % icm_object->key_frame == 0
                                                    ? ICCOMPRESS_KEYFRAME : 0;
  iccompress.lpbiOutput = (LPBITMAPINFOHEADER)icm_object->bmih_out;
  iccompress.lpOutput = output;
  iccompress.lpbiInput = (LPBITMAPINFOHEADER)icm_object->bmih_in;
  iccompress.lpInput = input;
  iccompress.lpckid = &dwID;
  iccompress.lpdwFlags = &dwFlags;
  iccompress.lFrameNum = icm_object->frame;
  iccompress.dwFrameSize = icm_object->size;
  iccompress.dwQuality = icm_object->quality;
  if (icm_object->frame > 0)
    {
      iccompress.lpbiPrev = (LPBITMAPINFOHEADER)icm_object->bmih_in;
      iccompress.lpPrev = icm_object->prev;
    }
  else
    {
      iccompress.lpbiPrev = NULL;
      iccompress.lpPrev = NULL;
    }
  if (SendDriverMessage (icm_object->hDrvr, ICM_COMPRESS,
                GPOINTER_TO_INT (&iccompress), sizeof (ICCOMPRESS)) == ICERR_OK
            && bmih_get_size_image (icm_object->bmih_out) <= icm_object->size)
    {
      icm_object->frame++;
      if (dwFlags & AVIIF_KEYFRAME)
        icm_object->key_point = 1;
      else
        icm_object->key_point++;
      g_memmove (icm_object->prev, input, icm_object->prev_size);
      if (key_frame)
        *key_frame = (dwFlags & AVIIF_KEYFRAME) != 0;
      return bmih_get_size_image (icm_object->bmih_out);
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
}


/******************************************************************************
*                                                                             *
* en:ICM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_decompress_query (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return icm_object && icm_object->hDrvr && bmih
            && SendDriverMessage (icm_object->hDrvr,
                ICM_DECOMPRESS_QUERY, GPOINTER_TO_INT (bmih), 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:展開後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_decompress_get_format_size (IcmObject              *icm_object,
                                const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gsize bmih_size = 0;

  if (icm_object && icm_object->hDrvr && bmih)
    {
      gssize size;

      size = SendDriverMessage (icm_object->hDrvr,
                        ICM_DECOMPRESS_GET_FORMAT, GPOINTER_TO_INT (bmih), 0);
      if (size > 0)
        {
          BitmapInfoHeader *bmih_out;

          bmih_out = g_malloc (size);
          if (SendDriverMessage (icm_object->hDrvr, ICM_DECOMPRESS_GET_FORMAT,
                                    GPOINTER_TO_INT (bmih),
                                    GPOINTER_TO_INT (bmih_out)) == ICERR_OK)
            {
              bmih_size = size;
              if (bmih_size == bmih_get_size (bmih_out)
                        && bm_color_bytes (bmih) == bm_color_bytes (bmih_out))
                bmih_size += bm_color_bytes (bmih_out);
            }
          g_free (bmih_out);
        }
    }
  return bmih_size;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:展開後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮されたビットマップのヘッタ
      bmih_out,展開されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_get_format (IcmObject              *icm_object,
                           const BitmapInfoHeader *bmih_in,
                           BitmapInfoHeader       *bmih_out)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (icm_object && icm_object->hDrvr && bmih_in && bmih_out
        && SendDriverMessage (icm_object->hDrvr, ICM_DECOMPRESS_GET_FORMAT,
            GPOINTER_TO_INT (bmih_in), GPOINTER_TO_INT (bmih_out)) == ICERR_OK)
    {
      gssize bmih_size;

      bmih_size = SendDriverMessage (icm_object->hDrvr,
                    ICM_DECOMPRESS_GET_FORMAT, GPOINTER_TO_INT (bmih_in), 0);
      if (bmih_size > 0)
        {
          if (bmih_size == bmih_get_size (bmih_out)
                    && bm_color_bytes (bmih_in) == bm_color_bytes (bmih_out))
            g_memmove ((guint8 *)bmih_out + bmih_get_size (bmih_out),
                       (guint8 *)bmih_in  + bmih_get_size (bmih_in),
                       bm_color_bytes (bmih_out));
          return TRUE;
        }
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
}


/*  ja:展開を開始する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_begin (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr
                    || !bmih || icm_object->bmih_in || icm_object->bmih_out)
    return FALSE;
  icm_object->bmih_out
                = g_malloc (icm_decompress_get_format_size (icm_object, bmih));
  if (!icm_decompress_get_format (icm_object,bmih, icm_object->bmih_out)
                || SendDriverMessage (icm_object->hDrvr, ICM_DECOMPRESS_BEGIN,
                        GPOINTER_TO_INT (bmih),
                        GPOINTER_TO_INT (icm_object->bmih_out)) != ICERR_OK)
    {
      g_free (icm_object->bmih_out);
      icm_object->bmih_out = NULL;
      return FALSE;
    }
  icm_object->bmih_in = g_memdup (bmih, bm_header_bytes (bmih));
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_end (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!icm_object || !icm_object->hDrvr
                            || (!icm_object->bmih_in && !icm_object->bmih_out))
    return FALSE;
  g_free (icm_object->bmih_in);
  g_free (icm_object->bmih_out);
  icm_object->bmih_in = NULL;
  icm_object->bmih_out = NULL;
  return SendDriverMessage (icm_object->hDrvr,
                                        ICM_DECOMPRESS_END, 0, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:展開する
    icm_object,IcmObject
         flags,フラグ
         input,圧縮されたデータ
        output,展開されたデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress (IcmObject      *icm_object,
                const guint     flags,
                const gpointer  input,
                gpointer        output)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  ICDECOMPRESS icdecompress;

  if (!icm_object || !icm_object->hDrvr || !input || !output
                            || !icm_object->bmih_in || !icm_object->bmih_out)
    return FALSE;
  icdecompress.dwFlags = 0;
  if (flags & ICM_DECOMPRESS_HURRYUP)
    icdecompress.dwFlags |= ICDECOMPRESS_HURRYUP;
  if (flags & ICM_DECOMPRESS_UPDATE)
    icdecompress.dwFlags |= ICDECOMPRESS_UPDATE;
  if (flags & ICM_DECOMPRESS_PREROLL)
    icdecompress.dwFlags |= ICDECOMPRESS_PREROLL;
  if (flags & ICM_DECOMPRESS_NULLFRAME)
    icdecompress.dwFlags |= ICDECOMPRESS_NULLFRAME;
  if (flags & ICM_DECOMPRESS_NOTKEYFRAME)
    icdecompress.dwFlags |= ICDECOMPRESS_NOTKEYFRAME;
  icdecompress.lpbiInput = (LPBITMAPINFOHEADER)icm_object->bmih_in;
  icdecompress.lpInput = input;
  icdecompress.lpbiOutput = (LPBITMAPINFOHEADER)icm_object->bmih_out;
  icdecompress.lpOutput = output;
  icdecompress.ckid = 0;
  return SendDriverMessage (icm_object->hDrvr, ICM_DECOMPRESS,
        GPOINTER_TO_INT (&icdecompress), sizeof (ICDECOMPRESS)) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined () */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}
