/*
    Video maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "change.h"
#include "codec.h"
#include "command.h"
#include "file.h"
#include "general.h"
#include "jump.h"
#include "merge.h"
#include "preview.h"
#include "prop.h"
#include "sigfile.h"
#include "size.h"
#include "thread.h"
#include "version.h"
#include "avicore/aviclip.h"
#include "avicore/aviextra.h"
#include "misc/fileio.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"
#include "orz/orzpref.h"


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (GtkWidget *widget,
             gpointer   user_data)
{
#ifdef USE_THREAD
  g_thread_create ((GThreadFunc)file_open_edit, NULL, TRUE, NULL);
#else /* not USE_THREAD */
  file_open_edit (NULL);
#endif /* not USE_THREAD */
}


typedef struct _CommandOpenOk CommandOpenOk;
struct _CommandOpenOk
{
  gboolean merge;
  gchar *file;
  GtkWidget *dialog, *check;
};


#if ! GTK_CHECK_VERSION(2,4,0)
static void
command_open_ok (GtkWidget     *widget,
                 CommandOpenOk *cmd_ok)
{
  const gchar *file;

  file = gtk_file_selection_get_filename (GTK_FILE_SELECTION (cmd_ok->dialog));
  cmd_ok->file = file && file[0] != '\0'
                && file[g_strlen (file) - 1] != '/' ? g_strdup (file) : NULL;
  cmd_ok->merge = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON (cmd_ok->check));
  gtk_widget_destroy (cmd_ok->dialog);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


void
command_open (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *path, *title;
  GtkStockItem stock_item;
  CommandOpenOk cmd_ok;
  FileOpen *file_open;

  gtk_stock_lookup (GTK_STOCK_OPEN, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  cmd_ok.file = NULL;
  cmd_ok.merge = file_merge;
  cmd_ok.check = gtk_check_button_new_with_label (_("Continual"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cmd_ok.check), file_merge);
#if GTK_CHECK_VERSION(2,4,0)
  cmd_ok.dialog = gtk_file_chooser_dialog_new (title,
                            GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
# ifdef G_OS_WIN32
                            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
# else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
# endif /* not G_OS_WIN32 */
                            NULL);
  if (open_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (cmd_ok.dialog),
                                                                    open_path);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (cmd_ok.dialog),
                                                                cmd_ok.check);
  if (gtk_dialog_run (GTK_DIALOG (cmd_ok.dialog)) == GTK_RESPONSE_ACCEPT)
    {
      cmd_ok.file = gtk_file_chooser_get_filename
                                            (GTK_FILE_CHOOSER (cmd_ok.dialog));
      cmd_ok.merge = gtk_toggle_button_get_active
                                            (GTK_TOGGLE_BUTTON (cmd_ok.check));
    }
  gtk_widget_destroy (cmd_ok.dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_ok.dialog = gtk_file_selection_new (title);
  g_signal_connect (G_OBJECT (cmd_ok.dialog), "destroy", gtk_main_quit, NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), &cmd_ok);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                    G_OBJECT (cmd_ok.dialog));
  if (open_path)
    {
      path = g_strconcat (open_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename
                                    (GTK_FILE_SELECTION (cmd_ok.dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_ok.dialog));
  gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (cmd_ok.dialog)->main_vbox),
                                                cmd_ok.check, FALSE, FALSE, 0);
  gtk_widget_show (cmd_ok.check);
  gtk_widget_show (cmd_ok.dialog);
  gtk_grab_add (cmd_ok.dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_free (title);
  if (cmd_ok.file)
    {
      g_free (open_path);
      path = fileio_get_full_path (cmd_ok.file);
      open_path = g_path_get_dirname (path);
      g_free (path);
      file_open = g_malloc (sizeof (FileOpen));
      file_open->file = cmd_ok.file;
      file_open->merge = file_merge = cmd_ok.merge;
#ifdef USE_THREAD
      g_thread_create ((GThreadFunc)file_open_edit, file_open, TRUE, NULL);
#else /* not USE_THREAD */
      file_open_edit (file_open);
#endif /* not USE_THREAD */
    }
}


void
command_close (GtkWidget *widget,
               gpointer   user_data)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0
    && prompt_close ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), page)))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
}


typedef struct _CommandSaveOk CommandSaveOk;
struct _CommandSaveOk
{
  gboolean addext;
  gchar *file;
  AviSave *avi_save;
  GtkWidget *dialog, *check, *extra;
};


#if ! GTK_CHECK_VERSION(2,4,0)
static void
command_save_ok (GtkWidget     *widget,
                 CommandSaveOk *cmd_ok)
{
  const gchar *file;

  file = gtk_file_selection_get_filename (GTK_FILE_SELECTION (cmd_ok->dialog));
  cmd_ok->file = file && file[0] != '\0'
                && file[g_strlen (file) - 1] != '/' ? g_strdup (file) : NULL;
  cmd_ok->avi_save = avi_extra_get_option (AVI_EXTRA (cmd_ok->extra));
  cmd_ok->addext = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON (cmd_ok->check));
  gtk_widget_destroy (cmd_ok->dialog);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


void
command_save (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *path, *title;
  gint i = 0;
  GtkStockItem stock_item;
  AviEdit *avi_edit[3];
  CommandSaveOk cmd_ok;
  FileSave *file_save;
  VmaidWindow *vmaid;
  GtkWidget *vbox;
#if ! GTK_CHECK_VERSION(2,4,0)
  GtkWidget *hbox_base, *hbox_button, *vbox_base;
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  if (vmaid->avi_edit[0])
    avi_edit[i++] = vmaid->avi_edit[0];
  if (vmaid->avi_edit[1])
    avi_edit[i++] = vmaid->avi_edit[1];
  avi_edit[i] = NULL;
  gtk_stock_lookup (GTK_STOCK_SAVE_AS, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  cmd_ok.addext = addext;
  cmd_ok.file = NULL;
  cmd_ok.avi_save = NULL;
  cmd_ok.extra = avi_extra_new ();
  avi_extra_set_handle (AVI_EXTRA (cmd_ok.extra), avi_edit);
  if (vmaid->avi_edit[0])
    avi_extra_set_range (AVI_EXTRA (cmd_ok.extra),
        MIN (vmaid->cursor.frame, avi_edit_length (avi_edit[0]) - 1),
        MIN (vmaid->select.stream < 0 ? vmaid->cursor.frame
                    : vmaid->select.frame, avi_edit_length (avi_edit[0]) - 1));
  cmd_ok.check = gtk_check_button_new_with_mnemonic
                                        (_("_Automatic file name extension"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cmd_ok.check), addext);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), cmd_ok.check, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), cmd_ok.extra, TRUE, TRUE, 0);
  gtk_widget_show_all (vbox);
#if GTK_CHECK_VERSION(2,4,0)
  cmd_ok.dialog = gtk_file_chooser_dialog_new (title,
                            GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
# ifdef G_OS_WIN32
                            GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
# else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
# endif /* not G_OS_WIN32 */
                            NULL);
  if (save_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (cmd_ok.dialog),
                                                                    save_path);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (cmd_ok.dialog), vbox);
  if (gtk_dialog_run (GTK_DIALOG (cmd_ok.dialog)) == GTK_RESPONSE_ACCEPT)
    {
      cmd_ok.file = gtk_file_chooser_get_filename
                                            (GTK_FILE_CHOOSER (cmd_ok.dialog));
      cmd_ok.avi_save = avi_extra_get_option (AVI_EXTRA (cmd_ok.extra));
      cmd_ok.addext = gtk_toggle_button_get_active
                                            (GTK_TOGGLE_BUTTON (cmd_ok.check));
    }
  gtk_widget_destroy (cmd_ok.dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_ok.dialog = gtk_file_selection_new (title);
  g_signal_connect (G_OBJECT (cmd_ok.dialog), "destroy", gtk_main_quit, NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_save_ok), &cmd_ok);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                    G_OBJECT (cmd_ok.dialog));
  if (save_path)
    {
      path = g_strconcat (save_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename
                                    (GTK_FILE_SELECTION (cmd_ok.dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_ok.dialog));
  /* ja:フレームとボックス */
  hbox_button = gtk_widget_get_parent
                            (GTK_FILE_SELECTION (cmd_ok.dialog)->ok_button);
  gtk_widget_ref (hbox_button);
  gtk_container_remove (GTK_CONTAINER (GTK_FILE_SELECTION
                                    (cmd_ok.dialog)->main_vbox), hbox_button);
  gtk_widget_ref (GTK_FILE_SELECTION (cmd_ok.dialog)->main_vbox);
  gtk_container_remove (GTK_CONTAINER (cmd_ok.dialog),
                                GTK_FILE_SELECTION (cmd_ok.dialog)->main_vbox);

  hbox_base = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox_base),
                GTK_FILE_SELECTION (cmd_ok.dialog)->main_vbox, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox_base), vbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox_base);

  vbox_base = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox_base), hbox_base, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (vbox_base), hbox_button, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (cmd_ok.dialog), vbox_base);
  gtk_widget_show (vbox_base);

  gtk_widget_show (cmd_ok.dialog);
  gtk_grab_add (cmd_ok.dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_free (title);
  if (cmd_ok.file && cmd_ok.avi_save)
    {
      if (cmd_ok.addext)
        {
          gchar *basename;

          basename = g_path_get_basename (cmd_ok.file);
          if (!g_strchr (basename, '.'))
            {
              const gchar *ext;

              ext = avi_save_get_extension (cmd_ok.avi_save);
              if (ext)
                {
                  gchar *tmp;

                  tmp = g_strconcat (cmd_ok.file, ".", ext, NULL);
                  g_free (cmd_ok.file);
                  cmd_ok.file = tmp;
                }
            }
          g_free (basename);
        }
      addext = cmd_ok.addext;
      g_free (save_path);
      path = fileio_get_full_path (cmd_ok.file);
      save_path = g_path_get_dirname (path);
      g_free (path);
      if (avi_save_is_all (avi_edit, cmd_ok.avi_save))
        {
          /* ja:すべて保存されるとき */
          orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), vmaid), FALSE);
          delete_list (&vmaid->undo);
          delete_list (&vmaid->redo);
          set_menu_bar (vmaid);
          orz_history_add_file (ORZ_HISTORY (history), cmd_ok.file);
        }
      file_save = g_malloc (sizeof (FileSave));
      file_save->file = cmd_ok.file;
      for (i = 0; avi_edit[i]; i++)
        file_save->avi_edit[i] = avi_edit_dup (avi_edit[i]);
      file_save->avi_edit[i] = NULL;
      file_save->avi_save = cmd_ok.avi_save;
      /* ja:保存 */
#ifdef USE_THREAD
      g_thread_create ((GThreadFunc)file_save_edit, file_save, TRUE, NULL);
#else /* not USE_THREAD */
      file_save_edit (file_save);
#endif /* not USE_THREAD */
    }
  else
    {
      g_free (cmd_ok.file);
      avi_save_free (cmd_ok.avi_save);
    }
}


void
command_preview (GtkWidget *widget,
                 gpointer   user_data)
{
  preview ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}


void
command_property (GtkWidget *widget,
                  gpointer   user_data)
{
  VmaidWindow *vmaid;
  PropertyConfig propcfg;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  g_memset (&propcfg, 0, sizeof (PropertyConfig));
  propcfg.rate = vmaid->rate;
  propcfg.scale = vmaid->scale;
  if (vmaid->avi_edit[0])
    {
      propcfg.priority0 = avi_edit_get_priority (vmaid->avi_edit[0]);
      propcfg.language0 = avi_edit_get_language (vmaid->avi_edit[0]);
      if (avi_edit_get_name (vmaid->avi_edit[0]))
        propcfg.name0 = g_strdup (avi_edit_get_name (vmaid->avi_edit[0]));
      propcfg.width = avi_edit_get_width (vmaid->avi_edit[0]);
      propcfg.height = avi_edit_get_height (vmaid->avi_edit[0]);
      propcfg.bit_count = avi_edit_get_bit_count (vmaid->avi_edit[0]);
      propcfg.x_pixels_per_meter
                        = avi_edit_get_x_pixels_per_meter (vmaid->avi_edit[0]);
      propcfg.y_pixels_per_meter
                        = avi_edit_get_y_pixels_per_meter (vmaid->avi_edit[0]);
    }
  else
    {
      propcfg.x_pixels_per_meter = -1;
      propcfg.y_pixels_per_meter = -1;
    }
  if (vmaid->avi_edit[1])
    {
      propcfg.priority1 = avi_edit_get_priority (vmaid->avi_edit[1]);
      propcfg.language1 = avi_edit_get_language (vmaid->avi_edit[1]);
      if (avi_edit_get_name (vmaid->avi_edit[1]))
        propcfg.name1 = g_strdup (avi_edit_get_name (vmaid->avi_edit[1]));
      propcfg.samples_per_sec
                        = avi_edit_get_samples_per_sec (vmaid->avi_edit[1]);
      propcfg.channels = avi_edit_get_channels (vmaid->avi_edit[1]);
      propcfg.bits_per_sample
                        = avi_edit_get_bits_per_sample (vmaid->avi_edit[1]);
    }
  if (property_dialog (&propcfg))
    {
      gint max, sx;
    
      vmaid->rate = propcfg.rate;
      vmaid->scale = propcfg.scale;
      if (vmaid->avi_edit[0])
        {
          avi_edit_set_priority (vmaid->avi_edit[0], propcfg.priority0);
          avi_edit_set_language (vmaid->avi_edit[0], propcfg.language0);
          avi_edit_set_name (vmaid->avi_edit[0], propcfg.name0);
          avi_edit_set_rate (vmaid->avi_edit[0], propcfg.rate);
          avi_edit_set_scale (vmaid->avi_edit[0], propcfg.scale);
          avi_edit_set_width (vmaid->avi_edit[0], propcfg.width);
          avi_edit_set_height (vmaid->avi_edit[0], propcfg.height);
          avi_edit_set_bit_count (vmaid->avi_edit[0], propcfg.bit_count);
          avi_edit_set_x_pixels_per_meter (vmaid->avi_edit[0],
                                                propcfg.x_pixels_per_meter);
          avi_edit_set_y_pixels_per_meter (vmaid->avi_edit[0],
                                                propcfg.y_pixels_per_meter);
          g_free (propcfg.name0);
        }
      if (vmaid->avi_edit[1])
        {
          avi_edit_set_priority (vmaid->avi_edit[1], propcfg.priority1);
          avi_edit_set_language (vmaid->avi_edit[1], propcfg.language1);
          avi_edit_set_name (vmaid->avi_edit[1], propcfg.name1);
          avi_edit_set_samples_per_sec (vmaid->avi_edit[1],
                                                    propcfg.samples_per_sec);
          avi_edit_set_channels (vmaid->avi_edit[1], propcfg.channels);
          avi_edit_set_bits_per_sample (vmaid->avi_edit[1],
                                                    propcfg.bits_per_sample);
          g_free (propcfg.name1);
        }
      delete_list (&vmaid->undo);
      delete_list (&vmaid->redo);
      max = get_max_frame (vmaid, -1) + 1;
      sx = get_sx_frame (vmaid);
      vmaid->select.stream = -1;
      if (vmaid->cursor.frame > max)
        vmaid->cursor.frame = max;
      /* ja:キャレットが表示されるように調整 */
      if (vmaid->top > MIN (vmaid->cursor.frame, max - sx))
        vmaid->top = MIN (vmaid->cursor.frame, max - sx);
      if (vmaid->top < MAX (vmaid->cursor.frame - sx, 0))
        vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
      /* ja:メニュー,ステートメント,子ウインドウ */
      set_menu_bar (vmaid);
      gtk_widget_queue_draw (vmaid->drawing);
      draw_caret (vmaid, NULL);
      misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
    }
}


void
command_exit (GtkWidget *widget,
              gpointer   user_data)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_undo (GtkWidget *widget,
              gpointer   user_data)
{
  history_operation ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1),
                                                                        FALSE);
}


void
command_redo (GtkWidget *widget,
              gpointer   user_data)
{
  history_operation ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1),
                                                                        TRUE);
}


void
command_cut (GtkWidget *widget,
             gpointer   user_data)
{
  command_copy (widget, user_data);
  command_delete (widget, user_data);
}


void
command_copy (GtkWidget *widget,
              gpointer   user_data)
{
  gint i = 0, start, end, length, max;
  AviEdit *avi_edit[3];
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  /* ja:ビデオ */
  if (MIN (vmaid->cursor.stream, vmaid->select.stream) == 0)
    {
      max = avi_edit_length (vmaid->avi_edit[0]);
      start = MAX (MIN (vmaid->cursor.frame, vmaid->select.frame), 0);
      end = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame), max);
      length = end - start;
      if (length > 0 && (avi_edit[i]
                        = avi_edit_copy (vmaid->avi_edit[0], start, length)))
        i++;
    }
  /* ja:オーディオ */
  if (MAX (vmaid->cursor.stream, vmaid->select.stream) == 1)
    {
      max = avi_edit_length (vmaid->avi_edit[1]);
      start = MAX (avi_time_to_sample (vmaid->avi_edit[1], (gint)
            ((glonglong)MIN (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                            * vmaid->scale / vmaid->rate)), 0);
      end = MIN (avi_time_to_sample (vmaid->avi_edit[1], (gint)
            ((glonglong)MAX (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                        * vmaid->scale / vmaid->rate)), max);
      length = end - start;
      if (length > 0 && (avi_edit[i]
                        = avi_edit_copy (vmaid->avi_edit[1], start, length)))
        i++;
    }
  if (i > 0)
    {
      avi_edit[i] = NULL;
      avi_clip_copy (AVI_CLIP (clipboard), avi_edit);
      for (i = 0; avi_edit[i]; i++)
        avi_edit_close (avi_edit[i]);
    }
}


void
command_paste (GtkWidget *widget,
               gpointer   user_data)
{
  avi_clip_paste (AVI_CLIP (clipboard));
}


void
command_delete (GtkWidget *widget,
                gpointer   user_data)
{
  gboolean result = TRUE;
  gint i, start, end, length, max, sx;
  VmaidHistory *d;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  /* ja:UNDOのための処理 */
  d = g_malloc0 (sizeof (VmaidHistory));
  for (i = 0; i < 2; i++)
    if (vmaid->avi_edit[i]
                    && !(d->avi_edit[i] = avi_edit_dup (vmaid->avi_edit[i])))
      result = FALSE;
  if (!result)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            _("AVI Clone Error"));
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      for (i = 0; i < 2; i++)
        if (d->avi_edit[i])
          avi_edit_close (d->avi_edit[i]);
      g_free (d);
      return;
    }
  d->cursor = vmaid->cursor;
  d->select = vmaid->select;
  /* ja:ビデオ */
  if (MIN (vmaid->cursor.stream, vmaid->select.stream) == 0)
    {
      max = avi_edit_length (vmaid->avi_edit[0]);
      start = MAX (MIN (vmaid->cursor.frame, vmaid->select.frame), 0);
      end = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame), max);
      if (start == 0 && end == max)
        {
          if (!avi_edit_close (vmaid->avi_edit[0]))
            result = FALSE;
          vmaid->avi_edit[0] = NULL;
        }
      else
        {
          length = end - start;
          if (length > 0
                    && !avi_edit_delete (vmaid->avi_edit[0], start, length))
            result = FALSE;
        }
    }
  /* ja:オーディオ */
  if (MAX (vmaid->cursor.stream, vmaid->select.stream) == 1)
    {
      max = avi_edit_length (vmaid->avi_edit[1]);
      start = MAX (avi_time_to_sample (vmaid->avi_edit[1], (gint)
            ((glonglong)MIN (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                            * vmaid->scale / vmaid->rate)), 0);
      end = MIN (avi_time_to_sample (vmaid->avi_edit[1], (gint)
            ((glonglong)MAX (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                        * vmaid->scale / vmaid->rate)), max);
      if (start == 0 && end == max)
        {
          if (!avi_edit_close (vmaid->avi_edit[1]))
            result=FALSE;
          vmaid->avi_edit[1] = NULL;
        }
      else
        {
          length = end - start;
          if (length > 0
                    && !avi_edit_delete (vmaid->avi_edit[1], start, length))
            result = FALSE;
        }
    }
  if (!result)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            _("AVI Delete Error"));
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      for (i = 0; i < 2; i++)
        {
          if (vmaid->avi_edit[i])
            avi_edit_close (vmaid->avi_edit[i]);
          vmaid->avi_edit[i] = d->avi_edit[i];
        }
      g_free (d);
      return;
    }
  d->next = vmaid->undo;
  vmaid->undo = d;
  delete_list (&vmaid->redo);
  max = get_max_frame (vmaid, -1) + 1;
  sx = get_sx_frame (vmaid);
  vmaid->cursor.frame = (vmaid->avi_edit[0] || vmaid->avi_edit[1])
                        ? MIN (vmaid->cursor.frame, vmaid->select.frame) : 0;
  vmaid->cursor.stream = !vmaid->avi_edit[0] && !vmaid->avi_edit[1]
                                            ? -1 : vmaid->avi_edit[0] ? 0 : 1;
  vmaid->select.stream = -1;
  if (vmaid->cursor.frame > max)
    vmaid->cursor.frame = max;
  /* ja:キャレットが表示されるように調整 */
  if (vmaid->top > MIN (vmaid->cursor.frame, max - sx))
    vmaid->top = MIN (vmaid->cursor.frame, max - sx);
  if (vmaid->top < MAX (vmaid->cursor.frame - sx, 0))
    vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
  /* ja:メニュー,ステートメント,子ウインドウ */
  set_menu_bar (vmaid);
  set_usize (vmaid);
  draw_caret (vmaid, NULL);
  misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), vmaid), TRUE);
}


void
command_merge (GtkWidget *widget,
               gpointer   user_data)
{
  merge_dialog ();
}


void
command_change (GtkWidget *widget,
                gpointer   user_data)
{
  gint st = 0, ed = 0;
  ChangeConfig chgcfg;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  if (vmaid->cursor.stream == 0 && vmaid->select.stream == 0)
    {
      st = MIN (vmaid->cursor.frame, vmaid->select.frame);
      ed = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame),
                                        avi_edit_length (vmaid->avi_edit[0]));
    }
  if (ed - st <= 1)
    {
      st = 0;
      ed = avi_edit_length (vmaid->avi_edit[0]);
      chgcfg.times = vmaid->avi_edit[1] ? avi_time_length (vmaid->avi_edit[1])
                                        : 0;
    }
  else
    {
      chgcfg.times = 0;
    }
  chgcfg.frames = ed - st;
  chgcfg.rate = vmaid->rate;
  chgcfg.scale = vmaid->scale;
  if (change_dialog (&chgcfg))
    {
      ChangeFrames *chgfrm;

      chgfrm = g_malloc (sizeof (ChangeFrames));
      chgfrm->frames = chgcfg.frames;
      chgfrm->avi_edit = avi_edit_copy (vmaid->avi_edit[0], st, ed - st);
#ifdef USE_THREAD
      g_thread_create ((GThreadFunc)change_frames, chgfrm, TRUE, NULL);
#else /* not USE_THREAD */
      change_frames (chgfrm);
#endif /* not USE_THREAD */
    }
}


void
command_reverse (GtkWidget *widget,
                 gpointer   user_data)
{
  gint st = 0, ed = 0;
  AviEdit *avi_edit;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  if (vmaid->cursor.stream == 0 || vmaid->select.stream == 0)
    {
      st = MIN (vmaid->cursor.frame, vmaid->select.frame);
      ed = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame),
                                        avi_edit_length (vmaid->avi_edit[0]));
    }
  if (ed - st <= 1)
    {
      st = 0;
      ed = avi_edit_length (vmaid->avi_edit[0]);
    }
  avi_edit = avi_edit_copy (vmaid->avi_edit[0], st, ed - st);
#ifdef USE_THREAD
  g_thread_create ((GThreadFunc)reverse_frame, avi_edit, TRUE, NULL);
#else /* not USE_THREAD */
  reverse_frame (avi_edit);
#endif /* not USE_THREAD */
}


void
command_jump (GtkWidget *widget,
              gpointer   user_data)
{
  gint sx, top;
  JumpConfig jmpcfg;
  VmaidCursor cursor, select;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  cursor = vmaid->cursor;
  select = vmaid->select;
  top = vmaid->top;
  jmpcfg.video = vmaid->avi_edit[0] != NULL;
  jmpcfg.audio = vmaid->avi_edit[1] != NULL;
  jmpcfg.max = get_max_frame (vmaid, -1) + 1;
  jmpcfg.rate = vmaid->rate;
  jmpcfg.scale = vmaid->scale;
  jmpcfg.cursor = vmaid->cursor;
  jmpcfg.select = vmaid->select;
  if (jump_dialog (&jmpcfg))
    {
      vmaid->cursor = jmpcfg.cursor;
      vmaid->select = jmpcfg.select;
      sx = get_sx_frame (vmaid);
      /* ja:キャレットが表示されるように調整 */
      if (vmaid->top > MIN (vmaid->cursor.frame, jmpcfg.max - sx))
        vmaid->top = MIN (vmaid->cursor.frame, jmpcfg.max - sx);
      if (vmaid->top < MAX (vmaid->cursor.frame - sx, 0))
        vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
      /* ja:ジャンプの例外 ジャンプ先のフレームを表示する */
      if (vmaid->cursor.frame < jmpcfg.max
                                    && vmaid->top + sx <= vmaid->cursor.frame)
        vmaid->top = vmaid->cursor.frame - sx + 1;
      if ((vmaid->select.stream >= 0 && select.stream < 0)
                        || (vmaid->select.stream < 0 && select.stream >= 0))
        set_menu_bar (vmaid);
      if (vmaid->top != top)
        misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                        vmaid, 0, jmpcfg.max, sx, vmaid->top);
      clear_sel (vmaid, &cursor, &select, top);
      draw_caret (vmaid, &cursor);
    }
}


void
command_all (GtkWidget *widget,
             gpointer   user_data)
{
  gint max, sx, top;
  VmaidCursor cursor, select;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  if (vmaid->avi_edit[0] || vmaid->avi_edit[1])
    {
      cursor = vmaid->cursor;
      select = vmaid->select;
      top = vmaid->top;
      sx = get_sx_frame (vmaid);
      max = get_max_frame (vmaid, -1) + 1;
      vmaid->select.frame = 0;
      vmaid->cursor.frame = max;
      vmaid->select.stream = vmaid->avi_edit[0] ? 0 : 1;
      vmaid->cursor.stream = vmaid->avi_edit[1] ? 1 : 0;
      /* ja:キャレットが表示されるように調整 */
      if (vmaid->top > MIN (vmaid->cursor.frame, max - sx))
        vmaid->top = MIN (vmaid->cursor.frame, max - sx);
      if (vmaid->top < MAX (vmaid->cursor.frame - sx, 0))
        vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
      if (select.stream < 0)
        set_menu_bar (vmaid);
      if (vmaid->top != top)
        misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
      clear_sel (vmaid, &cursor, &select, top);
      draw_caret (vmaid, &cursor);
    }
}


void
command_size (GtkWidget *widget,
              gpointer   user_data)
{
  SizeConfig sizecfg;
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  sizecfg.view = default_view;
  sizecfg.ratio = default_ratio;
  sizecfg.width = vmaid->width;
  sizecfg.height = vmaid->height;
  if (vmaid->avi_edit[0])
    {
      sizecfg.def_width = avi_edit_get_width (vmaid->avi_edit[0]);
      sizecfg.def_height = avi_edit_get_height (vmaid->avi_edit[0]);
    }
  else
    {
      sizecfg.def_width = 0;
      sizecfg.def_height = 0;
    }
  if (size_dialog (&sizecfg))
    {
      default_view = sizecfg.view;
      default_ratio = sizecfg.ratio;
      vmaid->width = sizecfg.width;
      vmaid->height = sizecfg.height;
      set_usize (vmaid);
    }
}


void
command_frame (GtkWidget *widget,
               gpointer   user_data)
{
  VmaidWindow *vmaid;

  vmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  vmaid->selfrm = gtk_check_menu_item_get_active
        (GTK_CHECK_MENU_ITEM (misc_find_menu (menu_entries, "/view/frame")));
  set_usize (vmaid);
  default_selfrm = vmaid->selfrm;
}


void
command_pref (GtkWidget *widget,
              gpointer   user_data)
{
  GtkWidget *dialog;

  dialog = orz_pref_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (window));
  orz_pref_set_history (ORZ_PREF (dialog),
                                orz_history_get_num (ORZ_HISTORY (history)));
  orz_pref_set_newfile (ORZ_PREF (dialog), newfile);
  orz_pref_set_second (ORZ_PREF (dialog), second);
  orz_pref_set_size (ORZ_PREF (dialog), def_width, def_height);
  orz_pref_set_tab (ORZ_PREF (dialog), n_pos);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      orz_history_set_num (ORZ_HISTORY (history),
                                    orz_pref_get_history (ORZ_PREF (dialog)));
      newfile = orz_pref_get_newfile (ORZ_PREF (dialog));
      second = orz_pref_get_second (ORZ_PREF (dialog));
      orz_pref_get_size (ORZ_PREF (dialog), &def_width, &def_height);
      n_pos = orz_pref_get_tab (ORZ_PREF (dialog));
      /* ja:ノートの位置 */
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
    }
  gtk_widget_destroy (dialog);
}


void
command_codec (GtkWidget *widget,
               gpointer   user_data)
{
  codec_dialog ();
}


void
command_version (GtkWidget *widget,
                 gpointer   user_data)
{
  version_dialog ();
}


void
command_cancel (GtkWidget *widget,
                gpointer   user_data)
{
  thread_break (FALSE);
}
