/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32ldr.h"
#include "image.h"
#include "advapi32.h"
#include "commctrl.h"
#include "commdlg.h"
#include "gdi32.h"
#include "kernel32.h"
#include "msvcrt.h"
#include "ole32.h"
#include "oleaut32.h"
#include "rasapi32.h"
#include "rasdlg.h"
#include "shell32.h"
#include "user32.h"
#include "version.h"
#include "winmm.h"
#include "winsock.h"
#include "misc/fileio.h"
#include "misc/profile.h"


#define USE_DEBUG 1


#define DLL_PROCESS_DETACH 0
#define DLL_PROCESS_ATTACH 1


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
static GList *glist_module = NULL;
# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */


/*
0x0001 = Cursor
0x0002 = Bitmap
0x0003 = Icon
0x0004 = Menu
0x0005 = Dialog
0x0006 = String Table
0x0007 = Font Directory
0x0008 = Font
0x0009 = Accelerators Table
0x000A = RC Data (custom binary data)
0x000B = Message table
0x000C = Group Cursor
0x000E = Group Icon
0x0010 = Version Information
0x0011 = Dialog Include
0x0013 = Plug'n'Play
0x0014 = VXD
0x0015 = Animated Cursor
0x2002 = Bitmap (new version)
0x2004 = Menu (new version)
0x2005 = Dialog (new version)
*/
static void
w32ldr_module_resource_directory (W32LdrModule              *module,
                                  PIMAGE_RESOURCE_DIRECTORY  pird,
                                  const gchar               *name)
{
  guint8 *image;
  gint i;
  PIMAGE_FILE_HEADER pifh;
  PIMAGE_OPTIONAL_HEADER pioh;

  image = (guint8 *)(module + 1);
  pifh = (PIMAGE_FILE_HEADER)
                        (image + *(LPDWORD)(image + 0x3c) + sizeof (DWORD));
  pioh = (PIMAGE_OPTIONAL_HEADER)(pifh + 1);
  for (i = 0; i < pird->NumberOfNamedEntries + pird->NumberOfIdEntries; i++)
    {
      gchar *new_name;
      PIMAGE_RESOURCE_DIRECTORY_ENTRY pirde;

      pirde = (PIMAGE_RESOURCE_DIRECTORY_ENTRY)
                            ((LPBYTE)pird + sizeof (IMAGE_RESOURCE_DIRECTORY)
                                + sizeof (IMAGE_RESOURCE_DIRECTORY_ENTRY) * i);
      if (pirde->Id & 0x80000000)
        {
          gchar *utf8str, *tmp;
          PIMAGE_RESOURCE_DIR_STRING_U pirdsu;

          pirdsu = (PIMAGE_RESOURCE_DIR_STRING_U)(image
                + pioh->DataDirectory
                                [IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
                + (pirde->Id & 0x7fffffff));
          utf8str = g_utf16_to_utf8 (pirdsu->NameString, pirdsu->Length,
                                                            NULL, NULL, NULL);
          tmp = g_utf8_strup (utf8str, -1);
          g_free (utf8str);
          new_name = g_strdup_printf ("%s/%s", name, tmp);
          g_free (tmp);
        }
      else
        {
          new_name = g_strdup_printf ("%s/0x%x", name, pirde->Id);
        }
      if (pirde->OffsetToData & 0x80000000)
        {
          w32ldr_module_resource_directory (module,
                (PIMAGE_RESOURCE_DIRECTORY)(image
                + pioh->DataDirectory
                                [IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
                + (pirde->OffsetToData & 0x7fffffff)), new_name);
          g_free (new_name);
        }
      else
        {
          guint8 *data;
          PIMAGE_RESOURCE_DATA_ENTRY de;

          de = (PIMAGE_RESOURCE_DATA_ENTRY)(image
                + pioh->DataDirectory
                                [IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
                + pirde->OffsetToData);
          data = g_malloc (de->Size + sizeof (DWORD));
          *(LPDWORD)data = de->Size;
          g_memmove (data + sizeof (DWORD), image + de->OffsetToData,
                                                                    de->Size);
          g_hash_table_insert (module->resource, new_name, data);
        }
    }
}


static gchar *
w32ldr_module_get_basename (const gchar *file)
{
  gchar *name;

  if (!file)
    return NULL;
  name = g_path_get_basename (file);
  if (!g_strrchr (name, '.'))
    {
      gchar *tmp;

      tmp = g_strconcat (name, ".dll", NULL);
      g_free (name);
      name = tmp;
    }
  return name;
}


/*  ja:既にロードされているモジュールを取得する
    file,モジュール名
     RET,モジュール,NULL:ロードされていない                                 */
W32LdrModule *
w32ldr_module_get_library (const gchar *file)
{
  gchar *base, *name;
  gint i;
  W32LdrModule *module = NULL;

  if (!file)
    return NULL;
  base = g_path_get_basename (file);
  name = w32ldr_module_get_basename (file);
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  for (i = g_list_length (glist_module) - 1; i >= 0; i--)
    {
      W32LdrModule *mod;

      mod = g_list_nth_data (glist_module, i);
      if (g_ascii_strcasecmp (mod->name, base) == 0
                                || g_ascii_strcasecmp (mod->name, name) == 0)
        {
          module = mod;
          break;
        }
    }
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  g_free (base);
  g_free (name);
  return module;
}


/*  ja:モジュールをロードする
    file,モジュール名
    exec,TRUE:DllMainを実行する,FALSE:ロードのみ
     RET,モジュール,NULL:エラー                                             */
W32LdrModule *
w32ldr_module_load_library (const gchar    *file,
                            const gboolean  exec)
{
  gboolean result = TRUE;
  gssize leng, length;
  guint8 *image;
  gint i;
  W32LdrModule *module;
  PIMAGE_FILE_HEADER pifh;
  PIMAGE_OPTIONAL_HEADER pioh;
  PIMAGE_SECTION_HEADER pish;
  struct {
    const gchar *name;
    W32LdrExport *exports;
  } osdll[] = {
    {"advapi32.dll", advapi32_exports},
    {"comctl32.dll", commctrl_exports},
    {"comdlg32.dll", commdlg_exports},
    {"gdi32.dll",    gdi32_exports},
    {"kernel32.dll", kernel32_exports},
    {"msvcrt.dll",   msvcrt_exports},
    {"ole32.dll",    ole32_exports},
    {"oleaut32.dll", oleaut32_exports},
    {"rasapi32.dll", rasapi32_exports},
    {"rasdlg.dll",   rasdlg_exports},
    {"shell32.dll",  shell32_exports},
    {"user32.dll",   user32_exports},
    {"version.dll",  version_exports},
    {"winmm.dll",    winmm_exports},
    {"wsock32.dll",  winsock_exports},
    {NULL,           NULL}};

  if (!file)
    return NULL;
  module = w32ldr_module_get_library (file);
  if (module)
    {
      /* ja:既読 */
      module->counter++;
      return module;
    }
  module = g_malloc0 (sizeof (W32LdrModule));
  module->name = w32ldr_module_get_basename (file);
  module->counter = 1;
  for (i = 0; osdll[i].name; i++)
    if (g_ascii_strcasecmp (osdll[i].name, module->name) == 0)
      {
        /* ja:内蔵DLL */
        gint j;

        module->file = g_build_filename
                (w32ldr_dir_get_path (W32LDR_DIR_SYSTEM), osdll[i].name, NULL);
        module->exports = g_hash_table_new (g_str_hash, g_str_equal);
        module->ordinals = g_hash_table_new (g_direct_hash, g_direct_equal);
        for (j = 0; osdll[i].exports[j].func; j++)
          if (osdll[i].exports[j].implement)
            {
              g_hash_table_insert (module->exports,
                        osdll[i].exports[j].name, osdll[i].exports[j].func);
              g_hash_table_insert (module->ordinals,
                        GUINT_TO_POINTER ((guint)osdll[i].exports[j].ordinal),
                                                    osdll[i].exports[j].func);
            }
#ifdef USE_THREAD
        G_LOCK (critical);
        critical = TRUE;
#endif /* USE_THREAD */
        glist_module = g_list_append (glist_module, module);
#ifdef USE_THREAD
        critical = FALSE;
        G_UNLOCK (critical);
#endif /* USE_THREAD */
        return module;
      }

  module->name = g_path_get_basename (file);
  module->file = fileio_get_full_path (file);
  image = fileio_load (module->file, &length);
  for (i = 0; i < 8; i++)
    if (!image)
      {
        g_free (module->name);
        g_free (module->file);
        module->name = i % 2 == 0 ? g_path_get_basename (file)
                                  : w32ldr_module_get_basename (file);
        module->file = i < 4 ? g_build_filename (w32ldr_dir_get_path
                        (i / 2 == 0 ? W32LDR_DIR_SYSTEM : W32LDR_DIR_WINDOWS),
                                                        module->name, NULL)
                    : g_build_filename (w32ldr_dir_get_path (W32LDR_DIR_HOME),
                                    i < 6 ? "bin" : "lib", module->name, NULL);
        image = fileio_load (module->file, &length);
      }
  if (!image || length < 0x40 || *(guint16 *)image != IMAGE_DOS_SIGNATURE
    || length <= *(guint32 *)(image + 0x3c)
    || *(guint32 *)(image + *(guint32 *)(image + 0x3c)) != IMAGE_NT_SIGNATURE)
    {
      g_free (image);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return NULL;
    }
  pifh = (PIMAGE_FILE_HEADER)
                        (image + *(LPDWORD)(image + 0x3c) + sizeof (DWORD));
  pioh = (PIMAGE_OPTIONAL_HEADER)(pifh + 1);
  pish = (PIMAGE_SECTION_HEADER)(pioh + 1);
  if (pifh->Machine != IMAGE_FILE_MACHINE_I386
                || pifh->SizeOfOptionalHeader != sizeof (IMAGE_OPTIONAL_HEADER)
                || pioh->Magic != IMAGE_NT_OPTIONAL_HDR_MAGIC)
    {
      g_free (image);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return NULL;
    }
  /* ja:イメージ作成 */
  leng = pioh->SizeOfImage;
  for (i = 0; i < pifh->NumberOfSections; i++)
    if (leng < pish[i].VirtualAddress + pish[i].Misc.VirtualSize)
      leng = pish[i].VirtualAddress + pish[i].Misc.VirtualSize;
  module = g_realloc (module, leng + sizeof (W32LdrModule));
  g_memset (module + 1, 0, leng);
  g_memmove (module + 1, image, pioh->SizeOfHeaders);
  for (i = 0; i < pifh->NumberOfSections; i++)
    g_memmove ((guint8 *)(module + 1) + pish[i].VirtualAddress,
                    image + pish[i].PointerToRawData, pish[i].SizeOfRawData);
  g_free (image);
  image = (guint8 *)(module + 1);
  pifh = (PIMAGE_FILE_HEADER)
                        (image + *(LPDWORD)(image + 0x3c) + sizeof (DWORD));
  pioh = (PIMAGE_OPTIONAL_HEADER)(pifh + 1);
  pish = (PIMAGE_SECTION_HEADER)(pioh + 1);

  /* ja:エクスポート */
  if (pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT].Size)
    {
      LPDWORD func, name;
      LPWORD ordinal;
      PIMAGE_EXPORT_DIRECTORY pied;

      pied = (PIMAGE_EXPORT_DIRECTORY)(image
        + pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT].VirtualAddress);
      func = (LPDWORD)(image + pied->AddressOfFunctions);
      name = (LPDWORD)(image + pied->AddressOfNames);
      ordinal = (LPWORD)(image + pied->AddressOfNameOrdinals);
      module->exports = g_hash_table_new (g_str_hash, g_str_equal);
      module->ordinals = g_hash_table_new (g_direct_hash, g_direct_equal);
      for (i = 0; i < pied->NumberOfNames; i++)
        {
          gpointer addr;

          addr = image + func[ordinal[i]];
          g_hash_table_insert (module->exports, image + name[i], addr);
          g_hash_table_insert (module->ordinals,
                            GUINT_TO_POINTER (ordinal[i] + pied->Base), addr);
        }
    }
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_module = g_list_append (glist_module, module);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */

  /* ja:インポート */
  if (pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size)
    {
      PIMAGE_IMPORT_DESCRIPTOR piid;

      piid = (PIMAGE_IMPORT_DESCRIPTOR)(image
        + pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress);
      for (i = 0; piid[i].Name; i++)
        {
          W32LdrModule *mod;

          mod = w32ldr_module_load_library ((gchar *)(image + piid[i].Name),
                                                                        TRUE);
          if (mod)
            {
              gint j;
              PIMAGE_THUNK_DATA piat, pilt;

              piat = (PIMAGE_THUNK_DATA)(image
                                + (piid[i].DUMMYUNIONNAME.OriginalFirstThunk
                                    ? piid[i].DUMMYUNIONNAME.OriginalFirstThunk
                                    : piid[i].FirstThunk));
              pilt = (PIMAGE_THUNK_DATA)(image + piid[i].FirstThunk);
              for (j = 0; piat[j].u1.AddressOfData; j++)
                {
                  gpointer func;
                  WORD ordinal;
                  PIMAGE_IMPORT_BY_NAME piibn;
#ifdef USE_DEBUG
                  const static gchar *fmt = "%s\n";
                  gsize leng;
                  guint8 *opcode, *p;
#endif /* USE_DEBUG */

                  ordinal = piat[j].u1.Ordinal & ~IMAGE_ORDINAL_FLAG;
                  piibn = (PIMAGE_IMPORT_BY_NAME)
                                            (image + piat[j].u1.AddressOfData);
                  func = piat[j].u1.Ordinal & IMAGE_ORDINAL_FLAG
                        ? w32ldr_module_get_proc (mod, ordinal)
                        : w32ldr_module_get_func (mod, (gchar *)piibn->Name);
                  if (!func)
                    {
                      const static guint8 ret_code = 0xc3;

#ifdef USE_DEBUG
                      if (piat[j].u1.Ordinal & IMAGE_ORDINAL_FLAG)
                        g_warning ("import-function:%s,%d\n",
                                                image + piid[i].Name, ordinal);
                      else
                        g_warning ("import-function:%s\n", piibn->Name);
#endif /* USE_DEBUG */
                      func = (gpointer)&ret_code;
                      result = FALSE;
                    }
#ifdef USE_DEBUG
                  leng = g_strlen ((gchar *)piibn->Name) + 1;
                  p = opcode = g_malloc (leng + 26);
                  *p++ = 0x60;
                  *p++ = 0xe8;
                  *(guint32 *)p = leng;
                  p += 4;
                  g_strcpy ((gchar *)p, (gchar *)piibn->Name);
                  p += leng;
                  *p++ = 0x68;
                  *(guint32 *)p = GPOINTER_TO_UINT (fmt);
                  p += 4;
                  *p++ = 0xe8;
                  *(guint32 *)p = GPOINTER_TO_UINT (g_print)
                                                - (GPOINTER_TO_UINT (p) + 4);
                  p += 4;
                  *p++ = 0x83;
                  *p++ = 0xc4;
                  *p++ = 0x08;
                  *p++ = 0x61;
                  *p++ = 0xe9;
                  *(guint32 *)p = GPOINTER_TO_UINT (func)
                                                - (GPOINTER_TO_INT (p) + 4);
                  pilt[j].u1.Function = GPOINTER_TO_UINT (opcode);
#else /* not USE_DEBUG */
                  pilt[j].u1.Function = GPOINTER_TO_UINT (func);
#endif /* not USE_DEBUG */
                }
              module->module = g_list_append (module->module, mod);
            }
#ifdef USE_DEBUG
          else
            {
              g_warning ("import-module:%s\n", image + piid[i].Name);
              result = FALSE;
            }
#endif /* USE_DEBUG */
        }
    }
  if (!result)
    {
      w32ldr_module_free (module);
      return NULL;
    }

  /* ja:リソース */
  if (pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE].Size)
    {
      PIMAGE_RESOURCE_DIRECTORY pird;

      pird = (PIMAGE_RESOURCE_DIRECTORY)(image
        + pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress);
      module->resource = g_hash_table_new (g_str_hash, g_str_equal);
      w32ldr_module_resource_directory (module, pird, "");
    }

  /* ja:再配置 */
  if (pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size)
    {
      gint delta;
      PIMAGE_BASE_RELOCATION pibr;

      pibr = (PIMAGE_BASE_RELOCATION)(image
        + pioh->DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress);
      delta = GPOINTER_TO_UINT (image) - pioh->ImageBase;
      while (pibr->VirtualAddress)
        {
          gint count;

          count = (pibr->SizeOfBlock - sizeof (DWORD) * 2) / sizeof (WORD);
          for (i = 0; i < count; i++)
            if (((pibr->TypeOffset[i] >> 12) & 0xf) == IMAGE_REL_BASED_HIGHLOW)
              *(LPDWORD)(image + pibr->VirtualAddress
                                    + (pibr->TypeOffset[i] & 0xfff)) += delta;
          pibr = (PIMAGE_BASE_RELOCATION)
                                        (((guint8 *)pibr) + pibr->SizeOfBlock);
        }
    }

  if (exec && (pifh->Characteristics & IMAGE_FILE_DLL))
    {
      module->DllMain = w32ldr_module_get_func (module, "DllMain");
      if (!module->DllMain && pioh->SizeOfHeaders < pioh->AddressOfEntryPoint
                                        && pioh->AddressOfEntryPoint < leng)
        module->DllMain = (DllEntryProc)(image + pioh->AddressOfEntryPoint);
      if (module->DllMain)
        module->DllMain ((W32LdrModule *)(module + 1),
                                                    DLL_PROCESS_ATTACH, NULL);
    }
  return module;
}


static void
g_hash_table_callback (gpointer key,
                       gpointer value,
                       gpointer hash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable*)hash, key,
                                                    &orig_key, &orig_value))
    {
      g_free (orig_key);
      g_free (orig_value);
    }
}


/*  ja:モジュールを解放する
    module,モジュール
       RET,TRUE:正常終了,NULL:エラー                                        */
gboolean
w32ldr_module_free (W32LdrModule *module)
{
  gboolean result;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  result = module && g_list_find (glist_module, module);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  if (!result)
    return FALSE;
  module->counter--;
  if (module->counter <= 0)
    {
      gint i;

      if (module->DllMain)
        module->DllMain ((W32LdrModule *)(module + 1),
                                                    DLL_PROCESS_DETACH, NULL);
#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      glist_module = g_list_remove (glist_module, module);
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      for (i = g_list_length (module->module) - 1; i >= 0; i--)
        if (!w32ldr_module_free (g_list_nth_data (module->module, i)))
          result = FALSE;
      if (module->resource)
        {
          g_hash_table_foreach (module->resource,
                                    g_hash_table_callback, module->resource);
          g_hash_table_destroy (module->resource);
        }
      if (module->exports)
        g_hash_table_destroy (module->exports);
      if (module->ordinals)
        g_hash_table_destroy (module->ordinals);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
    }
  return result;
}


/*  ja:モジュールからAPIを取得する
    module,モジュール
      name,API名
       RET,アドレス,NULL:エラー                                             */
gpointer
w32ldr_module_get_func (W32LdrModule *module,
                        const gchar  *name)
{
  return module && module->exports
                        ? g_hash_table_lookup (module->exports, name) : NULL;
}


/*  ja:モジュールからAPIを取得する
     module,モジュール
    ordinal,オーディナル値
        RET,アドレス,NULL:エラー                                            */
gpointer
w32ldr_module_get_proc (W32LdrModule  *module,
                        const guint16  ordinal)
{
  return module && module->ordinals ? g_hash_table_lookup (module->ordinals,
                                    GUINT_TO_POINTER ((guint)ordinal)) : NULL;
}


/*  ja:モジュールからリソースを取得する
    module,モジュール
       key,リソース
       RET,データ,NULL:エラー                                               */
gpointer
w32ldr_module_get_resource (W32LdrModule *module,
                            const gchar  *key)
{
  return module && module->resource
                        ? g_hash_table_lookup (module->resource, key) : NULL;
}


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを取得する
    dir,ディレクトリの種類
    RET,ディレクトリ                                                        */
const gchar *
w32ldr_dir_get_path (const guint dir)
{
  gchar *file;
  const gchar *key, *path = NULL;
  const static gchar *dir_windows = NULL, *dir_system = NULL, *dir_temp = NULL;
  const static gchar *dir_data = NULL, *dir_home = NULL;
  gint i;
  GKeyFile *key_file;

  switch (dir)
    {
      case W32LDR_DIR_WINDOWS: if (dir_windows) return dir_windows; break;
      case W32LDR_DIR_SYSTEM:  if (dir_system)  return dir_system;  break;
      case W32LDR_DIR_TEMP:    if (dir_temp)    return dir_temp;    break;
      case W32LDR_DIR_DATA:    if (dir_data)    return dir_data;    break;
      case W32LDR_DIR_HOME:    if (dir_home)    return dir_home;
    }
  for (i = 0; i < 2 && !path; i++)
    {
      switch (dir)
        {
          case W32LDR_DIR_WINDOWS:
            file = w32ldr_dir_get_filename ("system.ini",
                                i == 0 ? W32LDR_DIR_HOME : W32LDR_DIR_DATA);
            key = "windows";
            break;
          case W32LDR_DIR_SYSTEM:
            file = w32ldr_dir_get_filename ("system.ini",
                                i == 0 ? W32LDR_DIR_HOME : W32LDR_DIR_DATA);
            key = "system";
            break;
          case W32LDR_DIR_TEMP:
            file = w32ldr_dir_get_filename ("system.ini",
                                i == 0 ? W32LDR_DIR_HOME : W32LDR_DIR_DATA);
            key = "temp";
            break;
          case W32LDR_DIR_DATA:
            if (i == 0)
              {
                file = g_build_filename (g_get_home_dir (),
                                    ".maid.org", "w32ldr", "system.ini", NULL);
                key = "data";
                break;
              }
          default:
            file = NULL;
            key = NULL;
        }
      if (file && key)
        {
          gchar *tmp, *utf8str;

          key_file = g_key_file_new ();
          tmp = w32ldr_dir_get_case_insensitive (file);
          g_key_file_load_from_file (key_file, tmp, G_KEY_FILE_NONE, NULL);
          g_free (tmp);
          utf8str = g_key_file_get_string (key_file, "directory", key, NULL);
          g_key_file_free (key_file);
          if (utf8str)
            {
              path = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
        }
      g_free (file);
    }
  switch (dir)
    {
      case W32LDR_DIR_WINDOWS:
        return dir_windows = path ? path
#ifdef BINDIR
                                  : g_build_filename (BINDIR, "w32ldr", NULL);
#else /* not BINDIR */
                                  : "/usr/local/bin/w32ldr";
#endif /* not BINDIR */
      case W32LDR_DIR_SYSTEM:
        return dir_system = path ? path
#ifdef LIBDIR
                                 : g_build_filename (LIBDIR, "w32ldr", NULL);
#else /* not LIBDIR */
                                 : "/usr/local/lib/w32ldr";
#endif /* not LIBDIR */
      case W32LDR_DIR_TEMP:
        return dir_temp = path ? path : g_get_tmp_dir ();
      case W32LDR_DIR_DATA:
        return dir_data = path ? path
#ifdef SYSCONFDIR
                               : g_build_filename (SYSCONFDIR, "w32ldr", NULL);
#else /* not SYSCONFDIR */
                               : "/usr/local/etc/w32ldr";
#endif /* not SYSCONFDIR */
      case W32LDR_DIR_HOME:
        if (!dir_home)
          dir_home = g_build_filename (g_get_home_dir (),
                                                ".maid.org", "w32ldr", NULL);
        return dir_home;
    }
  return NULL;
}


/*  ja:大文字小文字を区別しないパス名を求める
    path,ファイル名
     RET,大文字小文字を区別しないパス名                                     */
gchar *
w32ldr_dir_get_case_insensitive (const gchar *path)
{
  gchar **element, *file = NULL;
  gint i;

  if (!path)
    return NULL;
  if (path[0] == '\0')
    return g_malloc0 (sizeof (gchar));
  element = g_strsplit (path, G_DIR_SEPARATOR_S, 0);
  for (i = 0; element[i]; i++)
    if ((element[i])[0] == '\0')
      {
        if (!file)
          file = g_strdup (G_DIR_SEPARATOR_S);
      }
    else
      {
        gchar *tmp;

        tmp = file ? g_build_filename (file, element[i], NULL)
                   : g_strdup (element[i]);
        if (g_file_test (tmp, G_FILE_TEST_EXISTS))
          {
            g_free (file);
            file = tmp;
          }
        else
          {
            const gchar *name;
            GDir *dir;

            g_free (tmp);
            tmp = NULL;
            dir = g_dir_open (file ? file : ".", 0, NULL);
            while ((name = g_dir_read_name (dir)))
              if (g_ascii_strcasecmp (element[i], name) == 0)
                {
                  tmp = file ? g_build_filename (file, name, NULL)
                             : g_strdup (name);
                  break;
                }
            g_dir_close (dir);
            if (!tmp)
              {
                while (element[i])
                  {
                    if ((element[i])[0] != '\0')
                      {
                        tmp = file ? g_build_filename (file, element[i], NULL)
                                   : g_strdup (element[i]);
                        g_free (file);
                        file = tmp;
                      }
                    i++;
                  }
                break;
              }
            g_free (file);
            file = tmp;
          }
      }
  g_strfreev (element);
  return file;
}


/*  ja:特定のディレクトリのファイルを取得する
    file,元のファイル名
     dir,ディレクトリの種類
     RET,ファイル                                                           */
gchar *
w32ldr_dir_get_filename (const gchar *file,
                         const guint  dir)
{
  gchar *name, *tmp, *result;

  name = g_path_get_basename (file);
  tmp = g_build_filename (w32ldr_dir_get_path (dir), name, NULL);
  g_free (name);
  result = w32ldr_dir_get_case_insensitive (tmp);
  g_free (tmp);
  return result;
}


/******************************************************************************
*                                                                             *
* ja:文字列関数群                                                             *
*                                                                             *
******************************************************************************/
/*  ja:UTF-8←MultiByte
     mb,MultiByte
    RET,UTF-8,NULL:エラー                                                   */
gchar *
w32ldr_utf8_from_mb (const gchar *mb)
{
  gssize leng;
  gchar *utf8str;
  gunichar2 *utf16str;

  if (!mb)
    return NULL;
  leng = MultiByteToWideChar (CP_ACP, 0, mb, -1, NULL, 0);
  utf16str = g_malloc (leng * sizeof (gunichar2));
  MultiByteToWideChar (CP_ACP, 0, mb, -1, utf16str, leng);
  utf8str = g_utf16_to_utf8 (utf16str, -1, NULL, NULL, NULL);
  g_free (utf16str);
  return utf8str;
}


/*  ja:UTF-8→MultiByte
    utf8str,UTF-8
        RET,MultiByte,NULL:エラー                                           */
gchar *
w32ldr_utf8_to_mb (const gchar *utf8str)
{
  gssize leng;
  gchar *mb;
  gunichar2 *utf16str;

  if (!utf8str)
    return NULL;
  utf16str = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  if (!utf16str)
    return NULL;
  leng = WideCharToMultiByte (CP_ACP, 0, utf16str, -1, NULL, 0, NULL, NULL);
  mb = g_malloc (leng * sizeof (gchar));
  WideCharToMultiByte (CP_ACP, 0, utf16str, -1, mb, leng, NULL, NULL);
  g_free (utf16str);
  return mb;
}


/*  ja:ファイル名←MultiByte
     mb,MultiByte
    RET,ファイル名,NULL:エラー                                              */
gchar *
w32ldr_filename_from_mb (const gchar *mb)
{
  gssize leng;
  gchar *file, *utf8str, *str;
  gunichar2 *utf16str;
  gint i = 0;

  if (!mb)
    return NULL;
  str = g_strdup (mb);
  while (str[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, str[i]))
      {
        i += 2;
      }
    else
      {
        if (str[i] == '\\')
          str[i] = G_DIR_SEPARATOR;
        i++;
      }
  leng = MultiByteToWideChar (CP_ACP, 0, str, -1, NULL, 0);
  utf16str = g_malloc (leng * sizeof (gunichar2));
  MultiByteToWideChar (CP_ACP, 0, str, -1, utf16str, leng);
  g_free (str);
  utf8str = g_utf16_to_utf8 (utf16str, -1, NULL, NULL, NULL);
  g_free (utf16str);
  if (!utf8str)
    return NULL;
  if (g_ascii_isalpha (utf8str[i]) && utf8str[i + 1] == ':')
    g_memmove (utf8str, utf8str + 2,
                                    (g_strlen (utf8str) - 1) * sizeof (gchar));
  str = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  file = w32ldr_dir_get_case_insensitive (str);
  g_free (str);
  return file;
}


/*  ja:ファイル名→MultiByte
    file,ファイル名
     RET,MultiByte,NULL:エラー                                              */
gchar *
w32ldr_filename_to_mb (const gchar *file)
{
  gssize leng;
  gchar *mb, *utf8str;
  gunichar2 *utf16str;
  gint i = 0;

  if (!file)
    return NULL;
  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8str)
    return NULL;
  if (G_IS_DIR_SEPARATOR (utf8str[0]) && !G_IS_DIR_SEPARATOR (utf8str[1]))
    {
      gchar *tmp;

      tmp = g_strconcat ("c:", utf8str, NULL);
      g_free (utf8str);
      utf8str = tmp;
    }
  utf16str = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  leng = WideCharToMultiByte (CP_ACP, 0, utf16str, -1, NULL, 0, NULL, NULL);
  mb = g_malloc (leng * sizeof (gchar));
  WideCharToMultiByte (CP_ACP, 0, utf16str, -1, (LPSTR)mb, leng, NULL, NULL);
  g_free (utf16str);
  while (mb[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, mb[i]))
      {
        i += 2;
      }
    else
      {
        if (G_IS_DIR_SEPARATOR (mb[i]))
          mb[i] = '\\';
        i++;
      }
  return mb;
}


/*  ja:ファイル名←WideChar
     wc,WideChar
    RET,ファイル名,NULL:エラー                                              */
gchar *
w32ldr_filename_from_wc (const gunichar2 *wc)
{
  gchar *file, *utf8str, *str;
  gint i;

  if (!wc)
    return NULL;
  utf8str = g_utf16_to_utf8 (wc, -1, NULL, NULL, NULL);
  for (i = 0; utf8str[i] != '\0'; i++)
    if (utf8str[i] == '\\')
      utf8str[i] = G_DIR_SEPARATOR;
  if (g_ascii_isalpha (utf8str[i]) && utf8str[i + 1] == ':')
    g_memmove (utf8str, utf8str + 2,
                                    (g_strlen (utf8str) - 1) * sizeof (gchar));
  str = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  file = w32ldr_dir_get_case_insensitive (str);
  g_free (str);
  return file;
}


/*  ja:ファイル名→WideChar
    file,ファイル名
     RET,WideChar,NULL:エラー                                               */
gunichar2 *
w32ldr_filename_to_wc (const gchar *file)
{
  gchar *utf8str;
  gunichar2 *wc;
  gint i;

  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8str)
    return NULL;
  for (i = 0; utf8str[i] != '\0'; i++)
    if (G_IS_DIR_SEPARATOR (utf8str[i]))
      utf8str[i] = '\\';
  if (utf8str[0] == '\\')
    {
      gchar *tmp;

      tmp = g_strconcat ("c:", utf8str, NULL);
      g_free (utf8str);
      utf8str = tmp;
    }
  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  return wc;
}


/******************************************************************************
*                                                                             *
* ja:tchar関数群                                                              *
*                                                                             *
******************************************************************************/
/*  ja:ソースコード文字列をMultiByteに変換する
    str,ソースコード文字列
    RET,MultiByte                                                           */
gchar *
w32ldr_tchar_to_mb (gchar *str)
{
  gchar *mb = NULL, *charset, *utf8str;
  static GHashTable *ghash = NULL;

  if (!str)
    return NULL;
  if (ghash)
    {
      mb = g_hash_table_lookup (ghash, str);
      if (mb)
        return mb;
    }
  else
    {
      ghash = g_hash_table_new (g_str_hash, g_str_equal);
    }
  /* ja:ANSIコードページ */
  charset = g_strdup_printf ("CP%d", GetACP ());
  utf8str = g_convert (str, -1, "UTF-8", charset, NULL, NULL, NULL);
  g_free (charset);
  if (utf8str)
    {
      g_free (utf8str);
      mb = str;
    }
  /* en:UTF-8 */
  if (!mb)
    mb = g_utf8_validate (str, -1, NULL) ? w32ldr_utf8_to_mb (str) : NULL;
  /* ja:ロケール */
  if (!mb)
    {
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
      mb = w32ldr_utf8_to_mb (utf8str);
      g_free (utf8str);
    }
  /* ja:文字符号化方式不明 */
  if (!mb)
    {
      gint i;

      mb = g_strdup (str);
      for (i = 0; mb[i] != '\0'; i++)
        mb[i] &= 0x7f;
    }
  g_hash_table_insert (ghash, str, mb);
  return mb;
}


/*  ja:ソースコード文字列をUTF-16に変換する
    str,ソースコード文字列
    RET,UTF-16                                                              */
gunichar2 *
w32ldr_tchar_to_wc (gchar *str)
{
  gchar *charset;
  gunichar2 *wc;
  static GHashTable *ghash = NULL;

  if (!str)
    return NULL;
  if (ghash)
    {
      wc = g_hash_table_lookup (ghash, str);
      if (wc)
        return wc;
    }
  else
    {
      ghash = g_hash_table_new (g_str_hash, g_str_equal);
    }
  /* ja:ANSIコードページ */
  charset = g_strdup_printf ("CP%d", GetACP ());
  wc = (gunichar2 *)g_convert (str, -1, "UTF-16LE", charset, NULL, NULL, NULL);
  g_free (charset);
  /* en:UTF-8 */
  if (!wc)
    wc = g_utf8_to_utf16 (str, -1, NULL, NULL, NULL);
  /* ja:ロケール */
  if (!wc)
    {
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
      wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
    }
  /* ja:文字符号化方式不明 */
  if (!wc)
    {
      gint i = 0;

      wc = g_malloc ((g_strlen (str) + 1) * sizeof (gunichar2));
      do
        wc[i] = str[i] & 0x7f;
      while (str[i++] != '\0');
    }
  g_hash_table_insert (ghash, str, wc);
  return wc;
}


/******************************************************************************
*                                                                             *
* ja:アトム関数群                                                             *
*                                                                             *
******************************************************************************/
#define ATOMBASE 0x0100
typedef struct _W32LdrAtom
{
  gint counter;
  gchar *name;
  guint16 atom;
} W32LdrAtom;
static GList *glist_atom = NULL;
static W32LdrAtom atom_table[] = {
    {0, "BUTTON",    0x0080},
    {0, "EDIT",      0x0081},
    {0, "STATIC",    0x0082},
    {0, "LISTBOX",   0x0083},
    {0, "SCROLLBAR", 0x0084},
    {0, "COMBOBOX",  0x0085},
    {0, NULL,        0x0000}};


/*  ja:アトムを追加する
    name,名前(UTF-8)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name (const gchar *name)
{
  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      guint16 atom = ATOMBASE - 1;
      gint i;
      GList *glist;
      W32LdrAtom *w32atom;

      for (i = 0; atom_table[i].name; i++)
        if (g_ascii_strcasecmp (atom_table[i].name, name) == 0)
          return atom_table[i].atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          w32atom = glist->data;
          if (g_ascii_strcasecmp (w32atom->name, name) == 0)
            {
              w32atom->counter++;
              return w32atom->atom;
            }
          if (atom < w32atom->atom)
            atom = w32atom->atom;
        }
      w32atom = g_malloc (sizeof (W32LdrAtom));
      w32atom->name = g_strdup (name);
      w32atom->counter = 1;
      w32atom->atom = atom + 1;
      glist_atom = g_list_append (glist_atom, w32atom);
      return w32atom->atom;
    }
  else if (GPOINTER_TO_UINT (name) > 0)
    {
      gint i;
      guint16 atom;
      GList *glist;

      atom = GPOINTER_TO_UINT (name);
      for (i = 0; atom_table[i].name; i++)
        if (atom_table[i].atom == atom)
          return atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          W32LdrAtom *w32atom;

          w32atom = glist->data;
          if (w32atom->atom == atom)
            {
              w32atom->counter++;
              return atom;
            }
        }
    }
  return 0;
}


/*  ja:アトムを追加する
    name,名前(MultiByte)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name_mb (const gchar *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = w32ldr_utf8_from_mb (name);
      atom = w32ldr_atom_add_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_add_name (name);
    }
  return atom;
}


/*  ja:アトムを追加する
    name,名前(WideChar)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name_wc (const gunichar2 *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (name, -1, NULL, NULL, NULL);
      atom = w32ldr_atom_add_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_add_name ((const gchar *)name);
    }
  return atom;
}


/*  ja:アトムをアトムから削除する
    atom,アトム
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_atom (const guint16 atom)
{
  GList *glist;

  for (glist = g_list_first (glist_atom); glist; glist = g_list_next (glist))
    {
      W32LdrAtom *w32atom;

      w32atom = glist->data;
      if (w32atom->atom == atom)
        {
          if (--w32atom->counter <= 0)
            {
              glist_atom = g_list_remove (glist_atom, w32atom);
              g_free (w32atom->name);
              g_free (w32atom);
            }
          return TRUE;
        }
    }
  return FALSE;
}


/*  ja:アトムを名前から削除する
    name,名前(UTF-8)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name (const gchar *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name (name));
}


/*  ja:アトムを名前から削除する
    name,名前(MultiByte)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name_mb (const gchar *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name_mb (name));
}


/*  ja:アトムを名前から削除する
    name,名前(WideChar)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name_wc (const gunichar2 *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name_wc (name));
}


/*  ja:アトムを検索する
    name,名前(UTF-8)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name (const gchar *name)
{
  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gint i;
      GList *glist;
      W32LdrAtom *w32atom;

      for (i = 0; atom_table[i].name; i++)
        if (g_ascii_strcasecmp (atom_table[i].name, name) == 0)
          return atom_table[i].atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          w32atom = glist->data;
          if (g_ascii_strcasecmp (w32atom->name, name) == 0)
            return w32atom->atom;
        }
    }
  else if (GPOINTER_TO_UINT (name) > 0)
    {
      gint i;
      guint16 atom;
      GList *glist;

      atom = GPOINTER_TO_UINT (name);
      for (i = 0; atom_table[i].name; i++)
        if (atom_table[i].atom == atom)
          return atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          W32LdrAtom *w32atom;

          w32atom = glist->data;
          if (w32atom->atom == atom)
            return atom;
        }
    }
  return 0;
}


/*  ja:アトムを追加する
    name,名前(MultiByte)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name_mb (const gchar *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = w32ldr_utf8_from_mb (name);
      atom = w32ldr_atom_find_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_find_name (name);
    }
  return atom;
}


/*  ja:アトムを検索する
    name,名前(WideChar)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name_wc (const gunichar2 *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (name, -1, NULL, NULL, NULL);
      atom = w32ldr_atom_find_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_find_name ((const gchar *)name);
    }
  return atom;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(UTF-8),NULL:エラー                                            */
gchar *
w32ldr_atom_get_name (const guint16 atom)
{
  gint i;
  GList *glist;

  for (i = 0; atom_table[i].name; i++)
    if (atom_table[i].atom == atom)
      return g_strdup (atom_table[i].name);
  for (glist = g_list_first (glist_atom); glist; glist = g_list_next (glist))
    {
      W32LdrAtom *w32atom;

      w32atom = glist->data;
      if (w32atom->atom == atom)
        return g_strdup (w32atom->name);
    }
  return NULL;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(MultiByte),NULL:エラー                                        */
gchar *
w32ldr_atom_get_name_mb (const guint16 atom)
{
  gchar *utf8str, *mb;

  utf8str = w32ldr_atom_get_name (atom);
  mb = w32ldr_utf8_to_mb (utf8str);
  g_free (utf8str);
  return mb;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(WideChar),NULL:エラー                                         */
gunichar2 *
w32ldr_atom_get_name_wc (const guint16 atom)
{
  gchar *utf8str;
  gunichar2 *wc;

  utf8str = w32ldr_atom_get_name (atom);
  wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
  g_free (utf8str);
  return wc;
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(UTF-8),NULL:エラー                                          */
gchar *
w32ldr_atom_get_string (const gchar *name)
{
  return GPOINTER_TO_UINT (name) > G_MAXUINT16 ? g_strdup (name)
                            : w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(MultiByte),NULL:エラー                                      */
gchar *
w32ldr_atom_get_string_mb (const gchar *name)
{
  gchar *mb;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      mb = w32ldr_utf8_to_mb (name);
    }
  else
    {
      gchar *utf8str;

      utf8str = w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
      mb = w32ldr_utf8_to_mb (name);
      g_free (utf8str);
    }
  return mb;
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(WideChar),NULL:エラー                                       */
gunichar2 *
w32ldr_atom_get_string_wc (const gchar *name)
{
  gunichar2 *wc;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      wc = g_utf8_to_utf16 (name, -1, NULL, NULL, NULL);
    }
  else
    {
      gchar *utf8str;

      utf8str = w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
      wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
    }
  return wc;
}
