/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "acm.h"
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
# include <tchar.h>
# include <windows.h>
# include <vfw.h>
# ifdef G_OS_WIN32
#  include <gdk/gdkwin32.h>
# endif /* G_OS_WIN32 */


# define ACM_BUFFER_SIZE (1024 * 1024)


# define MAKE_ACM_VERSION(mjr,mnr,bld) (((LONG)(mjr)<<24)| \
                                        ((LONG)(mnr)<<16)| \
                                        ((LONG)(bld)))
# define VERSION_MSACM                   MAKE_ACM_VERSION(3,50,0)
# define VERSION_ACM_DRIVER              MAKE_ACM_VERSION(4, 0,0)
# define ACMDM_DRIVER_NOTIFY             (ACMDM_BASE+1)
# define ACMDM_DRIVER_DETAILS            (ACMDM_BASE+10)
# define ACMDM_HARDWARE_WAVE_CAPS_INPUT  (ACMDM_BASE+20)
# define ACMDM_HARDWARE_WAVE_CAPS_OUTPUT (ACMDM_BASE+21)
# define ACMDM_FORMATTAG_DETAILS         (ACMDM_BASE+25)
# define ACMDM_FORMAT_DETAILS            (ACMDM_BASE+26)
# define ACMDM_FORMAT_SUGGEST            (ACMDM_BASE+27)
# define ACMDM_FILTERTAG_DETAILS         (ACMDM_BASE+50)
# define ACMDM_FILTER_DETAILS            (ACMDM_BASE+51)
# define ACMDM_STREAM_OPEN               (ACMDM_BASE+76)
# define ACMDM_STREAM_CLOSE              (ACMDM_BASE+77)
# define ACMDM_STREAM_SIZE               (ACMDM_BASE+78)
# define ACMDM_STREAM_CONVERT            (ACMDM_BASE+79)
# define ACMDM_STREAM_RESET              (ACMDM_BASE+80)
# define ACMDM_STREAM_PREPARE            (ACMDM_BASE+81)
# define ACMDM_STREAM_UNPREPARE          (ACMDM_BASE+82)


# include <pshpack1.h>
typedef struct tagACMDRVOPENDESC
{
  DWORD   cbStruct;
  FOURCC  fccType;
  FOURCC  fccComp;
  DWORD   dwVersion;
  DWORD   dwFlags;
  DWORD   dwError;
  LPCWSTR pszSectionName;
  LPCWSTR pszAliasName;
  DWORD   dnDevNode;
} ACMDRVOPENDESC, *PACMDRVOPENDESC, *LPACMDRVOPENDESC;
typedef struct tagACMDRVSTREAMINSTANCE
{
  DWORD          cbStruct;
  LPWAVEFORMATEX pwfxSrc;
  LPWAVEFORMATEX pwfxDst;
  LPWAVEFILTER   pwfltr;
  DWORD          dwCallback;
  DWORD          dwInstance;
  DWORD          fdwOpen;
  DWORD          fdwDriver;
  DWORD          dwDriver;
  HACMSTREAM     has;
} ACMDRVSTREAMINSTANCE, *PACMDRVSTREAMINSTANCE, *LPACMDRVSTREAMINSTANCE;
typedef struct tagACMDRVFORMATSUGGEST
{
  DWORD         cbStruct;
  DWORD         fdwSuggest;
  PWAVEFORMATEX pwfxSrc;
  DWORD         cbwfxSrc;
  PWAVEFORMATEX pwfxDst;
  DWORD         cbwfxDst;
} ACMDRVFORMATSUGGEST, *PACMDRVFORMATSUGGEST, *LPACMDRVFORMATSUGGEST;
typedef struct tagACMDRVSTREAMSIZE
{
  DWORD cbStruct;
  DWORD fdwSize;
  DWORD cbSrcLength;
  DWORD cbDstLength;
} ACMDRVSTREAMSIZE, *PACMDRVSTREAMSIZE, *LPACMDRVSTREAMSIZE;
typedef struct tagACMDRVSTREAMHEADER
{
  DWORD                         cbStruct;
  DWORD                         fdwStatus;
  DWORD                         dwUser;
  LPBYTE                        pbSrc;
  DWORD                         cbSrcLength;
  DWORD                         cbSrcLengthUsed;
  DWORD                         dwSrcUser;
  LPBYTE                        pbDst;
  DWORD                         cbDstLength;
  DWORD                         cbDstLengthUsed;
  DWORD                         dwDstUser;
  DWORD                         fdwConvert;
  struct tagACMDRVSTREAMHEADER *padshNext;
  DWORD                         fdwDriver;
  DWORD                         dwDriver;
  DWORD                         fdwPrepared;
  DWORD                         dwPrepared;
  LPBYTE                        pbPreparedSrc;
  DWORD                         cbPreparedSrcLength;
  LPBYTE                        pbPreparedDst;
  DWORD                         cbPreparedDstLength;
} ACMDRVSTREAMHEADER, *PACMDRVSTREAMHEADER, *LPACMDRVSTREAMHEADER;
# include <poppack.h>


const static gunichar2 drivers32[]
                        = {'D', 'r', 'i', 'v', 'e', 'r', 's', '3', '2', '\0'};
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


struct _AcmObject
{
  gint counter;
  guint16 tag;
  gsize format_size;
  gint format_count;
  gchar *name;
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gsize format_size_pcm;
  gint stat;
  HDRVR hDrvr;
  ACMDRVSTREAMINSTANCE acstminst;
  ACMDRVSTREAMSIZE acstmsize;
  ACMDRVSTREAMHEADER acstmhdr;
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
};


/******************************************************************************
*                                                                             *
* ja:ACM初期化                                                                *
*                                                                             *
******************************************************************************/
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static GHashTable *ghash_tag = NULL;
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ACMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_init (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  DWORD dwIndex, dwName, dwType, dwValue;
  HKEY hKey;
  TCHAR szName[MAX_PATH], szValue[MAX_PATH];

  if (ghash_tag)
    return FALSE;
  ghash_tag = g_hash_table_new (NULL, NULL);
  if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
        _T("Software\\Microsoft\\Windows NT\\CurrentVersion\\Drivers32"), 0,
                                    KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS)
    return FALSE;
  for (dwIndex = 0; dwName = MAX_PATH * sizeof (TCHAR),
                    dwValue = MAX_PATH * sizeof (TCHAR),
            RegEnumValue (hKey, dwIndex, szName, &dwName, NULL,
            &dwType, (LPBYTE)szValue, &dwValue) == ERROR_SUCCESS; dwIndex++)
    if (dwType == REG_SZ
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[0]))) == 'M'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[1]))) == 'S'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[2]))) == 'A'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[3]))) == 'C'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[4]))) == 'M'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[5]))) == '.')
      {
# ifndef UNICODE
        gchar *utf8str;
# endif /* not UNICODE */
        gunichar2 *name;
        ACMDRVOPENDESC acopen;
        HDRVR hDrvr;

# ifdef UNICODE
        name = g_memdup (szName, (lstrlenW (szName) + 1) * sizeof (gunichar2));
# else /* not UNICODE */
        utf8str = g_locale_to_utf8 (szName, -1, NULL, NULL, NULL);
        name = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
        g_free (utf8str);
# endif /* not UNICODE */
        g_memset (&acopen, 0, sizeof (ACMDRVOPENDESC));
        acopen.cbStruct = sizeof (ACMDRVOPENDESC);
        acopen.fccType = ACMDRIVERDETAILS_FCCTYPE_AUDIOCODEC;
        acopen.pszSectionName = drivers32;
        acopen.pszAliasName = name;
        hDrvr = OpenDriver (name, NULL, GPOINTER_TO_INT (&acopen));
        if (hDrvr)
          {
            ACMDRIVERDETAILSW acdetail;

            acdetail.cbStruct = sizeof (ACMDRIVERDETAILSW);
            if (SendDriverMessage (hDrvr, ACMDM_DRIVER_DETAILS,
                            GPOINTER_TO_INT (&acdetail), 0) == MMSYSERR_NOERROR
                    && (acdetail.fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC))
              {
                gint i;

                for (i = 0; i < acdetail.cFormatTags; i++)
                  {
                    ACMFORMATTAGDETAILSW acfmttag;

                    acfmttag.cbStruct = sizeof (ACMFORMATTAGDETAILSW);
                    acfmttag.dwFormatTagIndex = i;
                    if (SendDriverMessage (hDrvr, ACMDM_FORMATTAG_DETAILS,
                                                GPOINTER_TO_INT (&acfmttag),
                                                ACM_FORMATTAGDETAILSF_INDEX)
                                                        == MMSYSERR_NOERROR
                            && (acfmttag.fdwSupport
                                            & ACMDRIVERDETAILS_SUPPORTF_CODEC)
                            && acfmttag.dwFormatTag != WAVE_FORMAT_PCM
                            && !g_hash_table_lookup (ghash_tag,
                                    GUINT_TO_POINTER (acfmttag.dwFormatTag)))
                      {
                        gunichar2 *value;

                        value = g_memdup (name,
                                (lstrlenW (name) + 1) * sizeof (gunichar2));
                        g_hash_table_insert (ghash_tag,
                            GUINT_TO_POINTER (acfmttag.dwFormatTag), value);
                      }
                  }
              }
            CloseDriver (hDrvr, 0, 0);
          }
        g_free (name);
      }
  return RegCloseKey (hKey) == ERROR_SUCCESS;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static void
acm_exit_callback (gpointer key,
                   gpointer value,
                   gpointer ghash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable *)ghash, key,
                                                    &orig_key, &orig_value))
    g_free (orig_value);
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ACMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_exit (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (ghash_tag)
    {
      g_hash_table_foreach (ghash_tag, acm_exit_callback, ghash_tag);
      g_hash_table_destroy (ghash_tag);
      ghash_tag = NULL;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:ACM基本                                                                  *
*                                                                             *
******************************************************************************/
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static gint
acm_get_handler_list_compare (gconstpointer a,
                              gconstpointer b)
{
  return GPOINTER_TO_UINT (a) - GPOINTER_TO_UINT (b);
}


static void
acm_get_handler_list_callback (gpointer   key,
                               gpointer   value,
                               GList    **glist)
{
  *glist = g_list_insert_sorted (*glist, key, acm_get_handler_list_compare);
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:登録されているタグを取得する
    RET,タグのリスト,NULL:登録なし                                          */
GList *
acm_get_tag_list (void)
{
  GList *glist = NULL;

#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (ghash_tag)
    g_hash_table_foreach (ghash_tag,
                                (GHFunc)acm_get_handler_list_callback, &glist);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return glist;
}


/*  ja:Codecを開く
    tag,タグ
    RET,AcmObject,NULL:エラー                                               */
AcmObject *
acm_open (const guint16 tag)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gunichar2 *name;
  AcmObject *acm_object = NULL;

  name = g_hash_table_lookup (ghash_tag, GUINT_TO_POINTER ((guint)tag));
  if (name)
    {
      ACMDRVOPENDESC acopen;
      HDRVR hDrvr;

      g_memset (&acopen, 0, sizeof (ACMDRVOPENDESC));
      acopen.cbStruct = sizeof (ACMDRVOPENDESC);
      acopen.fccType = ACMDRIVERDETAILS_FCCTYPE_AUDIOCODEC;
      acopen.pszSectionName = drivers32;
      acopen.pszAliasName = name;
      hDrvr = OpenDriver (name, NULL, GPOINTER_TO_INT (&acopen));
      if (hDrvr)
        {
          ACMFORMATTAGDETAILSW acfmttag;

          acfmttag.cbStruct = sizeof (ACMFORMATTAGDETAILSW);
          acfmttag.dwFormatTag = WAVE_FORMAT_PCM;
          if (SendDriverMessage (hDrvr, ACMDM_FORMATTAG_DETAILS,
                                            GPOINTER_TO_INT (&acfmttag),
                                            ACM_FORMATTAGDETAILSF_FORMATTAG)
                                                        == MMSYSERR_NOERROR
                    && (acfmttag.fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC)
                    && acfmttag.dwFormatTag == WAVE_FORMAT_PCM
                    && (acfmttag.cbFormatSize == WFX_SIZE - sizeof (guint16)
                                        || acfmttag.cbFormatSize == WFX_SIZE))
            {
              gsize format_size_pcm;

              format_size_pcm = acfmttag.cbFormatSize;
              acfmttag.cbStruct = sizeof (ACMFORMATTAGDETAILSW);
              acfmttag.dwFormatTag = tag;
              if (SendDriverMessage (hDrvr, ACMDM_FORMATTAG_DETAILS,
                                            GPOINTER_TO_INT (&acfmttag),
                                            ACM_FORMATTAGDETAILSF_FORMATTAG)
                                                        == MMSYSERR_NOERROR
                    && (acfmttag.fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC)
                    && acfmttag.dwFormatTag == tag)
                {
                  acm_object = g_malloc0 (sizeof (AcmObject));
                  acm_object->counter = 1;
                  acm_object->tag = tag;
                  acm_object->format_size = acfmttag.cbFormatSize;
                  acm_object->format_count = acfmttag.cStandardFormats;
                  acm_object->name = g_utf16_to_utf8 (acfmttag.szFormatTag, -1,
                                                            NULL, NULL, NULL);
                  acm_object->format_size_pcm = format_size_pcm;
                  acm_object->hDrvr = hDrvr;
                  if (!acm_object->name || acm_object->name[0] == '\0')
                    {
                      ACMDRIVERDETAILSW acdetail;

                      acdetail.cbStruct = sizeof (ACMDRIVERDETAILSW);
                      if (SendDriverMessage (hDrvr, ACMDM_DRIVER_DETAILS,
                        GPOINTER_TO_INT (&acdetail), 0) == MMSYSERR_NOERROR)
                        {
                          g_free (acm_object->name);
                          acm_object->name = g_utf16_to_utf8
                                (acdetail.szLongName, -1, NULL, NULL, NULL);
                        }
                    }
                }
            }
          if (!acm_object)
            CloseDriver (hDrvr, 0, 0);
        }
    }
  return acm_object;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return NULL;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Codecの参照数を増やす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_ref (AcmObject *acm_object)
{
  return acm_object ? acm_object->counter++, TRUE : FALSE;
}


/*  ja:Codecの参照数を減らす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_unref (AcmObject *acm_object)
{
  if (acm_object && --(acm_object->counter) <= 0)
    {
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
      if (acm_object->stat != 0)
        acm_convert_end (acm_object, NULL, NULL);
      CloseDriver (acm_object->hDrvr, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
      g_free (acm_object->name);
      g_free (acm_object);
    }
  return TRUE;
}


/*  ja:Codecを複製する
    acm_object,AcmObject
           RET,AcmObject,NULL:エラー                                        */
AcmObject *
acm_dup (AcmObject *acm_object)
{
  return acm_object ? acm_open (acm_object->tag) : NULL;
}


/******************************************************************************
*                                                                             *
* ja:ACM情報                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:Codecのタグを取得する
    acm_object,AcmObject
           RET,タグ,0:エラー                                                */
guint16
acm_get_tag (AcmObject *acm_object)
{
  return acm_object ? acm_object->tag : 0;
}


/*  ja:Codecの名前を取得する
    acm_object,AcmObject
           RET,名前,NULL:エラー                                             */
const gchar *
acm_get_name (AcmObject *acm_object)
{
  return acm_object ? acm_object->name : NULL;
}


/*  ja:Codecの属性を取得する
    acm_object,AcmObject
           RET,ACM_ATTR_NATIVE,ACM_ATTR_WIN32                               */
gint
acm_get_attribute (AcmObject *acm_object)
{
  return ACM_ATTR_WIN32;
}


/*  ja:WaveFormatEx構造体の名前を取得する
    acm_object,AcmObject
           wfx,WaveFormatEx構造体へのポインタ
           RET,名前,NULL:エラー                                             */
gchar *
acm_get_format_name (AcmObject          *acm_object,
                     const WaveFormatEx *wfx)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gchar *name = NULL;
  gint i;
  WaveFormatEx *wfx_x;

  if (!acm_object || !acm_object->hDrvr || !wfx
                        || wfx_get_format_tag (wfx) != acm_object->tag
                        || wf_header_bytes (wfx) != acm_object->format_size)
    return FALSE;
  wfx_x = g_malloc (acm_object->format_size);
  for (i = 0; i < acm_object->format_count; i++)
    {
      ACMFORMATDETAILSW acfmt;

      g_memset (&acfmt, 0, sizeof (ACMFORMATDETAILSW));
      acfmt.cbStruct = sizeof (ACMFORMATDETAILSW);
      acfmt.dwFormatIndex = i;
      acfmt.dwFormatTag = acm_object->tag;
      acfmt.pwfx = (LPWAVEFORMATEX)wfx_x;
      acfmt.cbwfx = acm_object->format_size;
      if (SendDriverMessage (acm_object->hDrvr, ACMDM_FORMAT_DETAILS,
                                GPOINTER_TO_INT (&acfmt),
                                ACM_FORMATDETAILSF_INDEX) == MMSYSERR_NOERROR
                        && g_memcmp (wfx, wfx_x, acm_object->format_size) == 0)
        {
          name = g_utf16_to_utf8 (acfmt.szFormat, -1, NULL, NULL, NULL);
          if (!name || name[0] == '\0')
            {
              gchar *samples_per_sec, *bits_per_sample, *channels;
              gchar *average_bytes_per_sec;

              samples_per_sec = wfx_get_samples_per_sec (wfx) < 1000
                ? g_strdup_printf ("%d Hz, ", wfx_get_samples_per_sec (wfx))
                : g_strdup_printf ("%d.%03d kHz, ",
                                        wfx_get_samples_per_sec (wfx) / 1000,
                                        wfx_get_samples_per_sec (wfx) % 1000);
              bits_per_sample = wfx_get_bits_per_sample (wfx) > 1
                ? g_strdup_printf (_("%d bits, "),
                                                wfx_get_bits_per_sample (wfx))
                : wfx_get_bits_per_sample (wfx) == 1 ? g_strdup (_("1 bit, "))
                : g_strdup ("");
              channels
                    = wfx_get_channels (wfx) == 0 ? g_strdup (_("0 channel, "))
                    : wfx_get_channels (wfx) == 1 ? g_strdup (_("Monophone, "))
                    : wfx_get_channels (wfx) == 2 ? g_strdup (_("Stereo, "))
                    : g_strdup_printf ("%d channels, ",
                                                    wfx_get_channels (wfx));
              average_bytes_per_sec
                                 = wfx_get_average_bytes_per_sec (wfx) < 1000
                    ? g_strdup_printf ("%d B/Sec",
                                        wfx_get_average_bytes_per_sec (wfx))
                    : g_strdup_printf ("%d.%03d KB/Sec",
                                wfx_get_average_bytes_per_sec (wfx) / 1000,
                                wfx_get_average_bytes_per_sec (wfx) % 1000);
              name = g_strconcat (samples_per_sec, bits_per_sample, channels,
                                                average_bytes_per_sec, NULL);
              g_free (samples_per_sec);
              g_free (bits_per_sample);
              g_free (channels);
              g_free (average_bytes_per_sec);
            }
          break;
        }
    }
  g_free (wfx_x);
  return name;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/******************************************************************************
*                                                                             *
* ja:ACMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    acm_object,AcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
acm_is_dialog_about (AcmObject *acm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return acm_object && acm_object->hDrvr
        && SendDriverMessage (acm_object->hDrvr, ACMDM_DRIVER_ABOUT, -1, 0)
                                                        == MMSYSERR_NOERROR;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Aboutダイアログを表示する
    acm_object,AcmObject
        widget,親ウインドウ                                                 */
void
acm_dialog_about (AcmObject *acm_object,
                  GtkWidget *widget)

{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (acm_object && acm_object->hDrvr)
    SendDriverMessage (acm_object->hDrvr, ACMDM_DRIVER_ABOUT,
# if defined (G_OS_WIN32)
        widget ? GPOINTER_TO_INT (GDK_WINDOW_HWND (widget->window)) : 0, 0);
# elif defined (USE_W32LDR)
                                                GPOINTER_TO_INT (widget), 0);
# endif /* defined (USE_W32LDR) */
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:設定ダイアログのサポートを確認する
    acm_object,AcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
acm_is_dialog_preference (AcmObject *acm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return acm_object && acm_object->hDrvr
            && SendDriverMessage (acm_object->hDrvr, DRV_QUERYCONFIGURE, 0, 0);
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:設定ダイアログを表示する
    acm_object,AcmObject
        widget,親ウインドウ                                                 */
void
acm_dialog_preference (AcmObject *acm_object,
                       GtkWidget *widget)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (acm_object && acm_object->hDrvr)
    SendDriverMessage (acm_object->hDrvr, DRV_CONFIGURE,
# if defined (G_OS_WIN32)
        widget ? GPOINTER_TO_INT (GDK_WINDOW_HWND (widget->window)) : 0, 0);
# elif defined (USE_W32LDR)
                                                GPOINTER_TO_INT (widget), 0);
# endif /* defined (USE_W32LDR) */
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/******************************************************************************
*                                                                             *
* en:ACM Convert                                                              *
*                                                                             *
******************************************************************************/
/*  ja:変換可能か確認する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_convert_query (AcmObject          *acm_object,
                   const WaveFormatEx *wfx_in,
                   const WaveFormatEx *wfx_out)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  ACMDRVSTREAMINSTANCE acstminst;

  if (!acm_object || !acm_object->hDrvr || !wfx_in || !wfx_out
                    || ((wfx_get_format_tag (wfx_in) != WAVE_FMT_PCM
                    || wf_header_bytes (wfx_in) != WFX_SIZE
                    || wfx_get_format_tag (wfx_out) != acm_object->tag
                    || wf_header_bytes (wfx_out) != acm_object->format_size)
                    && (wfx_get_format_tag (wfx_in) != acm_object->tag
                    || wf_header_bytes (wfx_in) != acm_object->format_size
                    || wfx_get_format_tag (wfx_out) != WAVE_FMT_PCM
                    || wf_header_bytes (wfx_out) != WFX_SIZE)))
    return FALSE;
  g_memset (&acstminst, 0, sizeof (ACMDRVSTREAMINSTANCE));
  acstminst.cbStruct = sizeof (ACMDRVSTREAMINSTANCE);
  acstminst.pwfxSrc = (LPWAVEFORMATEX)wfx_in;
  acstminst.pwfxDst = (LPWAVEFORMATEX)wfx_out;
  acstminst.fdwOpen = ACM_STREAMOPENF_NONREALTIME | ACM_STREAMOPENF_QUERY;
  return SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_OPEN,
                        GPOINTER_TO_INT (&acstminst), 0) == MMSYSERR_NOERROR;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:変換後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_convert_get_format_size (AcmObject          *acm_object,
                             const WaveFormatEx *wfx)
{
  return acm_object && wfx ? wfx_get_format_tag (wfx) == acm_object->tag
                               ? WFX_SIZE
                               : wfx_get_format_tag (wfx) == WAVE_FMT_PCM
                                   ? acm_object->format_size
                                   : 0
                           : 0;
}


/*  ja:変換を開始する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_convert_begin (AcmObject          *acm_object,
                   const WaveFormatEx *wfx_in,
                   const WaveFormatEx *wfx_out)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (!acm_object || !acm_object->hDrvr || !wfx_in || !wfx_out
                    || ((wfx_get_format_tag (wfx_in) != WAVE_FMT_PCM
                    || wf_header_bytes (wfx_in) != WFX_SIZE
                    || wfx_get_format_tag (wfx_out) != acm_object->tag
                    || wf_header_bytes (wfx_out) != acm_object->format_size)
                    && (wfx_get_format_tag (wfx_in) != acm_object->tag
                    || wf_header_bytes (wfx_in) != acm_object->format_size
                    || wfx_get_format_tag (wfx_out) != WAVE_FMT_PCM
                    || wf_header_bytes (wfx_out) != WFX_SIZE)))
    return FALSE;
  g_memset (&acm_object->acstminst, 0, sizeof (ACMDRVSTREAMINSTANCE));
  acm_object->acstminst.cbStruct = sizeof (ACMDRVSTREAMINSTANCE);
  acm_object->acstminst.pwfxSrc = g_memdup (wfx_in, wf_header_bytes (wfx_in));
  acm_object->acstminst.pwfxDst = g_memdup
                                        (wfx_out, wf_header_bytes (wfx_out));
  acm_object->acstminst.fdwOpen = ACM_STREAMOPENF_NONREALTIME;
  if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_OPEN,
            GPOINTER_TO_INT (&acm_object->acstminst), 0) != MMSYSERR_NOERROR)
    {
      g_free (acm_object->acstminst.pwfxSrc);
      g_free (acm_object->acstminst.pwfxDst);
      acm_object->acstminst.pwfxSrc = NULL;
      acm_object->acstminst.pwfxDst = NULL;
      return FALSE;
    }
  acm_object->acstmsize.cbStruct = sizeof (ACMDRVSTREAMSIZE);
  acm_object->acstmsize.fdwSize = ACM_STREAMSIZEF_DESTINATION;
  acm_object->acstmsize.cbSrcLength = 0;
  acm_object->acstmsize.cbDstLength
                    = MAX (ACM_BUFFER_SIZE / wfx_get_block_align (wfx_out), 1)
                                            * wfx_get_block_align (wfx_out);
  if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_SIZE,
                GPOINTER_TO_INT (&acm_object->acstminst),
                GPOINTER_TO_INT (&acm_object->acstmsize)) != MMSYSERR_NOERROR
                                    || acm_object->acstmsize.cbSrcLength <= 0
                                    || acm_object->acstmsize.cbDstLength <= 0)
    {
      acm_object->acstmsize.cbStruct = sizeof (ACMDRVSTREAMSIZE);
      acm_object->acstmsize.fdwSize = ACM_STREAMSIZEF_SOURCE;
      acm_object->acstmsize.cbSrcLength
                    = MAX (ACM_BUFFER_SIZE / wfx_get_block_align (wfx_in), 1)
                                                * wfx_get_block_align (wfx_in);
      acm_object->acstmsize.cbDstLength = 0;
      if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_SIZE,
                GPOINTER_TO_INT (&acm_object->acstminst),
                GPOINTER_TO_INT (&acm_object->acstmsize)) != MMSYSERR_NOERROR
                                    || acm_object->acstmsize.cbSrcLength <= 0
                                    || acm_object->acstmsize.cbDstLength <= 0)
        {
          SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_CLOSE,
                                GPOINTER_TO_INT (&acm_object->acstminst), 0);
          g_free (acm_object->acstminst.pwfxSrc);
          g_free (acm_object->acstminst.pwfxDst);
          acm_object->acstminst.pwfxSrc = NULL;
          acm_object->acstminst.pwfxDst = NULL;
          return FALSE;
        }
    }
  acm_object->acstmsize.cbSrcLength = MAX (acm_object->acstmsize.cbSrcLength
        / wfx_get_block_align (wfx_in), 1) * wfx_get_block_align (wfx_in);
  acm_object->acstmsize.cbDstLength = MAX (acm_object->acstmsize.cbDstLength
        / wfx_get_block_align (wfx_out), 1) * wfx_get_block_align (wfx_out);
  g_memset (&acm_object->acstmhdr, 0, sizeof (ACMDRVSTREAMHEADER));
  acm_object->acstmhdr.cbStruct = sizeof (ACMDRVSTREAMHEADER);
  acm_object->acstmhdr.pbSrc = g_malloc (acm_object->acstmsize.cbSrcLength);
  acm_object->acstmhdr.cbSrcLength = acm_object->acstmsize.cbSrcLength;
  acm_object->acstmhdr.pbDst = g_malloc (acm_object->acstmsize.cbDstLength);
  acm_object->acstmhdr.cbDstLength = acm_object->acstmsize.cbDstLength;
  switch (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_PREPARE,
                                    GPOINTER_TO_INT (&acm_object->acstminst),
                                    GPOINTER_TO_INT (&acm_object->acstmhdr)))
    {
      case MMSYSERR_NOERROR:
      case MMSYSERR_NOTSUPPORTED:
        break;
      default:
        SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_CLOSE,
                                GPOINTER_TO_INT (&acm_object->acstminst), 0);
        g_free (acm_object->acstminst.pwfxSrc);
        g_free (acm_object->acstminst.pwfxDst);
        g_free (acm_object->acstmhdr.pbSrc);
        g_free (acm_object->acstmhdr.pbDst);
        acm_object->acstminst.pwfxSrc = NULL;
        acm_object->acstminst.pwfxDst = NULL;
        acm_object->acstmhdr.pbSrc = NULL;
        acm_object->acstmhdr.pbDst = NULL;
        return FALSE;
    }
  acm_object->acstmhdr.fdwStatus |= ACMSTREAMHEADER_STATUSF_PREPARED;
  acm_object->acstmhdr.cbSrcLength = 0;
  acm_object->stat = 1;
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:変換を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_convert_end (AcmObject *acm_object,
                 gpointer  *out_buf,
                 gsize     *out_samples)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gboolean result = TRUE;

  if (!acm_object || !acm_object->hDrvr || acm_object->stat == 0)
    return FALSE;
  if (out_buf)
    *out_buf = NULL;
  if (out_samples)
    *out_samples = 0;
  if (/*acm_object->acstmhdr.cbSrcLength > 0 && */out_buf && out_samples)
    {
      acm_object->acstmhdr.fdwStatus = 0;
      acm_object->acstmhdr.cbSrcLengthUsed = 0;
      acm_object->acstmhdr.cbDstLengthUsed = 0;
      acm_object->acstmhdr.fdwConvert = acm_object->stat == 1
                            ? ACM_STREAMCONVERTF_START | ACM_STREAMCONVERTF_END
                            : ACM_STREAMCONVERTF_END;
      if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_CONVERT,
                GPOINTER_TO_INT (&acm_object->acstminst),
                GPOINTER_TO_INT (&acm_object->acstmhdr)) == MMSYSERR_NOERROR)
        {
          *out_buf = g_memdup (acm_object->acstmhdr.pbDst,
                                        acm_object->acstmhdr.cbDstLengthUsed);
          *out_samples = acm_object->acstmhdr.cbDstLengthUsed
                        / wfx_get_block_align (acm_object->acstminst.pwfxDst);
        }
      else
        {
          result = FALSE;
        }
    }
  acm_object->acstmhdr.cbSrcLength = acm_object->acstmsize.cbSrcLength;
  acm_object->acstmhdr.cbDstLength = acm_object->acstmsize.cbDstLength;
  switch (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_UNPREPARE,
                                    GPOINTER_TO_INT (&acm_object->acstminst),
                                    GPOINTER_TO_INT (&acm_object->acstmhdr)))
    {
      case MMSYSERR_NOERROR:
      case MMSYSERR_NOTSUPPORTED:
        break;
      default:
        result = FALSE;
    }
  if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_CLOSE,
            GPOINTER_TO_INT (&acm_object->acstminst), 0) != MMSYSERR_NOERROR)
    result = FALSE;
  g_free (acm_object->acstminst.pwfxSrc);
  g_free (acm_object->acstminst.pwfxDst);
  g_free (acm_object->acstmhdr.pbSrc);
  g_free (acm_object->acstmhdr.pbDst);
  acm_object->acstminst.pwfxSrc = NULL;
  acm_object->acstminst.pwfxDst = NULL;
  acm_object->acstmhdr.pbSrc = NULL;
  acm_object->acstmhdr.pbDst = NULL;
  acm_object->stat = 0;
  return result;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:変換する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_convert (AcmObject      *acm_object,
             const gpointer  in_buf,
             const gsize     in_samples,
             gpointer       *out_buf,
             gsize          *out_samples)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  const guint8 *in_p;
  guint8 *out_p = NULL;
  gsize in_bytes, out_bytes = 0;

  if (!acm_object || !acm_object->hDrvr || acm_object->stat == 0)
    return FALSE;
  in_p = in_buf;
  in_bytes = in_samples * wfx_get_block_align (acm_object->acstminst.pwfxSrc);
  while (in_bytes > 0
    && acm_object->acstmhdr.cbSrcLength < acm_object->acstmsize.cbSrcLength)
    {
      gsize leng;

      leng = MIN (in_bytes, acm_object->acstmsize.cbSrcLength
                                        - acm_object->acstmhdr.cbSrcLength);
      g_memmove (acm_object->acstmhdr.pbSrc + acm_object->acstmhdr.cbSrcLength,
                                                                in_p, leng);
      acm_object->acstmhdr.fdwStatus = 0;
      acm_object->acstmhdr.cbSrcLength += leng;
      acm_object->acstmhdr.cbSrcLengthUsed = 0;
      acm_object->acstmhdr.cbDstLengthUsed = 0;
      acm_object->acstmhdr.fdwConvert = ACM_STREAMCONVERTF_BLOCKALIGN
                    | (acm_object->stat == 1 ? ACM_STREAMCONVERTF_START : 0);
      acm_object->stat = 2;
      in_bytes -= leng;
      in_p += leng;
      if (SendDriverMessage (acm_object->hDrvr, ACMDM_STREAM_CONVERT,
                GPOINTER_TO_INT (&acm_object->acstminst),
                GPOINTER_TO_INT (&acm_object->acstmhdr)) != MMSYSERR_NOERROR)
        break;
      g_memmove (acm_object->acstmhdr.pbSrc,
            acm_object->acstmhdr.pbSrc + acm_object->acstmhdr.cbSrcLengthUsed,
            acm_object->acstmhdr.cbSrcLength
                                    - acm_object->acstmhdr.cbSrcLengthUsed);
      acm_object->acstmhdr.cbSrcLength -= acm_object->acstmhdr.cbSrcLengthUsed;
      out_p = g_realloc (out_p,
                            out_bytes + acm_object->acstmhdr.cbDstLengthUsed);
      g_memmove (out_p + out_bytes, acm_object->acstmhdr.pbDst,
                                        acm_object->acstmhdr.cbDstLengthUsed);
      out_bytes += acm_object->acstmhdr.cbDstLengthUsed;
    }
  if (in_bytes > 0)
    {
      g_free (out_p);
      return FALSE;
    }
  if (out_buf)
    *out_buf = out_p;
  else
    g_free (out_p);
  if (out_samples)
    *out_samples = out_bytes
                        / wfx_get_block_align (acm_object->acstminst.pwfxDst);
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/******************************************************************************
*                                                                             *
* en:ACM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_compress_query (AcmObject          *acm_object,
                    const WaveFormatEx *wfx_in,
                    const WaveFormatEx *wfx_out)
{
  return acm_object && wfx_in && wfx_out
                    && wfx_get_format_tag (wfx_in) == WAVE_FMT_PCM
                    && wf_header_bytes (wfx_in) == WFX_SIZE
                    && wfx_get_format_tag (wfx_out) == acm_object->tag
                    && wf_header_bytes (wfx_out) == acm_object->format_size
                    && acm_convert_query (acm_object, wfx_in, wfx_out);
}


/*  ja:圧縮後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_compress_get_format_size (AcmObject          *acm_object,
                              const WaveFormatEx *wfx)
{
  return acm_object && wfx && wfx_get_format_tag (wfx) == WAVE_FMT_PCM
                        ? acm_convert_get_format_size (acm_object, wfx) : 0;
}


/*  ja:圧縮後のフォーマットを取得する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
         index,インデックス(0...)
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_compress_enum_format (AcmObject          *acm_object,
                          const WaveFormatEx *wfx_in,
                          WaveFormatEx       *wfx_out,
                          const gint          index)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gboolean result = FALSE;
  gint i, count = 0;
  WaveFormatEx *wfx;

  if (!acm_object || !acm_object->hDrvr || !wfx_in || !wfx_out
                                || wfx_get_format_tag (wfx_in) != WAVE_FMT_PCM
                                || wf_header_bytes (wfx_in) != WFX_SIZE)
    return FALSE;
  wfx = g_malloc (acm_object->format_size);
  for (i = 0; i < acm_object->format_count; i++)
    {
      ACMFORMATDETAILSW acfmt;

      g_memset (&acfmt, 0, sizeof (ACMFORMATDETAILSW));
      acfmt.cbStruct = sizeof (ACMFORMATDETAILSW);
      acfmt.dwFormatIndex = i;
      acfmt.dwFormatTag = acm_object->tag;
      acfmt.pwfx = (LPWAVEFORMATEX)wfx;
      acfmt.cbwfx = acm_object->format_size;
      if (SendDriverMessage (acm_object->hDrvr, ACMDM_FORMAT_DETAILS,
                                GPOINTER_TO_INT (&acfmt),
                                ACM_FORMATDETAILSF_INDEX) == MMSYSERR_NOERROR
                            && acm_compress_query (acm_object, wfx_in, wfx)
                            && index == count++)
        {
          g_memmove (wfx_out, wfx, acm_object->format_size);
          result = TRUE;
          break;
        }
    }
  g_free (wfx);
  return result;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:圧縮を開始する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_compress_begin (AcmObject          *acm_object,
                    const WaveFormatEx *wfx_in,
                    const WaveFormatEx *wfx_out)
{
  return acm_object && wfx_in && wfx_out
                    && wfx_get_format_tag (wfx_in) == WAVE_FMT_PCM
                    && wf_header_bytes (wfx_in) == WFX_SIZE
                    && wfx_get_format_tag (wfx_out) == acm_object->tag
                    && wf_header_bytes (wfx_out) == acm_object->format_size
                    && acm_convert_begin (acm_object, wfx_in, wfx_out);
}


/*  ja:圧縮を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_compress_end (AcmObject *acm_object,
                  gpointer  *out_buf,
                  gsize     *out_samples)
{
  return acm_convert_end (acm_object, out_buf, out_samples);
}


/*  ja:圧縮する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_compress (AcmObject      *acm_object,
              const gpointer  in_buf,
              const gsize     in_samples,
              gpointer       *out_buf,
              gsize          *out_samples)
{
  return acm_convert (acm_object, in_buf, in_samples, out_buf, out_samples);
}


/******************************************************************************
*                                                                             *
* en:ACM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_decompress_query (AcmObject          *acm_object,
                      const WaveFormatEx *wfx)
{
  gboolean result;
  WaveFormatEx *wfx_out;

  if (!acm_object || !wfx || wfx_get_format_tag (wfx) != acm_object->tag
                        || wf_header_bytes (wfx) != acm_object->format_size)
    return FALSE;
  wfx_out = g_malloc (acm_decompress_get_format_size (acm_object, wfx));
  result = acm_decompress_get_format (acm_object, wfx, wfx_out)
                            && acm_convert_query (acm_object, wfx, wfx_out);
  g_free (wfx_out);
  return result;
}


/*  ja:展開後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_decompress_get_format_size (AcmObject          *acm_object,
                                const WaveFormatEx *wfx)
{
  return acm_object && wfx && wfx_get_format_tag (wfx) == acm_object->tag
                        ? acm_convert_get_format_size (acm_object, wfx) : 0;
}


/*  ja:展開後のフォーマットを取得する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_decompress_get_format (AcmObject          *acm_object,
                           const WaveFormatEx *wfx_in,
                           WaveFormatEx       *wfx_out)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  ACMDRVFORMATSUGGEST acfmtsgst;

  if (!acm_object || !acm_object->hDrvr || !wfx_in
                        || wfx_get_format_tag (wfx_in) != acm_object->tag
                        || wf_header_bytes (wfx_in) != acm_object->format_size)
    return FALSE;
  acfmtsgst.cbStruct = sizeof (ACMDRVFORMATSUGGEST);
  acfmtsgst.fdwSuggest = ACM_FORMATSUGGESTF_WFORMATTAG;
  acfmtsgst.pwfxSrc = (LPWAVEFORMATEX)wfx_in;
  acfmtsgst.cbwfxSrc = acm_object->format_size;
  acfmtsgst.pwfxDst = (LPWAVEFORMATEX)wfx_out;
  acfmtsgst.cbwfxDst = acm_object->format_size_pcm;
  wfx_set_format_tag (wfx_out, WAVE_FMT_PCM);
  if (SendDriverMessage (acm_object->hDrvr, ACMDM_FORMAT_SUGGEST,
                        GPOINTER_TO_INT (&acfmtsgst), 0) != MMSYSERR_NOERROR)
    return FALSE;
  wfx_set_block_align (wfx_out,
        wfx_get_channels (wfx_out) * wfx_get_bits_per_sample (wfx_out) / 8);
  wfx_set_average_bytes_per_sec (wfx_out,
            wfx_get_samples_per_sec (wfx_out) * wfx_get_block_align (wfx_out));
  wfx_set_size (wfx_out, 0);
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:展開を開始する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_decompress_begin (AcmObject          *acm_object,
                      const WaveFormatEx *wfx)
{
  gboolean result;
  WaveFormatEx *wfx_out;

  if (!acm_object
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
                        || !acm_object->hDrvr
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
                        || !wfx
                        || wfx_get_format_tag (wfx) != acm_object->tag
                        || wf_header_bytes (wfx) != acm_object->format_size
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
                        || acm_object->stat != 0
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
                        )
    return FALSE;
  wfx_out = g_malloc (acm_decompress_get_format_size (acm_object, wfx));
  result = acm_decompress_get_format (acm_object, wfx, wfx_out)
                            && acm_convert_begin (acm_object, wfx, wfx_out);
  g_free (wfx_out);
  return result;
}


/*  ja:展開を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_decompress_end (AcmObject *acm_object,
                    gpointer  *out_buf,
                    gsize     *out_samples)
{
  return acm_convert_end (acm_object, out_buf, out_samples);
}


/*  ja:展開する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_decompress (AcmObject      *acm_object,
                const gpointer  in_buf,
                const gsize     in_samples,
                gpointer       *out_buf,
                gsize          *out_samples)
{
  return acm_convert (acm_object, in_buf, in_samples, out_buf, out_samples);
}
