/*
    orzpref
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "orzpref.h"
#include "orzhistory.h"
#include "misc/misc.h"
#include <gdk/gdkkeysyms.h>


static void orz_pref_class_init (OrzPrefClass *klass);
static void orz_pref_init       (OrzPref      *dialog);
static void orz_pref_escape     (OrzPref      *dialog);


static GtkDialogClass *parent_class = NULL;


enum
{
  ESCAPE_SIGNAL,
  LAST_SIGNAL
};


static gint orz_pref_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
orz_pref_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      const static GTypeInfo info =
      {
        sizeof (OrzPrefClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)orz_pref_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (OrzPref),
        0,                  /* n_preallocs */
        (GInstanceInitFunc)orz_pref_init,
      };

      type = g_type_register_static (GTK_TYPE_DIALOG, "OrzPref", &info, 0);
    }

  return type;
}


static void
orz_pref_class_init (OrzPrefClass *klass)
{
  GtkBindingSet *binding_set;

  parent_class = g_type_class_peek_parent (klass);

  klass->escape = orz_pref_escape;

  orz_pref_signals[ESCAPE_SIGNAL]
                = g_signal_new ("escape",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (OrzPrefClass, escape),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);

  binding_set = gtk_binding_set_by_class (klass);
  gtk_binding_entry_add_signal (binding_set, GDK_Escape, 0, "escape", 0);
}


/* ja:OKボタンを表示 */
#define orz_pref_toggled orz_pref_changed
static void
orz_pref_changed (GtkWidget *widget,
                  OrzPref   *pref)
{
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK, TRUE);
}


/* ja:スピンボタンでリターンが押された */
static void
orz_pref_activate (GtkWidget *widget,
                   OrzPref   *pref)
{
  if (GTK_WIDGET_VISIBLE (pref->ok_button)
                                    && GTK_WIDGET_SENSITIVE (pref->ok_button))
    g_signal_emit_by_name (G_OBJECT (pref->ok_button), "clicked");
}


static void
orz_pref_escape (OrzPref *pref)
{
  if (GTK_WIDGET_VISIBLE (pref->cancel_button)
                                && GTK_WIDGET_SENSITIVE (pref->cancel_button))
    g_signal_emit_by_name (G_OBJECT (pref->cancel_button), "clicked");
}


/* ja:閉じるボタンが押された */
static gboolean
orz_pref_delete (GtkWidget *widget,
                 GdkEvent  *event,
                 gpointer   user_data)
{
  if (GTK_WIDGET_VISIBLE (ORZ_PREF (widget)->cancel_button)
                    && GTK_WIDGET_SENSITIVE (ORZ_PREF (widget)->cancel_button))
    g_signal_emit_by_name
                    (G_OBJECT (ORZ_PREF (widget)->cancel_button), "clicked");
  return TRUE;
}


static void
orz_pref_init (OrzPref *pref)
{
  gchar *title;
  gint width, height;
  GtkStockItem stock_item;
  GtkWidget *label0, *label1, *label2;
  GtkWidget *frame0, *frame1, *hbox0, *hbox1, *vbox;

  width = gdk_screen_width ();
  height = gdk_screen_height ();

  /* ja:メインウインドウ */
  gtk_stock_lookup (GTK_STOCK_PREFERENCES, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  gtk_window_set_title (GTK_WINDOW (pref), title);
  g_free (title);
  g_signal_connect (G_OBJECT (pref), "delete-event",
                                        G_CALLBACK (orz_pref_delete), NULL);
  /* ja:ボタン */
#ifdef G_OS_WIN32
  pref->ok_button = gtk_dialog_add_button (GTK_DIALOG (pref),
                                        GTK_STOCK_OK, GTK_RESPONSE_OK);
  pref->cancel_button = gtk_dialog_add_button (GTK_DIALOG (pref),
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
#else /* not G_OS_WIN32 */
  pref->cancel_button = gtk_dialog_add_button (GTK_DIALOG (pref),
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  pref->ok_button = gtk_dialog_add_button (GTK_DIALOG (pref),
                                        GTK_STOCK_OK, GTK_RESPONSE_OK);
#endif /* not G_OS_WIN32 */
  /* ja:チェックボタン */
  pref->newfile_check = gtk_check_button_new_with_mnemonic (_("_New File"));
  pref->second_check  = gtk_check_button_new_with_mnemonic (_("_Open Window"));
  g_signal_connect (G_OBJECT (pref->newfile_check), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  g_signal_connect (G_OBJECT (pref->second_check), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  /* ja:スピンボタン */
  pref->history_spin = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (ORZ_HISTORY_NUM_DEF,
                                 ORZ_HISTORY_NUM_MIN,
                                 ORZ_HISTORY_NUM_MAX, 1, 5, 0)), 0, 0);
  pref->width_spin = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (width * 2 / 3, 0, width, 1, 5, 0)), 0, 0);
  pref->height_spin = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (height * 2 / 3, 0, height, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->history_spin), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->width_spin), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->height_spin), TRUE);
  g_signal_connect (G_OBJECT (pref->history_spin), "activate",
                                        G_CALLBACK (orz_pref_activate), pref);
  g_signal_connect (G_OBJECT (pref->width_spin), "activate",
                                        G_CALLBACK (orz_pref_activate), pref);
  g_signal_connect (G_OBJECT (pref->height_spin), "activate",
                                        G_CALLBACK (orz_pref_activate), pref);
  g_signal_connect (G_OBJECT (pref->history_spin), "changed",
                                        G_CALLBACK (orz_pref_changed), pref);
  g_signal_connect (G_OBJECT (pref->width_spin), "changed",
                                        G_CALLBACK (orz_pref_changed), pref);
  g_signal_connect (G_OBJECT (pref->height_spin), "changed",
                                        G_CALLBACK (orz_pref_changed), pref);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("H_istory"));
  label1 = gtk_label_new_with_mnemonic (_("_Width"));
  label2 = gtk_label_new_with_mnemonic (_("_Height"));
  gtk_label_set_mnemonic_widget (GTK_LABEL(label0), pref->history_spin);
  gtk_label_set_mnemonic_widget (GTK_LABEL(label1), pref->width_spin);
  gtk_label_set_mnemonic_widget (GTK_LABEL(label2), pref->height_spin);
  /* ja:ラジオボタン */
  pref->top_radio = gtk_radio_button_new_with_mnemonic (NULL, _("_Top"));
  pref->left_radio = gtk_radio_button_new_with_mnemonic_from_widget
                        (GTK_RADIO_BUTTON (pref->top_radio), _("_Left"));
  pref->right_radio = gtk_radio_button_new_with_mnemonic_from_widget
                        (GTK_RADIO_BUTTON (pref->left_radio), _("_Right"));
  pref->bottom_radio = gtk_radio_button_new_with_mnemonic_from_widget
                        (GTK_RADIO_BUTTON (pref->right_radio), _("_Bottom"));
  g_signal_connect (G_OBJECT (pref->top_radio), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  g_signal_connect (G_OBJECT (pref->left_radio), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  g_signal_connect (G_OBJECT (pref->right_radio), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  g_signal_connect (G_OBJECT (pref->bottom_radio), "toggled",
                                        G_CALLBACK (orz_pref_toggled), pref);
  /* ja:フレームとボックス */
  frame0 = gtk_frame_new (_("Window Size"));
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), pref->width_spin, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), pref->height_spin, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (frame0), hbox0);

  frame1 = gtk_frame_new (_("Tab Position"));
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), pref->top_radio, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), pref->left_radio, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox0), pref->right_radio, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), pref->bottom_radio, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame1), vbox);

  vbox = gtk_vbox_new (FALSE, SPACING);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), pref->history_spin, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), pref->newfile_check, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), pref->second_check, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame0, FALSE, FALSE, 0);

  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), vbox, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), frame1, TRUE, TRUE, 0);

  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (pref)->vbox), vbox);

  /* ja:表示 */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref),
                                                    GTK_RESPONSE_OK, FALSE);
  gtk_widget_show_all (vbox);
}


/******************************************************************************
*                                                                             *
* ja:更新関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
orz_pref_new (void)
{
  return GTK_WIDGET (g_object_new (ORZ_TYPE_PREF, NULL));
}


/*  ja:履歴を設定する
    pref,ウィジェット
     RET,履歴                                                               */
gint
orz_pref_get_history (OrzPref *pref)
{
  return gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (pref->history_spin));
}


/*  ja:新規ファイルを設定する
    pref,ウィジェット
     RET,新規ファイル                                                       */
gboolean
orz_pref_get_newfile (OrzPref *pref)
{
  return gtk_toggle_button_get_active
                                    (GTK_TOGGLE_BUTTON (pref->newfile_check));
}


/*  ja:新規ウインドウを設定する
    pref,ウィジェット
     RET,新規ウインドウ                                                     */
gboolean
orz_pref_get_second (OrzPref *pref)
{
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->second_check));
}


/*  ja:ウインドウサイズを設定する
      pref,ウィジェット
     width,幅
    height,高さ                                                             */
void
orz_pref_get_size (OrzPref *pref,
                   gint    *width,
                   gint    *height)
{
  if (width)
    *width = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (pref->width_spin));
  if (height)
    *height = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (pref->height_spin));
}


/*  ja:ノートブックの位置を設定する
    pref,ウィジェット
     RET,ノートブックの位置                                                 */
GtkPositionType
orz_pref_get_tab (OrzPref *pref)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->left_radio)))
    return GTK_POS_LEFT;
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->right_radio)))
    return GTK_POS_RIGHT;
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->bottom_radio)))
    return GTK_POS_BOTTOM;
  return GTK_POS_TOP;
}


/*  ja:履歴を設定する
       pref,ウィジェット
    history,履歴                                                            */
void
orz_pref_set_history (OrzPref    *pref,
                      const gint  history)
{
  gboolean sensitive;

  sensitive = GTK_WIDGET_SENSITIVE (pref->ok_button);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->history_spin), history);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK,
                                                                    sensitive);
}


/*  ja:新規ファイルを設定する
       pref,ウィジェット
    newfile,新規ファイル                                                    */
void
orz_pref_set_newfile (OrzPref        *pref,
                      const gboolean  newfile)
{
  gboolean sensitive;

  sensitive = GTK_WIDGET_SENSITIVE (pref->ok_button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->newfile_check),
                                                                    newfile);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK,
                                                                    sensitive);
}


/*  ja:新規ウインドウを設定する
      pref,ウィジェット
    second,新規ウインドウ                                                   */
void
orz_pref_set_second (OrzPref        *pref,
                     const gboolean  second)
{
  gboolean sensitive;

  sensitive = GTK_WIDGET_SENSITIVE (pref->ok_button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->second_check),
                                                                    second);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK,
                                                                    sensitive);
}


/*  ja:ウインドウサイズを設定する
      pref,ウィジェット
     width,幅
    height,高さ                                                             */
void
orz_pref_set_size (OrzPref    *pref,
                   const gint  width,
                   const gint  height)
{
  gboolean sensitive;

  sensitive = GTK_WIDGET_SENSITIVE (pref->ok_button);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->width_spin), width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->height_spin), height);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK,
                                                                    sensitive);
}


/*  ja:ノートブックの位置を設定する
     pref,ウィジェット
    n_pos,ノートブックの位置                                                */
void
orz_pref_set_tab (OrzPref               *pref,
                  const GtkPositionType  n_pos)
{
  gboolean sensitive;

  sensitive = GTK_WIDGET_SENSITIVE (pref->ok_button);
  switch (n_pos)
    {
      case GTK_POS_TOP:
        gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (pref->top_radio), TRUE);
        break;
      case GTK_POS_LEFT:
        gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (pref->left_radio), TRUE);
        break;
      case GTK_POS_RIGHT:
        gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (pref->right_radio), TRUE);
        break;
      case GTK_POS_BOTTOM:
        gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (pref->bottom_radio), TRUE);
    }
  gtk_dialog_set_response_sensitive (GTK_DIALOG (pref), GTK_RESPONSE_OK,
                                                                    sensitive);
}
