/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "bitmap.h"


/******************************************************************************
*                                                                             *
* ja:ビットマップ加工関数                                                     *
*                                                                             *
******************************************************************************/
typedef struct _PaletteTableList
{
  gint blue_min, blue_max, green_min, green_max, red_min, red_max;
  guint count;
} PaletteTableList;


/*  ja:パレットの構造体の範囲を最小化する
      ptl,パレット構造体
    table,パレットの分布                                                    */
static void
optimize (PaletteTableList *ptl,
          const guint       table[32][32][32])
{
  gint i, j;

  while (ptl->blue_min < ptl->blue_max)
    {
      for (i = ptl->red_min; i <= ptl->red_max; i++)
        for (j = ptl->green_min; j <= ptl->green_max; j++)
          if (table[i][j][ptl->blue_min] > 0)
            goto loop_blue0;
      ptl->blue_min++;
    }
  loop_blue0:
  while (ptl->blue_min < ptl->blue_max)
    {
      for (i = ptl->red_min; i <= ptl->red_max; i++)
        for (j = ptl->green_min; j <= ptl->green_max; j++)
          if (table[i][j][ptl->blue_max] > 0)
            goto loop_blue1;
      ptl->blue_max--;
    }
  loop_blue1:
  while (ptl->green_min < ptl->green_max)
    {
      for (i = ptl->red_min; i <= ptl->red_max; i++)
        for (j = ptl->blue_min; j <= ptl->blue_max; j++)
          if (table[i][ptl->green_min][j] > 0)
            goto loop_green0;
      ptl->green_min++;
    }
  loop_green0:
  while (ptl->green_min < ptl->green_max)
    {
      for (i = ptl->red_min; i <= ptl->red_max; i++)
        for (j = ptl->blue_min; j <= ptl->blue_max; j++)
          if (table[i][ptl->green_max][j] > 0)
            goto loop_green1;
      ptl->green_max--;
    }
  loop_green1:
  while (ptl->red_min < ptl->red_max)
    {
      for (i = ptl->green_min; i <= ptl->green_max; i++)
        for (j = ptl->blue_min; j <= ptl->blue_max; j++)
          if (table[ptl->red_min][i][j] > 0)
            goto loop_red0;
      ptl->red_min++;
    }
  loop_red0:
  while (ptl->red_min < ptl->red_max)
    {
      for (i = ptl->green_min; i <= ptl->green_max; i++)
        for (j = ptl->blue_min; j <= ptl->blue_max; j++)
          if (table[ptl->red_max][i][j] > 0)
            return;
      ptl->red_max--;
    }
}


/*  ja:パレットの構造体の範囲を密度が等しくなるように分割する
       p0,分割前のパレットの構造体
       p1,分割後のパレットの構造体(暗い)
       p2,分割後のパレットの構造体(明るい)
    table,パレットの分布                                                    */
static void
divide_palette_count (PaletteTableList *p0,
                      PaletteTableList *p1,
                      PaletteTableList *p2,
                      const guint       table[32][32][32])
{
  gint i, j;

  *p2 = *p1 = *p0;
  p1->count = p2->count = 0;
  if (p0->green_max - p0->green_min >= p0->blue_max - p0->blue_min
                && p0->green_max - p0->green_min >= p0->red_max - p0->red_min)
    {
      p1->green_max = p1->green_min - 1;
      p2->green_min = p2->green_max + 1;
      do
        {
          if (p1->count < p2->count)
            {
              p1->green_max++;
              for (i = p1->red_min; i <= p1->red_max; i++)
                for (j = p1->blue_min; j <= p1->blue_max; j++)
                  p1->count += table[i][p1->green_max][j];
            }
          else
            {
              p2->green_min--;
              for (i = p2->red_min; i <= p2->red_max; i++)
                for (j = p2->blue_min; j <= p2->blue_max; j++)
                  p2->count += table[i][p2->green_min][j];
            }
        }
      while (p1->green_max + 1 < p2->green_min);
    }
  else if (p0->red_max - p0->red_min >= p0->blue_max - p0->blue_min)
    {
      p1->red_max = p1->red_min - 1;
      p2->red_min = p2->red_max + 1;
      do
        {
          if (p1->count < p2->count)
            {
              p1->red_max++;
              for (i = p1->green_min; i <= p1->green_max; i++)
                for (j = p1->blue_min; j <= p1->blue_max; j++)
                  p1->count += table[p1->red_max][i][j];
            }
          else
            {
              p2->red_min--;
              for (i = p2->green_min; i <= p2->green_max; i++)
                for (j = p2->blue_min; j <= p2->blue_max; j++)
                  p2->count += table[p2->red_min][i][j];
            }
        }
      while (p1->red_max + 1 < p2->red_min);
    }
  else
    {
      p1->blue_max = p1->blue_min - 1;
      p2->blue_min = p2->blue_max + 1;
      do
        {
          if (p1->count < p2->count)
            {
              p1->blue_max++;
              for (i = p1->red_min; i <= p1->red_max; i++)
                for (j = p1->green_min; j <= p1->green_max; j++)
                  p1->count += table[i][j][p1->blue_max];
            }
          else
            {
              p2->blue_min--;
              for (i = p2->red_min; i <= p2->red_max; i++)
                for (j = p2->green_min; j <= p2->green_max; j++)
                  p2->count+=table[i][j][p2->blue_min];
            }
        }
      while (p1->blue_max + 1 < p2->blue_min);
    }
  optimize (p1, table);
  optimize (p2, table);
}


/*  ja:パレットの構造体の範囲を体積が等しくなるように分割する
       p0,分割前のパレットの構造体
       p1,分割後のパレットの構造体(暗い)
       p2,分割後のパレットの構造体(明るい)
    table,パレットの分布                                                    */
static void
divide_palette_area (PaletteTableList *p0,
                     PaletteTableList *p1,
                     PaletteTableList *p2,
                     const guint       table[32][32][32])
{
  gint i, j, k, medium;

  *p2 = *p1 = *p0;
  if (p0->green_max - p0->green_min >= p0->blue_max - p0->blue_min
                && p0->green_max - p0->green_min >= p0->red_max - p0->red_min)
    {
      medium = p0->green_max + p0->green_min;
      p1->green_max = medium / 2 + ((medium % 2 == 0 && medium >= 31) ? 1 : 0);
      p2->green_min = p1->green_max + 1;
    }
  else if (p0->red_max - p0->red_min >= p0->blue_max - p0->blue_min)
    {
      medium = p0->red_max + p0->red_min;
      p1->red_max = medium / 2 + ((medium % 2 == 0 && medium >= 31) ? 1 : 0);
      p2->red_min = p1->red_max + 1;
    }
  else
    {
      medium = p0->blue_max + p0->blue_min;
      p1->blue_max = medium / 2 + ((medium % 2 == 0 && medium >= 31) ? 1 : 0);
      p2->blue_min = p1->blue_max + 1;
    }
  p1->count = p2->count = 0;
  for (i = p1->red_min; i <= p1->red_max; i++)
    for (j = p1->green_min; j <= p1->green_max; j++)
      for (k = p1->blue_min; k <= p1->blue_max; k++)
        p1->count += table[i][j][k];
  for (i = p2->red_min; i <= p2->red_max; i++)
    for (j = p2->green_min; j <= p2->green_max; j++)
      for (k = p2->blue_min; k <= p2->blue_max; k++)
        p2->count += table[i][j][k];
  optimize (p1, table);
  optimize (p2, table);
}


/*  ja:任意のパレットを求める
    count,求めるパレットの総数
      rgb,パレットを格納するバッファ
    table,パレット分布立方体                                                */
void
bitmap_make_palette (const gint   count,
                     RgbQuad     *rgb,
                     const guint  table[32][32][32])
{
  gint i, j, k;
  guint max;
  PaletteTableList p, ptl[256];

  g_memset (ptl, 0, count * sizeof (PaletteTableList));
  ptl[0].blue_max = ptl[0].green_max = ptl[0].red_max = 31;
  for (i = 0; i < 32; i++)
    for (j = 0; j < 32; j++)
      for (k = 0; k < 32; k++)
        ptl[0].count += table[i][j][k];
  optimize (&ptl[0], table);

  for (i = 1; i < count * 3 / 4; i++)
    {
      /* ja:最初の3/4は密度の多いところを探す */
      max = 0;
      for (j = 0; j < i; j++)
        if (ptl[j].count > max && (ptl[j].blue_min < ptl[j].blue_max
                                        || ptl[j].green_min < ptl[j].green_max
                                        || ptl[j].red_min < ptl[j].red_max))
          {
            max = ptl[j].count;
            k = j;
          }
      if (max == 0)
        break;
      p = ptl[k];
      g_memmove (&ptl[k], &ptl[k + 1], (i - k) * sizeof (PaletteTableList));
      divide_palette_count (&p, &ptl[i - 1], &ptl[i], table);
    }
  while (i < count)
    {
      /* ja:最後の1/4は辺の長いところを探す */
      max = 0;
      for (j = 0; j < i; j++)
        {
          if (ptl[j].count == 0)
            break;
          if (ptl[j].blue_max - ptl[j].blue_min > max)
            {
              max = ptl[j].blue_max - ptl[j].blue_min;
              k = j;
            }
          if (ptl[j].green_max - ptl[j].green_min > max)
            {
              max = ptl[j].green_max - ptl[j].green_min;
              k = j;
            }
          if (ptl[j].red_max - ptl[j].red_min > max)
            {
              max = ptl[j].red_max - ptl[j].red_min;
              k = j;
            }
        }
      if (max == 0)
        break;
      p = ptl[k];
      g_memmove (&ptl[k], &ptl[k + 1], (i - k) * sizeof (PaletteTableList));
      divide_palette_area (&p, &ptl[i - 1], &ptl[i], table);
      i++;
    }

  for (i = 0; i < count; i++)
    {
      j = (ptl[i].blue_min + ptl[i].blue_max) / 2 * 8;
      if (j & 8)
        j |= 7;
      rgbquad_set (rgb, i, RGB_BLUE, j);
      j = (ptl[i].green_min + ptl[i].green_max) / 2 * 8;
      if (j&8)
        j |= 7;
      rgbquad_set (rgb, i, RGB_GREEN, j);
      j = (ptl[i].red_min + ptl[i].red_max) / 2 * 8;
      if (j & 8)
        j |= 7;
      rgbquad_set (rgb, i, RGB_RED, j);
      rgbquad_set (rgb, i, RGB_RESERVED, j);
    }
}


/*  ja:圧縮されたビットマップを復元する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_expand (const BitmapInfoHeader *bmih0,
               BitmapInfoHeader       *bmih1)
{
  gsize length;
  gint c, i, n, x = 0, y = 0;
  guint8 *p, *q;

  length = bm_header_bytes (bmih0);
  p = (guint8 *)bmih0 + length;
  q = (guint8 *)bmih1 + length;
  g_memmove (bmih1, bmih0, length);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length = bm_line_bytes (bmih0);
  if (bmih_get_compression (bmih0) == BI_COMP_RLE8)
    while (*p != 0 || *(p + 1) != 1)
      if (*p == 0)
        {
          p++;
          switch (*p)
            {
              case 0:
                x=0;
                y++;
                p++;
                break;
              case 2:
                p++;
                x += *p;
                p++;
                y += *p;
                p++;
                break;
              default:
                n = *p;
                p++;
                for (i = 0; i < n; i++)
                  {
                    if (x >= length)
                      return;
                    q[x + y * length] = *p;
                    p++;
                    x++;
                  }
                p+=n%2;
            }
        }
      else
        {
          n = *p;
          p++;
          for (i = 0; i < n; i++)
            {
              if (x >= length)
                return;
              q[x + y * length] = *p;
              x++;
            }
           p++;
        }
  else
    while (*p != 0 || *(p + 1) != 1)
      if (*p == 0)
        {
          p++;
          switch (*p)
            {
              case 0:
                x = 0;
                y++;
                p++;
                break;
              case 2:
                p++;
                x += *p;
                p++;
                y += *p;
                p++;
                break;
              default:
                n = *p;
                p++;
                for (i = 0; i < n; i++)
                  {
                    if (i % 2 == 0)
                      {
                        c = (guint8)(*p >> 4);
                      }
                    else
                      {
                        c = (guint8)(*p & 15);
                        p++;
                      }
                    if (x / 2 >= length)
                      return;
                    if (x % 2 == 0)
                      q[x / 2 + y * length] = c << 4;
                    else
                      q[x / 2 + y * length] |= c;
                    x++;
                  }
                p += (n % 4 == 1) + (n % 4 != 0);
            }
        }
      else
        {
          n = *p;
          p++;
          for (i = 0; i < n; i++)
            {
              if (x / 2 >= length)
                return;
              if (x % 2 == 0)
                q[x / 2 + y * length] = i % 2 == 0 ? *p & 0xf0 : *p << 4;
              else
                q[x / 2 + y * length] |= i % 2 == 0 ? *p >> 4 : *p & 15;
              x++;
            }
          p++;
        }
}


/*  ja:フルカラー/32ビットビットマップを1ビットビットマップに変換する(近似)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_approx_1 (const BitmapInfoHeader *bmih0,
                 BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j, k;
  gint blue, green, red;
  gint distance, dist_red, dist_blue, dist_green;
  gint index, min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 1);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes];
          green = p[j * pixel_bytes + 1];
          red = p[j * pixel_bytes + 2];
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red - rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS (dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          q[j / 8]
                = (q[j / 8] & ~(1 << ( 7 - j % 8))) | (index << ( 7 - j % 8));
        }
      p += length0;
      q += length1;
    }
}


/*  ja:フルカラー/32ビットビットマップを4ビットビットマップに変換する(近似)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_approx_4 (const BitmapInfoHeader *bmih0,
                 BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j, k;
  gint blue, green, red;
  gint distance, dist_blue, dist_green, dist_red;
  gint index, min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 4);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes];
          green = p[j * pixel_bytes + 1];
          red = p[j * pixel_bytes + 2];
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red - rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS (dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          q[j / 2] = (j % 2 == 0) ? index << 4 : (q[j / 2] & 0xf0) | index;
        }
      p += length0;
      q += length1;
    }
}


/*  ja:フルカラー/32ビットビットマップを8ビットビットマップに変換する(近似)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_approx_8 (const BitmapInfoHeader *bmih0,
                 BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j, k;
  gint blue, green, red;
  gint distance, dist_blue, dist_green, dist_red;
  gint index, min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 8);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes];
          green = p[j * pixel_bytes + 1];
          red = p[j * pixel_bytes + 2];
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red - rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS(dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          q[j] = index;
        }
      p += length0;
      q += length1;
    }
}


/*  ja:フルカラー/32ビットビットマップを1ビットビットマップに変換する(誤差拡散)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_diffuse_1 (const BitmapInfoHeader *bmih0,
                  BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gshort *buffer;
  gint i, j, k;
  gint blue, green, red;
  gint diagonal_blue, diagonal_green, diagonal_red;
  gint distance, dist_blue, dist_green, dist_red;
  gint next_blue, next_green, next_red;
  gint index,min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 1);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  buffer = g_malloc0 (bmih_get_width (bmih0) * sizeof (gshort) * 3);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      next_blue = next_green = next_red
                        = diagonal_blue = diagonal_green = diagonal_red = 0;
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes] + buffer[j * 3] + next_blue;
          green = p[j * pixel_bytes + 1] + buffer[j * 3 + 1] + next_green;
          red = p[j * pixel_bytes + 2] + buffer[j * 3 +2 ] + next_red;
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red-rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS (dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          dist_blue = p[j * pixel_bytes] - rgb_get(bmih1, index, RGB_BLUE);
          dist_green = p[j * pixel_bytes + 1]
                                        - rgb_get (bmih1, index, RGB_GREEN);
          dist_red = p[j * pixel_bytes + 2] - rgb_get (bmih1, index, RGB_RED);
          next_blue = dist_blue * 3 / 8;
          next_green = dist_green * 3 / 8;
          next_red = dist_red * 3 / 8;
          buffer[j * 3] = next_blue + diagonal_blue;
          buffer[j * 3 + 1] = next_green + diagonal_green;
          buffer[j * 3 + 2] = next_red + diagonal_red;
          diagonal_blue = dist_blue / 4;
          diagonal_green = dist_green / 4;
          diagonal_red = dist_red / 4;
          q[j / 8] = (q[j / 8] & ~(1 << (7 - j % 8))) | (index << (7 - j % 8));
        }
      p += length0;
      q += length1;
    }
  g_free (buffer);
}


/*  ja:フルカラー/32ビットビットマップを4ビットビットマップに変換する(誤差拡散)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_diffuse_4 (const BitmapInfoHeader *bmih0,
                  BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gshort *buffer;
  gint i, j, k;
  gint blue, green, red;
  gint diagonal_blue, diagonal_green, diagonal_red;
  gint distance, dist_blue, dist_green, dist_red;
  gint next_blue, next_green, next_red;
  gint index, min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 4);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  buffer = g_malloc0 (bmih_get_width (bmih0) * sizeof (gshort) * 3);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      next_blue = next_green = next_red
                        = diagonal_blue = diagonal_green = diagonal_red = 0;
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes] + buffer[j * 3] + next_blue;
          green = p[j * pixel_bytes + 1] + buffer[j * 3 + 1] + next_green;
          red = p[j * pixel_bytes + 2] + buffer[j * 3 + 2] + next_red;
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red - rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS (dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          dist_blue = p[j * pixel_bytes] - rgb_get (bmih1, index, RGB_BLUE);
          dist_green = p[j * pixel_bytes + 1]
                                        - rgb_get (bmih1, index, RGB_GREEN);
          dist_red = p[j * pixel_bytes + 2] - rgb_get (bmih1, index, RGB_RED);
          next_blue = dist_blue * 3 / 8;
          next_green = dist_green * 3 / 8;
          next_red = dist_red * 3 / 8;
          buffer[j * 3] = next_blue + diagonal_blue;
          buffer[j * 3 + 1] = next_green + diagonal_green;
          buffer[j * 3 + 2] = next_red + diagonal_red;
          diagonal_blue = dist_blue / 4;
          diagonal_green = dist_green / 4;
          diagonal_red = dist_red / 4;
          q[j / 2] = (j % 2 == 0) ? index << 4 : (q[j / 2] & 0xf0) | index;
        }
      p += length0;
      q += length1;
    }
  g_free (buffer);
}


/*  ja:フルカラー/32ビットビットマップを8ビットビットマップに変換する(誤差拡散)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_diffuse_8 (const BitmapInfoHeader *bmih0,
                  BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gshort *buffer;
  gint i, j, k;
  gint blue, green, red;
  gint diagonal_blue, diagonal_green, diagonal_red;
  gint distance, dist_blue, dist_green, dist_red;
  gint next_blue, next_green, next_red;
  gint index, min, color_used;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 8);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  color_used = bm_color_used (bmih1);
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  buffer = g_malloc0 (bmih_get_width (bmih0) * sizeof (gshort) * 3);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      next_blue = next_green = next_red
                        = diagonal_blue = diagonal_green = diagonal_red = 0;
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          min = G_MAXINT;
          index = 0;
          blue = p[j * pixel_bytes] + buffer[j * 3] + next_blue;
          green = p[j * pixel_bytes + 1] + buffer[j * 3 + 1] + next_green;
          red = p[j * pixel_bytes + 2] + buffer[j * 3 + 2] + next_red;
          for (k = 0; k < color_used; k++)
            {
              dist_blue = blue - rgb_get (bmih1, k, RGB_BLUE);
              dist_green = green - rgb_get (bmih1, k, RGB_GREEN);
              dist_red = red - rgb_get (bmih1, k, RGB_RED);
              distance = dist_blue * dist_blue + dist_green * dist_green
                                                        + dist_red * dist_red;
              if (distance < min || (distance == min
                    && (ABS (dist_green)
                            < ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                    || (ABS (dist_green)
                            == ABS (green - rgb_get (bmih1, index, RGB_GREEN))
                        && ABS (dist_red)
                            < ABS (red - rgb_get (bmih1, index, RGB_RED))))))
                {
                  min = distance;
                  index = k;
                  if (min <= 0)
                    break;
                }
            }
          dist_blue = p[j * pixel_bytes] - rgb_get (bmih1, index, RGB_BLUE);
          dist_green = p[j * pixel_bytes + 1]
                                        - rgb_get (bmih1, index, RGB_GREEN);
          dist_red = p[j * pixel_bytes + 2] - rgb_get (bmih1, index, RGB_RED);
          next_blue = dist_blue * 3 / 8;
          next_green = dist_green * 3 / 8;
          next_red = dist_red * 3 / 8;
          buffer[j * 3] = next_blue + diagonal_blue;
          buffer[j * 3 + 1] = next_green + diagonal_green;
          buffer[j * 3 + 2] = next_red + diagonal_red;
          diagonal_blue = dist_blue / 4;
          diagonal_green = dist_green / 4;
          diagonal_red = dist_red / 4;
          q[j] = index;
        }
      p += length0;
      q += length1;
    }
  g_free (buffer);
}


/*  ja:フルカラー/32ビットビットマップを1ビットビットマップに変換する(識域)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_order_1 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j;
  guint8 *p, *q;
  const gint bayer[2][2] = {{51, 153}, {204, 102}};

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 1);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        q[j / 8] = (q[j / 8] & ~(1 << (7 - j % 8)))
                                                | ((bayer[i % 2][j % 2] * 1000
                        < p[j * pixel_bytes + 2] * 299
                        + p[j * pixel_bytes + 1] * 587
                        + p[j * pixel_bytes] * 114 ? 1 : 0) << (7 - j % 8));
      p += length0;
      q += length1;
    }
}


/*  ja:フルカラー/32ビットビットマップを4ビットビットマップに変換する(識域)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_order_4 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j, index, order;
  guint8 *p, *q;
  const gint bayer[2][2] = {{51, 153}, {204, 102}};

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 4);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  if (bm_color_used (bmih1) < 8)
    for (i = 0; i < bmih_get_height (bmih0); i++)
      {
        for (j = 0; j < bmih_get_width (bmih0); j++)
          {
            index = bayer[i % 2][j % 2] * 1000 < p[j * pixel_bytes + 2] * 299
                                            + p[j * pixel_bytes + 1] * 587
                                            + p[j * pixel_bytes] * 114 ? 1 : 0;
            q[j / 2] = (j % 2 == 0) ? index << 4 : (q[j / 2] & 0xf0) | index;
          }
        p += length0;
        q += length1;
      }
  else
    for (i = 0; i < bmih_get_height (bmih0); i++)
      {
        for (j = 0; j < bmih_get_width (bmih0); j++)
          {
            index = 0;
            order = bayer[i % 2][j % 2];
            if (order < p[j * pixel_bytes])
              index |= 4;
            if (order < p[j * pixel_bytes + 1])
              index |= 2;
            if (order < p[j * pixel_bytes + 2])
              index |= 1;
            q[j / 2] = (j % 2 == 0) ? index << 4 : (q[j / 2] & 0xf0) | index;
          }
        p += length0;
        q += length1;
      }
}


/*  ja:フルカラー/32ビットビットマップを8ビットビットマップに変換する(識域)
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_order_8 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j, order;
  guint8 *p, *q;
  const gint bayer[2][2] = {{51, 153}, {204, 102}};

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 8);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  if (bm_color_used (bmih1) < 8)
    for (i = 0; i < bmih_get_height (bmih0); i++)
      {
        for (j = 0; j <bmih_get_width (bmih0); j++)
          q[j] = bayer[i % 2][j % 2] * 1000 < p[j * pixel_bytes + 2] * 299
                                            + p[j * pixel_bytes + 1] * 587
                                            + p[j * pixel_bytes] * 114 ? 1 : 0;
        p += length0;
        q += length1;
      }
  else
    for (i = 0; i < bmih_get_height (bmih0); i++)
      {
        for (j = 0; j < bmih_get_width (bmih0); j++)
          {
            q[j] = 0;
            order = bayer[i % 2][j % 2];
            if (order < p[j * pixel_bytes])
              q[j] |= 4;
            if (order < p[j * pixel_bytes + 1])
              q[j] |= 2;
            if (order < p[j * pixel_bytes + 2])
              q[j] |= 1;
          }
        p += length0;
        q += length1;
      }
}


/*  ja:フルカラー/32ビットビットマップを16ビットビットマップに変換する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_convert_16 (const BitmapInfoHeader *bmih0,
                   BitmapInfoHeader       *bmih1)
{
  gsize length0, length1, pixel_bytes;
  gint i, j;
  guint8 *p;
  guint16 *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 16);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, 0);
  bmih_set_color_important (bmih1, 0);
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  pixel_bytes = bmih_get_bit_count (bmih0) < 32 ? 3 : 4;
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        q[j] = GUINT16_TO_LE (((p[j * pixel_bytes] >> 3) & 0x001f)
                                | ((p[j * pixel_bytes + 1] << 2) & 0x03e0)
                                | ((p[j * pixel_bytes + 2] << 7) & 0x7c00));
      p += length0;
      q = (guint16 *)((guint8 *)q + length1);
    }
}


/*  ja:ビットマップをフルカラービットマップに変換する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_convert_24 (const BitmapInfoHeader *bmih0,
                   BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint i, j, k, bit, index;
  guint32 mask[3], shift;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 24);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, 0);
  bmih_set_color_important (bmih1, 0);
  bmih_set_size_image (bmih1,bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  if (bmih_get_bit_count (bmih0) == 16 || bmih_get_bit_count (bmih0) == 32)
    {
      if (bmih_get_compression (bmih0) == BI_COMP_RGB)
        if (bmih_get_bit_count (bmih0) == 16)
          {
            mask[0] = 0x001f;
            mask[1] = 0x03e0;
            mask[2] = 0x7c00;
          }
        else
          {
            mask[0] = 0x000000ff;
            mask[1] = 0x0000ff00;
            mask[2] = 0x00ff0000;
          }
      else
        for (i = 0; i < 3; i++)
          mask[i] = ((guint32 *)((guint8 *)bmih0 + bmih_get_size (bmih0)))[i];
    }
  switch (bmih_get_bit_count (bmih0))
    {
      case 1:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = (p[j / 8] >> (7 - j % 8)) & 1;
                q[j * 3] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 3 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 3 + 2] = rgb_get (bmih0, index, RGB_RED);
              }
            p += length0;
            q += length1;
          }
        break;
      case 4:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = (j % 2 == 0 ? p[j / 2] >> 4 : p[j / 2]) & 15;
                q[j * 3] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 3 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 3+ 2 ] = rgb_get (bmih0, index, RGB_RED);
              }
              p += length0;
              q += length1;
          }
        break;
      case 8:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = p[j];
                q[j * 3] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 3 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 3 + 2] = rgb_get (bmih0, index, RGB_RED);
              }
              p += length0;
              q += length1;
          }
          break;
      case 16:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              for (k = 0; k < 3; k++)
                {
                  bit = 0;
                  for (shift = 0x8000; shift != 0; shift >>= 1)
                    {
                      if (mask[k] & shift)
                        {
                          q[j * 3 + k] <<= 1;
                          if (GUINT16_FROM_LE (((guint16 *)p)[j]) & shift)
                            q[j * 3 + k] |= 1;
                          if (++bit>=8)
                            break;
                        }
                    }
                  if (bit < 8)
                    {
                      q[j * 3 + k] <<= 8 - bit;
                      q[j * 3 + k] |= (q[j * 3 + k] - 1)
                                                & ~(0xffffffff << (8 - bit));
                    }
                }
              p += length0;
              q += length1;
          }
          break;
      case 24:
        g_memmove (q, p, bm_image_bytes (bmih0));
        break;
      case 32:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              for (k = 0; k < 3; k++)
                {
                  bit = 0;
                  for (shift = 0x80000000; shift != 0; shift >>= 1)
                    {
                      if (mask[k] & shift)
                        {
                          q[j * 3 + k] <<= 1;
                          if (GUINT32_FROM_LE (((guint32 *)p)[j]) & shift)
                            q[j * 3 + k] |= 1;
                          if (++bit >= 8)
                            break;
                        }
                    }
                  if (bit < 8)
                    {
                      q[j * 3 + k] <<= 8 - bit;
                      q[j * 3 + k] |= (q[j * 3 + k] - 1)
                                                & ~(0xffffffff << (8 - bit));
                    }
                }
              p += length0;
              q += length1;
          }
    }
}


/*  ja:32ビットビットマップをフルカラービットマップに変換する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ
      rgb,背景色                                                            */
void
bitmap_convert_24_with_bgcolor (const BitmapInfoHeader *bmih0,
                                BitmapInfoHeader       *bmih1,
                                const RgbQuad          *rgb)
{
  gsize length;
  gint i, j, alpha, bgcolor;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 24);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, 0);
  bmih_set_color_important (bmih1, 0);
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  for (i = 0; i < bmih_get_height (bmih0); i++)
    {
      for (j = 0; j < bmih_get_width (bmih0); j++)
        {
          alpha = p[j * 4 + 3];
          bgcolor = 255 - alpha;
          q[j * 3] = (p[j * 4] * alpha
                            + rgbquad_get (rgb, 0, RGB_BLUE) * bgcolor) / 255;
          q[j * 3 + 1] = (p[j * 4 + 1] * alpha
                            + rgbquad_get (rgb, 0, RGB_GREEN) * bgcolor) / 255;
          q[j * 3 + 2] = (p[j * 4 + 2] * alpha
                            + rgbquad_get (rgb, 0, RGB_RED) * bgcolor) / 255;
        }
      p += bmih_get_width (bmih0) * 4;
      q += length;
    }
}


/*  ja:ビットマップを32ビットビットマップに変換する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_convert_32 (const BitmapInfoHeader *bmih0,
                   BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint i, j, k, bit, index;
  guint32 mask[3], shift;
  guint8 *p, *q;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_width (bmih1, bmih_get_width (bmih0));
  bmih_set_height (bmih1, bmih_get_height (bmih0));
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 32);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, 0);
  bmih_set_color_important (bmih1, 0);
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  p = bm_image_pointer (bmih0);
  q = bm_image_pointer (bmih1);
  if (bmih_get_bit_count (bmih0) == 16 || bmih_get_bit_count (bmih0) == 32)
    {
      if (bmih_get_compression (bmih0) == BI_COMP_RGB)
        if (bmih_get_bit_count (bmih0) == 16)
          {
            mask[0] = 0x001f;
            mask[1] = 0x03e0;
            mask[2] = 0x7c00;
          }
        else
          {
            mask[0] = 0x000000ff;
            mask[1] = 0x0000ff00;
            mask[2] = 0x00ff0000;
          }
      else
        for (i = 0; i < 3; i++)
          mask[i] = ((guint32 *)((guint8 *)bmih0 + bmih_get_size (bmih0)))[i];
    }
  switch (bmih_get_bit_count (bmih0))
    {
      case 1:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = (p[j / 8] >> (7 - j % 8)) & 1;
                q[j * 4] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 4 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 4 + 2] = rgb_get (bmih0, index, RGB_RED);
                q[j * 4 + 3] = 0xff;
              }
            p += length0;
            q += length1;
          }
        break;
      case 4:
        for (i = 0; i <bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = (j % 2 == 0 ? p[j / 2] >> 4 : p[j / 2]) & 15;
                q[j * 4] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 4 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 4 + 2] = rgb_get (bmih0, index, RGB_RED);
                q[j * 4 + 3] = 0xff;
              }
            p += length0;
            q += length1;
          }
          break;
      case 8:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                index = p[j];
                q[j * 4] = rgb_get (bmih0, index, RGB_BLUE);
                q[j * 4 + 1] = rgb_get (bmih0, index, RGB_GREEN);
                q[j * 4 + 2] = rgb_get (bmih0, index, RGB_RED);
                q[j * 4 + 3] = 0xff;
              }
            p += length0;
            q += length1;
          }
        break;
      case 16:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                for (k = 0; k < 3; k++)
                  {
                    bit = 0;
                    for (shift = 0x8000; shift != 0; shift >>= 1)
                      {
                        if (mask[k] & shift)
                          {
                            q[j * 4 + k] <<= 1;
                            if (GUINT16_FROM_LE (((guint16 *)p)[j]) & shift)
                              q[j * 4 + k] |= 1;
                            if (++bit >= 8)
                              break;
                          }
                      }
                    if (bit < 8)
                      {
                        q[j * 4 + k] <<= 8 - bit;
                        q[j * 4 + k] |= (q[j * 4 + k] - 1)
                                                & ~(0xffffffff << (8 - bit));
                      }
                  }
                q[j * 4 + 3] = 0xff;
              }
            p += length0;
            q += length1;
          }
        break;
      case 24:
        for (i = 0; i < bmih_get_height (bmih0); i++)
          {
            for (j = 0; j < bmih_get_width (bmih0); j++)
              {
                q[j * 4] = p[j * 3];
                q[j * 4 + 1] = p[j * 3 + 1];
                q[j * 4 + 2] = p[j * 3 + 2];
                q[j * 4 + 3] = 0xff;
              }
             p += length0;
             q += length1;
          }
        break;
      case 32:
        if (bmih_get_compression (bmih0) == BI_COMP_RGB)
          g_memmove (q, p, bm_image_bytes (bmih0));
        else
          for (i = 0;i < bmih_get_height (bmih0); i++)
            {
              for (j = 0; j < bmih_get_width (bmih0); j++)
                {
                  for (k = 0; k < 3; k++)
                    {
                      bit = 0;
                      for (shift = 0x80000000; shift != 0; shift >>= 1)
                        {
                          if (mask[k] & shift)
                            {
                              q[j * 4 + k] <<= 1;
                              if (GUINT32_FROM_LE (((guint32 *)p)[j]) & shift)
                                q[j * 4 + k] |= 1;
                              if (++bit >= 8)
                                break;
                            }
                        }
                      if (bit < 8)
                        {
                          q[j * 4 + k] <<= 8 - bit;
                          q[j * 4 + k] |= (q[j * 4 + k] - 1)
                                                & ~(0xffffffff << (8 - bit));
                        }
                    }
                  q[j * 4 + 3] = 0xff;
                }
              p += length0;
              q += length1;
            }
    }
}


/*  ja:1ビットビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_1 (const BitmapInfoHeader *bmih0,
               BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint x0, y0, x1, y1;
  guint8 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 1);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  g_memset (dst, 0, bmih_get_height (bmih1) * length1);
  for (y1 = 0; y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          if (src[x0 / 8 + y0 * length0] & (0x80 >> (x0 % 8)))
            dst[x1 / 8 + y1 * length1] |= 0x80 >> (x1 % 8);
        }
    }
}


/*  ja:4ビットビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_4 (const BitmapInfoHeader *bmih0,
               BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint x0, y0, x1, y1;
  guint8 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 4);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  g_memset (dst, 0, bmih_get_height (bmih1) * length1);
  for (y1 = 0; y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          dst[x1 / 2 + y1 * length1] |= ((src[x0 / 2 + y0 * length0]
                    >> (x0 % 2 == 0 ? 4 : 0)) & 0x0f) << (x1 % 2 == 0 ? 4 : 0);
        }
    }
}


/*  ja:8ビットビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_8 (const BitmapInfoHeader *bmih0,
               BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint x0, y0, x1, y1;
  guint8 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 8);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  for (y1 = 0; y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          dst[x1 + y1 * length1] = src[x0 + y0 * length0];
        }
    }
}


/*  ja:16ビットビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_16 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint x0, y0, x1, y1;
  guint16 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 16);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0) / sizeof (guint16);
  length1 = bm_line_bytes(bmih1) / sizeof (guint16);
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  for (y1 = 0;y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          dst[x1 + y1 * length1] = src[x0 + y0 * length0];
        }
    }
}


/*  ja:フルカラービットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_24 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  gsize length0, length1;
  gint x0, y0, x1, y1;
  guint8 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 24);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  length0 = bm_line_bytes (bmih0);
  length1 = bm_line_bytes (bmih1);
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  for (y1 = 0; y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          dst[x1 * 3 + y1 * length1] = src[x0 * 3 + y0 * length0];
          dst[x1 * 3 + y1 * length1 + 1] = src[x0 * 3 + y0 * length0 + 1];
          dst[x1 * 3 + y1 * length1 + 2] = src[x0 * 3 + y0 * length0 + 2];
        }
    }
}


/*  ja:32ビットビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom_32 (const BitmapInfoHeader *bmih0,
                BitmapInfoHeader       *bmih1)
{
  guint x0, y0, x1, y1;
  guint32 *src, *dst;

  bmih_set_size (bmih1, BMIH_SIZE);
  bmih_set_planes (bmih1, 1);
  bmih_set_bit_count (bmih1, 32);
  bmih_set_compression (bmih1, BI_COMP_RGB);
  bmih_set_x_pixels_per_meter (bmih1, bmih_get_x_pixels_per_meter (bmih0));
  bmih_set_y_pixels_per_meter (bmih1, bmih_get_y_pixels_per_meter (bmih0));
  bmih_set_color_used (bmih1, bmih_get_color_used (bmih0));
  bmih_set_color_important (bmih1, bmih_get_color_important (bmih0));
  bmih_set_size_image (bmih1, bm_image_bytes (bmih1));
  src = bm_image_pointer (bmih0);
  dst = bm_image_pointer (bmih1);
  for (y1 = 0; y1 < bmih_get_height (bmih1); y1++)
    {
      y0 = y1 * bmih_get_height (bmih0) / bmih_get_height (bmih1);
      for (x1 = 0; x1 < bmih_get_width (bmih1); x1++)
        {
          x0 = x1 * bmih_get_width (bmih0) / bmih_get_width (bmih1);
          dst[x1 + y1 * bmih_get_width (bmih1)]
                                    = src[x0 + y0 * bmih_get_width (bmih0)];
        }
    }
}


/*  ja:ビットマップのサイズを変更する
    bmih0,元のビットマップへのポインタ
    bmih1,新しいビットマップへのポインタ                                    */
void
bitmap_zoom (const BitmapInfoHeader *bmih0,
             BitmapInfoHeader       *bmih1)
{
  switch (bmih_get_bit_count (bmih0))
    {
      case  1: bitmap_zoom_1 (bmih0, bmih1); break;
      case  4: bitmap_zoom_4 (bmih0, bmih1); break;
      case  8: bitmap_zoom_8 (bmih0, bmih1); break;
      case 16: bitmap_zoom_16 (bmih0, bmih1); break;
      case 24: bitmap_zoom_24 (bmih0, bmih1); break;
      default: bitmap_zoom_32 (bmih0, bmih1);
    }
}


/*  ja:32ビットビットマップを重ね合わせる
        bmih0,下になるビットマップへのポインタ
        bmih1,上になるビットマップへのポインタ
    x_overlay,X座標
    y_overlay,X座標
        trans,透過色,TRUE:有効,FALSE:無効                                   */
void
bitmap_overlay_32 (const BitmapInfoHeader *bmih0,
                   BitmapInfoHeader       *bmih1,
                   const gint              x_overlay,
                   const gint              y_overlay,
                   const gboolean          trans)
{
  gint i, index, alpha, bgcolor;
  gint x, y, width, height;
  gint x_under, y_under, x_over, y_over, under_leng, over_leng;
  guint8 *p0, *p1;
  guint32 *q0, *q1;

  if (x_overlay < 0)
    {
      x_under = 0;
      x_over = -x_overlay;
    }
  else
    {
      x_under = x_overlay;
      x_over = 0;
    }
  if (y_overlay < 0)
    {
      y_under = 0;
      y_over = -y_overlay;
    }
  else
    {
      y_under = y_overlay;
      y_over = 0;
    }
  width = MIN (bmih_get_width (bmih0) - x_under,
                                            bmih_get_width (bmih1) - x_over);
  height = MIN (bmih_get_height (bmih0) - y_under,
                                            bmih_get_height (bmih1) - y_over);
  under_leng = bmih_get_height (bmih0) - y_under - 1;
  over_leng = bmih_get_height (bmih1) - y_over - 1;
  if (trans)
    {
      p0 = bm_image_pointer (bmih0);
      p1 = bm_image_pointer (bmih1);
      for (y = 0; y < height; y++)
        for (x = 0; x < width; x++)
          {
            index = ((under_leng - y) * bmih_get_width (bmih0) + x_under + x)
                                                                        * 4;
            alpha = p1[index + 3];
            bgcolor = 255 - alpha;
            for (i = 0; i < 3; i++)
              p0[index + i] = (p0[index + i] * bgcolor
                    +p1[((over_leng - y) * bmih_get_width (bmih1) + x_over + x)
                                                    * 4 + i] * alpha) / 255;
            p0[index + 3] = MIN (p0[index + 3]
                    +p1[((over_leng - y) * bmih_get_width (bmih1) + x_over + x)
                                                                * 4 + 3], 255);
          }
    }
  else
    {
      q0 = bm_image_pointer (bmih0);
      q1 = bm_image_pointer (bmih1);
      width *= sizeof (guint32);
      for (y = 0; y < height; y++)
        g_memmove (q0 + (under_leng - y) * bmih_get_width (bmih0) + x_under,
                q1 + (over_leng - y) * bmih_get_width (bmih1) + x_over, width);
    }
}
