/*
    Profile
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef __PROFILE_H__
#define __PROFILE_H__


#include "common.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#define PROFILE_VALUE_TYPE_BOOL 0
#define PROFILE_VALUE_TYPE_INT 1
#define PROFILE_VALUE_TYPE_STRING 2
#define PROFILE_VALUE_TYPE_ARRAY 3


typedef struct _Profile Profile;


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(独自)                                               *
*                                                                             *
******************************************************************************/
/*  ja:初期化ファイルを開く
    file,ファイル名
     RET,プロファイル                                                       */
Profile *
profile_open (const gchar *file);


/*  ja:初期化ファイルを閉じる
    profile,プロファイル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_close (Profile *profile);


/*  ja:初期化ファイルから値のサイズ取得する
    profile,プロファイル
    section,セクション
        key,キー
       type,タイプ
        RET,バイト数,0:エラー                                               */
gsize
profile_get_size (Profile     *profile,
                  const gchar *section,
                  const gchar *key,
                  const guint  type);


/*  ja:初期化ファイルから文字列を取得する
    profile,プロファイル
    section,セクション
        key,キー
        RET,文字列,NULL:エラー                                              */
gchar *
profile_get_string (Profile     *profile,
                    const gchar *section,
                    const gchar *key);


/*  ja:初期化ファイルから値を取得する
    profile,プロファイル
    section,セクション
        key,キー
      value,値を入れるバッファ
       size,値を入れるバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_get_value (Profile     *profile,
                   const gchar *section,
                   const gchar *key,
                   gpointer     value,
                   const gsize  size,
                   const guint  type);


/*  ja:初期化ファイルに値を設定する
    profile,プロファイル
    section,セクション
        key,キー
      value,値が入っているバッファ
       size,値が入っているバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_set_value (Profile       *profile,
                   const gchar   *section,
                   const gchar   *key,
                   gconstpointer  value,
                   const gsize    size,
                   const guint    type);


/*  ja:初期化ファイルのセクションを削除する
    profile,プロファイル
    section,セクション
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_section (Profile     *profile,
                        const gchar *section);


/*  ja:初期化ファイルのキーを削除する
    profile,プロファイル
    section,セクション
        key,キー
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_key (Profile     *profile,
                    const gchar *section,
                    const gchar *key);


/*  ja:初期化ファイルのセクションを列挙する
    profile,プロファイル
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_section (Profile *profile);


/*  ja:初期化ファイルのキーを列挙する
    profile,プロファイル
    section,セクション
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_key (Profile     *profile,
                  const gchar *section);


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(互換)                                               *
*                                                                             *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,6,0)
#define GKeyFile Profile
#define GKeyFileError guint
#define GKeyFileFlags guint
#define G_KEY_FILE_NONE              0
#define G_KEY_FILE_KEEP_COMMENTS     1
#define G_KEY_FILE_KEEP_TRANSLATIONS 2


GKeyFile *
g_key_file_new (void);


void
g_key_file_free (GKeyFile *key_file);


gboolean
g_key_file_load_from_file (GKeyFile       *key_file,
                           const gchar    *file,
                           GKeyFileFlags   flags,
                           GError        **error);


gboolean
g_key_file_load_from_data (GKeyFile       *key_file,
                           const gchar    *data,
                           gsize           length,
                           GKeyFileFlags   flags,
                           GError        **error);


gchar *
g_key_file_to_data (GKeyFile  *key_file,
                    gsize     *length,
                    GError   **error);


gchar **
g_key_file_get_groups (GKeyFile *key_file,
                       gsize    *length);


gchar **
g_key_file_get_keys (GKeyFile     *key_file,
                     const gchar  *group_name,
                     gsize        *length,
                     GError      **error);


gboolean
g_key_file_has_group (GKeyFile    *key_file,
                      const gchar *group_name);


gboolean
g_key_file_has_key (GKeyFile     *key_file,
                    const gchar  *group_name,
                    const gchar  *key,
                    GError      **error);


gchar *
g_key_file_get_string (GKeyFile     *key_file,
                       const gchar  *group_name,
                       const gchar  *key,
                       GError      **error);


gboolean
g_key_file_get_boolean (GKeyFile     *key_file,
                        const gchar  *group_name,
                        const gchar  *key,
                        GError      **error);


gint
g_key_file_get_integer (GKeyFile     *key_file,
                        const gchar  *group_name,
                        const gchar  *key,
                        GError      **error);


gint *
g_key_file_get_integer_list (GKeyFile     *key_file,
                             const gchar  *group_name,
                             const gchar  *key,
                             gsize        *length,
                             GError      **error);


void
g_key_file_set_string (GKeyFile    *key_file,
                       const gchar *group_name,
                       const gchar *key,
                       const gchar *string);


void
g_key_file_set_boolean (GKeyFile    *key_file,
                        const gchar *group_name,
                        const gchar *key,
                        gboolean     value);


void
g_key_file_set_integer (GKeyFile    *key_file,
                        const gchar *group_name,
                        const gchar *key,
                        gint         value);


void
g_key_file_set_integer_list (GKeyFile    *key_file,
                             const gchar *group_name,
                             const gchar *key,
                             gint         list[],
                             gsize        length);


void
g_key_file_remove_group (GKeyFile     *key_file,
                         const gchar  *group_name,
                         GError      **error);


void
g_key_file_remove_key (GKeyFile     *key_file,
                       const gchar  *group_name,
                       const gchar  *key,
                       GError      **error);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __PROFILE_H__ */
