/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef __ACM_H__
#define __ACM_H__


#include "wave.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#define ACM_ATTR_NATIVE             0
#define ACM_ATTR_WIN32              1


typedef struct _AcmObject AcmObject;


/******************************************************************************
*                                                                             *
* ja:ACM初期化                                                                *
*                                                                             *
******************************************************************************/
/*  ja:ACMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_init (void);


/*  ja:ACMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_exit (void);


/******************************************************************************
*                                                                             *
* ja:ACM基本                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:登録されているタグを取得する
    RET,タグのリスト,NULL:登録なし                                          */
GList *
acm_get_tag_list (void);


/*  ja:Codecを開く
    tag,タグ
    RET,AcmObject,NULL:エラー                                               */
AcmObject *
acm_open (const guint16 tag);


/*  ja:Codecを閉じる
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_close (AcmObject *acm_object);


/*  ja:Codecの参照数を増やす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_ref (AcmObject *acm_object);


/*  ja:Codecの参照数を減らす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_unref (AcmObject *acm_object);


/*  ja:Codecをコピーする
    acm_object,AcmObject
           RET,AcmObject,NULL:エラー                                        */
AcmObject *
acm_dup (AcmObject *acm_object);


/******************************************************************************
*                                                                             *
* ja:ACM情報                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:Codecのタグを取得する
    acm_object,AcmObject
           RET,タグ,0:エラー                                                */
guint16
acm_get_tag (AcmObject *acm_object);


/*  ja:Codecの名前を取得する
    acm_object,AcmObject
           RET,名前,NULL:エラー                                             */
const gchar *
acm_get_name (AcmObject *acm_object);


/*  ja:Codecの属性を取得する
    acm_object,AcmObject
           RET,ACM_ATTR_NATIVE,ACM_ATTR_WIN32                               */
gint
acm_get_attribute (AcmObject *acm_object);


/*  ja:WaveFormatEx構造体の名前を取得する
    acm_object,AcmObject
           wfx,WaveFormatEx構造体へのポインタ
           RET,名前,NULL:エラー                                             */
gchar *
acm_get_format_name (AcmObject          *acm_object,
                     const WaveFormatEx *wfx);


/******************************************************************************
*                                                                             *
* ja:ACMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    acm_object,AcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
acm_is_dialog_about (AcmObject *acm_object);


/*  ja:Aboutダイアログを表示する
    acm_object,AcmObject
        widget,親ウインドウ                                                 */
void
acm_dialog_about (AcmObject *acm_object,
                  GtkWidget *widget);


/******************************************************************************
*                                                                             *
* en:ACM Convert                                                              *
*                                                                             *
******************************************************************************/
/*  ja:変換可能か確認する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_convert_query (AcmObject          *acm_object,
                   const WaveFormatEx *wfx_in,
                   const WaveFormatEx *wfx_out);


/*  ja:変換後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_convert_get_format_size (AcmObject          *acm_object,
                             const WaveFormatEx *wfx);


/*  ja:変換を開始する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_convert_begin (AcmObject          *acm_object,
                   const WaveFormatEx *wfx_in,
                   const WaveFormatEx *wfx_out);


/*  ja:変換を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_convert_end (AcmObject *acm_object,
                 gpointer  *out_buf,
                 gsize     *out_samples);


/*  ja:変換する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_convert (AcmObject      *acm_object,
             const gpointer  in_buf,
             const gsize     in_samples,
             gpointer       *out_buf,
             gsize          *out_samples);


/******************************************************************************
*                                                                             *
* en:ACM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_compress_query (AcmObject          *acm_object,
                    const WaveFormatEx *wfx_in,
                    const WaveFormatEx *wfx_out);


/*  ja:圧縮後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_compress_get_format_size (AcmObject          *acm_object,
                              const WaveFormatEx *wfx);


/*  ja:圧縮後のフォーマットを取得する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
         index,インデックス(0...)
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_compress_enum_format (AcmObject          *acm_object,
                          const WaveFormatEx *wfx_in,
                          WaveFormatEx       *wfx_out,
                          const gint          index);


/*  ja:圧縮を開始する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_compress_begin (AcmObject          *acm_object,
                    const WaveFormatEx *wfx_in,
                    const WaveFormatEx *wfx_out);


/*  ja:圧縮を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_compress_end (AcmObject *acm_object,
                  gpointer  *out_buf,
                  gsize     *out_samples);


/*  ja:圧縮する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_compress (AcmObject      *acm_object,
              const gpointer  in_buf,
              const gsize     in_samples,
              gpointer       *out_buf,
              gsize          *out_samples);


/******************************************************************************
*                                                                             *
* en:ACM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_compress_get_format_size (AcmObject          *acm_object,
                              const WaveFormatEx *wfx);


/******************************************************************************
*                                                                             *
* en:ACM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
acm_decompress_query (AcmObject          *acm_object,
                      const WaveFormatEx *wfx);


/*  ja:展開後のフォーマットのサイズを取得する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,バイト数,0:エラー                                            */
gsize
acm_decompress_get_format_size (AcmObject          *acm_object,
                                const WaveFormatEx *wfx);


/*  ja:展開後のフォーマットを取得する
    acm_object,AcmObject
        wfx_in,元のWaveFormatEx構造体へのポインタ
       wfx_out,新しいWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_decompress_get_format (AcmObject          *acm_object,
                           const WaveFormatEx *wfx_in,
                           WaveFormatEx       *wfx_out);


/*  ja:展開を開始する
    acm_object,AcmObject
           wfx,元のWaveFormatEx構造体へのポインタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_decompress_begin (AcmObject          *acm_object,
                      const WaveFormatEx *wfx);


/*  ja:展開を終了する
     acm_object,AcmObject
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_decompress_end (AcmObject *acm_object,
                    gpointer  *out_buf,
                    gsize     *out_samples);


/*  ja:展開する
     acm_object,AcmObject
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
acm_decompress (AcmObject      *acm_object,
                const gpointer  in_buf,
                const gsize     in_samples,
                gpointer       *out_buf,
                gsize          *out_samples);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __ACM_H__ */
