/*
    Video maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "size.h"
#include <gdk/gdkkeysyms.h>


/******************************************************************************
*                                                                             *
* ja:サイズ変更関数群                                                         *
*                                                                             *
******************************************************************************/
/*  ja:フレームサイズの設定
          view,設定値
         width,幅
        height,高さ
     def_width,デフォルトの幅
    def_height,デフォルトの高さ                                             */
void
size_set_scale (const gint  view,
                gint       *width,
                gint       *height,
                const gint  def_width,
                const gint  def_height)
{
  switch (def_width <= 0 || def_height <= 0 ? view | 4 : view)
    {
      case 0: *width = def_width;     *height = def_height; break;
      case 1: *width = def_width / 2; *height = def_height / 2; break;
      case 2: *width = def_width / 4; *height = def_height / 4; break;
      case 3: *width = def_width / 8; *height = def_height / 8; break;
      case 4: *width = 320;           *height = 240; return;
      case 5: *width = 160;           *height = 120; return;
      case 6: *width = 80;            *height = 60; return;
      case 7: *width = 40;            *height = 30; return;
    }
  if (*width < 1)
    *width = 1;
  if (*height < 1)
    *height = 1;
}


/******************************************************************************
*                                                                             *
* ja:サイズ変更ダイアログ関数群                                               *
*                                                                             *
******************************************************************************/
typedef struct _SizeDialog
{
  gboolean edit;
  gint def_view, def_width, def_height;
  GtkWidget *dialog, *check, *spin0, *spin1;
} SizeDialog;


/* ja:スピンボタンの値が変化した */
static void
size_dialog_changed (GtkWidget  *widget,
                     SizeDialog *sizedlg)
{
  if (sizedlg->edit)
    {
      gint width, height;

      sizedlg->edit = FALSE;
      width = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (sizedlg->spin0));
      height = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (sizedlg->spin1));
      if (sizedlg->def_width > 0 && sizedlg->def_height > 0
        && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (sizedlg->check)))
        {
          /* ja:縦/横の比率を保持する */
          if (widget == sizedlg->spin0)
            {
              height = MAX (width * sizedlg->def_height
                                                    / sizedlg->def_width, 1);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON (sizedlg->spin1),
                                                                    height);
            }
          else
            {
              width = MAX (height * sizedlg->def_width
                                                    / sizedlg->def_height, 1);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON (sizedlg->spin0),
                                                                        width);
            }
        }
      gtk_dialog_set_response_sensitive (GTK_DIALOG (sizedlg->dialog),
                                    GTK_RESPONSE_OK, 0 < width && 0 < height);
      sizedlg->edit = TRUE;
    }
}


/* ja:ボタンが押された */
static void
size_dialog_clicked_button (GtkWidget  *widget,
                            SizeDialog *sizedlg)
{
  gint width, height;

  sizedlg->def_view = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget),
                                                                "user_data"));
  size_set_scale (sizedlg->def_view, &width, &height,
                                    sizedlg->def_width, sizedlg->def_height);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (sizedlg->spin0), width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (sizedlg->spin1), height);
}


/* ja:スピンボタンでリターンが押された */
static void
size_dialog_activate (GtkWidget  *widget,
                      SizeDialog *sizedlg)
{
  gint width, height;

  width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sizedlg->spin0));
  height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sizedlg->spin1));
  if (0 < width && 0 < height)
    gtk_dialog_response (GTK_DIALOG (sizedlg->dialog), GTK_RESPONSE_OK);
}


/* ja:ESCが押された */
static gboolean
size_dialog_key_press (GtkWidget   *widget,
                       GdkEventKey *event,
                       gpointer     user_data)
{
  if (event->keyval == GDK_Escape)
    gtk_dialog_response (GTK_DIALOG (widget), GTK_RESPONSE_NONE);
  return FALSE;
}


/*  ja:サイズ変更ダイアログ表示
    sizecfg,サイズ変更構造体
        RET,TRUE:OK,FALSE:Cancel                                            */
gboolean
size_dialog (SizeConfig *sizecfg)
{
  gboolean result;
  GtkWidget *button0, *button1, *button2, *button3;
  GtkWidget *button4, *button5, *button6, *button7;
  GtkWidget *label0, *label1, *table, *hbox0, *hbox1, *vbox;
  SizeDialog sizedlg;

  sizedlg.edit = FALSE;
  sizedlg.def_view = -1;
  sizedlg.def_width  = sizecfg->def_width;
  sizedlg.def_height  = sizecfg->def_height;
  /* ja:メインウインドウ */
  sizedlg.dialog = gtk_dialog_new_with_buttons
                            (_("Size"),
                            GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            NULL);
  g_signal_connect_after (G_OBJECT (sizedlg.dialog), "key-press-event",
                                    G_CALLBACK (size_dialog_key_press), NULL);
  /* ja:ボタン */
  if (sizecfg->def_width > 0 && sizecfg->def_height > 0)
    {
      button0 = gtk_button_new_from_stock (GTK_STOCK_ZOOM_100);
      button1 = gtk_button_new_with_label (_("1/2"));
      button2 = gtk_button_new_with_label (_("1/4"));
      button3 = gtk_button_new_with_label (_("1/8"));
      g_object_set_data (G_OBJECT (button0), "user_data", GINT_TO_POINTER (0));
      g_object_set_data (G_OBJECT (button1), "user_data", GINT_TO_POINTER (1));
      g_object_set_data (G_OBJECT (button2), "user_data", GINT_TO_POINTER (2));
      g_object_set_data (G_OBJECT (button3), "user_data", GINT_TO_POINTER (3));
      g_signal_connect (G_OBJECT (button0), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
      g_signal_connect (G_OBJECT (button1),"clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
      g_signal_connect (G_OBJECT (button2), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
      g_signal_connect (G_OBJECT (button3), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
      GTK_WIDGET_SET_FLAGS (button0, GTK_CAN_DEFAULT);
      GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);
      GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);
      GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);
    }
  else
    {
      button0 = button1 = button2 = button3 = NULL;
    }
  button4 = gtk_button_new_with_label (_("320x240"));
  button5 = gtk_button_new_with_label (_("160x120"));
  button6 = gtk_button_new_with_label (_("80x60"));
  button7 = gtk_button_new_with_label (_("40x30"));
  g_object_set_data (G_OBJECT (button4), "user_data", GINT_TO_POINTER (4));
  g_object_set_data (G_OBJECT (button5), "user_data", GINT_TO_POINTER (5));
  g_object_set_data (G_OBJECT (button6), "user_data", GINT_TO_POINTER (6));
  g_object_set_data (G_OBJECT (button7), "user_data", GINT_TO_POINTER (7));
  g_signal_connect (G_OBJECT (button4), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
  g_signal_connect (G_OBJECT (button5), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
  g_signal_connect (G_OBJECT (button6), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
  g_signal_connect (G_OBJECT (button7), "clicked",
                            G_CALLBACK (size_dialog_clicked_button), &sizedlg);
  GTK_WIDGET_SET_FLAGS (button4, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button5, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button6, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button7, GTK_CAN_DEFAULT);
  /* ja:チェックボタン */
  if (sizecfg->def_width > 0 && sizecfg->def_height > 0)
    {
      sizedlg.check = gtk_check_button_new_with_mnemonic
                                                (_("Constrain _aspect ratio"));
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sizedlg.check),
                                                            sizecfg->ratio);
    }
  else
    {
      sizedlg.check = NULL;
    }
  /* ja:スピンボタン */
  sizedlg.spin0 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                            (sizecfg->width, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  sizedlg.spin1 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                            (sizecfg->height, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (sizedlg.spin0), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (sizedlg.spin1), TRUE);
  g_signal_connect (G_OBJECT (sizedlg.spin0), "activate",
                                G_CALLBACK (size_dialog_activate), &sizedlg);
  g_signal_connect (G_OBJECT (sizedlg.spin1), "activate",
                                G_CALLBACK (size_dialog_activate), &sizedlg);
  g_signal_connect (G_OBJECT (sizedlg.spin0), "changed",
                                G_CALLBACK (size_dialog_changed), &sizedlg);
  g_signal_connect (G_OBJECT (sizedlg.spin1), "changed",
                                G_CALLBACK (size_dialog_changed), &sizedlg);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("_Width"));
  label1 = gtk_label_new_with_mnemonic (_("_Height"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label0), sizedlg.spin0);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label1), sizedlg.spin1);
  /* ja:テーブル */
  if (button0 && button1 && button2 && button3)
    {
      table = gtk_table_new (2, 4, FALSE);
      gtk_table_attach_defaults (GTK_TABLE (table), button0, 0, 1, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button1, 1, 2, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button2, 2, 3, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button3, 3, 4, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button4, 0, 1, 1, 2);
      gtk_table_attach_defaults (GTK_TABLE (table), button5, 1, 2, 1, 2);
      gtk_table_attach_defaults (GTK_TABLE (table), button6, 2, 3, 1, 2);
      gtk_table_attach_defaults (GTK_TABLE (table), button7, 3, 4, 1, 2);
    }
  else
    {
      table = gtk_table_new (1, 4, FALSE);
      gtk_table_attach_defaults (GTK_TABLE (table), button4, 0, 1, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button5, 1, 2, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button6, 2, 3, 0, 1);
      gtk_table_attach_defaults (GTK_TABLE (table), button7, 3, 4, 0, 1);
    }

  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);

  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), sizedlg.spin0, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), sizedlg.spin1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);

  if (sizedlg.check)
    gtk_box_pack_start (GTK_BOX (vbox), sizedlg.check, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (sizedlg.dialog)->vbox), vbox);

  /* ja:表示 */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (sizedlg.dialog),
                                                    GTK_RESPONSE_OK, FALSE);
  gtk_widget_show_all (vbox);

  sizedlg.edit = TRUE;

  result = gtk_dialog_run (GTK_DIALOG (sizedlg.dialog)) == GTK_RESPONSE_OK;
  if (result)
    {
      if (sizedlg.def_view > 0)
        sizecfg->view = sizedlg.def_view;
      if (sizecfg->def_width > 0 && sizecfg->def_height > 0)
        sizecfg->ratio = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON (sizedlg.check));
      sizecfg->width = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (sizedlg.spin0));
      sizecfg->height = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (sizedlg.spin1));
    }
  gtk_widget_destroy (sizedlg.dialog);

  return result;
}
