/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"


struct _AviFrame
{
  guint32 handler;              /* ja:4文字コード */
  gint pos;                     /* ja:最後に取得したフレーム */
  AviEdit *avi_edit;
  IcmObject *icm_object;        /* ja:ICM展開 */
  BitmapInfoHeader *bmih;       /* ja:展開後のフォーマット */
  gsize bmih_size;              /* ja:展開後のフォーマットのサイズ */
};


/******************************************************************************
*                                                                             *
* ja:フレーム関数                                                             *
*                                                                             *
******************************************************************************/
/*  ja:DIBに展開されたフレームを開く
    avi_edit,AVI編集ハンドル
         RET,AVIフレーム構造体,NULL:エラー                                  */
AviFrame *
avi_get_frame_open (AviEdit *avi_edit)
{
  AviFrame *avi_frame;

  if (!avi_edit || avi_type (avi_edit) != AVI_TYPE_VIDEO)
    return NULL;
  avi_frame = g_malloc0 (sizeof (AviFrame));
  avi_frame->pos = -1;
  avi_frame->avi_edit = avi_edit;
  return avi_frame;
}


/*  ja:DIBに展開されたフレームを閉じる
    avi_frame,AVIフレーム構造体
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_get_frame_close (AviFrame *avi_frame)
{
  if (!avi_frame)
    return FALSE;
  if (avi_frame->icm_object)
    {
      icm_decompress_end (avi_frame->icm_object);
      icm_close (avi_frame->icm_object);
    }
  g_free (avi_frame->bmih);
  g_free (avi_frame);
  return TRUE;
}


/*  ja:DIBに展開されたフレームを取得する
     avi_edit,AVI編集ハンドル
          pos,取得するフレーム(サンプル番号)
          RET,パックDIBへのポインタ,NULL:エラー                             */
static BitmapInfoHeader *
avi_get_frame_real (AviFrame   *avi_frame,
                    const gint  pos)
{
  gpointer input = NULL, output;
  gint i, j, size;
  AviEdit *avi_edit;
  BitmapInfoHeader *bmih;

  avi_edit = avi_frame->avi_edit;
  if (pos < 0 || avi_length (avi_edit) <= pos)
    return NULL;                    /* ja:範囲外 */
  avi_get_file (avi_edit, pos);     /* ja:AVIファイルを求める */
  if (avi_edit->file->handler == AVI_STREAM_COMP_DIB)
    {
      /* ja:ヌルフレームのチェック */
      for (i = pos - avi_edit->offset + avi_edit->file->start; i > 0; i--)
        if (avi_edit->file->index[i].size > 0)
          break;
      /* ja:イメージのサイズよりもデータのサイズが小さいときエラー */
      if (bm_image_bytes (avi_edit->file->bmih)
                                            > avi_edit->file->index[i].size)
        return NULL;
      size = bm_header_bytes (avi_edit->file->bmih);
      bmih = g_malloc (size + avi_edit->file->index[i].size);
      /* ja:ヘッタ */
      g_memmove (bmih, avi_edit->file->bmih, size);
      if (bmih_get_compression (bmih) == AVI_STREAM_COMP_DIB)
        bmih_set_compression (bmih, BI_COMP_RGB);
      if (bmih_get_size_image (bmih) <= 0)
        bmih_set_size_image (bmih, bm_image_bytes (bmih));
      /* ja:イメージ */
      if (!avi_edit->file->data)
        {
          /* ja:ファイル */
          fileio_seek (avi_edit->file->fio, avi_edit->file->index[i].offset,
                                                            FILEIO_SEEK_SET);
          fileio_read (avi_edit->file->fio, (guint8 *)bmih + size,
                                                avi_edit->file->index[i].size);
        }
      else
        {
          /* ja:メモリ */
          g_memmove ((guint8 *)bmih + size,
            (guint8 *)avi_edit->file->data + avi_edit->file->index[i].offset,
                                                avi_edit->file->index[i].size);
        }
    }
  else
    {
      if (avi_edit->file->handler != avi_frame->handler)
        {
          /* ja:圧縮が異なるとき */
          if (avi_frame->icm_object)
            {
              icm_decompress_end (avi_frame->icm_object);
              icm_close (avi_frame->icm_object);
              g_free (avi_frame->bmih);
              avi_frame->bmih = NULL;
              avi_frame->bmih_size = 0;
            }
          avi_frame->handler = avi_edit->file->handler;
          avi_frame->pos = -1;
          avi_frame->icm_object = icm_open (avi_frame->handler,
                                            ICM_MODE_DECOMPRESS);
          if (!avi_frame->icm_object)
            {
              avi_frame->handler = 0;
              return NULL;
            }
          if (!icm_decompress_begin (avi_frame->icm_object,
                                                        avi_edit->file->bmih))
            {
              icm_close (avi_frame->icm_object);
              avi_frame->handler = 0;
              avi_frame->icm_object = NULL;
              g_free (avi_frame->bmih);
              avi_frame->bmih = NULL;
              avi_frame->bmih_size = 0;
              return NULL;
            }
          avi_frame->bmih_size = icm_decompress_get_format_size
                                                        (avi_frame->icm_object,
                                                        avi_edit->file->bmih);
          avi_frame->bmih = g_malloc (avi_frame->bmih_size);
          if (avi_frame->bmih_size <= 0 || !avi_frame->bmih
                        || !icm_decompress_get_format (avi_frame->icm_object,
                                        avi_edit->file->bmih, avi_frame->bmih))
            {
              icm_close (avi_frame->icm_object);
              avi_frame->handler = 0;
              avi_frame->icm_object = NULL;
              g_free (avi_frame->bmih);
              avi_frame->bmih = NULL;
              avi_frame->bmih_size = 0;
              return NULL;
            }
        }

      /* ja:非キーフレームのチェック */
      i = j = pos - avi_edit->offset + avi_edit->file->start;
      if (avi_frame->pos != pos - 1 || pos <= avi_edit->offset)
        while (j > 0)
          { /* ja:連続していないとき */
            if ((avi_edit->file->index[j].flags & AVI_IF_KEYFRAME)
                                        && avi_edit->file->index[j].size > 0)
              break;
            j--;
          }
      bmih = g_malloc (bm_all_bytes (avi_frame->bmih));
      g_memmove (bmih, avi_frame->bmih, avi_frame->bmih_size);
      output = (guint8 *)bmih + bm_header_bytes (bmih);
      while (j <= i)
        {
          if (avi_edit->file->index[j].size > 0)
            {
              /* ja:イメージ */
              input = g_realloc (input, avi_edit->file->index[j].size);
              if (!avi_edit->file->data)
                {
                  /* ja:ファイル */
                  fileio_seek (avi_edit->file->fio,
                            avi_edit->file->index[j].offset, FILEIO_SEEK_SET);
                  fileio_read (avi_edit->file->fio, input,
                                                avi_edit->file->index[j].size);
                }
              else
                {
                  /* ja:メモリ */
                  g_memmove (input, (guint8 *)avi_edit->file->data
                                            + avi_edit->file->index[j].offset,
                                                avi_edit->file->index[j].size);
                }
              icm_decompress (avi_frame->icm_object,
                        avi_edit->file->index[j].flags & AVI_IF_KEYFRAME
                                            ? 0 : ICM_DECOMPRESS_NOTKEYFRAME,
                                                                input, output);
            }
          j++;
        }
      g_free (input);
    }
  return bmih;
}


/*  ja:DIBに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
    bit_count,色数
          RET,パックDIBへのポインタ,NULL:エラー                             */
BitmapInfoHeader *
avi_get_frame (AviFrame   *avi_frame,
               const gint  pos,
               const gint  width,
               const gint  height,
               const gint  bit_count)
{
  gint i, n;
  AviBuffer avi_buf;
  AviEdit *avi_edit;
  BitmapInfoHeader *bmih0, *bmih1;

  if (!avi_frame)
    return NULL;
  avi_edit = avi_frame->avi_edit;
  /* ja:ヌルフレームのチェック */
  avi_get_file (avi_edit,pos);      /* ja:AVIファイルを求める */
  for (n = pos-avi_edit->offset + avi_edit->file->start; n > 0; n--)
    if (avi_edit->file->index[n].size > 0)
      break;
  n += avi_edit->offset - avi_edit->file->start;
  /* ja:バッファをチェック */
  for (i = 0; i < AVI_EDIT_CACHE && avi_edit->buf[i].data; i++)
    if ((avi_edit->buf[i].pos == n || avi_edit->buf[i].pos == pos)
                    && !avi_edit->buf[i].raw
                    && bmih_get_width (avi_edit->buf[i].data) == width
                    && bmih_get_height (avi_edit->buf[i].data) == height
                    && bmih_get_bit_count (avi_edit->buf[i].data) == bit_count)
        {
          /* ja:キャッシュに同じものがあるとき */
          avi_buf = avi_edit->buf[i];
          g_memmove (avi_edit->buf + 1, avi_edit->buf, i * sizeof (AviBuffer));
          avi_edit->buf[0] = avi_buf;
          return avi_buf.data;
        }

  /* ja:展開された元のフレームを取得 */
  bmih0 = avi_get_frame_real (avi_frame,pos);
  if (!bmih0)
    return NULL;

  if (bmih_get_width (bmih0) != width || bmih_get_height (bmih0) != height
                || bmih_get_bit_count (bmih0) != bit_count || bit_count < 16)
    {
      /* ja:色数を32ビットに変換する */
      if (bmih_get_bit_count (bmih0) != 32)
        {
          bmih1 = g_malloc (bx_all_bytes (bmih_get_width (bmih0),
                                bmih_get_height (bmih0), 32, BI_COMP_RGB, 0));
          bitmap_convert_32 (bmih0, bmih1);
          g_free (bmih0);
          bmih0 = bmih1;
        }
      /* ja:サイズを変換する */
      if (bmih_get_width (bmih0) != width || bmih_get_height (bmih0) != height)
        {
          bmih1 = g_malloc (bx_all_bytes (width, height, 32, BI_COMP_RGB, 0));
          bmih_set_width (bmih1, width);
          bmih_set_height (bmih1, height);
          bitmap_zoom_32 (bmih0, bmih1);
          g_free (bmih0);
          bmih0 = bmih1;
        }
      /* ja:色数を変換する */
      if (bit_count != 32)
        {
          bmih1 = g_malloc (bx_all_bytes
                                (width, height, bit_count, BI_COMP_RGB, 0));
          bmih_set_color_used (bmih1, 0);
          bmih_set_color_important (bmih1, 0);
          switch (bit_count)
            {
              case 1:
                g_memmove ((guint8 *)bmih1 + BMIH_SIZE, rgb2,
                                                        RGBQUAD_SIZE * 2);
                bitmap_diffuse_1 (bmih0, bmih1);
                break;
              case 4:
                g_memmove ((guint8 *)bmih1 + BMIH_SIZE, rgb16,
                                                        RGBQUAD_SIZE * 16);
                bitmap_diffuse_4 (bmih0, bmih1);
                break;
              case 8:
                g_memmove ((guint8 *)bmih1 + BMIH_SIZE, rgb256,
                                                        RGBQUAD_SIZE * 256);
                bitmap_diffuse_8 (bmih0, bmih1);
                break;
              case 16:
                bitmap_convert_16 (bmih0, bmih1);
                break;
              case 24:
                bitmap_convert_24 (bmih0, bmih1);
            }
          g_free (bmih0);
          bmih0 = bmih1;
        }
    }
  g_free (avi_edit->buf[AVI_EDIT_CACHE-1].data);
  g_memmove (avi_edit->buf + 1, avi_edit->buf,
                                    (AVI_EDIT_CACHE - 1) * sizeof (AviBuffer));
  avi_edit->buf[0].data = bmih0;
  avi_edit->buf[0].raw = FALSE;
  avi_edit->buf[0].pos = pos;
  return bmih0;
}


/*  ja:RAWに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
          RET,RAWへのポインタ,NULL:エラー                               */
guint8 *
avi_get_frame32 (AviFrame  *avi_frame,
                const gint  pos,
                const gint  width,
                const gint  height)
{
  gint i, n, xx, yy, src, dst;
  guchar *p;
  guint8 *q;
  AviBuffer avi_buf;
  AviEdit *avi_edit;
  BitmapInfoHeader *bmih;

  if (!avi_frame)
    return NULL;
  avi_edit = avi_frame->avi_edit;
  /* ja:ヌルフレームのチェック */
  avi_get_file (avi_edit, pos);     /* ja:AVIファイルを求める */
  for (n = pos - avi_edit->offset + avi_edit->file->start; n > 0; n--)
    if (avi_edit->file->index[n].size > 0)
        break;
    n += avi_edit->offset - avi_edit->file->start;
  /* ja:バッファをチェック */
  for (i = 0; i < AVI_EDIT_CACHE && avi_edit->buf[i].data; i++)
    if ((avi_edit->buf[i].pos == n || avi_edit->buf[i].pos == pos)
                                        && avi_edit->buf[i].raw
                                        && avi_edit->buf[i].width == width
                                        && avi_edit->buf[i].height == height)
      {
        /* ja:キャッシュに同じものがあるとき */
        avi_buf = avi_edit->buf[i];
        g_memmove (avi_edit->buf + 1, avi_edit->buf, i * sizeof (AviBuffer));
        avi_edit->buf[0] = avi_buf;
        return avi_buf.data;
      }
  bmih = avi_get_frame (avi_frame, pos, width, height, 32);
  if (!bmih)
    return NULL;
  p = g_malloc (width * height * 4 * sizeof (guchar));
  q = (guchar *)bmih + bm_header_bytes (bmih);
  for (yy = 0; yy < height; yy++)
    for (xx = 0; xx < width; xx++)
      {
        src = xx * 4 + (height - yy - 1) * width * 4;
        dst = xx * 4 + yy * width * 4;
        p[dst]  = q[src + 2];
        p[dst + 1] = q[src + 1];
        p[dst + 2] = q[src];
        p[dst + 3] = q[src + 3];
      }
  g_free (avi_edit->buf[0].data);
  avi_edit->buf[0].data = p;
  avi_edit->buf[0].raw = TRUE;
  avi_edit->buf[0].width = width;
  avi_edit->buf[0].height = height;
  return p;
}


/*  ja:フレームがキーフレームか判定する
    avi_edit,AVI編集ハンドル
         pos,取得するフレーム(サンプル番号)
         RET,TRUE:キーフレーム,FALSE:非キーフレーム                         */
gboolean
avi_is_keyframe (AviEdit    *avi_edit,
                 const gint  pos)
{
  if (avi_edit && avi_type (avi_edit) == AVI_TYPE_VIDEO)
    {
      /* ja:ビデオ */
      avi_get_file (avi_edit, pos);/* ja:AVIファイルを求める */
      return avi_edit->file->index
                        [pos-avi_edit->offset + avi_edit->file->start].flags
                                                            & AVI_IF_KEYFRAME;
    }
  return TRUE;
}
