/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* ja:シナリオオブジェクト関数                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIの一部をシナリオオブジェクトに変換する
    avi_edit,AVI編集ハンドルへのポインタ
      stream,ストリーム番号
       start,変換をはじめるサンプル番号
     samples,サンプル数
         RET,シナリオオブジェクト,NULL:エラー                               */
static gchar *
avi_at_scenario (AviEdit    *avi_edit,
                 const gint  stream,
                 const gint  start,
                 const gint  samples)
{
  gchar *scenario = NULL;
  gint length = 0, start_n, samples_n;

  /* ja:スタート位置とサンプル数の調整 */
  if (start < 0)
    return NULL;
  samples_n = MIN (avi_length (avi_edit) - start, samples);
  if (samples_n <= 0 || avi_length (avi_edit) < samples_n)
    return NULL;

  start_n = start;
  while (samples_n > 0)
    {
      gchar *text;
      gint leng;

      avi_get_file (avi_edit, start_n);/* ja:AVIファイルを求める */
      if (avi_edit->file->name)
        {
          /* ja:ファイル */
          text = g_strdup_printf ("%d,file,\'%s\',%d,%d,%d\n", stream,
                avi_edit->file->name, avi_edit->file->param,
                avi_edit->file->start + start_n - avi_edit->offset,
                MIN (avi_edit->file->length - (start_n - avi_edit->offset),
                                                                samples_n));
          samples_n -= avi_edit->file->length - (start_n - avi_edit->offset);
          start_n += avi_edit->file->length - (start_n - avi_edit->offset);
        }
      else if (avi_edit->file->data && avi_edit->file->bmih)
        {
          gint i, j = 0, size;

          /* ja:メモリ内ビットマップ */
          text = g_strdup_printf ("%d,bitmap,%d,\\\n",
                                stream, bm_all_bytes (avi_edit->file->bmih));
          leng = g_strlen (text);
          size = bm_header_bytes (avi_edit->file->bmih);
          for (i = 0; i < size; i++)
            {
              text = g_realloc (text, (leng + 4 + (j % 16 == 0 ? 1 : 0)
                                + (j % 16 == 15 ? 2 : 0)) * sizeof (gchar));
              if (j % 16 == 0)
                text[leng++] = '\t';
              g_snprintf (text + leng, 4, "%02X,",
                                        ((guint8 *)avi_edit->file->bmih)[i]);
              leng += 3;
              if (j % 15 == 0)
                {
                  text[leng++] = '\\';
                  text[leng++] = '\n';
                }
              j++;
            }
          size = bm_image_bytes (avi_edit->file->bmih);
          for (i = 0; i < size; i++)
            {
              text = g_realloc (text, (leng + 4 + (j % 16 == 0 ? 1 : 0)
                                + (j % 16 == 15 ? 2 : 0)) * sizeof (gchar));
              if (j % 16 == 0)
                text[leng++] = '\t';
              g_snprintf (text + leng, 4, "%02X,",
                                        ((guint8 *)avi_edit->file->data)[i]);
              leng += 3;
              if (j % 15 == 0 && i < size - 1)
                {
                  text[leng++] = '\\';
                  text[leng++] = '\n';
                }
              j++;
            }
          text[leng - 1] = '\n';
          start_n++;
          samples_n--;
        }
      else
        {
          g_free (scenario);
          return NULL;
        }
      leng = g_strlen (text);
      scenario = g_realloc (scenario, (length + leng + 1) * sizeof (gchar));
      g_memmove (scenario + length, text, leng);
      g_free (text);
      length += leng;
    }
  scenario[length] = '\0';
  return scenario;
}


/*  ja:AVIをシナリオオブジェクトに変換する
      avi_edit,AVI編集ハンドルへのポインタ
    interleave,TRUE:インターリーブする,FALSE:インターリーブしない
           RET,シナリオオブジェクト,NULL:エラー                             */
gchar *
avi_to_scenario (AviEdit        *avi_edit[],
                 const gboolean  interleave)
{
  gchar *text, *scenario = NULL;
  gint i, leng, length = 0;

  if (!avi_edit)
    return NULL;
  /* ja:ヘッタ */
  scenario = g_strdup ("#VMAID\n-1,Video maid scenario object,0\n\n");
  length = g_strlen (scenario);
  for (i = 0; avi_edit[i]; i++)
    {
      switch (avi_type (avi_edit[i]))
        {
          case AVI_TYPE_AUDIO:
            text = g_strdup_printf ("%d,type,audio\n"
                                    "%d,priority,%u\n"
                                    "%d,language,%X\n",
                                i,
                                i, avi_get_priority (avi_edit[i]),
                                i, avi_get_language (avi_edit[i]));
            break;
          case AVI_TYPE_VIDEO:
            text = g_strdup_printf ("%d,type,video\n"
                                    "%d,rate,%u\n"
                                    "%d,scale,%u\n"
                                    "%d,priority,%u\n"
                                    "%d,language,%X\n"
                                    "%d,bmih,%d,%d,%d,%d,%d\n",
                        i,
                        i, avi_get_rate (avi_edit[i]),
                        i, avi_get_scale (avi_edit[i]),
                        i, avi_get_priority (avi_edit[i]),
                        i, avi_get_language (avi_edit[i]),
                        i,  bmih_get_width (avi_edit[i]->bmih),
                            bmih_get_height (avi_edit[i]->bmih),
                            bmih_get_bit_count (avi_edit[i]->bmih),
                            bmih_get_x_pixels_per_meter (avi_edit[i]->bmih),
                            bmih_get_y_pixels_per_meter (avi_edit[i]->bmih));
            break;
          default:
            g_free (scenario);
            return NULL;
        }
      leng = g_strlen (text);
      scenario = g_realloc (scenario, (length + leng + 2) * sizeof (gchar));
      g_memmove (scenario + length, text, leng);
      g_free (text);
      length += leng;
      if (avi_get_name (avi_edit[i]))
        {
          text = g_strdup_printf ("name,\'%s\'\n", avi_get_name (avi_edit[i]));
          leng = g_strlen (text);
          scenario = g_realloc (scenario,
                                        (length + leng + 2) * sizeof (gchar));
          g_memmove (scenario + length, text, leng);
          length += leng;
          g_free (text);
        }
    }
  if (i <= 0)
    {
      g_free (scenario);
      return NULL;
    }
  scenario[length++] = '\n';
  scenario[length] = '\0';
  /* ja:本体 */
  if (interleave)
    {
      gint current = 0, *pos, streams, timer = 0, times;

      /* ja:インターリーブ */
      for (streams = 0; avi_edit[streams]; streams++);
      /* ja:最も1秒間のフレーム数が少ないストリームを探す */
      for (current = 0; current < streams; current++)
        if (avi_type (avi_edit[current]) == AVI_TYPE_VIDEO)
          break;
      if (current >= streams)
        {
          current = -1;
          times = 1000;
        }
      else
        {
          for (i = 0; i < streams; i++)
            if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO
                        && (glonglong)avi_get_rate (avi_edit[i])
                                            * avi_get_scale (avi_edit[current])
                            < (glonglong)avi_get_rate (avi_edit[current])
                                                * avi_get_scale (avi_edit[i]))
              current = i;
            times = avi_length_time (avi_edit[current])
                                            / avi_length (avi_edit[current]);
        }
      pos = g_malloc0 (streams * sizeof (gint));
      while (TRUE)
        {
          /* ja:すべて出力したら終了 */
          for (i = 0; i < streams; i++)
            if (pos[i] < avi_length (avi_edit[i]))
              break;
          if (i >= streams)
            break;
          for (i = 0; i < streams; i++)
            if (pos[i] < avi_length (avi_edit[i]))
              {
                gint samples;

                samples = current > 0 ? i == current ? 1 : avi_time_to_sample
                        (avi_edit[i], avi_sample_to_time (avi_edit[current],
                                                    pos[current] + 1)) - pos[i]
                            : avi_time_to_sample (avi_edit[i], timer) - pos[i];
                text = avi_at_scenario (avi_edit[i], i, pos[i], samples);
                if (text)
                  {
                    leng = g_strlen (text);
                    scenario = g_realloc (scenario,
                                        (length + leng + 1) * sizeof (gchar));
                    g_memmove (scenario + length, text, leng);
                    g_free (text);
                    length += leng;
                    if (i != current)
                      pos[i] += samples;
                  }
              }
          if (current >= 0)
            {
              pos[current]++;
              if (avi_length (avi_edit[current]) <= pos[current])
                current = -1;
            }
          else
            {
              timer += times;
            }
        }
      g_free (pos);
    }
  else
    {
      /* text非インターリーブ */
      for (i = 0; avi_edit[i]; i++)
        {
          text = avi_at_scenario (avi_edit[i], i, 0, avi_length (avi_edit[i]));
          leng = g_strlen (text);
          scenario = g_realloc (scenario,
                                        (length + leng + 1) * sizeof (gchar));
          g_memmove (scenario + length, text, leng);
          g_free (text);
          length += leng;
        }
    }
  scenario[length] = '\0';
  return scenario;
}


/*  ja:シナリオオブジェクトからAVIを作る
    scenario,シナリオオブジェクト
         RET,AVI編集ハンドルへのポインタ,NULL:エラー                        */
AviEdit **
avi_from_scenario (const gchar *scenario)
{
  gchar **lines, *text;
  gint i, j, count = 0;
  AviEdit **avi_edit = NULL;
  AviFile *avi_file;

  if (!scenario)
    return NULL;
  /* ja:便宜改行を真の改行に変換する */
  text = g_strdup (scenario);
  j = 0;
  for (i = 0; text[i] != '\0'; i++)
    if (text[i] != '\\' || text[i + 1] != '\n')
      text[j++] = text[i];
  text[j] = '\0';
  lines = g_strsplit (text, "\n", 0);
  g_free (text);
  for (i = 0; lines[i]; i++)
    {
      gboolean quote[2];
      gchar **cmd;
      gint stream;

      /* ja:コメント部分を削除,カンマを改行にする */
      quote[0] = quote[1] = FALSE;
      for (j = 0; (lines[i])[j] != '\0'; j++)
        if ((lines[i])[j] == '\'' && !quote[1])
          {
            quote[0] = !quote[0];
          }
        else if ((lines[i])[j] == '\"' && !quote[0])
          {
            quote[1] = !quote[1];
          }
        else if (!quote[0] && !quote[1])
          {
            if ((lines[i])[j] == ',')
              {
                (lines[i])[j] = '\n';
              }
            else if ((lines[i])[j] == '#')
              {
                (lines[i])[j] = '\0';
                break;
              }
          }
      g_strstrip (lines[i]);
      cmd = g_strsplit (lines[i], "\n", 0);
      for (j = 0; cmd[j]; j++)
        {
          gint leng;

          g_strstrip (cmd[j]);
          leng = g_strlen (cmd[j]) - 1;
          if (leng >= 1 && (((cmd[j])[0] == '\'' && (cmd[j])[leng] == '\'')
                        || ((cmd[j])[0] == '\"' && (cmd[j])[leng] == '\"')))
            {
              g_memmove (cmd[j], cmd[j] + 1,--leng * sizeof (gchar));
              (cmd[j])[leng] = '\0';
            }
        }
      if (cmd[0] && cmd[1]
                && misc_str_to_val (&stream, cmd[0], 10, TRUE) && stream >= 0)
        {
          gint length, size, value;

          if (count <= stream)/* ja:拡張 */
            {
              avi_edit = g_realloc (avi_edit,
                                            (stream + 1) * sizeof (AviEdit *));
              g_memset (avi_edit + count, 0,
                                    (stream - count + 1) * sizeof (AviEdit *));
              count = stream + 1;
            }
          if (!avi_edit[stream])/* ja:新規 */
            avi_edit[stream] = g_malloc0 (sizeof (AviEdit));
          if (g_strcasecmp (cmd[1], "type") == 0 && cmd[2])
            {
              if (g_strcasecmp (cmd[2], "audio") == 0)
                avi_edit[stream]->type = AVI_TYPE_AUDIO;
              else if (g_strcasecmp (cmd[2], "video") == 0)
                avi_edit[stream]->type = AVI_TYPE_VIDEO;
            }
          else if (g_strcasecmp (cmd[1], "rate") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_set_rate (avi_edit[stream], value);
            }
          else if (g_strcasecmp (cmd[1], "scale") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_set_scale (avi_edit[stream], value);
            }
          else if (g_strcasecmp (cmd[1], "priority") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_set_priority (avi_edit[stream], (guint16)value);
            }
          else if (g_strcasecmp (cmd[1], "language") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 16, FALSE))
                avi_set_language (avi_edit[stream], (guint16)value);
            }
          else if (g_strcasecmp (cmd[1], "name") == 0 && cmd[2])
            {
              avi_set_name (avi_edit[stream], cmd[2]);
            }
          else if (g_strcasecmp (cmd[1], "bmih") == 0)
            {
              for (j = 2; cmd[j]; j++);
                if (j >= 7)
                  {
                    if (!avi_edit[stream]->bmih)/* ja:新規 */
                      {
                        avi_edit[stream]->bmih = g_malloc0 (BMIH_SIZE);
                        bmih_set_size (avi_edit[stream]->bmih, BMIH_SIZE);
                        bmih_set_planes (avi_edit[stream]->bmih, 1);
                        bmih_set_compression (avi_edit[stream]->bmih,
                                                                BI_COMP_RGB);
                      }
                    if (misc_str_to_val (&value, cmd[2], 10, TRUE)
                                                                && value > 0)
                        bmih_set_width (avi_edit[stream]->bmih, value);
                    if (misc_str_to_val (&value, cmd[3], 10, TRUE)
                                                                && value > 0)
                        bmih_set_height (avi_edit[stream]->bmih, value);
                    if (misc_str_to_val (&value, cmd[4], 10, FALSE))
                        bmih_set_bit_count (avi_edit[stream]->bmih, value);
                        switch (value)
                          {
                            case  1:
                            bmih_set_color_used (avi_edit[stream]->bmih, 2);
                            break;
                            case  4:
                            bmih_set_color_used (avi_edit[stream]->bmih, 16);
                            break;
                            case  8:
                            bmih_set_color_used (avi_edit[stream]->bmih, 256);
                            break;
                            case 16:
                            case 24:
                            case 32:
                            bmih_set_color_used (avi_edit[stream]->bmih, 0);
                        }
                    if (misc_str_to_val (&value, cmd[5], 10, FALSE))
                        bmih_set_x_pixels_per_meter (avi_edit[stream]->bmih,
                                                                        value);
                    if (misc_str_to_val (&value, cmd[6], 10, FALSE))
                        bmih_set_x_pixels_per_meter (avi_edit[stream]->bmih,
                                                                        value);
                    if (bmih_get_width (avi_edit[stream]->bmih) <= 0
                        || bmih_get_height (avi_edit[stream]->bmih) <= 0
                        || bmih_get_bit_count (avi_edit[stream]->bmih) == 0)
                      {
                        g_free (avi_edit[stream]->bmih);
                        avi_edit[stream]->bmih = NULL;
                      }
                    else
                      {
                        avi_edit[stream]->bmih
                                = g_realloc (avi_edit[stream]->bmih,
                                    bm_header_bytes (avi_edit[stream]->bmih));
                      }
                  }
            }
          else if (g_strcasecmp (cmd[1], "file") == 0 && cmd[2])
            {
              gint param;
              guint32 type;

              if (!cmd[3] || !misc_str_to_val (&param, cmd[3], 10, FALSE))
                param = 0;
              avi_file = avi_open_file (cmd[2], param, &type,
                                avi_get_rate (avi_edit[stream]) <= 0
                                        ? &avi_edit[stream]->rate : NULL,
                                avi_get_scale (avi_edit[stream]) <= 0
                                        ? &avi_edit[stream]->scale : NULL,
                                !avi_edit[stream]->file
                                        ? &avi_edit[stream]->priority : NULL,
                                !avi_edit[stream]->file
                                        ? &avi_edit[stream]->language : NULL,
                                !avi_get_name (avi_edit[stream])
                                        ? &avi_edit[stream]->name : NULL);
              if (avi_type (avi_edit[stream]) == AVI_TYPE_UNKNOWN)
                {
                  avi_edit[stream]->type = type;
                }
              else if (avi_type (avi_edit[stream]) != type)
                {
                  avi_release_file (avi_file);
                  avi_file = NULL;
                }
              if (avi_file)
                {
                  gint start;

                  if (cmd[3] || cmd[4]
                            || !misc_str_to_val (&start, cmd[4], 10, TRUE))
                    start = -1;
                  if (cmd[3] || cmd[4] || cmd[5]
                            || !misc_str_to_val (&length, cmd[5], 10, TRUE))
                    length = -1;
                  if (0 < start && start < avi_file->length)
                    {
                      avi_file->start = start;
                      avi_file->length -= start;
                    }
                  if (0 < length && length < avi_file->length)
                      avi_file->length = length;
                  if (!avi_edit[stream]->file)
                    {
                      /* ja:先頭 */
                      avi_edit[stream]->file = avi_file;
                    }
                  else if (g_strfilecmp (avi_edit[stream]->file->name,
                                                        avi_file->name) == 0
                      && avi_edit[stream]->file->start
                        + avi_edit[stream]->file->length == avi_file->start)
                    {
                      /* ja:連続 */
                      avi_edit[stream]->file->length += avi_file->length;
                      avi_release_file (avi_file);
                    }
                  else
                    {
                      /* ja:追加 */
                      avi_edit[stream]->file->next = avi_file;
                      avi_file->prev = avi_edit[stream]->file;
                      avi_edit[stream]->file = avi_file;
                    }
                }
            }
          else if (g_strcasecmp (cmd[1], "bitmap") == 0 && cmd[2]
                                && misc_str_to_val (&size, cmd[2], 10, FALSE)
                                && size > BMIH_SIZE)
            {
              avi_file = g_malloc (sizeof (AviFile));
              avi_file->data = NULL;            /* ja:メモリ */
              /* ja:ヘッタを読み込む */
              avi_file->bmih = g_malloc (BMIH_SIZE);
              for (j = 0; j < BMIH_SIZE && cmd[j + 3]; j++)
                {
                  if (!misc_str_to_val (&value, cmd[j + 3], 16, FALSE))
                    break;
                  ((guint8 *)avi_file->bmih)[j] = value;
                }
              if (j != BMIH_SIZE)
                {
                  g_free (avi_file->bmih);
                  g_free (avi_file);
                  avi_file = NULL;
                }
              else
                {
                  /* ja:ヘッタの残りを読み込む */
                  length = bm_header_bytes (avi_file->bmih);
                  if (length >= size || bm_all_bytes (avi_file->bmih) != size)
                    {
                      g_free (avi_file->bmih);
                      g_free (avi_file);
                    }
                  else
                    {
                      avi_file->bmih = g_realloc (avi_file->bmih, length);
                      while (j < length && cmd[j + 3])
                        {
                          if (!misc_str_to_val
                                            (&value, cmd[j + 3], 16, FALSE))
                            break;
                          ((guint8 *)avi_file->bmih)[j++] = value;
                        }
                      if (j != length)
                        {
                          g_free (avi_file->bmih);
                          g_free (avi_file);
                          avi_file = NULL;
                        }
                      else
                        {
                          /* ja:本体を読み込む */
                          avi_file->data = g_malloc (size - length);
                          while (j < size && cmd[j + 3])
                            {
                              if (!misc_str_to_val
                                            (&value, cmd[j + 3], 16, FALSE))
                                break;
                              ((guint8 *)avi_file->data)[j - length] = value;
                              j++;
                            }
                          if (j != size)
                            {
                              g_free (avi_file->data);
                              g_free (avi_file->bmih);
                              g_free (avi_file);
                              avi_file = NULL;
                            }
                        }
                    }
                }
              if (avi_file)
                {
                  avi_file->fio = NULL;     /* ja:ファイルポインタ */
                  avi_file->name = NULL;    /* ja:ファイル名 */
                  avi_file->start = 0;
                  avi_file->length = 1;     /* ja:有効な範囲 */
                  avi_file->entries = 1;    /* ja:インデックスの数 */
                  avi_file->param = 0;      /* ja:ストリーム番号 */
                  /* ja:インデックス */
                  avi_file->index = g_malloc (sizeof (AviIndex));
                  avi_file->index[0].offset = 0;
                  avi_file->index[0].size = size;
                  avi_file->index[0].flags = 0;
                  /* ja:オーディオ固有 */
                  avi_file->wfx = NULL;     /* ja:オリジナルのフォーマット */
                  /* ja:リスト */
                  avi_file->next = NULL;
                  if (avi_edit[stream]->file)
                    {
                      /* ja:先頭 */
                      avi_file->prev = NULL;
                      avi_edit[stream]->file = avi_file;
                    }
                  else
                    {
                      /* ja:追加 */
                      avi_edit[stream]->file->next = avi_file;
                      avi_file->prev = avi_edit[stream]->file;
                      avi_edit[stream]->file = avi_file;
                    }
                }
            }
        }
      g_strfreev (cmd);
    }
  g_strfreev (lines);
  if (avi_edit)
    {
      for (i = 0; i < count; i++)
        {
          /* ja:サンプル数 */
          if (avi_edit[i]->file)
            {
              while (avi_edit[i]->file->prev)
                avi_edit[i]->file = avi_edit[i]->file->prev;
              for (avi_file = avi_edit[i]->file; avi_file;
                                                    avi_file = avi_file->next)
                avi_edit[i]->length += avi_file->length;
            }
          if (avi_length (avi_edit[i]) <= 0)
            {
              avi_release (avi_edit[i]);
              avi_edit[i] = NULL;
            }
          /* ja:展開後のフォーマット */
          if (avi_edit[i] && !avi_edit[i]->bmih
                                && avi_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            {
              if (avi_edit[i]->file->handler == AVI_STREAM_COMP_DIB)
                {
                  avi_edit[i]->bmih = g_memdup (avi_edit[i]->file->bmih,
                                    bm_header_bytes (avi_edit[i]->file->bmih));
                  if (bmih_get_compression (avi_edit[i]->bmih)
                                                        == AVI_STREAM_COMP_DIB)
                    bmih_set_compression (avi_edit[i]->bmih, BI_COMP_RGB);
                  if (bmih_get_size_image (avi_edit[i]->bmih) <= 0)
                    bmih_set_size_image (avi_edit[i]->bmih,
                                        bm_image_bytes (avi_edit[i]->bmih));
                }
              else
                {
                  IcmObject *icm_object;

                  /* ja:圧縮 */
                  icm_object = icm_open (avi_edit[i]->file->handler,
                                                        ICM_MODE_DECOMPRESS);
                  if (icm_object)
                    {
                      avi_edit[i]->bmih = g_malloc
                                (icm_decompress_get_format_size (icm_object,
                                                    avi_edit[i]->file->bmih));
                      if (icm_decompress_get_format (icm_object,
                                    avi_edit[i]->file->bmih, avi_edit[i]->bmih)
                                                    & icm_close (icm_object))
                        {
                          avi_edit[i]->bmih = g_realloc (avi_edit[i]->bmih,
                                        bm_header_bytes (avi_edit[i]->bmih));
                        }
                      else
                        {
                          avi_release (avi_edit[i]);
                          avi_edit[i] = NULL;
                        }
                    }
                  else
                    {
                      avi_release (avi_edit[i]);
                      avi_edit[i] = NULL;
                    }
                }
            }
          else if (avi_edit[i] && !avi_edit[i]->wfx
                                && avi_type (avi_edit[i]) == AVI_TYPE_AUDIO)
            {
              if (wfx_get_format_tag (avi_edit[i]->file->wfx)
                                                            == WAVE_FORMAT_PCM)
                {
                  avi_edit[i]->wfx = g_memdup (avi_edit[i]->file->wfx,
                                    wf_header_bytes (avi_edit[i]->file->wfx));
                }
              else
                {
                  /* ja:圧縮 */
                  avi_release (avi_edit[i]);
                  avi_edit[i] = NULL;
                }
            }
#ifdef XXX
          g_print("type=%X,rate=%u,scale=%u,length=%d\n",avi_edit[i]->type,avi_edit[i]->rate,avi_edit[i]->scale,avi_edit[i]->length);
          for (avi_file=avi_edit[i]->file;avi_file!=NULL;
                                                  avi_file=avi_file->next)
              g_print("%s,start=%d,length=%d\n",avi_file->name,avi_file->start,avi_file->length);
#endif
        }
      j = 0;
      for (i = 0; i < count; i++)
        if (avi_edit[i])
          avi_edit[j++] = avi_edit[i];
      if (j > 0)
        {
          avi_edit = g_realloc (avi_edit, (j + 1) * sizeof (AviEdit *));
          avi_edit[j] = NULL;
        }
      else
        {
          g_free (avi_edit);
          avi_edit = NULL;
        }
    }
  return avi_edit;
}
