/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "acm.h"
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
# include <tchar.h>
# include <windows.h>
# include <vfw.h>


# define MAKE_ACM_VERSION(mjr,mnr,bld) (((LONG)(mjr)<<24)| \
                                        ((LONG)(mnr)<<16)| \
                                        ((LONG)(bld)))
# define VERSION_MSACM                   MAKE_ACM_VERSION(3,50,0)
# define VERSION_ACM_DRIVER              MAKE_ACM_VERSION(4, 0,0)
# define ACMDM_DRIVER_NOTIFY             (ACMDM_BASE+1)
# define ACMDM_DRIVER_DETAILS            (ACMDM_BASE+10)
# define ACMDM_HARDWARE_WAVE_CAPS_INPUT  (ACMDM_BASE+20)
# define ACMDM_HARDWARE_WAVE_CAPS_OUTPUT (ACMDM_BASE+21)
# define ACMDM_FORMATTAG_DETAILS         (ACMDM_BASE+25)
# define ACMDM_FORMAT_DETAILS            (ACMDM_BASE+26)
# define ACMDM_FORMAT_SUGGEST            (ACMDM_BASE+27)
# define ACMDM_FILTERTAG_DETAILS         (ACMDM_BASE+50)
# define ACMDM_FILTER_DETAILS            (ACMDM_BASE+51)
# define ACMDM_STREAM_OPEN               (ACMDM_BASE+76)
# define ACMDM_STREAM_CLOSE              (ACMDM_BASE+77)
# define ACMDM_STREAM_SIZE               (ACMDM_BASE+78)
# define ACMDM_STREAM_CONVERT            (ACMDM_BASE+79)
# define ACMDM_STREAM_RESET              (ACMDM_BASE+80)
# define ACMDM_STREAM_PREPARE            (ACMDM_BASE+81)
# define ACMDM_STREAM_UNPREPARE          (ACMDM_BASE+82)


# include <pshpack1.h>
typedef struct tagACMDRVOPENDESC
{
  DWORD   cbStruct;
  FOURCC  fccType;
  FOURCC  fccComp;
  DWORD   dwVersion;
  DWORD   dwFlags;
  DWORD   dwError;
  LPCWSTR pszSectionName;
  LPCWSTR pszAliasName;
  DWORD   dnDevNode;
} ACMDRVOPENDESC, *PACMDRVOPENDESC, *LPACMDRVOPENDESC;
# include <poppack.h>


const static gunichar2 drivers32[]
                        = {'D', 'r', 'i', 'v', 'e', 'r', 's', '3', '2', '\0'};
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


struct _AcmObject
{
  gint counter;
  guint16 tag;
  gchar *name;
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  HDRVR hDrvr;
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
};


/******************************************************************************
*                                                                             *
* ja:ACM初期化                                                                *
*                                                                             *
******************************************************************************/
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static GHashTable *ghash_tag = NULL;
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ACMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_init (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  DWORD dwIndex, dwName, dwType, dwValue;
  HKEY hKey;
  TCHAR szName[MAX_PATH], szValue[MAX_PATH];

  if (ghash_tag)
    return FALSE;
  ghash_tag = g_hash_table_new (NULL, NULL);
  if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
        _T("Software\\Microsoft\\Windows NT\\CurrentVersion\\Drivers32"), 0,
                                    KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS)
    return FALSE;
  for (dwIndex = 0; dwName = MAX_PATH * sizeof (TCHAR),
                    dwValue = MAX_PATH * sizeof (TCHAR),
            RegEnumValue (hKey, dwIndex, szName, &dwName, NULL,
            &dwType, (LPBYTE)szValue, &dwValue) == ERROR_SUCCESS; dwIndex++)
    if (dwType == REG_SZ
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[0]))) == 'M'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[1]))) == 'S'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[2]))) == 'A'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[3]))) == 'C'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[4]))) == 'M'
    && GPOINTER_TO_INT (CharUpper (GINT_TO_POINTER ((gint)szName[5]))) == '.')
      {
# ifndef UNICODE
        gchar *utf8str;
# endif /* not UNICODE */
        gunichar2 *name;
        ACMDRVOPENDESC acopen;
        HDRVR hDrvr;

# ifdef UNICODE
        name = g_memdup (szName, (lstrlenW (szName) + 1) * sizeof (gunichar2));
# else /* not UNICODE */
        utf8str = g_locale_to_utf8 (szName, -1, NULL, NULL, NULL);
        name = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
        g_free (utf8str);
# endif /* not UNICODE */
        g_memset (&acopen, 0, sizeof (ACMDRVOPENDESC));
        acopen.cbStruct = sizeof (ACMDRVOPENDESC);
        acopen.fccType = ACMDRIVERDETAILS_FCCTYPE_AUDIOCODEC;
        acopen.pszSectionName = drivers32;
        acopen.pszAliasName = name;
        hDrvr = OpenDriver (name, NULL, GPOINTER_TO_INT (&acopen));
        if (hDrvr)
          {
            ACMDRIVERDETAILSW acdetail;

            acdetail.cbStruct = sizeof (ACMDRIVERDETAILSW);
            if (SendDriverMessage (hDrvr, ACMDM_DRIVER_DETAILS,
                            GPOINTER_TO_INT (&acdetail), 0) == MMSYSERR_NOERROR
                    && (acdetail.fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC))
              {
                gint i;

                for (i = 0; i < acdetail.cFormatTags; i++)
                  {
                    ACMFORMATTAGDETAILSW acfmttag;

                    acfmttag.cbStruct = sizeof (ACMFORMATTAGDETAILSW);
                    acfmttag.dwFormatTagIndex = i;
                    if (SendDriverMessage (hDrvr, ACMDM_FORMATTAG_DETAILS,
                                                GPOINTER_TO_INT (&acfmttag),
                                                ACM_FORMATTAGDETAILSF_INDEX)
                                                        == MMSYSERR_NOERROR
                            && (acfmttag.fdwSupport
                                            & ACMDRIVERDETAILS_SUPPORTF_CODEC)
                            && acfmttag.dwFormatTag != 0
                            && !g_hash_table_lookup (ghash_tag,
                                    GUINT_TO_POINTER (acfmttag.dwFormatTag)))
                      {
                        gunichar2 *value;

                        value = g_memdup (name,
                                (lstrlenW (name) + 1) * sizeof (gunichar2));
                        g_hash_table_insert (ghash_tag,
                            GUINT_TO_POINTER (acfmttag.dwFormatTag), value);
                      }
                  }
              }
            CloseDriver (hDrvr, 0, 0);
          }
        g_free (name);
      }
  return RegCloseKey (hKey) == ERROR_SUCCESS;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


#if defined (G_OS_WIN32) || defined (USE_W32LDR)
static void
acm_exit_callback (gpointer key,
                   gpointer value,
                   gpointer ghash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable *)ghash, key,
                                                    &orig_key, &orig_value))
    g_free (orig_value);
}
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */


/*  ja:ACMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
acm_exit (void)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (ghash_tag)
    {
      g_hash_table_foreach (ghash_tag, acm_exit_callback, ghash_tag);
      g_hash_table_destroy (ghash_tag);
      ghash_tag = NULL;
    }
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:ACM基本                                                                  *
*                                                                             *
******************************************************************************/
static gint
acm_get_handler_list_compare (gconstpointer a,
                              gconstpointer b)
{
  return GPOINTER_TO_UINT (a) - GPOINTER_TO_UINT (b);
}


static void
acm_get_handler_list_callback (gpointer   key,
                               gpointer   value,
                               GList    **glist)
{
  *glist = g_list_insert_sorted (*glist, key, acm_get_handler_list_compare);
}


/*  ja:登録されているタグを取得する
    RET,タグのリスト,NULL:登録なし                                          */
GList *
acm_get_tag_list (void)
{
  GList *glist = NULL;

  if (ghash_tag)
    g_hash_table_foreach (ghash_tag,
                                (GHFunc)acm_get_handler_list_callback, &glist);
  return glist;
}


/*  ja:Codecを開く
    tag,タグ
    RET,AcmObject,NULL:エラー                                               */
AcmObject *
acm_open (const guint16 tag)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  gunichar2 *name;
  AcmObject *acm_object = NULL;

  name = g_hash_table_lookup (ghash_tag, GUINT_TO_POINTER ((guint)tag));
  if (name)
    {
      ACMDRVOPENDESC acopen;
      HDRVR hDrvr;

      g_memset (&acopen, 0, sizeof (ACMDRVOPENDESC));
      acopen.cbStruct = sizeof (ACMDRVOPENDESC);
      acopen.fccType = ACMDRIVERDETAILS_FCCTYPE_AUDIOCODEC;
      acopen.pszSectionName = drivers32;
      acopen.pszAliasName = name;
      hDrvr = OpenDriver (name, NULL, GPOINTER_TO_INT (&acopen));
      if (hDrvr)
        {
          ACMFORMATTAGDETAILSW acfmttag;

          acfmttag.cbStruct = sizeof (ACMFORMATTAGDETAILSW);
          acfmttag.dwFormatTag = tag;
          if (SendDriverMessage (hDrvr, ACMDM_FORMATTAG_DETAILS,
                                            GPOINTER_TO_INT (&acfmttag),
                                            ACM_FORMATTAGDETAILSF_FORMATTAG)
                                                        == MMSYSERR_NOERROR
                            && (acfmttag.fdwSupport
                                            & ACMDRIVERDETAILS_SUPPORTF_CODEC)
                            && acfmttag.dwFormatTag == tag)
            {
              acm_object = g_malloc0 (sizeof (AcmObject));
              acm_object->counter = 1;
              acm_object->tag = tag;
              acm_object->name = g_utf16_to_utf8 (acfmttag.szFormatTag, -1,
                                                            NULL, NULL, NULL);
              acm_object->hDrvr = hDrvr;
              if (!acm_object->name || acm_object->name[0] == '\0')
                {
                  ACMDRIVERDETAILSW acdetail;

                  acdetail.cbStruct = sizeof (ACMDRIVERDETAILSW);
                  if (SendDriverMessage (hDrvr, ACMDM_DRIVER_DETAILS,
                        GPOINTER_TO_INT (&acdetail), 0) == MMSYSERR_NOERROR)
                    {
                      g_free (acm_object->name);
                      acm_object->name = g_utf16_to_utf8 (acdetail.szLongName,
                                                        -1, NULL, NULL, NULL);
                    }
                }
            }
          else
            {
              CloseDriver (hDrvr, 0, 0);
            }
        }
    }
  return acm_object;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return NULL;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Codecを閉じる
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_close (AcmObject *acm_object)
{
  if (acm_object && --(acm_object->counter) <= 0)
    {
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
      CloseDriver (acm_object->hDrvr, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
      g_free (acm_object->name);
      g_free (acm_object);
    }
  return TRUE;
}


/*  ja:Codecの参照数を増やす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_ref (AcmObject *acm_object)
{
  return acm_object ? acm_object->counter++, TRUE : FALSE;
}


/*  ja:Codecの参照数を減らす
    acm_object,AcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
acm_unref (AcmObject *acm_object)
{
  return acm_close (acm_object);
}


/*  ja:Codecをコピーする
    acm_object,AcmObject
           RET,AcmObject,NULL:エラー                                        */
AcmObject *
acm_dup (AcmObject *acm_object)
{
  return acm_object ? acm_open (acm_object->tag) : NULL;
}


/******************************************************************************
*                                                                             *
* ja:ACM情報                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:Codecのタグを取得する
    acm_object,AcmObject
           RET,タグ,0:エラー                                                */
guint16
acm_get_tag (AcmObject *acm_object)
{
  return acm_object ? acm_object->tag : 0;
}


/*  ja:Codecの名前を取得する
    acm_object,AcmObject
           RET,名前,NULL:エラー                                             */
const gchar *
acm_get_name (AcmObject *acm_object)
{
  return acm_object ? acm_object->name : NULL;
}


/*  ja:Codecの属性を取得する
    acm_object,AcmObject
           RET,ACM_ATTR_NATIVE,ACM_ATTR_WIN32                               */
gint
acm_get_attribute (AcmObject *acm_object)
{
  return ACM_ATTR_WIN32;
}


/******************************************************************************
*                                                                             *
* ja:ACMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    acm_object,AcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
acm_is_dialog_about (AcmObject *acm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  return acm_object && acm_object->hDrvr
        && SendDriverMessage (acm_object->hDrvr, ACMDM_DRIVER_ABOUT, -1, 0)
                                                        == MMSYSERR_NOERROR;
#else /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (USE_W32LDR) */
}


/*  ja:Aboutダイアログを表示する
    acm_object,AcmObject                                                    */
void
acm_dialog_about (AcmObject *acm_object)
{
#if defined (G_OS_WIN32) || defined (USE_W32LDR)
  if (acm_object && acm_object->hDrvr)
    SendDriverMessage (acm_object->hDrvr, ACMDM_DRIVER_ABOUT, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (USE_W32LDR) */
}
