require "disks_block"
require "disks_guess"
require "disks_detect"
require "vfs"
require "log"
# XXX: $paramで検出対象にしない/dev名の指定が必要


class VolumeFindError < RuntimeError
	def initialize(file, hints)
		@file = file
		@hints = hints
		@hints.empty? && @hints = ["none"]
	end
	def to_s
		return "Can't find volume with file: #{@file},  hints: #{@hints.join(",")}"
	end
end


class DiskManager

	def initialize(kmodule, initrdfs, arkfs, sysfs)
		@kmodule = kmodule
		@initrdfs = initrdfs
		@arkfs = arkfs
		@sysfs = sysfs
		BlockEntry.setInitrdFS(initrdfs)
		BlockEntry.setArkFS(arkfs)

		@block_entries = Array.new
		@detected = false
		@modules_loaded = false
	end


	class VolumeMatch
		def initialize(entry, num_matched_hints)
			@entry = entry
			@num_matched_hints = num_matched_hints
		end
		attr_reader :entry, :num_matched_hints

		def <=>(other)
			# マッチしたヒント数, hdd == flash > optical, external > internal, サイズ）

			# マッチしたヒント数
			if @num_matched_hints != other.num_matched_hints
				return @num_matched_hints <=> other.num_matched_hints
			end

			my_type = @entry.type
			other_type = other.entry.type

			# hdd == flash > optical
			if (my_type.include?("hdd") || my_type.include?("flash")) && other_type.include?("optical")
				return 1
			elsif my_type.include("optical") && (other_type.include?("hdd") || other_type.include?("flash"))
				return -1
			end

			# external > internal
			if my_type.include?("external") && other_type.include?("internal")
				return 1
			elsif my_type.include?("internal") && other_type.include?("external")
				return -1
			end

			# サイズ
			return @entry.size <=> other.entry.size

		end
	end


	def getAllVolume(file)
		$log.debug "searching all volumes by file #{file}"

		volumes = Array.new
		@block_entries.each {|b|
			if ! b.probed_succeeded?
				next
			end

			if !File.file?(b.point + "/" + file)
				next
			end
			volumes.push(b)
		}
		return volumes
	end


	def getBestVolumeByDirOrFile(path, hints)  # hints: Array
		$log.debug "searching best volume by file #{path}, hints #{hints.join(",")}"

		found_volumes = Array.new
		@block_entries.each {|b|
			b.probed_succeeded? || next

			if File.file?(b.point + "/#{path}")
				num_matched_hints = 50
			elsif File.directory?(b.point + "/#{path}")
				num_matched_hints = 0
			else
				next
			end

			hints.each {|h|
				b.type.include?(h) && num_matched_hints += 1
			}

			found_volumes.push( VolumeMatch.new(b, num_matched_hints) )
		}


		if found_volumes.empty?
			raise VolumeFindError.new(path, hints)
		else
			found_volumes.sort!
			found_volumes.reverse!
			vol = found_volumes[0].entry
			$log.debug "found volume #{vol.name} (file #{path}, type #{vol.type.join(",")})"
			return vol
		end
	end


	def getBestVolumeByFile(path, hints)  # hints: Array
		$log.debug "searching best volume by file #{path}, hints #{hints.join(",")}"

		found_volumes = Array.new
		@block_entries.each {|b|
			b.probed_succeeded? || next
			File.file?(b.point + "/#{path}") || next

			num_matched_hints = 0
			hints.each {|h|
				b.type.include?(h) && num_matched_hints += 1
			}

			found_volumes.push( VolumeMatch.new(b, num_matched_hints) )
		}


		if found_volumes.empty?
			raise VolumeFindError.new(path, hints)
		else
			found_volumes.sort!
			found_volumes.reverse!
			vol = found_volumes[0].entry
			$log.debug "found volume #{vol.name} (file #{path}, type #{vol.type.join(",")})"
			return vol
		end
	end


	def clean
		@block_entries.each {|b|
			if b.mountedOnTrial?
				b.umount
			end
		}
	end

	def each
		@block_entries.each {|b|
			yield b
		}
	end

end



$log.debug "#{File.basename(__FILE__, ".*")} loaded"
