/*
	vfddrv.c

	Virtual Floppy Disk kernel-mode driver for Windows NT platform
	Copyright (C) 2003 Kenji Kato
*/

#pragma warning(push,3)
#include <ntddk.h>
#include <ntdddisk.h>
#include <ntverp.h>
#pragma warning(pop)
// disable warnings : anonymous structure / unreferenced inline function
#pragma warning(disable: 4201 4514)

#include "vfd.h"
#include "imports.h"
#include "utility.h"

//
//	device extension for Virtual FD device
//
typedef struct _DEVICE_EXTENSION {
	//	Security context to access files on network drive
	PSECURITY_CLIENT_CONTEXT	scurity_context;

	//	IRP queue list
	LIST_ENTRY					list_head;
	KSPIN_LOCK					list_lock;

	//	device thread
	KEVENT						request_event;
	PVOID						thread_pointer;
	BOOLEAN 					terminate_thread;

	//	drive status
	ULONG						media_change_count;
	HANDLE						file_handle;

	ANSI_STRING					file_name;
	ULONG						file_size;
	BOOLEAN 					read_only;
} DEVICE_EXTENSION, *PDEVICE_EXTENSION;

//
//	Bytes per sector constant
//
#define VFD_BYTES_PER_SECTOR		512
#define VFD_SECTOR_ALIGNMENT_MASK	(VFD_BYTES_PER_SECTOR - 1)

//
//	Fill character for formatting media
//
#define MEDIA_FORMAT_FILL_DATA	0xf6

//
//	Media type constant table
//
#define MEDIA_TYPE_144	0
#define MEDIA_TYPE_720	1
#define MEDIA_TYPE_288	2

static DISK_GEOMETRY media_table[] = {
	{ { 80 }, F3_1Pt44_512, 2, 18, VFD_BYTES_PER_SECTOR },
	{ { 80 }, F3_720_512,	2,	9, VFD_BYTES_PER_SECTOR },
	{ { 80 }, F3_2Pt88_512, 2, 36, VFD_BYTES_PER_SECTOR }
};

//
//	Default symbolic link name (\??\VirtualFD)
//
#define VFD_DEVICE_SYMLINK		L"\\??\\"	VFD_DEVICE_BASENAME

//
//	Prototypes for stanard driver routines
//
NTSTATUS
DriverEntry (
	IN	PDRIVER_OBJECT	DriverObject,
	IN	PUNICODE_STRING RegistryPath
);

VOID
VfdUnloadDriver (
	IN PDRIVER_OBJECT	DriverObject
);

NTSTATUS
VfdCreateClose (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp
);

NTSTATUS
VfdReadWrite (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp
);

NTSTATUS
VfdDeviceControl (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp
);

//
//	Prototypes for private routines
//

VOID
VfdThread (
	IN PVOID				Context
);

NTSTATUS
VfdOpen (
	IN PDEVICE_OBJECT		DeviceObject,
	IN PIRP 				Irp
);

NTSTATUS
VfdClose (
	IN PDEVICE_EXTENSION	Extension,
	IN PIRP 				Irp
);

NTSTATUS
VfdFormat (
	IN PDEVICE_EXTENSION	Extension,
	IN PIRP 				Irp
);

#pragma code_seg("INIT")

//
//	Driver Entry routine
//
NTSTATUS
DriverEntry (
	IN PDRIVER_OBJECT	DriverObject,
	IN PUNICODE_STRING	RegistryPath)
{
	NTSTATUS			status;
	UNICODE_STRING		device_name;
	UNICODE_STRING		symbolic_link;
	PDEVICE_OBJECT		device_object;
	PDEVICE_EXTENSION	extension;
	HANDLE				thread_handle;

	KdPrint(("VirtualFD: DriverEntry - IN\n"));

	ASSERT(DriverObject);
	UNREFERENCED_PARAMETER(RegistryPath);

	//	Create the VirtualFD device object

	RtlInitUnicodeString(&device_name, VFD_DEVICE_FULLNAME);

	status = IoCreateDevice(
		DriverObject,
		sizeof(DEVICE_EXTENSION),
		&device_name,
		FILE_DEVICE_DISK,
		FILE_REMOVABLE_MEDIA | FILE_FLOPPY_DISKETTE,
		FALSE,
		&device_object);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: IoCreateDevice() 0x%x\n", status));

		return status;
	}

	//	Initialize the device object

	device_object->Flags |= DO_DIRECT_IO;

	extension = (PDEVICE_EXTENSION)device_object->DeviceExtension;

	RtlZeroMemory(extension, sizeof(DEVICE_EXTENSION));

	//	Create the default link(\??\VirtualFD)

	RtlInitUnicodeString(&symbolic_link, VFD_DEVICE_SYMLINK);

	status = IoCreateSymbolicLink(&symbolic_link, &device_name);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: IoCreateSymbolicLink() 0x%x\n", status));

		IoDeleteDevice(device_object);

		return status;
	}

	//	Prepare IRP queue list

	InitializeListHead(&extension->list_head);

	KeInitializeSpinLock(&extension->list_lock);

	KeInitializeEvent(
		&extension->request_event,
		SynchronizationEvent,
		FALSE);

	//	Create device dedicated thread

	extension->terminate_thread = FALSE;

	status = PsCreateSystemThread(
		&thread_handle,
		(ACCESS_MASK) 0L,
		NULL,
		NULL,
		NULL,
		VfdThread,
		device_object);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: PsCreateSystemThread() 0x%x\n", status));

		IoDeleteSymbolicLink(&symbolic_link);
		IoDeleteDevice(device_object);

		return status;
	}

	//	get reference to the thread

	status = ObReferenceObjectByHandle(
		thread_handle,
		THREAD_ALL_ACCESS,
		NULL,
		KernelMode,
		&extension->thread_pointer,
		NULL);

	ZwClose(thread_handle);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: ObReferenceObjectByHandle() 0x%x\n", status));

		extension->terminate_thread = TRUE;

		KeSetEvent(
			&extension->request_event,
			(KPRIORITY) 0,
			FALSE);

		IoDeleteSymbolicLink(&symbolic_link);
		IoDeleteDevice(device_object);

		return status;
	}

	//	Setup dispatch table

	DriverObject->MajorFunction[IRP_MJ_CREATE]		   = VfdCreateClose;
	DriverObject->MajorFunction[IRP_MJ_CLOSE]		   = VfdCreateClose;
	DriverObject->MajorFunction[IRP_MJ_READ]		   = VfdReadWrite;
	DriverObject->MajorFunction[IRP_MJ_WRITE]		   = VfdReadWrite;
	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = VfdDeviceControl;

	DriverObject->DriverUnload = VfdUnloadDriver;

	//	Done !

	KdPrint(("VirtualFD: DriverEntry - OUT\n"));

	return STATUS_SUCCESS;
}

#pragma code_seg("PAGE")

//
//	Driver unload routine
//
VOID
VfdUnloadDriver (
	IN PDRIVER_OBJECT DriverObject)
{
	PDEVICE_EXTENSION	extension;
	UNICODE_STRING		symbolic_link;

	KdPrint(("VirtualFD: VfdUnloadDriver - IN\n"));

	extension = (PDEVICE_EXTENSION)DriverObject->DeviceObject->DeviceExtension;

	//	Terminate the device dedicated thread

	extension->terminate_thread = TRUE;

	KeSetEvent(
		&extension->request_event,
		(KPRIORITY) 0,
		FALSE);

	KeWaitForSingleObject(
		extension->thread_pointer,
		Executive,
		KernelMode,
		FALSE,
		NULL);

	ObDereferenceObject(extension->thread_pointer);

	//	Delete security context object

	if (extension->scurity_context) {
		SeDeleteClientSecurity(extension->scurity_context);
		ExFreePool(extension->scurity_context);
	}

	//	Close the image file

	if (extension->file_handle) {
		ZwClose(extension->file_handle);
	}

	//	Free image path buffer

	if (extension->file_name.Buffer) {
		ExFreePool(extension->file_name.Buffer);
	}

	//	Delete Default Symbolic Link

	RtlInitUnicodeString(&symbolic_link, VFD_DEVICE_SYMLINK);
	IoDeleteSymbolicLink(&symbolic_link);

	//	Delete the device object

	IoDeleteDevice(DriverObject->DeviceObject);

	//	Done!

	KdPrint(("VirtualFD: VfdUnloadDriver - OUT\n"));

	return;
}

//
//	IRP_MJ_CREATE and IRP_MJ_CLOSE handler
//	Really nothing to do here...
//
NTSTATUS
VfdCreateClose (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp)
{
	UNREFERENCED_PARAMETER(DeviceObject);

	Irp->IoStatus.Status = STATUS_SUCCESS;
	Irp->IoStatus.Information = FILE_OPENED;

	IoCompleteRequest(Irp, IO_NO_INCREMENT);

	return STATUS_SUCCESS;
}

#pragma code_seg()

//
//	IRP_MJ_READ and IRP_MJ_WRITE handler
//	Insert the IRP into queue list.
//	Actual read/write operation is performed by device thread
//
#define IO_READ_OFF(p)	(p)->Parameters.Read.ByteOffset.QuadPart
#define IO_READ_LEN(p)	(p)->Parameters.Read.Length

NTSTATUS
VfdReadWrite (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp)
{
	PDEVICE_EXTENSION	extension;
	PIO_STACK_LOCATION	io_stack;

	extension = (PDEVICE_EXTENSION)DeviceObject->DeviceExtension;

	io_stack = IoGetCurrentIrpStackLocation(Irp);

	//	Check if an image file is mounted

	if (!extension->file_handle)	{

		KdPrint(("VfdReadWrite: No media in device\n"));

		Irp->IoStatus.Status = STATUS_NO_MEDIA_IN_DEVICE;
		Irp->IoStatus.Information = 0;

		IoCompleteRequest(Irp, IO_NO_INCREMENT);

		return STATUS_NO_MEDIA_IN_DEVICE;
	}

	//
	// Check if write operation is allowed
	//
	if (io_stack->MajorFunction == IRP_MJ_WRITE && extension->read_only) {

		KdPrint(("IRP_MJ_WRITE: Media write protected\n"));

		Irp->IoStatus.Status = STATUS_MEDIA_WRITE_PROTECTED;
		Irp->IoStatus.Information = 0;

		IoCompleteRequest(Irp, IO_NO_INCREMENT);

		return STATUS_MEDIA_WRITE_PROTECTED;
	}

	//
	// Check for invalid parameters.  It is an error for the starting offset
	// + length to go past the end of the partition, or for the length or
	// offset to not be a proper multiple of the sector size.
	//
	// Others are possible, but we don't check them since we trust the
	// file system and they aren't deadly.
	//
	if (IO_READ_OFF(io_stack) + IO_READ_LEN(io_stack) > extension->file_size) {

		KdPrint(("Offset:%I64u + Length:%u goes past the media size %lu\n",
			IO_READ_OFF(io_stack),
			IO_READ_LEN(io_stack),
			extension->file_size));

		Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
		Irp->IoStatus.Information = 0;

		IoCompleteRequest(Irp, IO_NO_INCREMENT);

		return STATUS_INVALID_PARAMETER;
	}

	if ((IO_READ_LEN(io_stack) & VFD_SECTOR_ALIGNMENT_MASK) ||
		(IO_READ_OFF(io_stack) & VFD_SECTOR_ALIGNMENT_MASK)) {

		KdPrint(("Invalid Alignment Offset:%I64u Length:%u\n",
			IO_READ_OFF(io_stack),
			IO_READ_LEN(io_stack)));

		Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
		Irp->IoStatus.Information = 0;

		IoCompleteRequest(Irp, IO_NO_INCREMENT);

		return STATUS_INVALID_PARAMETER;
	}

	//	If read/write data length is 0, we are done

	if (IO_READ_LEN(io_stack) == 0) {
		Irp->IoStatus.Status = STATUS_SUCCESS;
		Irp->IoStatus.Information = 0;

		IoCompleteRequest(Irp, IO_NO_INCREMENT);

		return STATUS_SUCCESS;
	}

	//	It seems that actual read/write operation is required
	//	so mark the IRP as pending, insert the IRP into queue list
	//	then signal the device thread to perform the operation

	IoMarkIrpPending(Irp);

	ExInterlockedInsertTailList(
		&extension->list_head,
		&Irp->Tail.Overlay.ListEntry,
		&extension->list_lock);

	KeSetEvent(
		&extension->request_event,
		(KPRIORITY) 0,
		FALSE);

	return STATUS_PENDING;
}

//
//	Handles various IOCTL commands
//
#define IO_INPUTLEN(p)	(p)->Parameters.DeviceIoControl.InputBufferLength
#define IO_OUTPUTLEN(p)	(p)->Parameters.DeviceIoControl.OutputBufferLength
#define IO_CTRLCODE(p)	(p)->Parameters.DeviceIoControl.IoControlCode

NTSTATUS
VfdDeviceControl (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp)
{
	PDEVICE_EXTENSION	extension;
	PIO_STACK_LOCATION	io_stack;
	NTSTATUS			status;

	extension = (PDEVICE_EXTENSION)DeviceObject->DeviceExtension;
	io_stack = IoGetCurrentIrpStackLocation(Irp);

	switch (IO_CTRLCODE(io_stack)) {
	case IOCTL_VFD_OPEN_FILE:
		{
			PVFD_FILE_INFO				file_info;
			SECURITY_QUALITY_OF_SERVICE sqos;

			//	Mount an image file as a virtual floppy disk.
			//	Only a few checks are done here.
			//	Actual operation is done in device thread

			//	Check media status

			if (extension->file_handle) {
				Irp->IoStatus.Information = 0;
				status = STATUS_DEVICE_BUSY;
				break;
			}

			//	Check input parameter length

			if (IO_INPUTLEN(io_stack) < sizeof(VFD_FILE_INFO)) {
				Irp->IoStatus.Information = 0;
				status = STATUS_INVALID_PARAMETER;
				break;
			}

			file_info = (PVFD_FILE_INFO)Irp->AssociatedIrp.SystemBuffer;

			if (IO_INPUTLEN(io_stack) <
				sizeof(VFD_FILE_INFO) + file_info->FileNameLength - sizeof(CHAR))
			{
				Irp->IoStatus.Information = 0;
				status = STATUS_INVALID_PARAMETER;
				break;
			}

			//	File name can be omitted only if an image has been mounted
			//	before and the path is still stored in device extension

			if (file_info->FileNameLength == 0 &&
				extension->file_name.Length == 0)
			{
				Irp->IoStatus.Information = 0;
				status = STATUS_INVALID_PARAMETER;
				break;
			}

			//	set security context to match the calling process' context

			if (extension->scurity_context != NULL) {
				SeDeleteClientSecurity(extension->scurity_context);
			}
			else {
				extension->scurity_context =
					ExAllocatePool(NonPagedPool, sizeof(SECURITY_CLIENT_CONTEXT));
			}

			RtlZeroMemory(&sqos, sizeof(SECURITY_QUALITY_OF_SERVICE));

			sqos.Length = sizeof(SECURITY_QUALITY_OF_SERVICE);
			sqos.ImpersonationLevel = SecurityImpersonation;
			sqos.ContextTrackingMode = SECURITY_STATIC_TRACKING;
			sqos.EffectiveOnly = FALSE;

			SeCreateClientSecurity(
				PsGetCurrentThread(),
				&sqos,
				FALSE,
				extension->scurity_context);

			// The device thread will do the rest of the job

			status = STATUS_PENDING;
		}
		break;

	case IOCTL_VFD_CLOSE_FILE:
		//	Unmount current image file
		//	Only status check is done here.
		//	Actual operation is done in device thread.

		if (!extension->file_handle) {
			Irp->IoStatus.Information = 0;
			status = STATUS_NO_MEDIA_IN_DEVICE;
		}
		else {
			status = STATUS_PENDING;
		}
		break;

	case IOCTL_VFD_QUERY_FILE:
		//	Returns current image file information
		{
			PVFD_FILE_INFO file_info;

			//	Check output buffer length

			if (IO_OUTPUTLEN(io_stack) <
				sizeof(VFD_FILE_INFO) + extension->file_name.Length - sizeof(CHAR))
			{
				Irp->IoStatus.Information = 0;
				status = STATUS_BUFFER_TOO_SMALL;
				break;
			}

			//	Store file information

			file_info = (PVFD_FILE_INFO)Irp->AssociatedIrp.SystemBuffer;

			file_info->FileNameLength	= extension->file_name.Length;
			file_info->FileSize			= extension->file_size;
			file_info->ReadOnly			= extension->read_only;

			if (extension->file_name.Length) {
				RtlCopyMemory(
					file_info->FileName,
					extension->file_name.Buffer,
					extension->file_name.Length);
			}

			Irp->IoStatus.Information
				= sizeof(VFD_FILE_INFO) + file_info->FileNameLength - sizeof(CHAR);

			status = STATUS_SUCCESS;
		}
		break;

	case IOCTL_DISK_CHECK_VERIFY:
	case IOCTL_STORAGE_CHECK_VERIFY:
	case IOCTL_STORAGE_CHECK_VERIFY2:
		//	Media verification is always unnecessary because the virtual
		//	media cannot be removed without the file system knowing it.

		if (IO_OUTPUTLEN(io_stack) < sizeof(ULONG)) {
			Irp->IoStatus.Information = 0;
		}
		else {
			*(PULONG)Irp->AssociatedIrp.SystemBuffer
				= extension->media_change_count;

			Irp->IoStatus.Information = sizeof(ULONG);
		}

		status = STATUS_SUCCESS;
		break;

	case IOCTL_DISK_FORMAT_TRACKS:
	case IOCTL_DISK_FORMAT_TRACKS_EX:
		//	Only several checks are done here
		//	Actual operation is done by the device thread
		{
			PFORMAT_PARAMETERS param;
			PDISK_GEOMETRY geometry;

			//	Check media status

			if (!extension->file_handle) {
				Irp->IoStatus.Information = 0;
				status = STATUS_NO_MEDIA_IN_DEVICE;
				break;
			}

			//	Media is writable?

			if (extension->read_only) {
				Irp->IoStatus.Information = 0;
				status = STATUS_MEDIA_WRITE_PROTECTED;
				break;
			}

			//	Check input parameter size

			if (IO_INPUTLEN(io_stack) < sizeof(FORMAT_PARAMETERS)) {
				Irp->IoStatus.Information = 0;
				status = STATUS_INVALID_PARAMETER;
				break;
			}

			//	Choose appropriate DISK_GEOMETRY for current image size

			if (extension->file_size == VFD_FILESIZE_720KB) {
				geometry = &media_table[MEDIA_TYPE_720];
			}
			else if (extension->file_size == VFD_FILESIZE_2P88MB) {
				geometry = &media_table[MEDIA_TYPE_288];
			}
			else {
				geometry = &media_table[MEDIA_TYPE_144];
			}

			//	Input parameter sanity check

			param = (PFORMAT_PARAMETERS)Irp->AssociatedIrp.SystemBuffer;

			if ((param->StartHeadNumber 	> geometry->TracksPerCylinder - 1)		||
				(param->EndHeadNumber		> geometry->TracksPerCylinder - 1)		||
				(param->StartCylinderNumber > (ULONG)geometry->Cylinders.QuadPart)	||
				(param->EndCylinderNumber	> (ULONG)geometry->Cylinders.QuadPart)	||
				(param->EndCylinderNumber	< param->StartCylinderNumber))
			{
				Irp->IoStatus.Information = 0;
				status = STATUS_INVALID_PARAMETER;
				break;
			}

			//	If this is an EX request then make a couple of extra checks

			if (IO_CTRLCODE(io_stack) ==
				IOCTL_DISK_FORMAT_TRACKS_EX)
			{
				PFORMAT_EX_PARAMETERS exparam;
				ULONG paramsize;

				if (IO_INPUTLEN(io_stack) < sizeof(FORMAT_EX_PARAMETERS)) {
					Irp->IoStatus.Information = 0;
					status = STATUS_INVALID_PARAMETER;
					break;
				}

				exparam = (PFORMAT_EX_PARAMETERS)Irp->AssociatedIrp.SystemBuffer;

				paramsize = sizeof(FORMAT_EX_PARAMETERS)
					+ exparam->SectorsPerTrack * sizeof(USHORT)
					- sizeof(USHORT);

				if (IO_INPUTLEN(io_stack) < paramsize ||
					exparam->FormatGapLength > geometry->SectorsPerTrack ||
					exparam->SectorsPerTrack != geometry->SectorsPerTrack)
				{
					Irp->IoStatus.Information = 0;
					status = STATUS_INVALID_PARAMETER;
					break;
				}
			}

			status = STATUS_PENDING;
		}
		break;

	case IOCTL_DISK_GET_DRIVE_GEOMETRY:
		//	Returns the geometry of current media

		if (!extension->file_handle) {
			Irp->IoStatus.Information = 0;
			status = STATUS_NO_MEDIA_IN_DEVICE;
			break;
		}

		//	The rest is the same as xxx_GET_MEDIA_TYPES

	case IOCTL_DISK_GET_MEDIA_TYPES:
	case IOCTL_STORAGE_GET_MEDIA_TYPES:
		//	Return *the last mounted* disk geometry, although xxx_GET_MEDIA_TYPES
		//	commands are supposed to return all supported media types.
		//	This makes the matter much simpler...;-)
		//	If an image has not been mounted, 1.44MB media is assumed.

		//	Check output buffer size

		if (IO_OUTPUTLEN(io_stack) < sizeof(DISK_GEOMETRY)) {
			Irp->IoStatus.Information = 0;
			status = STATUS_BUFFER_TOO_SMALL;
			break;
		}

		//	Copy appropriate DISK_GEOMETRY into output buffer

		RtlCopyMemory(
			Irp->AssociatedIrp.SystemBuffer,
			(extension->file_size == VFD_FILESIZE_720KB)
				? &media_table[MEDIA_TYPE_720]
				: ((extension->file_size == VFD_FILESIZE_2P88MB)
					? &media_table[MEDIA_TYPE_288]
					: &media_table[MEDIA_TYPE_144]),
			sizeof(DISK_GEOMETRY));

		Irp->IoStatus.Information = sizeof(DISK_GEOMETRY);

		status = STATUS_SUCCESS;
		break;

	case IOCTL_DISK_GET_LENGTH_INFO:
		//	Return disk length information

		//	Check media status

		if (!extension->file_handle) {
			Irp->IoStatus.Information = 0;
			status = STATUS_NO_MEDIA_IN_DEVICE;
			break;
		}

		if (IO_OUTPUTLEN(io_stack) < sizeof(GET_LENGTH_INFORMATION)) {
			Irp->IoStatus.Information = 0;
			status = STATUS_BUFFER_TOO_SMALL;
			break;
		}

		((PGET_LENGTH_INFORMATION)Irp->AssociatedIrp.SystemBuffer)->Length.QuadPart
			= extension->file_size;

		Irp->IoStatus.Information = sizeof(GET_LENGTH_INFORMATION);

		status = STATUS_SUCCESS;
		break;

	case IOCTL_DISK_IS_WRITABLE:
		//	Checks if current media is writable

		Irp->IoStatus.Information = 0;

		if (!extension->file_handle) {
			status = STATUS_NO_MEDIA_IN_DEVICE;
		}
		else if (extension->read_only) {
			status = STATUS_MEDIA_WRITE_PROTECTED;
		}
		else {
			status = STATUS_SUCCESS;
		}
		break;

	case IOCTL_DISK_MEDIA_REMOVAL:
	case IOCTL_STORAGE_MEDIA_REMOVAL:
		//	Since removal lock is irrelevant for virtual disks,
		//	there's really nothing to do here...

		Irp->IoStatus.Information = 0;
		status = STATUS_SUCCESS;
		break;

	default:
		//	Unknown IOCTL request

		Irp->IoStatus.Information = 0;
		status = STATUS_INVALID_DEVICE_REQUEST;
	}

	DBG_IOCTL_STATUS(IO_CTRLCODE(io_stack), status);

	if (status == STATUS_PENDING) {
		//	Let the device thread perform the operation

		IoMarkIrpPending(Irp);

		ExInterlockedInsertTailList(
			&extension->list_head,
			&Irp->Tail.Overlay.ListEntry,
			&extension->list_lock);

		KeSetEvent(
			&extension->request_event,
			(KPRIORITY) 0,
			FALSE);
	}
	else {
		//	complete the operation

		Irp->IoStatus.Status = status;
		IoCompleteRequest(Irp, IO_NO_INCREMENT);
	}

	return status;
}

//
//	Device dedicated thread routine
//	Handles read, write, mount, unmount, and format operation
//
VOID
VfdThread (
	IN PVOID Context)
{
	PDEVICE_OBJECT		device_object;
	PDEVICE_EXTENSION	extension;
	PLIST_ENTRY 		request;
	PIRP				irp;
	PIO_STACK_LOCATION	io_stack;

	ASSERT(Context != NULL);

	device_object = (PDEVICE_OBJECT)Context;

	extension = (PDEVICE_EXTENSION)device_object->DeviceExtension;

	KeSetPriorityThread(KeGetCurrentThread(), LOW_REALTIME_PRIORITY);

	for (;;) {
		//	wait for the request event to be signalled
		KeWaitForSingleObject(
			&extension->request_event,
			Executive,
			KernelMode,
			FALSE,
			NULL);

		//	terminate request ?
		if (extension->terminate_thread) {
			PsTerminateSystemThread(STATUS_SUCCESS);
		}

		//	perform requested task

		while ((request = ExInterlockedRemoveHeadList(
			&extension->list_head, &extension->list_lock)) != NULL)
		{
			irp = CONTAINING_RECORD(request, IRP, Tail.Overlay.ListEntry);

			io_stack = IoGetCurrentIrpStackLocation(irp);

			switch (io_stack->MajorFunction) {
			case IRP_MJ_READ:
				{
					PVOID buf;

					buf = MmGetSystemAddressForMdlPrettySafe(
						irp->MdlAddress, NormalPagePriority);

					if (!buf) {
						irp->IoStatus.Status = STATUS_DRIVER_INTERNAL_ERROR;
						irp->IoStatus.Information = 0;
						break;
					}

					irp->IoStatus.Status = ZwReadFile(
						extension->file_handle,
						NULL,
						NULL,
						NULL,
						&irp->IoStatus,
						buf,
						io_stack->Parameters.Read.Length,
						&io_stack->Parameters.Read.ByteOffset,
						NULL);

					if (NT_SUCCESS(irp->IoStatus.Status)) {
						irp->IoStatus.Information = io_stack->Parameters.Read.Length;
					}
					else {
						irp->IoStatus.Information = 0;
					}
				}
				break;

			case IRP_MJ_WRITE:
				{
					PVOID buf;

					buf = MmGetSystemAddressForMdlPrettySafe(
						irp->MdlAddress, NormalPagePriority);

					if (!buf) {
						irp->IoStatus.Status = STATUS_DRIVER_INTERNAL_ERROR;
						irp->IoStatus.Information = 0;
						break;
					}

					irp->IoStatus.Status = ZwWriteFile(
						extension->file_handle,
						NULL,
						NULL,
						NULL,
						&irp->IoStatus,
						buf,
						io_stack->Parameters.Write.Length,
						&io_stack->Parameters.Write.ByteOffset,
						NULL);

					if (NT_SUCCESS(irp->IoStatus.Status)) {
						irp->IoStatus.Information = io_stack->Parameters.Read.Length;
					}
					else {
						irp->IoStatus.Information = 0;
					}
				}
				break;

			case IRP_MJ_DEVICE_CONTROL:
				switch (IO_CTRLCODE(io_stack)) {
				case IOCTL_VFD_OPEN_FILE:

					SeImpersonateClient(extension->scurity_context, NULL);

					irp->IoStatus.Status = VfdOpen(device_object, irp);

					PsRevertToSelf();

					break;

				case IOCTL_VFD_CLOSE_FILE:
					irp->IoStatus.Status = VfdClose(extension, irp);
					break;

				case IOCTL_DISK_FORMAT_TRACKS:
				case IOCTL_DISK_FORMAT_TRACKS_EX:
					irp->IoStatus.Status = VfdFormat(extension, irp);
					break;

				default:
					//	This shouldn't happen...
					irp->IoStatus.Status = STATUS_DRIVER_INTERNAL_ERROR;
				}

				DBG_IOCTL_STATUS(IO_CTRLCODE(io_stack), irp->IoStatus.Status);
				break;

			default:
				//	This shouldn't happen...
				irp->IoStatus.Status = STATUS_DRIVER_INTERNAL_ERROR;
			}

			IoCompleteRequest(
				irp,
				(CCHAR)(NT_SUCCESS(irp->IoStatus.Status)
					? IO_DISK_INCREMENT
					: IO_NO_INCREMENT));
		} // while

	} // for (;;)
}

#pragma code_seg("PAGE")

//
//	Mount an image file as a virtual floppy disk
//
NTSTATUS
VfdOpen (
	IN PDEVICE_OBJECT	DeviceObject,
	IN PIRP 			Irp)
{
	PDEVICE_EXTENSION			extension;
	PVFD_FILE_INFO				file_info;
	OBJECT_ATTRIBUTES			attributes;
	UNICODE_STRING				unicode_name;
	FILE_BASIC_INFORMATION		file_basic;
	FILE_STANDARD_INFORMATION	file_standard;
	FILE_ALIGNMENT_INFORMATION	file_alignment;
	NTSTATUS					status;


	KdPrint(("VirtualFD: VfdOpen - IN\n"));

	ASSERT(DeviceObject != NULL);
	ASSERT(Irp != NULL);

	extension = (PDEVICE_EXTENSION)DeviceObject->DeviceExtension;
	file_info = (PVFD_FILE_INFO)Irp->AssociatedIrp.SystemBuffer;

	if (file_info->FileNameLength) {

		//	Store the new filename in the device extension

		if (file_info->FileNameLength > extension->file_name.MaximumLength) {

			//	expand the filename buffer

			if (extension->file_name.Buffer) {
				ExFreePool(extension->file_name.Buffer);
				RtlZeroMemory(&extension->file_name, sizeof(ANSI_STRING));
			}

			extension->file_name.Buffer =
				ExAllocatePool(NonPagedPool, file_info->FileNameLength);

			if (extension->file_name.Buffer == NULL) {
				KdPrint(("VirtualFD: Can't allocate memory for image path\n"));

				Irp->IoStatus.Information = 0;
				return STATUS_INSUFFICIENT_RESOURCES;
			}

			extension->file_name.MaximumLength = file_info->FileNameLength;
		}

		extension->file_name.Length = file_info->FileNameLength;

		RtlCopyMemory(
			extension->file_name.Buffer,
			file_info->FileName,
			file_info->FileNameLength);
	}

	if (extension->file_name.Length == 0) {
		//	shouldn't come here, but just in case...
		KdPrint(("VirtualFD: No image file name to remount.\n"));

		Irp->IoStatus.Information = 0;
		return STATUS_INVALID_PARAMETER;
	}

	//	Convert filename into Unicode string

	status = RtlAnsiStringToUnicodeString(
		&unicode_name, 	&extension->file_name, TRUE);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: Failed to convert filename to UNICODE\n"));

		Irp->IoStatus.Information = 0;
		return status;
	}

	//	OK, now file name is clear so try to open it

	extension->read_only = file_info->ReadOnly;

	InitializeObjectAttributes(
		&attributes,
		&unicode_name,
		OBJ_CASE_INSENSITIVE,
		NULL,
		NULL);

	status = ZwCreateFile(
		&extension->file_handle,
		extension->read_only ? GENERIC_READ : GENERIC_READ | GENERIC_WRITE,
		&attributes,
		&Irp->IoStatus,
		NULL,
		FILE_ATTRIBUTE_NORMAL,
		extension->read_only ? FILE_SHARE_READ : 0,
		FILE_OPEN,
		FILE_NON_DIRECTORY_FILE |
		FILE_RANDOM_ACCESS |
		FILE_NO_INTERMEDIATE_BUFFERING |
		FILE_SYNCHRONOUS_IO_NONALERT,
		NULL,
		0);

	if (status == STATUS_ACCESS_DENIED) {

		if (!extension->read_only) {
			//	Failed to open read-write, so try read-only

			status = ZwCreateFile(
				&extension->file_handle,
				GENERIC_READ,
				&attributes,
				&Irp->IoStatus,
				NULL,
				FILE_ATTRIBUTE_NORMAL,
				FILE_SHARE_READ,
				FILE_OPEN,
				FILE_NON_DIRECTORY_FILE |
				FILE_RANDOM_ACCESS |
				FILE_NO_INTERMEDIATE_BUFFERING |
				FILE_SYNCHRONOUS_IO_NONALERT,
				NULL,
				0);

			if (NT_SUCCESS(status)) {
				//	succeeded - file is read-only
				extension->read_only = TRUE;
			}
		}
	}
	else if (status == STATUS_OBJECT_NAME_NOT_FOUND ||
		status == STATUS_NO_SUCH_FILE) {
		//	The file doesn't seem to exist so try creating it

		LARGE_INTEGER	file_size;

		if (extension->read_only ||
			(file_info->FileSize != VFD_FILESIZE_720KB &&
			file_info->FileSize != VFD_FILESIZE_1P44MB &&
			file_info->FileSize != VFD_FILESIZE_2P88MB))
		{
			//	cannot create with these parameters
			KdPrint(("VirtualFD: Cannot create a new file.\n"));

			RtlFreeUnicodeString(&unicode_name);

			return status;
		}

		file_size.QuadPart = file_info->FileSize;

		//	create a new file

		status = ZwCreateFile(
			&extension->file_handle,
			GENERIC_READ | GENERIC_WRITE,
			&attributes,
			&Irp->IoStatus,
			&file_size,
			FILE_ATTRIBUTE_NORMAL,
			0,
			FILE_OPEN_IF,
			FILE_NON_DIRECTORY_FILE |
			FILE_RANDOM_ACCESS |
			FILE_NO_INTERMEDIATE_BUFFERING |
			FILE_SYNCHRONOUS_IO_NONALERT,
			NULL,
			0);

		if (!NT_SUCCESS(status)) {
			KdPrint(("VirtualFD: Failed to create file\n"));

			RtlFreeUnicodeString(&unicode_name);

			return status;
		}

		if (Irp->IoStatus.Information == FILE_CREATED) {

			//	Adjust file size

			FILE_END_OF_FILE_INFORMATION	file_eof;

			file_eof.EndOfFile.QuadPart = file_info->FileSize;

			status = ZwSetInformationFile(
				extension->file_handle,
				&Irp->IoStatus,
				&file_eof,
				sizeof(FILE_END_OF_FILE_INFORMATION),
				FileEndOfFileInformation);

			if (!NT_SUCCESS(status)) {
				KdPrint(("VirtualFD: Failed to adjust file size\n"));

				ZwClose(extension->file_handle);
				extension->file_handle = NULL;
				RtlFreeUnicodeString(&unicode_name);

				return status;
			}
		}
	}

	RtlFreeUnicodeString(&unicode_name);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: File open error - 0x%x\n", status));

		return status;
	}

	// The NT cache manager can deadlock if a filesystem that is using the cache
	// manager is used in a virtual disk that stores its file on a filesystem
	// that is also using the cache manager, this is why we open the file with
	// FILE_NO_INTERMEDIATE_BUFFERING above, however if the file is compressed
	// or encrypted NT will not honor this request and cache it anyway since it
	// need to store the decompressed/unencrypted data somewhere, therefor we put
	// an extra check here and don't alow disk images to be compressed/encrypted.

	status = ZwQueryInformationFile(
		extension->file_handle,
		&Irp->IoStatus,
		&file_basic,
		sizeof(FILE_BASIC_INFORMATION),
		FileBasicInformation);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: ZwQueryInformationFile - FILE_BASIC_INFORMATION\n"));

		ZwClose(extension->file_handle);
		extension->file_handle = NULL;

		return status;
	}

	if (file_basic.FileAttributes
		& (FILE_ATTRIBUTE_COMPRESSED | FILE_ATTRIBUTE_ENCRYPTED))
	{
		KdPrint(("VirtualFD: Image file is compressed and/or encrypted\n"));

		ZwClose(extension->file_handle);
		extension->file_handle = NULL;

		Irp->IoStatus.Information = 0;
		return STATUS_ACCESS_DENIED;
	}

	//	Determine logical file size (supports sparse file)

	status = ZwQueryInformationFile(
		extension->file_handle,
		&Irp->IoStatus,
		&file_standard,
		sizeof(FILE_STANDARD_INFORMATION),
		FileStandardInformation);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: ZwQueryInformationFile - FILE_STANDARD_INFORMATION\n"));

		ZwClose(extension->file_handle);
		extension->file_handle = NULL;

		return status;
	}

	//	Check actual file size

	if (file_standard.EndOfFile.QuadPart != VFD_FILESIZE_720KB &&
		file_standard.EndOfFile.QuadPart != VFD_FILESIZE_1P44MB &&
		file_standard.EndOfFile.QuadPart != VFD_FILESIZE_2P88MB)
	{
		KdPrint(("VirtualFD: Invalid file size\n"));

		ZwClose(extension->file_handle);
		extension->file_handle = NULL;

		Irp->IoStatus.Information = 0;
		return STATUS_ACCESS_DENIED;
	}

	extension->file_size = (ULONG)file_standard.EndOfFile.QuadPart;

	//	Retrieve alignment requirement

	status = ZwQueryInformationFile(
		extension->file_handle,
		&Irp->IoStatus,
		&file_alignment,
		sizeof(FILE_ALIGNMENT_INFORMATION),
		FileAlignmentInformation);

	if (!NT_SUCCESS(status)) {
		KdPrint(("VirtualFD: ZwQueryInformationFile - FILE_ALIGNMENT_INFORMATION\n"));

		ZwClose(extension->file_handle);
		extension->file_handle = NULL;

		return status;
	}

	DeviceObject->AlignmentRequirement = file_alignment.AlignmentRequirement;

	//	Adjust device characteristics

	if (extension->read_only) {
		DeviceObject->Characteristics |= FILE_READ_ONLY_DEVICE;
	}
	else {
		DeviceObject->Characteristics &= ~FILE_READ_ONLY_DEVICE;
	}

	//	Adjust media status

	extension->media_change_count++;

	//	Done!

	Irp->IoStatus.Information = 0;

	KdPrint(("VirtualFD: VfdOpen - OUT\n"));
	return STATUS_SUCCESS;
}

//
//	Unmount current image file
//
NTSTATUS
VfdClose (
	IN PDEVICE_EXTENSION	Extension,
	IN PIRP 				Irp
	)
{
	KdPrint(("VirtualFD: VfdClose - IN\n"));

	ASSERT(Extension != NULL);
	ASSERT(Irp != NULL);

	ZwClose(Extension->file_handle);
	Extension->file_handle = NULL;

	Irp->IoStatus.Information = 0;

	KdPrint(("VirtualFD: VfdClose - OUT\n"));
	return STATUS_SUCCESS;
}

//
//	Format tracks
//	Actually, just fills specified range of tracks with fill characters
//
NTSTATUS
VfdFormat (
	IN PDEVICE_EXTENSION	Extension,
	IN PIRP 				Irp)
{
	PFORMAT_PARAMETERS	param;
	PDISK_GEOMETRY		geometry;
	ULONG				track_length;
	PUCHAR				format_buffer;
	LARGE_INTEGER		start_offset;
	LARGE_INTEGER		end_offset;
	NTSTATUS			status;

	KdPrint(("VirtualFD: VfdFormat - IN\n"));

	ASSERT(Extension != NULL);
	ASSERT(Irp != NULL);

	param = (PFORMAT_PARAMETERS)Irp->AssociatedIrp.SystemBuffer;

	if (Extension->file_size == VFD_FILESIZE_720KB) {
		geometry = &media_table[MEDIA_TYPE_720];
	}
	else if (Extension->file_size == VFD_FILESIZE_2P88MB) {
		geometry = &media_table[MEDIA_TYPE_288];
	}
	else {
		geometry = &media_table[MEDIA_TYPE_144];
	}

	track_length = geometry->BytesPerSector * geometry->SectorsPerTrack;

	format_buffer = ExAllocatePool(PagedPool, track_length);

	if (format_buffer == NULL) {
		KdPrint(("VirtualFD: cannot allocate format buffer\n"));
		Irp->IoStatus.Information = 0;
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	RtlFillMemory(format_buffer, track_length, MEDIA_FORMAT_FILL_DATA);

	start_offset.QuadPart =
		param->StartCylinderNumber * geometry->TracksPerCylinder * track_length +
		param->StartHeadNumber * track_length;

	end_offset.QuadPart =
		param->EndCylinderNumber * geometry->TracksPerCylinder * track_length +
		param->EndHeadNumber * track_length;

	do {
		status = ZwWriteFile(
			Extension->file_handle,
			NULL,
			NULL,
			NULL,
			&Irp->IoStatus,
			format_buffer,
			track_length,
			&start_offset,
			NULL);

		if (!NT_SUCCESS(status)) {
			KdPrint(("VirtualFD: ZwWriteFile\n"));
			Irp->IoStatus.Information = 0;
			break;
		}

		start_offset.QuadPart += track_length;
	}
	while (start_offset.QuadPart <= end_offset.QuadPart);

	ExFreePool(format_buffer);

	Irp->IoStatus.Information = 0;

	KdPrint(("VirtualFD: VfdFormat - OUT\n"));
	return status;
}

//	End Of File
