/*
	utility.c

	Kernel mode driver utility routines
	Copyright (C) 2003 Kenji Kato
*/

#pragma warning(push,3)
#include <ntddk.h>
#include <ntdddisk.h>
#include <ntverp.h>
#pragma warning(pop)
// disable warnings : anonymous structure / unreferenced inline function
#pragma warning(disable: 4201 4514)

#include "imports.h"
#include "utility.h"

static ULONG	OsMajorVersion 	= 0;
static ULONG	OsMinorVersion 	= 0;
static ULONG	OsBuildNumber 	= 0;

//
//	Store current OS version for later use
//
VOID
StoreCurrentOsVersion()
{
  	PsGetVersion(
		&OsMajorVersion,
		&OsMinorVersion,
		&OsBuildNumber,
		NULL
	);
}

//
//	Substitute for MmGetSystemAddressForMdlSafe
//	for NT 4.0 DDK does not provide its equivqlent.
//	Originally written by Bruce Engle for filedisk
//  program (http://www.acc.umu.se/~bosse/).
//
PVOID
MmGetSystemAddressForMdlPrettySafe (
	IN PMDL 			Mdl,
	IN MM_PAGE_PRIORITY	Priority)
{
	if (OsMajorVersion == 0) {
		StoreCurrentOsVersion();
	}

	if (OsMajorVersion >= 5) {
		return MmGetSystemAddressForMdlSafe(Mdl, Priority);
	}
	else {
		CSHORT	MdlMappingCanFail;
		PVOID	MappedSystemVa;

		MdlMappingCanFail = (CSHORT)(Mdl->MdlFlags & MDL_MAPPING_CAN_FAIL);

		Mdl->MdlFlags |= MDL_MAPPING_CAN_FAIL;

		MappedSystemVa = MmGetSystemAddressForMdl(Mdl);

		if (!MdlMappingCanFail) {
			Mdl->MdlFlags &= ~MDL_MAPPING_CAN_FAIL;
		}

		return MappedSystemVa;
	}
}

#if DBG
//
// Output IO Control code and result status strings
//
VOID
DBG_IOCTL_STATUS(
	ULONG		ctrl_code,
	NTSTATUS	status)
{
	PCSTR 	ioctl;
	PCSTR 	stat;
	CHAR 	severity;

	ioctl 	= IoControlCodeToStr(ctrl_code);
	stat 	= NtStatusToStr(status);

	switch (status & 0xf0000000) {
	case 0x00000000:	severity = 'S';	break;
	case 0x40000000:	severity = 'I';	break;
	case 0x80000000:	severity = 'W';	break;
	case 0xC0000000:	severity = 'E';	break;
	default:			severity = '?';
	}

	if (ioctl) {
		if (stat) {
			KdPrint(("%s - (%c)%s\n", ioctl, severity, stat));
		}
		else {
			KdPrint(("%s - 0x%x\n", ioctl, status));
		}
	}
	else {
		if (stat) {
			KdPrint((
				"Unknown IoControlCode %#x - (%c)%s\n",
				ctrl_code, severity, stat));
		}
		else {
			KdPrint((
				"Unknown IoControlCode 0x%x - 0x%x\n",
				ctrl_code, status));
		}
	}
}

PCSTR
IoControlCodeToStr(
	ULONG ctrl_code)
{
	switch (ctrl_code) {
	//
	//	VFD control codes
	//
	case CTL_CODE(IOCTL_DISK_BASE, 0x800, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS):
		return "IOCTL_VFD_OPEN_FILE";

	case CTL_CODE(IOCTL_DISK_BASE, 0x801, METHOD_NEITHER, FILE_READ_ACCESS | FILE_WRITE_ACCESS):
		return "IOCTL_VFD_CLOSE_FILE";

	case CTL_CODE(IOCTL_DISK_BASE, 0x802, METHOD_BUFFERED, FILE_ANY_ACCESS):
		return "IOCTL_VFD_QUERY_FILE";

	//
	// Standard control codes
	//
	case IOCTL_CANCEL_DEVICE_WAKE:			return "IOCTL_CANCEL_DEVICE_WAKE";
	case IOCTL_QUERY_DEVICE_POWER_STATE:	return "IOCTL_QUERY_DEVICE_POWER_STATE";
	case IOCTL_SET_DEVICE_WAKE:				return "IOCTL_SET_DEVICE_WAKE";

	case IOCTL_DISK_BASE:					return "IOCTL_DISK_BASE";
	case IOCTL_DISK_CHECK_VERIFY:			return "IOCTL_DISK_CHECK_VERIFY";
	case IOCTL_DISK_CONTROLLER_NUMBER:		return "IOCTL_DISK_CONTROLLER_NUMBER";
	case IOCTL_DISK_DELETE_DRIVE_LAYOUT:	return "IOCTL_DISK_DELETE_DRIVE_LAYOUT";
	case IOCTL_DISK_EJECT_MEDIA:			return "IOCTL_DISK_EJECT_MEDIA";
	case IOCTL_DISK_FIND_NEW_DEVICES:		return "IOCTL_DISK_FIND_NEW_DEVICES";
	case IOCTL_DISK_FORMAT_DRIVE:			return "IOCTL_DISK_FORMAT_DRIVE";
	case IOCTL_DISK_FORMAT_TRACKS:			return "IOCTL_DISK_FORMAT_TRACKS";
	case IOCTL_DISK_FORMAT_TRACKS_EX:		return "IOCTL_DISK_FORMAT_TRACKS_EX";
	case IOCTL_DISK_GET_CACHE_INFORMATION:	return "IOCTL_DISK_GET_CACHE_INFORMATION";
	case IOCTL_DISK_GET_DRIVE_GEOMETRY:		return "IOCTL_DISK_GET_DRIVE_GEOMETRY";
	case IOCTL_DISK_GET_DRIVE_LAYOUT:		return "IOCTL_DISK_GET_DRIVE_LAYOUT";
	case IOCTL_DISK_GET_MEDIA_TYPES:		return "IOCTL_DISK_GET_MEDIA_TYPES";
	case IOCTL_DISK_GET_PARTITION_INFO:		return "IOCTL_DISK_GET_PARTITION_INFO";
	case IOCTL_DISK_GROW_PARTITION:			return "IOCTL_DISK_GROW_PARTITION";
	case IOCTL_DISK_HISTOGRAM_DATA:			return "IOCTL_DISK_HISTOGRAM_DATA";
	case IOCTL_DISK_HISTOGRAM_RESET:		return "IOCTL_DISK_HISTOGRAM_RESET";
	case IOCTL_DISK_HISTOGRAM_STRUCTURE:	return "IOCTL_DISK_HISTOGRAM_STRUCTURE";
	case IOCTL_DISK_IS_WRITABLE:			return "IOCTL_DISK_IS_WRITABLE";
	case IOCTL_DISK_LOAD_MEDIA:				return "IOCTL_DISK_LOAD_MEDIA";
	case IOCTL_DISK_LOGGING:				return "IOCTL_DISK_LOGGING";
	case IOCTL_DISK_MEDIA_REMOVAL:			return "IOCTL_DISK_MEDIA_REMOVAL";
	case IOCTL_DISK_PERFORMANCE:			return "IOCTL_DISK_PERFORMANCE";
	case IOCTL_DISK_REASSIGN_BLOCKS:		return "IOCTL_DISK_REASSIGN_BLOCKS";
	case IOCTL_DISK_RELEASE:				return "IOCTL_DISK_RELEASE";
	case IOCTL_DISK_REQUEST_DATA:			return "IOCTL_DISK_REQUEST_DATA";
	case IOCTL_DISK_REQUEST_STRUCTURE:		return "IOCTL_DISK_REQUEST_STRUCTURE";
	case IOCTL_DISK_RESERVE:				return "IOCTL_DISK_RESERVE";
	case IOCTL_DISK_SENSE_DEVICE:			return "IOCTL_DISK_SENSE_DEVICE";
	case IOCTL_DISK_SET_CACHE_INFORMATION:	return "IOCTL_DISK_SET_CACHE_INFORMATION";
	case IOCTL_DISK_SET_DRIVE_LAYOUT:		return "IOCTL_DISK_SET_DRIVE_LAYOUT";
	case IOCTL_DISK_SET_PARTITION_INFO:		return "IOCTL_DISK_SET_PARTITION_INFO";
	case IOCTL_DISK_SIMBAD:					return "IOCTL_DISK_SIMBAD";
	case IOCTL_DISK_UPDATE_DRIVE_SIZE:		return "IOCTL_DISK_UPDATE_DRIVE_SIZE";
	case IOCTL_DISK_VERIFY:					return "IOCTL_DISK_VERIFY";

	case IOCTL_STORAGE_BASE:				return "IOCTL_STORAGE_BASE";
	case IOCTL_STORAGE_CHECK_VERIFY:		return "IOCTL_STORAGE_CHECK_VERIFY";
	case IOCTL_STORAGE_CHECK_VERIFY2:		return "IOCTL_STORAGE_CHECK_VERIFY2";
	case IOCTL_STORAGE_EJECT_MEDIA:			return "IOCTL_STORAGE_EJECT_MEDIA";
	case IOCTL_STORAGE_EJECTION_CONTROL:	return "IOCTL_STORAGE_EJECTION_CONTROL";
	case IOCTL_STORAGE_FIND_NEW_DEVICES:	return "IOCTL_STORAGE_FIND_NEW_DEVICES";
	case IOCTL_STORAGE_GET_DEVICE_NUMBER:	return "IOCTL_STORAGE_GET_DEVICE_NUMBER";
	case IOCTL_STORAGE_GET_MEDIA_TYPES:		return "IOCTL_STORAGE_GET_MEDIA_TYPES";
	case IOCTL_STORAGE_GET_MEDIA_TYPES_EX:	return "IOCTL_STORAGE_GET_MEDIA_TYPES_EX";
	case IOCTL_STORAGE_LOAD_MEDIA:			return "IOCTL_STORAGE_LOAD_MEDIA";
	case IOCTL_STORAGE_LOAD_MEDIA2:			return "IOCTL_STORAGE_LOAD_MEDIA2";
	case IOCTL_STORAGE_MCN_CONTROL:			return "IOCTL_STORAGE_MCN_CONTROL";
	case IOCTL_STORAGE_MEDIA_REMOVAL:		return "IOCTL_STORAGE_MEDIA_REMOVAL";
	case IOCTL_STORAGE_PREDICT_FAILURE:		return "IOCTL_STORAGE_PREDICT_FAILURE";
	case IOCTL_STORAGE_QUERY_PROPERTY:		return "IOCTL_STORAGE_QUERY_PROPERTY";
	case IOCTL_STORAGE_RELEASE:				return "IOCTL_STORAGE_RELEASE";
	case IOCTL_STORAGE_RESERVE:				return "IOCTL_STORAGE_RESERVE";
	case IOCTL_STORAGE_RESET_BUS:			return "IOCTL_STORAGE_RESET_BUS";
	case IOCTL_STORAGE_RESET_DEVICE:		return "IOCTL_STORAGE_RESET_DEVICE";

	// ntddft.h	- fault tolearnce (mirror)
	case FT_INITIALIZE_SET:					return "FT_INITIALIZE_SET";
	case FT_REGENERATE:						return "FT_REGENERATE";
	case FT_CONFIGURE:						return "FT_CONFIGURE";
	case FT_VERIFY:							return "FT_VERIFY";
	case FT_SECONDARY_READ:					return "FT_SECONDARY_READ";
	case FT_PRIMARY_READ:					return "FT_PRIMARY_READ	";
	case FT_BALANCED_READ_MODE:				return "FT_BALANCED_READ_MODE";
	case FT_SYNC_REDUNDANT_COPY:			return "FT_SYNC_REDUNDANT_COPY";
	case FT_SEQUENTIAL_WRITE_MODE:			return "FT_SEQUENTIAL_WRITE_MODE";
	case FT_PARALLEL_WRITE_MODE:			return "FT_PARALLEL_WRITE_MODE";
	case FT_QUERY_SET_STATE:				return "FT_QUERY_SET_STATE";
	case FT_CLUSTER_SET_MEMBER_STATE:		return "FT_CLUSTER_SET_MEMBER_STATE";
	case FT_CLUSTER_GET_MEMBER_STATE:		return "FT_CLUSTER_GET_MEMBER_STATE";

	// mountmgr.h/mountdev.h
	case IOCTL_MOUNTDEV_QUERY_UNIQUE_ID:	return "IOCTL_MOUNTDEV_QUERY_UNIQUE_ID";
	case IOCTL_MOUNTDEV_QUERY_DEVICE_NAME:	return "IOCTL_MOUNTDEV_QUERY_DEVICE_NAME";
	case IOCTL_MOUNTDEV_UNIQUE_ID_CHANGE_NOTIFY:
											return "IOCTL_MOUNTDEV_UNIQUE_ID_CHANGE_NOTIFY";
	case IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME:
											return "IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME";
	case IOCTL_MOUNTDEV_LINK_CREATED:		return "IOCTL_MOUNTDEV_LINK_CREATED";
	case IOCTL_MOUNTDEV_LINK_DELETED:		return "IOCTL_MOUNTDEV_LINK_DELETED";

	default:	return NULL;
	}
}

PCSTR
NtStatusToStr(
	NTSTATUS status)
{
	switch (status) {
	case STATUS_SUCCESS:					return "STATUS_SUCCESS";
/*
	case STATUS_WAIT_0: 					return "STATUS_WAIT_0";
*/
	case STATUS_WAIT_1: 					return "STATUS_WAIT_1";
	case STATUS_WAIT_2: 					return "STATUS_WAIT_2";
	case STATUS_WAIT_3: 					return "STATUS_WAIT_3";
	case STATUS_WAIT_63:					return "STATUS_WAIT_63";
	case STATUS_ABANDONED:					return "STATUS_ABANDONED";
/*
	case STATUS_ABANDONED_WAIT_0:			return "STATUS_ABANDONED_WAIT_0";
*/
	case STATUS_ABANDONED_WAIT_63:			return "STATUS_ABANDONED_WAIT_63";
	case STATUS_USER_APC:					return "STATUS_USER_APC";
	case STATUS_KERNEL_APC: 				return "STATUS_KERNEL_APC";
	case STATUS_ALERTED:					return "STATUS_ALERTED";
	case STATUS_TIMEOUT:					return "STATUS_TIMEOUT";
	case STATUS_PENDING:					return "STATUS_PENDING";
	case STATUS_REPARSE:					return "STATUS_REPARSE";
	case STATUS_MORE_ENTRIES:				return "STATUS_MORE_ENTRIES";
	case STATUS_NOT_ALL_ASSIGNED:			return "STATUS_NOT_ALL_ASSIGNED";
	case STATUS_SOME_NOT_MAPPED:			return "STATUS_SOME_NOT_MAPPED";
	case STATUS_OPLOCK_BREAK_IN_PROGRESS:	return "STATUS_OPLOCK_BREAK_IN_PROGRESS";
	case STATUS_VOLUME_MOUNTED: 			return "STATUS_VOLUME_MOUNTED";
	case STATUS_RXACT_COMMITTED:			return "STATUS_RXACT_COMMITTED";
	case STATUS_NOTIFY_CLEANUP: 			return "STATUS_NOTIFY_CLEANUP";
	case STATUS_NOTIFY_ENUM_DIR:			return "STATUS_NOTIFY_ENUM_DIR";
	case STATUS_NO_QUOTAS_FOR_ACCOUNT:		return "STATUS_NO_QUOTAS_FOR_ACCOUNT";
	case STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED:
											return "STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED";
	case STATUS_PAGE_FAULT_TRANSITION:		return "STATUS_PAGE_FAULT_TRANSITION";
	case STATUS_PAGE_FAULT_DEMAND_ZERO: 	return "STATUS_PAGE_FAULT_DEMAND_ZERO";
	case STATUS_PAGE_FAULT_COPY_ON_WRITE:	return "STATUS_PAGE_FAULT_COPY_ON_WRITE";
	case STATUS_PAGE_FAULT_GUARD_PAGE:		return "STATUS_PAGE_FAULT_GUARD_PAGE";
	case STATUS_PAGE_FAULT_PAGING_FILE: 	return "STATUS_PAGE_FAULT_PAGING_FILE";
	case STATUS_CACHE_PAGE_LOCKED:			return "STATUS_CACHE_PAGE_LOCKED";
	case STATUS_CRASH_DUMP: 				return "STATUS_CRASH_DUMP";
	case STATUS_BUFFER_ALL_ZEROS:			return "STATUS_BUFFER_ALL_ZEROS";
	case STATUS_REPARSE_OBJECT: 			return "STATUS_REPARSE_OBJECT";
	case STATUS_RESOURCE_REQUIREMENTS_CHANGED:
											return "STATUS_RESOURCE_REQUIREMENTS_CHANGED";
	case STATUS_TRANSLATION_COMPLETE:		return "STATUS_TRANSLATION_COMPLETE";
	case STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY:
											return "STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY";
	case DBG_EXCEPTION_HANDLED: 			return "DBG_EXCEPTION_HANDLED";
	case DBG_CONTINUE:						return "DBG_CONTINUE";
	case STATUS_OBJECT_NAME_EXISTS: 		return "STATUS_OBJECT_NAME_EXISTS";
	case STATUS_THREAD_WAS_SUSPENDED:		return "STATUS_THREAD_WAS_SUSPENDED";
	case STATUS_WORKING_SET_LIMIT_RANGE:	return "STATUS_WORKING_SET_LIMIT_RANGE";
	case STATUS_IMAGE_NOT_AT_BASE:			return "STATUS_IMAGE_NOT_AT_BASE";
	case STATUS_RXACT_STATE_CREATED:		return "STATUS_RXACT_STATE_CREATED";
	case STATUS_SEGMENT_NOTIFICATION:		return "STATUS_SEGMENT_NOTIFICATION";
	case STATUS_LOCAL_USER_SESSION_KEY: 	return "STATUS_LOCAL_USER_SESSION_KEY";
	case STATUS_BAD_CURRENT_DIRECTORY:		return "STATUS_BAD_CURRENT_DIRECTORY";
	case STATUS_SERIAL_MORE_WRITES: 		return "STATUS_SERIAL_MORE_WRITES";
	case STATUS_REGISTRY_RECOVERED: 		return "STATUS_REGISTRY_RECOVERED";
	case STATUS_FT_READ_RECOVERY_FROM_BACKUP:
											return "STATUS_FT_READ_RECOVERY_FROM_BACKUP";
	case STATUS_FT_WRITE_RECOVERY:			return "STATUS_FT_WRITE_RECOVERY";
	case STATUS_SERIAL_COUNTER_TIMEOUT: 	return "STATUS_SERIAL_COUNTER_TIMEOUT";
	case STATUS_NULL_LM_PASSWORD:			return "STATUS_NULL_LM_PASSWORD";
	case STATUS_IMAGE_MACHINE_TYPE_MISMATCH:return "STATUS_IMAGE_MACHINE_TYPE_MISMATCH";
	case STATUS_RECEIVE_PARTIAL:			return "STATUS_RECEIVE_PARTIAL";
	case STATUS_RECEIVE_EXPEDITED:			return "STATUS_RECEIVE_EXPEDITED";
	case STATUS_RECEIVE_PARTIAL_EXPEDITED:	return "STATUS_RECEIVE_PARTIAL_EXPEDITED";
	case STATUS_EVENT_DONE: 				return "STATUS_EVENT_DONE";
	case STATUS_EVENT_PENDING:				return "STATUS_EVENT_PENDING";
	case STATUS_CHECKING_FILE_SYSTEM:		return "STATUS_CHECKING_FILE_SYSTEM";
	case STATUS_FATAL_APP_EXIT: 			return "STATUS_FATAL_APP_EXIT";
	case STATUS_PREDEFINED_HANDLE:			return "STATUS_PREDEFINED_HANDLE";
	case STATUS_WAS_UNLOCKED:				return "STATUS_WAS_UNLOCKED";
	case STATUS_SERVICE_NOTIFICATION:		return "STATUS_SERVICE_NOTIFICATION";
	case STATUS_WAS_LOCKED: 				return "STATUS_WAS_LOCKED";
	case STATUS_LOG_HARD_ERROR: 			return "STATUS_LOG_HARD_ERROR";
	case STATUS_ALREADY_WIN32:				return "STATUS_ALREADY_WIN32";
	case STATUS_WX86_UNSIMULATE:			return "STATUS_WX86_UNSIMULATE";
	case STATUS_WX86_CONTINUE:				return "STATUS_WX86_CONTINUE";
	case STATUS_WX86_SINGLE_STEP:			return "STATUS_WX86_SINGLE_STEP";
	case STATUS_WX86_BREAKPOINT:			return "STATUS_WX86_BREAKPOINT";
	case STATUS_WX86_EXCEPTION_CONTINUE:	return "STATUS_WX86_EXCEPTION_CONTINUE";
	case STATUS_WX86_EXCEPTION_LASTCHANCE:	return "STATUS_WX86_EXCEPTION_LASTCHANCE";
	case STATUS_WX86_EXCEPTION_CHAIN:		return "STATUS_WX86_EXCEPTION_CHAIN";
	case STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE:
											return "STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE";
	case STATUS_NO_YIELD_PERFORMED: 		return "STATUS_NO_YIELD_PERFORMED";
	case STATUS_TIMER_RESUME_IGNORED:		return "STATUS_TIMER_RESUME_IGNORED";
	case STATUS_ARBITRATION_UNHANDLED:		return "STATUS_ARBITRATION_UNHANDLED";
	case STATUS_CARDBUS_NOT_SUPPORTED:		return "STATUS_CARDBUS_NOT_SUPPORTED";
	case STATUS_WX86_CREATEWX86TIB: 		return "STATUS_WX86_CREATEWX86TIB";
	case STATUS_MP_PROCESSOR_MISMATCH:		return "STATUS_MP_PROCESSOR_MISMATCH";
	case DBG_REPLY_LATER:					return "DBG_REPLY_LATER";
	case DBG_UNABLE_TO_PROVIDE_HANDLE:		return "DBG_UNABLE_TO_PROVIDE_HANDLE";
	case DBG_TERMINATE_THREAD:				return "DBG_TERMINATE_THREAD";
	case DBG_TERMINATE_PROCESS: 			return "DBG_TERMINATE_PROCESS";
	case DBG_CONTROL_C: 					return "DBG_CONTROL_C";
	case DBG_PRINTEXCEPTION_C:				return "DBG_PRINTEXCEPTION_C";
	case DBG_RIPEXCEPTION:					return "DBG_RIPEXCEPTION";
	case DBG_CONTROL_BREAK: 				return "DBG_CONTROL_BREAK";
	case STATUS_GUARD_PAGE_VIOLATION:		return "STATUS_GUARD_PAGE_VIOLATION";
	case STATUS_DATATYPE_MISALIGNMENT:		return "STATUS_DATATYPE_MISALIGNMENT";
	case STATUS_BREAKPOINT: 				return "STATUS_BREAKPOINT";
	case STATUS_SINGLE_STEP:				return "STATUS_SINGLE_STEP";
	case STATUS_BUFFER_OVERFLOW:			return "STATUS_BUFFER_OVERFLOW";
	case STATUS_NO_MORE_FILES:				return "STATUS_NO_MORE_FILES";
	case STATUS_WAKE_SYSTEM_DEBUGGER:		return "STATUS_WAKE_SYSTEM_DEBUGGER";
	case STATUS_HANDLES_CLOSED: 			return "STATUS_HANDLES_CLOSED";
	case STATUS_NO_INHERITANCE: 			return "STATUS_NO_INHERITANCE";
	case STATUS_GUID_SUBSTITUTION_MADE: 	return "STATUS_GUID_SUBSTITUTION_MADE";
	case STATUS_PARTIAL_COPY:				return "STATUS_PARTIAL_COPY";
	case STATUS_DEVICE_PAPER_EMPTY: 		return "STATUS_DEVICE_PAPER_EMPTY";
	case STATUS_DEVICE_POWERED_OFF: 		return "STATUS_DEVICE_POWERED_OFF";
	case STATUS_DEVICE_OFF_LINE:			return "STATUS_DEVICE_OFF_LINE";
	case STATUS_DEVICE_BUSY:				return "STATUS_DEVICE_BUSY";
	case STATUS_NO_MORE_EAS:				return "STATUS_NO_MORE_EAS";
	case STATUS_INVALID_EA_NAME:			return "STATUS_INVALID_EA_NAME";
	case STATUS_EA_LIST_INCONSISTENT:		return "STATUS_EA_LIST_INCONSISTENT";
	case STATUS_INVALID_EA_FLAG:			return "STATUS_INVALID_EA_FLAG";
	case STATUS_VERIFY_REQUIRED:			return "STATUS_VERIFY_REQUIRED";
	case STATUS_EXTRANEOUS_INFORMATION: 	return "STATUS_EXTRANEOUS_INFORMATION";
	case STATUS_RXACT_COMMIT_NECESSARY: 	return "STATUS_RXACT_COMMIT_NECESSARY";
	case STATUS_NO_MORE_ENTRIES:			return "STATUS_NO_MORE_ENTRIES";
	case STATUS_FILEMARK_DETECTED:			return "STATUS_FILEMARK_DETECTED";
	case STATUS_MEDIA_CHANGED:				return "STATUS_MEDIA_CHANGED";
	case STATUS_BUS_RESET:					return "STATUS_BUS_RESET";
	case STATUS_END_OF_MEDIA:				return "STATUS_END_OF_MEDIA";
	case STATUS_BEGINNING_OF_MEDIA: 		return "STATUS_BEGINNING_OF_MEDIA";
	case STATUS_MEDIA_CHECK:				return "STATUS_MEDIA_CHECK";
	case STATUS_SETMARK_DETECTED:			return "STATUS_SETMARK_DETECTED";
	case STATUS_NO_DATA_DETECTED:			return "STATUS_NO_DATA_DETECTED";
	case STATUS_REDIRECTOR_HAS_OPEN_HANDLES:return "STATUS_REDIRECTOR_HAS_OPEN_HANDLES";
	case STATUS_SERVER_HAS_OPEN_HANDLES:	return "STATUS_SERVER_HAS_OPEN_HANDLES";
	case STATUS_ALREADY_DISCONNECTED:		return "STATUS_ALREADY_DISCONNECTED";
	case STATUS_LONGJUMP:					return "STATUS_LONGJUMP";
	case DBG_EXCEPTION_NOT_HANDLED: 		return "DBG_EXCEPTION_NOT_HANDLED";
	case STATUS_UNSUCCESSFUL:				return "STATUS_UNSUCCESSFUL";
	case STATUS_NOT_IMPLEMENTED:			return "STATUS_NOT_IMPLEMENTED";
	case STATUS_INVALID_INFO_CLASS: 		return "STATUS_INVALID_INFO_CLASS";
	case STATUS_INFO_LENGTH_MISMATCH:		return "STATUS_INFO_LENGTH_MISMATCH";
	case STATUS_ACCESS_VIOLATION:			return "STATUS_ACCESS_VIOLATION";
	case STATUS_IN_PAGE_ERROR:				return "STATUS_IN_PAGE_ERROR";
	case STATUS_PAGEFILE_QUOTA: 			return "STATUS_PAGEFILE_QUOTA";
	case STATUS_INVALID_HANDLE: 			return "STATUS_INVALID_HANDLE";
	case STATUS_BAD_INITIAL_STACK:			return "STATUS_BAD_INITIAL_STACK";
	case STATUS_BAD_INITIAL_PC: 			return "STATUS_BAD_INITIAL_PC";
	case STATUS_INVALID_CID:				return "STATUS_INVALID_CID";
	case STATUS_TIMER_NOT_CANCELED: 		return "STATUS_TIMER_NOT_CANCELED";
	case STATUS_INVALID_PARAMETER:			return "STATUS_INVALID_PARAMETER";
	case STATUS_NO_SUCH_DEVICE: 			return "STATUS_NO_SUCH_DEVICE";
	case STATUS_NO_SUCH_FILE:				return "STATUS_NO_SUCH_FILE";
	case STATUS_INVALID_DEVICE_REQUEST: 	return "STATUS_INVALID_DEVICE_REQUEST";
	case STATUS_END_OF_FILE:				return "STATUS_END_OF_FILE";
	case STATUS_WRONG_VOLUME:				return "STATUS_WRONG_VOLUME";
	case STATUS_NO_MEDIA_IN_DEVICE: 		return "STATUS_NO_MEDIA_IN_DEVICE";
	case STATUS_UNRECOGNIZED_MEDIA: 		return "STATUS_UNRECOGNIZED_MEDIA";
	case STATUS_NONEXISTENT_SECTOR: 		return "STATUS_NONEXISTENT_SECTOR";
	case STATUS_MORE_PROCESSING_REQUIRED:	return "STATUS_MORE_PROCESSING_REQUIRED";
	case STATUS_NO_MEMORY:					return "STATUS_NO_MEMORY";
	case STATUS_CONFLICTING_ADDRESSES:		return "STATUS_CONFLICTING_ADDRESSES";
	case STATUS_NOT_MAPPED_VIEW:			return "STATUS_NOT_MAPPED_VIEW";
	case STATUS_UNABLE_TO_FREE_VM:			return "STATUS_UNABLE_TO_FREE_VM";
	case STATUS_UNABLE_TO_DELETE_SECTION:	return "STATUS_UNABLE_TO_DELETE_SECTION";
	case STATUS_INVALID_SYSTEM_SERVICE: 	return "STATUS_INVALID_SYSTEM_SERVICE";
	case STATUS_ILLEGAL_INSTRUCTION:		return "STATUS_ILLEGAL_INSTRUCTION";
	case STATUS_INVALID_LOCK_SEQUENCE:		return "STATUS_INVALID_LOCK_SEQUENCE";
	case STATUS_INVALID_VIEW_SIZE:			return "STATUS_INVALID_VIEW_SIZE";
	case STATUS_INVALID_FILE_FOR_SECTION:	return "STATUS_INVALID_FILE_FOR_SECTION";
	case STATUS_ALREADY_COMMITTED:			return "STATUS_ALREADY_COMMITTED";
	case STATUS_ACCESS_DENIED:				return "STATUS_ACCESS_DENIED";
	case STATUS_BUFFER_TOO_SMALL:			return "STATUS_BUFFER_TOO_SMALL";
	case STATUS_OBJECT_TYPE_MISMATCH:		return "STATUS_OBJECT_TYPE_MISMATCH";
	case STATUS_NONCONTINUABLE_EXCEPTION:	return "STATUS_NONCONTINUABLE_EXCEPTION";
	case STATUS_INVALID_DISPOSITION:		return "STATUS_INVALID_DISPOSITION";
	case STATUS_UNWIND: 					return "STATUS_UNWIND";
	case STATUS_BAD_STACK:					return "STATUS_BAD_STACK";
	case STATUS_INVALID_UNWIND_TARGET:		return "STATUS_INVALID_UNWIND_TARGET";
	case STATUS_NOT_LOCKED: 				return "STATUS_NOT_LOCKED";
	case STATUS_PARITY_ERROR:				return "STATUS_PARITY_ERROR";
	case STATUS_UNABLE_TO_DECOMMIT_VM:		return "STATUS_UNABLE_TO_DECOMMIT_VM";
	case STATUS_NOT_COMMITTED:				return "STATUS_NOT_COMMITTED";
	case STATUS_INVALID_PORT_ATTRIBUTES:	return "STATUS_INVALID_PORT_ATTRIBUTES";
	case STATUS_PORT_MESSAGE_TOO_LONG:		return "STATUS_PORT_MESSAGE_TOO_LONG";
	case STATUS_INVALID_PARAMETER_MIX:		return "STATUS_INVALID_PARAMETER_MIX";
	case STATUS_INVALID_QUOTA_LOWER:		return "STATUS_INVALID_QUOTA_LOWER";
	case STATUS_DISK_CORRUPT_ERROR: 		return "STATUS_DISK_CORRUPT_ERROR";
	case STATUS_OBJECT_NAME_INVALID:		return "STATUS_OBJECT_NAME_INVALID";
	case STATUS_OBJECT_NAME_NOT_FOUND:		return "STATUS_OBJECT_NAME_NOT_FOUND";
	case STATUS_OBJECT_NAME_COLLISION:		return "STATUS_OBJECT_NAME_COLLISION";
	case STATUS_PORT_DISCONNECTED:			return "STATUS_PORT_DISCONNECTED";
	case STATUS_DEVICE_ALREADY_ATTACHED:	return "STATUS_DEVICE_ALREADY_ATTACHED";
	case STATUS_OBJECT_PATH_INVALID:		return "STATUS_OBJECT_PATH_INVALID";
	case STATUS_OBJECT_PATH_NOT_FOUND:		return "STATUS_OBJECT_PATH_NOT_FOUND";
	case STATUS_OBJECT_PATH_SYNTAX_BAD: 	return "STATUS_OBJECT_PATH_SYNTAX_BAD";
	case STATUS_DATA_OVERRUN:				return "STATUS_DATA_OVERRUN";
	case STATUS_DATA_LATE_ERROR:			return "STATUS_DATA_LATE_ERROR";
	case STATUS_DATA_ERROR: 				return "STATUS_DATA_ERROR";
	case STATUS_CRC_ERROR:					return "STATUS_CRC_ERROR";
	case STATUS_SECTION_TOO_BIG:			return "STATUS_SECTION_TOO_BIG";
	case STATUS_PORT_CONNECTION_REFUSED:	return "STATUS_PORT_CONNECTION_REFUSED";
	case STATUS_INVALID_PORT_HANDLE:		return "STATUS_INVALID_PORT_HANDLE";
	case STATUS_SHARING_VIOLATION:			return "STATUS_SHARING_VIOLATION";
	case STATUS_QUOTA_EXCEEDED: 			return "STATUS_QUOTA_EXCEEDED";
	case STATUS_INVALID_PAGE_PROTECTION:	return "STATUS_INVALID_PAGE_PROTECTION";
	case STATUS_MUTANT_NOT_OWNED:			return "STATUS_MUTANT_NOT_OWNED";
	case STATUS_SEMAPHORE_LIMIT_EXCEEDED:	return "STATUS_SEMAPHORE_LIMIT_EXCEEDED";
	case STATUS_PORT_ALREADY_SET:			return "STATUS_PORT_ALREADY_SET";
	case STATUS_SECTION_NOT_IMAGE:			return "STATUS_SECTION_NOT_IMAGE";
	case STATUS_SUSPEND_COUNT_EXCEEDED: 	return "STATUS_SUSPEND_COUNT_EXCEEDED";
	case STATUS_THREAD_IS_TERMINATING:		return "STATUS_THREAD_IS_TERMINATING";
	case STATUS_BAD_WORKING_SET_LIMIT:		return "STATUS_BAD_WORKING_SET_LIMIT";
	case STATUS_INCOMPATIBLE_FILE_MAP:		return "STATUS_INCOMPATIBLE_FILE_MAP";
	case STATUS_SECTION_PROTECTION: 		return "STATUS_SECTION_PROTECTION";
	case STATUS_EAS_NOT_SUPPORTED:			return "STATUS_EAS_NOT_SUPPORTED";
	case STATUS_EA_TOO_LARGE:				return "STATUS_EA_TOO_LARGE";
	case STATUS_NONEXISTENT_EA_ENTRY:		return "STATUS_NONEXISTENT_EA_ENTRY";
	case STATUS_NO_EAS_ON_FILE: 			return "STATUS_NO_EAS_ON_FILE";
	case STATUS_EA_CORRUPT_ERROR:			return "STATUS_EA_CORRUPT_ERROR";
	case STATUS_FILE_LOCK_CONFLICT: 		return "STATUS_FILE_LOCK_CONFLICT";
	case STATUS_LOCK_NOT_GRANTED:			return "STATUS_LOCK_NOT_GRANTED";
	case STATUS_DELETE_PENDING: 			return "STATUS_DELETE_PENDING";
	case STATUS_CTL_FILE_NOT_SUPPORTED: 	return "STATUS_CTL_FILE_NOT_SUPPORTED";
	case STATUS_UNKNOWN_REVISION:			return "STATUS_UNKNOWN_REVISION";
	case STATUS_REVISION_MISMATCH:			return "STATUS_REVISION_MISMATCH";
	case STATUS_INVALID_OWNER:				return "STATUS_INVALID_OWNER";
	case STATUS_INVALID_PRIMARY_GROUP:		return "STATUS_INVALID_PRIMARY_GROUP";
	case STATUS_NO_IMPERSONATION_TOKEN: 	return "STATUS_NO_IMPERSONATION_TOKEN";
	case STATUS_CANT_DISABLE_MANDATORY: 	return "STATUS_CANT_DISABLE_MANDATORY";
	case STATUS_NO_LOGON_SERVERS:			return "STATUS_NO_LOGON_SERVERS";
	case STATUS_NO_SUCH_LOGON_SESSION:		return "STATUS_NO_SUCH_LOGON_SESSION";
	case STATUS_NO_SUCH_PRIVILEGE:			return "STATUS_NO_SUCH_PRIVILEGE";
	case STATUS_PRIVILEGE_NOT_HELD: 		return "STATUS_PRIVILEGE_NOT_HELD";
	case STATUS_INVALID_ACCOUNT_NAME:		return "STATUS_INVALID_ACCOUNT_NAME";
	case STATUS_USER_EXISTS:				return "STATUS_USER_EXISTS";
	case STATUS_NO_SUCH_USER:				return "STATUS_NO_SUCH_USER";
	case STATUS_GROUP_EXISTS:				return "STATUS_GROUP_EXISTS";
	case STATUS_NO_SUCH_GROUP:				return "STATUS_NO_SUCH_GROUP";
	case STATUS_MEMBER_IN_GROUP:			return "STATUS_MEMBER_IN_GROUP";
	case STATUS_MEMBER_NOT_IN_GROUP:		return "STATUS_MEMBER_NOT_IN_GROUP";
	case STATUS_LAST_ADMIN: 				return "STATUS_LAST_ADMIN";
	case STATUS_WRONG_PASSWORD: 			return "STATUS_WRONG_PASSWORD";
	case STATUS_ILL_FORMED_PASSWORD:		return "STATUS_ILL_FORMED_PASSWORD";
	case STATUS_PASSWORD_RESTRICTION:		return "STATUS_PASSWORD_RESTRICTION";
	case STATUS_LOGON_FAILURE:				return "STATUS_LOGON_FAILURE";
	case STATUS_ACCOUNT_RESTRICTION:		return "STATUS_ACCOUNT_RESTRICTION";
	case STATUS_INVALID_LOGON_HOURS:		return "STATUS_INVALID_LOGON_HOURS";
	case STATUS_INVALID_WORKSTATION:		return "STATUS_INVALID_WORKSTATION";
	case STATUS_PASSWORD_EXPIRED:			return "STATUS_PASSWORD_EXPIRED";
	case STATUS_ACCOUNT_DISABLED:			return "STATUS_ACCOUNT_DISABLED";
	case STATUS_NONE_MAPPED:				return "STATUS_NONE_MAPPED";
	case STATUS_TOO_MANY_LUIDS_REQUESTED:	return "STATUS_TOO_MANY_LUIDS_REQUESTED";
	case STATUS_LUIDS_EXHAUSTED:			return "STATUS_LUIDS_EXHAUSTED";
	case STATUS_INVALID_SUB_AUTHORITY:		return "STATUS_INVALID_SUB_AUTHORITY";
	case STATUS_INVALID_ACL:				return "STATUS_INVALID_ACL";
	case STATUS_INVALID_SID:				return "STATUS_INVALID_SID";
	case STATUS_INVALID_SECURITY_DESCR: 	return "STATUS_INVALID_SECURITY_DESCR";
	case STATUS_PROCEDURE_NOT_FOUND:		return "STATUS_PROCEDURE_NOT_FOUND";
	case STATUS_INVALID_IMAGE_FORMAT:		return "STATUS_INVALID_IMAGE_FORMAT";
	case STATUS_NO_TOKEN:					return "STATUS_NO_TOKEN";
	case STATUS_BAD_INHERITANCE_ACL:		return "STATUS_BAD_INHERITANCE_ACL";
	case STATUS_RANGE_NOT_LOCKED:			return "STATUS_RANGE_NOT_LOCKED";
	case STATUS_DISK_FULL:					return "STATUS_DISK_FULL";
	case STATUS_SERVER_DISABLED:			return "STATUS_SERVER_DISABLED";
	case STATUS_SERVER_NOT_DISABLED:		return "STATUS_SERVER_NOT_DISABLED";
	case STATUS_TOO_MANY_GUIDS_REQUESTED:	return "STATUS_TOO_MANY_GUIDS_REQUESTED";
	case STATUS_GUIDS_EXHAUSTED:			return "STATUS_GUIDS_EXHAUSTED";
	case STATUS_INVALID_ID_AUTHORITY:		return "STATUS_INVALID_ID_AUTHORITY";
	case STATUS_AGENTS_EXHAUSTED:			return "STATUS_AGENTS_EXHAUSTED";
	case STATUS_INVALID_VOLUME_LABEL:		return "STATUS_INVALID_VOLUME_LABEL";
	case STATUS_SECTION_NOT_EXTENDED:		return "STATUS_SECTION_NOT_EXTENDED";
	case STATUS_NOT_MAPPED_DATA:			return "STATUS_NOT_MAPPED_DATA";
	case STATUS_RESOURCE_DATA_NOT_FOUND:	return "STATUS_RESOURCE_DATA_NOT_FOUND";
	case STATUS_RESOURCE_TYPE_NOT_FOUND:	return "STATUS_RESOURCE_TYPE_NOT_FOUND";
	case STATUS_RESOURCE_NAME_NOT_FOUND:	return "STATUS_RESOURCE_NAME_NOT_FOUND";
	case STATUS_ARRAY_BOUNDS_EXCEEDED:		return "STATUS_ARRAY_BOUNDS_EXCEEDED";
	case STATUS_FLOAT_DENORMAL_OPERAND: 	return "STATUS_FLOAT_DENORMAL_OPERAND";
	case STATUS_FLOAT_DIVIDE_BY_ZERO:		return "STATUS_FLOAT_DIVIDE_BY_ZERO";
	case STATUS_FLOAT_INEXACT_RESULT:		return "STATUS_FLOAT_INEXACT_RESULT";
	case STATUS_FLOAT_INVALID_OPERATION:	return "STATUS_FLOAT_INVALID_OPERATION";
	case STATUS_FLOAT_OVERFLOW: 			return "STATUS_FLOAT_OVERFLOW";
	case STATUS_FLOAT_STACK_CHECK:			return "STATUS_FLOAT_STACK_CHECK";
	case STATUS_FLOAT_UNDERFLOW:			return "STATUS_FLOAT_UNDERFLOW";
	case STATUS_INTEGER_DIVIDE_BY_ZERO: 	return "STATUS_INTEGER_DIVIDE_BY_ZERO";
	case STATUS_INTEGER_OVERFLOW:			return "STATUS_INTEGER_OVERFLOW";
	case STATUS_PRIVILEGED_INSTRUCTION: 	return "STATUS_PRIVILEGED_INSTRUCTION";
	case STATUS_TOO_MANY_PAGING_FILES:		return "STATUS_TOO_MANY_PAGING_FILES";
	case STATUS_FILE_INVALID:				return "STATUS_FILE_INVALID";
	case STATUS_ALLOTTED_SPACE_EXCEEDED:	return "STATUS_ALLOTTED_SPACE_EXCEEDED";
	case STATUS_INSUFFICIENT_RESOURCES: 	return "STATUS_INSUFFICIENT_RESOURCES";
	case STATUS_DFS_EXIT_PATH_FOUND:		return "STATUS_DFS_EXIT_PATH_FOUND";
	case STATUS_DEVICE_DATA_ERROR:			return "STATUS_DEVICE_DATA_ERROR";
	case STATUS_DEVICE_NOT_CONNECTED:		return "STATUS_DEVICE_NOT_CONNECTED";
	case STATUS_DEVICE_POWER_FAILURE:		return "STATUS_DEVICE_POWER_FAILURE";
	case STATUS_FREE_VM_NOT_AT_BASE:		return "STATUS_FREE_VM_NOT_AT_BASE";
	case STATUS_MEMORY_NOT_ALLOCATED:		return "STATUS_MEMORY_NOT_ALLOCATED";
	case STATUS_WORKING_SET_QUOTA:			return "STATUS_WORKING_SET_QUOTA";
	case STATUS_MEDIA_WRITE_PROTECTED:		return "STATUS_MEDIA_WRITE_PROTECTED";
	case STATUS_DEVICE_NOT_READY:			return "STATUS_DEVICE_NOT_READY";
	case STATUS_INVALID_GROUP_ATTRIBUTES:	return "STATUS_INVALID_GROUP_ATTRIBUTES";
	case STATUS_BAD_IMPERSONATION_LEVEL:	return "STATUS_BAD_IMPERSONATION_LEVEL";
	case STATUS_CANT_OPEN_ANONYMOUS:		return "STATUS_CANT_OPEN_ANONYMOUS";
	case STATUS_BAD_VALIDATION_CLASS:		return "STATUS_BAD_VALIDATION_CLASS";
	case STATUS_BAD_TOKEN_TYPE: 			return "STATUS_BAD_TOKEN_TYPE";
	case STATUS_BAD_MASTER_BOOT_RECORD: 	return "STATUS_BAD_MASTER_BOOT_RECORD";
	case STATUS_INSTRUCTION_MISALIGNMENT:	return "STATUS_INSTRUCTION_MISALIGNMENT";
	case STATUS_INSTANCE_NOT_AVAILABLE: 	return "STATUS_INSTANCE_NOT_AVAILABLE";
	case STATUS_PIPE_NOT_AVAILABLE: 		return "STATUS_PIPE_NOT_AVAILABLE";
	case STATUS_INVALID_PIPE_STATE: 		return "STATUS_INVALID_PIPE_STATE";
	case STATUS_PIPE_BUSY:					return "STATUS_PIPE_BUSY";
	case STATUS_ILLEGAL_FUNCTION:			return "STATUS_ILLEGAL_FUNCTION";
	case STATUS_PIPE_DISCONNECTED:			return "STATUS_PIPE_DISCONNECTED";
	case STATUS_PIPE_CLOSING:				return "STATUS_PIPE_CLOSING";
	case STATUS_PIPE_CONNECTED: 			return "STATUS_PIPE_CONNECTED";
	case STATUS_PIPE_LISTENING: 			return "STATUS_PIPE_LISTENING";
	case STATUS_INVALID_READ_MODE:			return "STATUS_INVALID_READ_MODE";
	case STATUS_IO_TIMEOUT: 				return "STATUS_IO_TIMEOUT";
	case STATUS_FILE_FORCED_CLOSED: 		return "STATUS_FILE_FORCED_CLOSED";
	case STATUS_PROFILING_NOT_STARTED:		return "STATUS_PROFILING_NOT_STARTED";
	case STATUS_PROFILING_NOT_STOPPED:		return "STATUS_PROFILING_NOT_STOPPED";
	case STATUS_COULD_NOT_INTERPRET:		return "STATUS_COULD_NOT_INTERPRET";
	case STATUS_FILE_IS_A_DIRECTORY:		return "STATUS_FILE_IS_A_DIRECTORY";
	case STATUS_NOT_SUPPORTED:				return "STATUS_NOT_SUPPORTED";
	case STATUS_REMOTE_NOT_LISTENING:		return "STATUS_REMOTE_NOT_LISTENING";
	case STATUS_DUPLICATE_NAME: 			return "STATUS_DUPLICATE_NAME";
	case STATUS_BAD_NETWORK_PATH:			return "STATUS_BAD_NETWORK_PATH";
	case STATUS_NETWORK_BUSY:				return "STATUS_NETWORK_BUSY";
	case STATUS_DEVICE_DOES_NOT_EXIST:		return "STATUS_DEVICE_DOES_NOT_EXIST";
	case STATUS_TOO_MANY_COMMANDS:			return "STATUS_TOO_MANY_COMMANDS";
	case STATUS_ADAPTER_HARDWARE_ERROR: 	return "STATUS_ADAPTER_HARDWARE_ERROR";
	case STATUS_INVALID_NETWORK_RESPONSE:	return "STATUS_INVALID_NETWORK_RESPONSE";
	case STATUS_UNEXPECTED_NETWORK_ERROR:	return "STATUS_UNEXPECTED_NETWORK_ERROR";
	case STATUS_BAD_REMOTE_ADAPTER: 		return "STATUS_BAD_REMOTE_ADAPTER";
	case STATUS_PRINT_QUEUE_FULL:			return "STATUS_PRINT_QUEUE_FULL";
	case STATUS_NO_SPOOL_SPACE: 			return "STATUS_NO_SPOOL_SPACE";
	case STATUS_PRINT_CANCELLED:			return "STATUS_PRINT_CANCELLED";
	case STATUS_NETWORK_NAME_DELETED:		return "STATUS_NETWORK_NAME_DELETED";
	case STATUS_NETWORK_ACCESS_DENIED:		return "STATUS_NETWORK_ACCESS_DENIED";
	case STATUS_BAD_DEVICE_TYPE:			return "STATUS_BAD_DEVICE_TYPE";
	case STATUS_BAD_NETWORK_NAME:			return "STATUS_BAD_NETWORK_NAME";
	case STATUS_TOO_MANY_NAMES: 			return "STATUS_TOO_MANY_NAMES";
	case STATUS_TOO_MANY_SESSIONS:			return "STATUS_TOO_MANY_SESSIONS";
	case STATUS_SHARING_PAUSED: 			return "STATUS_SHARING_PAUSED";
	case STATUS_REQUEST_NOT_ACCEPTED:		return "STATUS_REQUEST_NOT_ACCEPTED";
	case STATUS_REDIRECTOR_PAUSED:			return "STATUS_REDIRECTOR_PAUSED";
	case STATUS_NET_WRITE_FAULT:			return "STATUS_NET_WRITE_FAULT";
	case STATUS_PROFILING_AT_LIMIT: 		return "STATUS_PROFILING_AT_LIMIT";
	case STATUS_NOT_SAME_DEVICE:			return "STATUS_NOT_SAME_DEVICE";
	case STATUS_FILE_RENAMED:				return "STATUS_FILE_RENAMED";
	case STATUS_VIRTUAL_CIRCUIT_CLOSED: 	return "STATUS_VIRTUAL_CIRCUIT_CLOSED";
	case STATUS_NO_SECURITY_ON_OBJECT:		return "STATUS_NO_SECURITY_ON_OBJECT";
	case STATUS_CANT_WAIT:					return "STATUS_CANT_WAIT";
	case STATUS_PIPE_EMPTY: 				return "STATUS_PIPE_EMPTY";
	case STATUS_CANT_ACCESS_DOMAIN_INFO:	return "STATUS_CANT_ACCESS_DOMAIN_INFO";
	case STATUS_CANT_TERMINATE_SELF:		return "STATUS_CANT_TERMINATE_SELF";
	case STATUS_INVALID_SERVER_STATE:		return "STATUS_INVALID_SERVER_STATE";
	case STATUS_INVALID_DOMAIN_STATE:		return "STATUS_INVALID_DOMAIN_STATE";
	case STATUS_INVALID_DOMAIN_ROLE:		return "STATUS_INVALID_DOMAIN_ROLE";
	case STATUS_NO_SUCH_DOMAIN: 			return "STATUS_NO_SUCH_DOMAIN";
	case STATUS_DOMAIN_EXISTS:				return "STATUS_DOMAIN_EXISTS";
	case STATUS_DOMAIN_LIMIT_EXCEEDED:		return "STATUS_DOMAIN_LIMIT_EXCEEDED";
	case STATUS_OPLOCK_NOT_GRANTED: 		return "STATUS_OPLOCK_NOT_GRANTED";
	case STATUS_INVALID_OPLOCK_PROTOCOL:	return "STATUS_INVALID_OPLOCK_PROTOCOL";
	case STATUS_INTERNAL_DB_CORRUPTION: 	return "STATUS_INTERNAL_DB_CORRUPTION";
	case STATUS_INTERNAL_ERROR: 			return "STATUS_INTERNAL_ERROR";
	case STATUS_GENERIC_NOT_MAPPED: 		return "STATUS_GENERIC_NOT_MAPPED";
	case STATUS_BAD_DESCRIPTOR_FORMAT:		return "STATUS_BAD_DESCRIPTOR_FORMAT";
	case STATUS_INVALID_USER_BUFFER:		return "STATUS_INVALID_USER_BUFFER";
	case STATUS_UNEXPECTED_IO_ERROR:		return "STATUS_UNEXPECTED_IO_ERROR";
	case STATUS_UNEXPECTED_MM_CREATE_ERR:	return "STATUS_UNEXPECTED_MM_CREATE_ERR";
	case STATUS_UNEXPECTED_MM_MAP_ERROR:	return "STATUS_UNEXPECTED_MM_MAP_ERROR";
	case STATUS_UNEXPECTED_MM_EXTEND_ERR:	return "STATUS_UNEXPECTED_MM_EXTEND_ERR";
	case STATUS_NOT_LOGON_PROCESS:			return "STATUS_NOT_LOGON_PROCESS";
	case STATUS_LOGON_SESSION_EXISTS:		return "STATUS_LOGON_SESSION_EXISTS";
	case STATUS_INVALID_PARAMETER_1:		return "STATUS_INVALID_PARAMETER_1";
	case STATUS_INVALID_PARAMETER_2:		return "STATUS_INVALID_PARAMETER_2";
	case STATUS_INVALID_PARAMETER_3:		return "STATUS_INVALID_PARAMETER_3";
	case STATUS_INVALID_PARAMETER_4:		return "STATUS_INVALID_PARAMETER_4";
	case STATUS_INVALID_PARAMETER_5:		return "STATUS_INVALID_PARAMETER_5";
	case STATUS_INVALID_PARAMETER_6:		return "STATUS_INVALID_PARAMETER_6";
	case STATUS_INVALID_PARAMETER_7:		return "STATUS_INVALID_PARAMETER_7";
	case STATUS_INVALID_PARAMETER_8:		return "STATUS_INVALID_PARAMETER_8";
	case STATUS_INVALID_PARAMETER_9:		return "STATUS_INVALID_PARAMETER_9";
	case STATUS_INVALID_PARAMETER_10:		return "STATUS_INVALID_PARAMETER_10";
	case STATUS_INVALID_PARAMETER_11:		return "STATUS_INVALID_PARAMETER_11";
	case STATUS_INVALID_PARAMETER_12:		return "STATUS_INVALID_PARAMETER_12";
	case STATUS_REDIRECTOR_NOT_STARTED: 	return "STATUS_REDIRECTOR_NOT_STARTED";
	case STATUS_REDIRECTOR_STARTED: 		return "STATUS_REDIRECTOR_STARTED";
	case STATUS_STACK_OVERFLOW: 			return "STATUS_STACK_OVERFLOW";
	case STATUS_NO_SUCH_PACKAGE:			return "STATUS_NO_SUCH_PACKAGE";
	case STATUS_BAD_FUNCTION_TABLE: 		return "STATUS_BAD_FUNCTION_TABLE";
	case STATUS_VARIABLE_NOT_FOUND: 		return "STATUS_VARIABLE_NOT_FOUND";
	case STATUS_DIRECTORY_NOT_EMPTY:		return "STATUS_DIRECTORY_NOT_EMPTY";
	case STATUS_FILE_CORRUPT_ERROR: 		return "STATUS_FILE_CORRUPT_ERROR";
	case STATUS_NOT_A_DIRECTORY:			return "STATUS_NOT_A_DIRECTORY";
	case STATUS_BAD_LOGON_SESSION_STATE:	return "STATUS_BAD_LOGON_SESSION_STATE";
	case STATUS_LOGON_SESSION_COLLISION:	return "STATUS_LOGON_SESSION_COLLISION";
	case STATUS_NAME_TOO_LONG:				return "STATUS_NAME_TOO_LONG";
	case STATUS_FILES_OPEN: 				return "STATUS_FILES_OPEN";
	case STATUS_CONNECTION_IN_USE:			return "STATUS_CONNECTION_IN_USE";
	case STATUS_MESSAGE_NOT_FOUND:			return "STATUS_MESSAGE_NOT_FOUND";
	case STATUS_PROCESS_IS_TERMINATING: 	return "STATUS_PROCESS_IS_TERMINATING";
	case STATUS_INVALID_LOGON_TYPE: 		return "STATUS_INVALID_LOGON_TYPE";
	case STATUS_NO_GUID_TRANSLATION:		return "STATUS_NO_GUID_TRANSLATION";
	case STATUS_CANNOT_IMPERSONATE: 		return "STATUS_CANNOT_IMPERSONATE";
	case STATUS_IMAGE_ALREADY_LOADED:		return "STATUS_IMAGE_ALREADY_LOADED";
	case STATUS_ABIOS_NOT_PRESENT:			return "STATUS_ABIOS_NOT_PRESENT";
	case STATUS_ABIOS_LID_NOT_EXIST:		return "STATUS_ABIOS_LID_NOT_EXIST";
	case STATUS_ABIOS_LID_ALREADY_OWNED:	return "STATUS_ABIOS_LID_ALREADY_OWNED";
	case STATUS_ABIOS_NOT_LID_OWNER:		return "STATUS_ABIOS_NOT_LID_OWNER";
	case STATUS_ABIOS_INVALID_COMMAND:		return "STATUS_ABIOS_INVALID_COMMAND";
	case STATUS_ABIOS_INVALID_LID:			return "STATUS_ABIOS_INVALID_LID";
	case STATUS_ABIOS_SELECTOR_NOT_AVAILABLE:
											return "STATUS_ABIOS_SELECTOR_NOT_AVAILABLE";
	case STATUS_ABIOS_INVALID_SELECTOR: 	return "STATUS_ABIOS_INVALID_SELECTOR";
	case STATUS_NO_LDT: 					return "STATUS_NO_LDT";
	case STATUS_INVALID_LDT_SIZE:			return "STATUS_INVALID_LDT_SIZE";
	case STATUS_INVALID_LDT_OFFSET: 		return "STATUS_INVALID_LDT_OFFSET";
	case STATUS_INVALID_LDT_DESCRIPTOR: 	return "STATUS_INVALID_LDT_DESCRIPTOR";
	case STATUS_INVALID_IMAGE_NE_FORMAT:	return "STATUS_INVALID_IMAGE_NE_FORMAT";
	case STATUS_RXACT_INVALID_STATE:		return "STATUS_RXACT_INVALID_STATE";
	case STATUS_RXACT_COMMIT_FAILURE:		return "STATUS_RXACT_COMMIT_FAILURE";
	case STATUS_MAPPED_FILE_SIZE_ZERO:		return "STATUS_MAPPED_FILE_SIZE_ZERO";
	case STATUS_TOO_MANY_OPENED_FILES:		return "STATUS_TOO_MANY_OPENED_FILES";
	case STATUS_CANCELLED:					return "STATUS_CANCELLED";
	case STATUS_CANNOT_DELETE:				return "STATUS_CANNOT_DELETE";
	case STATUS_INVALID_COMPUTER_NAME:		return "STATUS_INVALID_COMPUTER_NAME";
	case STATUS_FILE_DELETED:				return "STATUS_FILE_DELETED";
	case STATUS_SPECIAL_ACCOUNT:			return "STATUS_SPECIAL_ACCOUNT";
	case STATUS_SPECIAL_GROUP:				return "STATUS_SPECIAL_GROUP";
	case STATUS_SPECIAL_USER:				return "STATUS_SPECIAL_USER";
	case STATUS_MEMBERS_PRIMARY_GROUP:		return "STATUS_MEMBERS_PRIMARY_GROUP";
	case STATUS_FILE_CLOSED:				return "STATUS_FILE_CLOSED";
	case STATUS_TOO_MANY_THREADS:			return "STATUS_TOO_MANY_THREADS";
	case STATUS_THREAD_NOT_IN_PROCESS:		return "STATUS_THREAD_NOT_IN_PROCESS";
	case STATUS_TOKEN_ALREADY_IN_USE:		return "STATUS_TOKEN_ALREADY_IN_USE";
	case STATUS_PAGEFILE_QUOTA_EXCEEDED:	return "STATUS_PAGEFILE_QUOTA_EXCEEDED";
	case STATUS_COMMITMENT_LIMIT:			return "STATUS_COMMITMENT_LIMIT";
	case STATUS_INVALID_IMAGE_LE_FORMAT:	return "STATUS_INVALID_IMAGE_LE_FORMAT";
	case STATUS_INVALID_IMAGE_NOT_MZ:		return "STATUS_INVALID_IMAGE_NOT_MZ";
	case STATUS_INVALID_IMAGE_PROTECT:		return "STATUS_INVALID_IMAGE_PROTECT";
	case STATUS_INVALID_IMAGE_WIN_16:		return "STATUS_INVALID_IMAGE_WIN_16";
	case STATUS_LOGON_SERVER_CONFLICT:		return "STATUS_LOGON_SERVER_CONFLICT";
	case STATUS_TIME_DIFFERENCE_AT_DC:		return "STATUS_TIME_DIFFERENCE_AT_DC";
	case STATUS_SYNCHRONIZATION_REQUIRED:	return "STATUS_SYNCHRONIZATION_REQUIRED";
	case STATUS_DLL_NOT_FOUND:				return "STATUS_DLL_NOT_FOUND";
	case STATUS_OPEN_FAILED:				return "STATUS_OPEN_FAILED";
	case STATUS_IO_PRIVILEGE_FAILED:		return "STATUS_IO_PRIVILEGE_FAILED";
	case STATUS_ORDINAL_NOT_FOUND:			return "STATUS_ORDINAL_NOT_FOUND";
	case STATUS_ENTRYPOINT_NOT_FOUND:		return "STATUS_ENTRYPOINT_NOT_FOUND";
	case STATUS_CONTROL_C_EXIT: 			return "STATUS_CONTROL_C_EXIT";
	case STATUS_LOCAL_DISCONNECT:			return "STATUS_LOCAL_DISCONNECT";
	case STATUS_REMOTE_DISCONNECT:			return "STATUS_REMOTE_DISCONNECT";
	case STATUS_REMOTE_RESOURCES:			return "STATUS_REMOTE_RESOURCES";
	case STATUS_LINK_FAILED:				return "STATUS_LINK_FAILED";
	case STATUS_LINK_TIMEOUT:				return "STATUS_LINK_TIMEOUT";
	case STATUS_INVALID_CONNECTION: 		return "STATUS_INVALID_CONNECTION";
	case STATUS_INVALID_ADDRESS:			return "STATUS_INVALID_ADDRESS";
	case STATUS_DLL_INIT_FAILED:			return "STATUS_DLL_INIT_FAILED";
	case STATUS_MISSING_SYSTEMFILE: 		return "STATUS_MISSING_SYSTEMFILE";
	case STATUS_UNHANDLED_EXCEPTION:		return "STATUS_UNHANDLED_EXCEPTION";
	case STATUS_APP_INIT_FAILURE:			return "STATUS_APP_INIT_FAILURE";
	case STATUS_PAGEFILE_CREATE_FAILED: 	return "STATUS_PAGEFILE_CREATE_FAILED";
	case STATUS_NO_PAGEFILE:				return "STATUS_NO_PAGEFILE";
	case STATUS_INVALID_LEVEL:				return "STATUS_INVALID_LEVEL";
	case STATUS_WRONG_PASSWORD_CORE:		return "STATUS_WRONG_PASSWORD_CORE";
	case STATUS_ILLEGAL_FLOAT_CONTEXT:		return "STATUS_ILLEGAL_FLOAT_CONTEXT";
	case STATUS_PIPE_BROKEN:				return "STATUS_PIPE_BROKEN";
	case STATUS_REGISTRY_CORRUPT:			return "STATUS_REGISTRY_CORRUPT";
	case STATUS_REGISTRY_IO_FAILED: 		return "STATUS_REGISTRY_IO_FAILED";
	case STATUS_NO_EVENT_PAIR:				return "STATUS_NO_EVENT_PAIR";
	case STATUS_UNRECOGNIZED_VOLUME:		return "STATUS_UNRECOGNIZED_VOLUME";
	case STATUS_SERIAL_NO_DEVICE_INITED:	return "STATUS_SERIAL_NO_DEVICE_INITED";
	case STATUS_NO_SUCH_ALIAS:				return "STATUS_NO_SUCH_ALIAS";
	case STATUS_MEMBER_NOT_IN_ALIAS:		return "STATUS_MEMBER_NOT_IN_ALIAS";
	case STATUS_MEMBER_IN_ALIAS:			return "STATUS_MEMBER_IN_ALIAS";
	case STATUS_ALIAS_EXISTS:				return "STATUS_ALIAS_EXISTS";
	case STATUS_LOGON_NOT_GRANTED:			return "STATUS_LOGON_NOT_GRANTED";
	case STATUS_TOO_MANY_SECRETS:			return "STATUS_TOO_MANY_SECRETS";
	case STATUS_SECRET_TOO_LONG:			return "STATUS_SECRET_TOO_LONG";
	case STATUS_INTERNAL_DB_ERROR:			return "STATUS_INTERNAL_DB_ERROR";
	case STATUS_FULLSCREEN_MODE:			return "STATUS_FULLSCREEN_MODE";
	case STATUS_TOO_MANY_CONTEXT_IDS:		return "STATUS_TOO_MANY_CONTEXT_IDS";
	case STATUS_LOGON_TYPE_NOT_GRANTED: 	return "STATUS_LOGON_TYPE_NOT_GRANTED";
	case STATUS_NOT_REGISTRY_FILE:			return "STATUS_NOT_REGISTRY_FILE";
	case STATUS_NT_CROSS_ENCRYPTION_REQUIRED:
											return "STATUS_NT_CROSS_ENCRYPTION_REQUIRED";
	case STATUS_DOMAIN_CTRLR_CONFIG_ERROR:	return "STATUS_DOMAIN_CTRLR_CONFIG_ERROR";
	case STATUS_FT_MISSING_MEMBER:			return "STATUS_FT_MISSING_MEMBER";
	case STATUS_ILL_FORMED_SERVICE_ENTRY:	return "STATUS_ILL_FORMED_SERVICE_ENTRY";
	case STATUS_ILLEGAL_CHARACTER:			return "STATUS_ILLEGAL_CHARACTER";
	case STATUS_UNMAPPABLE_CHARACTER:		return "STATUS_UNMAPPABLE_CHARACTER";
	case STATUS_UNDEFINED_CHARACTER:		return "STATUS_UNDEFINED_CHARACTER";
	case STATUS_FLOPPY_VOLUME:				return "STATUS_FLOPPY_VOLUME";
	case STATUS_FLOPPY_ID_MARK_NOT_FOUND:	return "STATUS_FLOPPY_ID_MARK_NOT_FOUND";
	case STATUS_FLOPPY_WRONG_CYLINDER:		return "STATUS_FLOPPY_WRONG_CYLINDER";
	case STATUS_FLOPPY_UNKNOWN_ERROR:		return "STATUS_FLOPPY_UNKNOWN_ERROR";
	case STATUS_FLOPPY_BAD_REGISTERS:		return "STATUS_FLOPPY_BAD_REGISTERS";
	case STATUS_DISK_RECALIBRATE_FAILED:	return "STATUS_DISK_RECALIBRATE_FAILED";
	case STATUS_DISK_OPERATION_FAILED:		return "STATUS_DISK_OPERATION_FAILED";
	case STATUS_DISK_RESET_FAILED:			return "STATUS_DISK_RESET_FAILED";
	case STATUS_SHARED_IRQ_BUSY:			return "STATUS_SHARED_IRQ_BUSY";
	case STATUS_FT_ORPHANING:				return "STATUS_FT_ORPHANING";
	case STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT:
											return "STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT";
	case STATUS_PARTITION_FAILURE:			return "STATUS_PARTITION_FAILURE";
	case STATUS_INVALID_BLOCK_LENGTH:		return "STATUS_INVALID_BLOCK_LENGTH";
	case STATUS_DEVICE_NOT_PARTITIONED: 	return "STATUS_DEVICE_NOT_PARTITIONED";
	case STATUS_UNABLE_TO_LOCK_MEDIA:		return "STATUS_UNABLE_TO_LOCK_MEDIA";
	case STATUS_UNABLE_TO_UNLOAD_MEDIA: 	return "STATUS_UNABLE_TO_UNLOAD_MEDIA";
	case STATUS_EOM_OVERFLOW:				return "STATUS_EOM_OVERFLOW";
	case STATUS_NO_MEDIA:					return "STATUS_NO_MEDIA";
	case STATUS_NO_SUCH_MEMBER: 			return "STATUS_NO_SUCH_MEMBER";
	case STATUS_INVALID_MEMBER: 			return "STATUS_INVALID_MEMBER";
	case STATUS_KEY_DELETED:				return "STATUS_KEY_DELETED";
	case STATUS_NO_LOG_SPACE:				return "STATUS_NO_LOG_SPACE";
	case STATUS_TOO_MANY_SIDS:				return "STATUS_TOO_MANY_SIDS";
	case STATUS_LM_CROSS_ENCRYPTION_REQUIRED:
											return "STATUS_LM_CROSS_ENCRYPTION_REQUIRED";
	case STATUS_KEY_HAS_CHILDREN:			return "STATUS_KEY_HAS_CHILDREN";
	case STATUS_CHILD_MUST_BE_VOLATILE: 	return "STATUS_CHILD_MUST_BE_VOLATILE";
	case STATUS_DEVICE_CONFIGURATION_ERROR: return "STATUS_DEVICE_CONFIGURATION_ERROR";
	case STATUS_DRIVER_INTERNAL_ERROR:		return "STATUS_DRIVER_INTERNAL_ERROR";
	case STATUS_INVALID_DEVICE_STATE:		return "STATUS_INVALID_DEVICE_STATE";
	case STATUS_IO_DEVICE_ERROR:			return "STATUS_IO_DEVICE_ERROR";
	case STATUS_DEVICE_PROTOCOL_ERROR:		return "STATUS_DEVICE_PROTOCOL_ERROR";
	case STATUS_BACKUP_CONTROLLER:			return "STATUS_BACKUP_CONTROLLER";
	case STATUS_LOG_FILE_FULL:				return "STATUS_LOG_FILE_FULL";
	case STATUS_TOO_LATE:					return "STATUS_TOO_LATE";
	case STATUS_NO_TRUST_LSA_SECRET:		return "STATUS_NO_TRUST_LSA_SECRET";
	case STATUS_NO_TRUST_SAM_ACCOUNT:		return "STATUS_NO_TRUST_SAM_ACCOUNT";
	case STATUS_TRUSTED_DOMAIN_FAILURE: 	return "STATUS_TRUSTED_DOMAIN_FAILURE";
	case STATUS_TRUSTED_RELATIONSHIP_FAILURE:
											return "STATUS_TRUSTED_RELATIONSHIP_FAILURE";
	case STATUS_EVENTLOG_FILE_CORRUPT:		return "STATUS_EVENTLOG_FILE_CORRUPT";
	case STATUS_EVENTLOG_CANT_START:		return "STATUS_EVENTLOG_CANT_START";
	case STATUS_TRUST_FAILURE:				return "STATUS_TRUST_FAILURE";
	case STATUS_MUTANT_LIMIT_EXCEEDED:		return "STATUS_MUTANT_LIMIT_EXCEEDED";
	case STATUS_NETLOGON_NOT_STARTED:		return "STATUS_NETLOGON_NOT_STARTED";
	case STATUS_ACCOUNT_EXPIRED:			return "STATUS_ACCOUNT_EXPIRED";
	case STATUS_POSSIBLE_DEADLOCK:			return "STATUS_POSSIBLE_DEADLOCK";
	case STATUS_NETWORK_CREDENTIAL_CONFLICT:return "STATUS_NETWORK_CREDENTIAL_CONFLICT";
	case STATUS_REMOTE_SESSION_LIMIT:		return "STATUS_REMOTE_SESSION_LIMIT";
	case STATUS_EVENTLOG_FILE_CHANGED:		return "STATUS_EVENTLOG_FILE_CHANGED";
	case STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT:
											return "STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT";
	case STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT:
											return "STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT";
	case STATUS_NOLOGON_SERVER_TRUST_ACCOUNT:
											return "STATUS_NOLOGON_SERVER_TRUST_ACCOUNT";
	case STATUS_DOMAIN_TRUST_INCONSISTENT:	return "STATUS_DOMAIN_TRUST_INCONSISTENT";
	case STATUS_FS_DRIVER_REQUIRED: 		return "STATUS_FS_DRIVER_REQUIRED";
	case STATUS_NO_USER_SESSION_KEY:		return "STATUS_NO_USER_SESSION_KEY";
	case STATUS_USER_SESSION_DELETED:		return "STATUS_USER_SESSION_DELETED";
	case STATUS_RESOURCE_LANG_NOT_FOUND:	return "STATUS_RESOURCE_LANG_NOT_FOUND";
	case STATUS_INSUFF_SERVER_RESOURCES:	return "STATUS_INSUFF_SERVER_RESOURCES";
	case STATUS_INVALID_BUFFER_SIZE:		return "STATUS_INVALID_BUFFER_SIZE";
	case STATUS_INVALID_ADDRESS_COMPONENT:	return "STATUS_INVALID_ADDRESS_COMPONENT";
	case STATUS_INVALID_ADDRESS_WILDCARD:	return "STATUS_INVALID_ADDRESS_WILDCARD";
	case STATUS_TOO_MANY_ADDRESSES: 		return "STATUS_TOO_MANY_ADDRESSES";
	case STATUS_ADDRESS_ALREADY_EXISTS: 	return "STATUS_ADDRESS_ALREADY_EXISTS";
	case STATUS_ADDRESS_CLOSED: 			return "STATUS_ADDRESS_CLOSED";
	case STATUS_CONNECTION_DISCONNECTED:	return "STATUS_CONNECTION_DISCONNECTED";
	case STATUS_CONNECTION_RESET:			return "STATUS_CONNECTION_RESET";
	case STATUS_TOO_MANY_NODES: 			return "STATUS_TOO_MANY_NODES";
	case STATUS_TRANSACTION_ABORTED:		return "STATUS_TRANSACTION_ABORTED";
	case STATUS_TRANSACTION_TIMED_OUT:		return "STATUS_TRANSACTION_TIMED_OUT";
	case STATUS_TRANSACTION_NO_RELEASE: 	return "STATUS_TRANSACTION_NO_RELEASE";
	case STATUS_TRANSACTION_NO_MATCH:		return "STATUS_TRANSACTION_NO_MATCH";
	case STATUS_TRANSACTION_RESPONDED:		return "STATUS_TRANSACTION_RESPONDED";
	case STATUS_TRANSACTION_INVALID_ID: 	return "STATUS_TRANSACTION_INVALID_ID";
	case STATUS_TRANSACTION_INVALID_TYPE:	return "STATUS_TRANSACTION_INVALID_TYPE";
	case STATUS_NOT_SERVER_SESSION: 		return "STATUS_NOT_SERVER_SESSION";
	case STATUS_NOT_CLIENT_SESSION: 		return "STATUS_NOT_CLIENT_SESSION";
	case STATUS_CANNOT_LOAD_REGISTRY_FILE:	return "STATUS_CANNOT_LOAD_REGISTRY_FILE";
	case STATUS_DEBUG_ATTACH_FAILED:		return "STATUS_DEBUG_ATTACH_FAILED";
	case STATUS_SYSTEM_PROCESS_TERMINATED:	return "STATUS_SYSTEM_PROCESS_TERMINATED";
	case STATUS_DATA_NOT_ACCEPTED:			return "STATUS_DATA_NOT_ACCEPTED";
	case STATUS_NO_BROWSER_SERVERS_FOUND:	return "STATUS_NO_BROWSER_SERVERS_FOUND";
	case STATUS_VDM_HARD_ERROR: 			return "STATUS_VDM_HARD_ERROR";
	case STATUS_DRIVER_CANCEL_TIMEOUT:		return "STATUS_DRIVER_CANCEL_TIMEOUT";
	case STATUS_REPLY_MESSAGE_MISMATCH: 	return "STATUS_REPLY_MESSAGE_MISMATCH";
	case STATUS_MAPPED_ALIGNMENT:			return "STATUS_MAPPED_ALIGNMENT";
	case STATUS_IMAGE_CHECKSUM_MISMATCH:	return "STATUS_IMAGE_CHECKSUM_MISMATCH";
	case STATUS_LOST_WRITEBEHIND_DATA:		return "STATUS_LOST_WRITEBEHIND_DATA";
	case STATUS_CLIENT_SERVER_PARAMETERS_INVALID:
											return "STATUS_CLIENT_SERVER_PARAMETERS_INVALID";
	case STATUS_PASSWORD_MUST_CHANGE:		return "STATUS_PASSWORD_MUST_CHANGE";
	case STATUS_NOT_FOUND:					return "STATUS_NOT_FOUND";
	case STATUS_NOT_TINY_STREAM:			return "STATUS_NOT_TINY_STREAM";
	case STATUS_RECOVERY_FAILURE:			return "STATUS_RECOVERY_FAILURE";
	case STATUS_STACK_OVERFLOW_READ:		return "STATUS_STACK_OVERFLOW_READ";
	case STATUS_FAIL_CHECK: 				return "STATUS_FAIL_CHECK";
	case STATUS_DUPLICATE_OBJECTID: 		return "STATUS_DUPLICATE_OBJECTID";
	case STATUS_OBJECTID_EXISTS:			return "STATUS_OBJECTID_EXISTS";
	case STATUS_CONVERT_TO_LARGE:			return "STATUS_CONVERT_TO_LARGE";
	case STATUS_RETRY:						return "STATUS_RETRY";
	case STATUS_FOUND_OUT_OF_SCOPE: 		return "STATUS_FOUND_OUT_OF_SCOPE";
	case STATUS_ALLOCATE_BUCKET:			return "STATUS_ALLOCATE_BUCKET";
	case STATUS_PROPSET_NOT_FOUND:			return "STATUS_PROPSET_NOT_FOUND";
	case STATUS_MARSHALL_OVERFLOW:			return "STATUS_MARSHALL_OVERFLOW";
	case STATUS_INVALID_VARIANT:			return "STATUS_INVALID_VARIANT";
	case STATUS_DOMAIN_CONTROLLER_NOT_FOUND:return "STATUS_DOMAIN_CONTROLLER_NOT_FOUND";
	case STATUS_ACCOUNT_LOCKED_OUT: 		return "STATUS_ACCOUNT_LOCKED_OUT";
	case STATUS_HANDLE_NOT_CLOSABLE:		return "STATUS_HANDLE_NOT_CLOSABLE";
	case STATUS_CONNECTION_REFUSED: 		return "STATUS_CONNECTION_REFUSED";
	case STATUS_GRACEFUL_DISCONNECT:		return "STATUS_GRACEFUL_DISCONNECT";
	case STATUS_ADDRESS_ALREADY_ASSOCIATED: return "STATUS_ADDRESS_ALREADY_ASSOCIATED";
	case STATUS_ADDRESS_NOT_ASSOCIATED: 	return "STATUS_ADDRESS_NOT_ASSOCIATED";
	case STATUS_CONNECTION_INVALID: 		return "STATUS_CONNECTION_INVALID";
	case STATUS_CONNECTION_ACTIVE:			return "STATUS_CONNECTION_ACTIVE";
	case STATUS_NETWORK_UNREACHABLE:		return "STATUS_NETWORK_UNREACHABLE";
	case STATUS_HOST_UNREACHABLE:			return "STATUS_HOST_UNREACHABLE";
	case STATUS_PROTOCOL_UNREACHABLE:		return "STATUS_PROTOCOL_UNREACHABLE";
	case STATUS_PORT_UNREACHABLE:			return "STATUS_PORT_UNREACHABLE";
	case STATUS_REQUEST_ABORTED:			return "STATUS_REQUEST_ABORTED";
	case STATUS_CONNECTION_ABORTED: 		return "STATUS_CONNECTION_ABORTED";
	case STATUS_BAD_COMPRESSION_BUFFER: 	return "STATUS_BAD_COMPRESSION_BUFFER";
	case STATUS_USER_MAPPED_FILE:			return "STATUS_USER_MAPPED_FILE";
	case STATUS_AUDIT_FAILED:				return "STATUS_AUDIT_FAILED";
	case STATUS_TIMER_RESOLUTION_NOT_SET:	return "STATUS_TIMER_RESOLUTION_NOT_SET";
	case STATUS_CONNECTION_COUNT_LIMIT: 	return "STATUS_CONNECTION_COUNT_LIMIT";
	case STATUS_LOGIN_TIME_RESTRICTION: 	return "STATUS_LOGIN_TIME_RESTRICTION";
	case STATUS_LOGIN_WKSTA_RESTRICTION:	return "STATUS_LOGIN_WKSTA_RESTRICTION";
	case STATUS_IMAGE_MP_UP_MISMATCH:		return "STATUS_IMAGE_MP_UP_MISMATCH";
	case STATUS_INSUFFICIENT_LOGON_INFO:	return "STATUS_INSUFFICIENT_LOGON_INFO";
	case STATUS_BAD_DLL_ENTRYPOINT: 		return "STATUS_BAD_DLL_ENTRYPOINT";
	case STATUS_BAD_SERVICE_ENTRYPOINT: 	return "STATUS_BAD_SERVICE_ENTRYPOINT";
	case STATUS_LPC_REPLY_LOST: 			return "STATUS_LPC_REPLY_LOST";
	case STATUS_IP_ADDRESS_CONFLICT1:		return "STATUS_IP_ADDRESS_CONFLICT1";
	case STATUS_IP_ADDRESS_CONFLICT2:		return "STATUS_IP_ADDRESS_CONFLICT2";
	case STATUS_REGISTRY_QUOTA_LIMIT:		return "STATUS_REGISTRY_QUOTA_LIMIT";
	case STATUS_PATH_NOT_COVERED:			return "STATUS_PATH_NOT_COVERED";
	case STATUS_NO_CALLBACK_ACTIVE: 		return "STATUS_NO_CALLBACK_ACTIVE";
	case STATUS_LICENSE_QUOTA_EXCEEDED: 	return "STATUS_LICENSE_QUOTA_EXCEEDED";
	case STATUS_PWD_TOO_SHORT:				return "STATUS_PWD_TOO_SHORT";
	case STATUS_PWD_TOO_RECENT: 			return "STATUS_PWD_TOO_RECENT";
	case STATUS_PWD_HISTORY_CONFLICT:		return "STATUS_PWD_HISTORY_CONFLICT";
	case STATUS_PLUGPLAY_NO_DEVICE: 		return "STATUS_PLUGPLAY_NO_DEVICE";
	case STATUS_UNSUPPORTED_COMPRESSION:	return "STATUS_UNSUPPORTED_COMPRESSION";
	case STATUS_INVALID_HW_PROFILE: 		return "STATUS_INVALID_HW_PROFILE";
	case STATUS_INVALID_PLUGPLAY_DEVICE_PATH:
											return "STATUS_INVALID_PLUGPLAY_DEVICE_PATH";
	case STATUS_DRIVER_ORDINAL_NOT_FOUND:	return "STATUS_DRIVER_ORDINAL_NOT_FOUND";
	case STATUS_DRIVER_ENTRYPOINT_NOT_FOUND:return "STATUS_DRIVER_ENTRYPOINT_NOT_FOUND";
	case STATUS_RESOURCE_NOT_OWNED: 		return "STATUS_RESOURCE_NOT_OWNED";
	case STATUS_TOO_MANY_LINKS: 			return "STATUS_TOO_MANY_LINKS";
	case STATUS_QUOTA_LIST_INCONSISTENT:	return "STATUS_QUOTA_LIST_INCONSISTENT";
	case STATUS_FILE_IS_OFFLINE:			return "STATUS_FILE_IS_OFFLINE";
	case STATUS_EVALUATION_EXPIRATION:		return "STATUS_EVALUATION_EXPIRATION";
	case STATUS_ILLEGAL_DLL_RELOCATION: 	return "STATUS_ILLEGAL_DLL_RELOCATION";
	case STATUS_LICENSE_VIOLATION:			return "STATUS_LICENSE_VIOLATION";
	case STATUS_DLL_INIT_FAILED_LOGOFF: 	return "STATUS_DLL_INIT_FAILED_LOGOFF";
	case STATUS_DRIVER_UNABLE_TO_LOAD:		return "STATUS_DRIVER_UNABLE_TO_LOAD";
	case STATUS_DFS_UNAVAILABLE:			return "STATUS_DFS_UNAVAILABLE";
	case STATUS_VOLUME_DISMOUNTED:			return "STATUS_VOLUME_DISMOUNTED";
	case STATUS_WX86_INTERNAL_ERROR:		return "STATUS_WX86_INTERNAL_ERROR";
	case STATUS_WX86_FLOAT_STACK_CHECK: 	return "STATUS_WX86_FLOAT_STACK_CHECK";
	case STATUS_VALIDATE_CONTINUE:			return "STATUS_VALIDATE_CONTINUE";
	case STATUS_NO_MATCH:					return "STATUS_NO_MATCH";
	case STATUS_NO_MORE_MATCHES:			return "STATUS_NO_MORE_MATCHES";
	case STATUS_NOT_A_REPARSE_POINT:		return "STATUS_NOT_A_REPARSE_POINT";
	case STATUS_IO_REPARSE_TAG_INVALID: 	return "STATUS_IO_REPARSE_TAG_INVALID";
	case STATUS_IO_REPARSE_TAG_MISMATCH:	return "STATUS_IO_REPARSE_TAG_MISMATCH";
	case STATUS_IO_REPARSE_DATA_INVALID:	return "STATUS_IO_REPARSE_DATA_INVALID";
	case STATUS_IO_REPARSE_TAG_NOT_HANDLED: return "STATUS_IO_REPARSE_TAG_NOT_HANDLED";
	case STATUS_REPARSE_POINT_NOT_RESOLVED: return "STATUS_REPARSE_POINT_NOT_RESOLVED";
	case STATUS_DIRECTORY_IS_A_REPARSE_POINT:
											return "STATUS_DIRECTORY_IS_A_REPARSE_POINT";
	case STATUS_RANGE_LIST_CONFLICT:		return "STATUS_RANGE_LIST_CONFLICT";
	case STATUS_SOURCE_ELEMENT_EMPTY:		return "STATUS_SOURCE_ELEMENT_EMPTY";
	case STATUS_DESTINATION_ELEMENT_FULL:	return "STATUS_DESTINATION_ELEMENT_FULL";
	case STATUS_ILLEGAL_ELEMENT_ADDRESS:	return "STATUS_ILLEGAL_ELEMENT_ADDRESS";
	case STATUS_MAGAZINE_NOT_PRESENT:		return "STATUS_MAGAZINE_NOT_PRESENT";
	case STATUS_REINITIALIZATION_NEEDED:	return "STATUS_REINITIALIZATION_NEEDED";
	case STATUS_DEVICE_REQUIRES_CLEANING:	return "STATUS_DEVICE_REQUIRES_CLEANING";
	case STATUS_DEVICE_DOOR_OPEN:			return "STATUS_DEVICE_DOOR_OPEN";
	case STATUS_ENCRYPTION_FAILED:			return "STATUS_ENCRYPTION_FAILED";
	case STATUS_DECRYPTION_FAILED:			return "STATUS_DECRYPTION_FAILED";
	case STATUS_RANGE_NOT_FOUND:			return "STATUS_RANGE_NOT_FOUND";
	case STATUS_NO_RECOVERY_POLICY: 		return "STATUS_NO_RECOVERY_POLICY";
	case STATUS_NO_EFS: 					return "STATUS_NO_EFS";
	case STATUS_WRONG_EFS:					return "STATUS_WRONG_EFS";
	case STATUS_NO_USER_KEYS:				return "STATUS_NO_USER_KEYS";
	case STATUS_FILE_NOT_ENCRYPTED: 		return "STATUS_FILE_NOT_ENCRYPTED";
	case STATUS_NOT_EXPORT_FORMAT:			return "STATUS_NOT_EXPORT_FORMAT";
	case STATUS_FILE_ENCRYPTED: 			return "STATUS_FILE_ENCRYPTED";
	case STATUS_WAKE_SYSTEM:				return "STATUS_WAKE_SYSTEM";
	case STATUS_WMI_GUID_NOT_FOUND: 		return "STATUS_WMI_GUID_NOT_FOUND";
	case STATUS_WMI_INSTANCE_NOT_FOUND: 	return "STATUS_WMI_INSTANCE_NOT_FOUND";
	case STATUS_WMI_ITEMID_NOT_FOUND:		return "STATUS_WMI_ITEMID_NOT_FOUND";
	case STATUS_WMI_TRY_AGAIN:				return "STATUS_WMI_TRY_AGAIN";
	case STATUS_SHARED_POLICY:				return "STATUS_SHARED_POLICY";
	case STATUS_POLICY_OBJECT_NOT_FOUND:	return "STATUS_POLICY_OBJECT_NOT_FOUND";
	case STATUS_POLICY_ONLY_IN_DS:			return "STATUS_POLICY_ONLY_IN_DS";
	case STATUS_VOLUME_NOT_UPGRADED:		return "STATUS_VOLUME_NOT_UPGRADED";
	case STATUS_REMOTE_STORAGE_NOT_ACTIVE:	return "STATUS_REMOTE_STORAGE_NOT_ACTIVE";
	case STATUS_REMOTE_STORAGE_MEDIA_ERROR: return "STATUS_REMOTE_STORAGE_MEDIA_ERROR";
	case STATUS_NO_TRACKING_SERVICE:		return "STATUS_NO_TRACKING_SERVICE";
	case STATUS_SERVER_SID_MISMATCH:		return "STATUS_SERVER_SID_MISMATCH";
	case STATUS_DS_NO_ATTRIBUTE_OR_VALUE:	return "STATUS_DS_NO_ATTRIBUTE_OR_VALUE";
	case STATUS_DS_INVALID_ATTRIBUTE_SYNTAX:return "STATUS_DS_INVALID_ATTRIBUTE_SYNTAX";
	case STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED:return "STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED";
	case STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS:
											return "STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS";
	case STATUS_DS_BUSY:					return "STATUS_DS_BUSY";
	case STATUS_DS_UNAVAILABLE: 			return "STATUS_DS_UNAVAILABLE";
	case STATUS_DS_NO_RIDS_ALLOCATED:		return "STATUS_DS_NO_RIDS_ALLOCATED";
	case STATUS_DS_NO_MORE_RIDS:			return "STATUS_DS_NO_MORE_RIDS";
	case STATUS_DS_INCORRECT_ROLE_OWNER:	return "STATUS_DS_INCORRECT_ROLE_OWNER";
	case STATUS_DS_RIDMGR_INIT_ERROR:		return "STATUS_DS_RIDMGR_INIT_ERROR";
	case STATUS_DS_OBJ_CLASS_VIOLATION: 	return "STATUS_DS_OBJ_CLASS_VIOLATION";
	case STATUS_DS_CANT_ON_NON_LEAF:		return "STATUS_DS_CANT_ON_NON_LEAF";
	case STATUS_DS_CANT_ON_RDN: 			return "STATUS_DS_CANT_ON_RDN";
	case STATUS_DS_CANT_MOD_OBJ_CLASS:		return "STATUS_DS_CANT_MOD_OBJ_CLASS";
	case STATUS_DS_CROSS_DOM_MOVE_FAILED:	return "STATUS_DS_CROSS_DOM_MOVE_FAILED";
	case STATUS_DS_GC_NOT_AVAILABLE:		return "STATUS_DS_GC_NOT_AVAILABLE";
	case STATUS_DIRECTORY_SERVICE_REQUIRED: return "STATUS_DIRECTORY_SERVICE_REQUIRED";
	case STATUS_REPARSE_ATTRIBUTE_CONFLICT: return "STATUS_REPARSE_ATTRIBUTE_CONFLICT";
	case STATUS_CANT_ENABLE_DENY_ONLY:		return "STATUS_CANT_ENABLE_DENY_ONLY";
	case STATUS_FLOAT_MULTIPLE_FAULTS:		return "STATUS_FLOAT_MULTIPLE_FAULTS";
	case STATUS_FLOAT_MULTIPLE_TRAPS:		return "STATUS_FLOAT_MULTIPLE_TRAPS";
	case STATUS_DEVICE_REMOVED: 			return "STATUS_DEVICE_REMOVED";
	case STATUS_JOURNAL_DELETE_IN_PROGRESS: return "STATUS_JOURNAL_DELETE_IN_PROGRESS";
	case STATUS_JOURNAL_NOT_ACTIVE: 		return "STATUS_JOURNAL_NOT_ACTIVE";
	case STATUS_NOINTERFACE:				return "STATUS_NOINTERFACE";
	case STATUS_DS_ADMIN_LIMIT_EXCEEDED:	return "STATUS_DS_ADMIN_LIMIT_EXCEEDED";
	case STATUS_DRIVER_FAILED_SLEEP:		return "STATUS_DRIVER_FAILED_SLEEP";
	case STATUS_MUTUAL_AUTHENTICATION_FAILED:
											return "STATUS_MUTUAL_AUTHENTICATION_FAILED";
	case STATUS_CORRUPT_SYSTEM_FILE:		return "STATUS_CORRUPT_SYSTEM_FILE";
	case STATUS_DATATYPE_MISALIGNMENT_ERROR:return "STATUS_DATATYPE_MISALIGNMENT_ERROR";
	case STATUS_WMI_READ_ONLY:				return "STATUS_WMI_READ_ONLY";
	case STATUS_WMI_SET_FAILURE:			return "STATUS_WMI_SET_FAILURE";
	case STATUS_COMMITMENT_MINIMUM: 		return "STATUS_COMMITMENT_MINIMUM";
	case STATUS_REG_NAT_CONSUMPTION:		return "STATUS_REG_NAT_CONSUMPTION";
	case STATUS_TRANSPORT_FULL: 			return "STATUS_TRANSPORT_FULL";
	case STATUS_DS_SAM_INIT_FAILURE:		return "STATUS_DS_SAM_INIT_FAILURE";
	case STATUS_ONLY_IF_CONNECTED:			return "STATUS_ONLY_IF_CONNECTED";
	case STATUS_DS_SENSITIVE_GROUP_VIOLATION:
											return "STATUS_DS_SENSITIVE_GROUP_VIOLATION";
	case STATUS_PNP_RESTART_ENUMERATION:	return "STATUS_PNP_RESTART_ENUMERATION";
	case STATUS_JOURNAL_ENTRY_DELETED:		return "STATUS_JOURNAL_ENTRY_DELETED";
	case STATUS_DS_CANT_MOD_PRIMARYGROUPID: return "STATUS_DS_CANT_MOD_PRIMARYGROUPID";
	case STATUS_SYSTEM_IMAGE_BAD_SIGNATURE: return "STATUS_SYSTEM_IMAGE_BAD_SIGNATURE";
	case STATUS_PNP_REBOOT_REQUIRED:		return "STATUS_PNP_REBOOT_REQUIRED";
	case STATUS_POWER_STATE_INVALID:		return "STATUS_POWER_STATE_INVALID";
	case STATUS_DS_INVALID_GROUP_TYPE:		return "STATUS_DS_INVALID_GROUP_TYPE";
	case STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN:
											return "STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN";
	case STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN:
											return "STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN";
	case STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER:
											return "STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER";
	case STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER:
											return "STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER";
	case STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER:
											return "STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER";
	case STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER:
									 		return "STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER";
	case STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER:
											return "STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER";
	case STATUS_DS_HAVE_PRIMARY_MEMBERS:	return "STATUS_DS_HAVE_PRIMARY_MEMBERS";
	case STATUS_WMI_NOT_SUPPORTED:			return "STATUS_WMI_NOT_SUPPORTED";
	case STATUS_INSUFFICIENT_POWER: 		return "STATUS_INSUFFICIENT_POWER";
	case STATUS_SAM_NEED_BOOTKEY_PASSWORD:	return "STATUS_SAM_NEED_BOOTKEY_PASSWORD";
	case STATUS_SAM_NEED_BOOTKEY_FLOPPY:	return "STATUS_SAM_NEED_BOOTKEY_FLOPPY";
	case STATUS_DS_CANT_START:				return "STATUS_DS_CANT_START";
	case STATUS_DS_INIT_FAILURE:			return "STATUS_DS_INIT_FAILURE";
	case STATUS_SAM_INIT_FAILURE:			return "STATUS_SAM_INIT_FAILURE";
	case STATUS_DS_GC_REQUIRED: 			return "STATUS_DS_GC_REQUIRED";
	case STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY:
											return "STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY";
	case STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS:
											return "STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS";
	case STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED:
											return "STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED";
	case STATUS_MULTIPLE_FAULT_VIOLATION:	return "STATUS_MULTIPLE_FAULT_VIOLATION";
	case STATUS_NOT_SUPPORTED_ON_SBS:		return "STATUS_NOT_SUPPORTED_ON_SBS";
	case STATUS_WOW_ASSERTION:				return "STATUS_WOW_ASSERTION";
	case DBG_NO_STATE_CHANGE:				return "DBG_NO_STATE_CHANGE";
	case DBG_APP_NOT_IDLE:					return "DBG_APP_NOT_IDLE";
	case RPC_NT_INVALID_STRING_BINDING: 	return "RPC_NT_INVALID_STRING_BINDING";
	case RPC_NT_WRONG_KIND_OF_BINDING:		return "RPC_NT_WRONG_KIND_OF_BINDING";
	case RPC_NT_INVALID_BINDING:			return "RPC_NT_INVALID_BINDING";
	case RPC_NT_PROTSEQ_NOT_SUPPORTED:		return "RPC_NT_PROTSEQ_NOT_SUPPORTED";
	case RPC_NT_INVALID_RPC_PROTSEQ:		return "RPC_NT_INVALID_RPC_PROTSEQ";
	case RPC_NT_INVALID_STRING_UUID:		return "RPC_NT_INVALID_STRING_UUID";
	case RPC_NT_INVALID_ENDPOINT_FORMAT:	return "RPC_NT_INVALID_ENDPOINT_FORMAT";
	case RPC_NT_INVALID_NET_ADDR:			return "RPC_NT_INVALID_NET_ADDR";
	case RPC_NT_NO_ENDPOINT_FOUND:			return "RPC_NT_NO_ENDPOINT_FOUND";
	case RPC_NT_INVALID_TIMEOUT:			return "RPC_NT_INVALID_TIMEOUT";
	case RPC_NT_OBJECT_NOT_FOUND:			return "RPC_NT_OBJECT_NOT_FOUND";
	case RPC_NT_ALREADY_REGISTERED: 		return "RPC_NT_ALREADY_REGISTERED";
	case RPC_NT_TYPE_ALREADY_REGISTERED:	return "RPC_NT_TYPE_ALREADY_REGISTERED";
	case RPC_NT_ALREADY_LISTENING:			return "RPC_NT_ALREADY_LISTENING";
	case RPC_NT_NO_PROTSEQS_REGISTERED: 	return "RPC_NT_NO_PROTSEQS_REGISTERED";
	case RPC_NT_NOT_LISTENING:				return "RPC_NT_NOT_LISTENING";
	case RPC_NT_UNKNOWN_MGR_TYPE:			return "RPC_NT_UNKNOWN_MGR_TYPE";
	case RPC_NT_UNKNOWN_IF: 				return "RPC_NT_UNKNOWN_IF";
	case RPC_NT_NO_BINDINGS:				return "RPC_NT_NO_BINDINGS";
	case RPC_NT_NO_PROTSEQS:				return "RPC_NT_NO_PROTSEQS";
	case RPC_NT_CANT_CREATE_ENDPOINT:		return "RPC_NT_CANT_CREATE_ENDPOINT";
	case RPC_NT_OUT_OF_RESOURCES:			return "RPC_NT_OUT_OF_RESOURCES";
	case RPC_NT_SERVER_UNAVAILABLE: 		return "RPC_NT_SERVER_UNAVAILABLE";
	case RPC_NT_SERVER_TOO_BUSY:			return "RPC_NT_SERVER_TOO_BUSY";
	case RPC_NT_INVALID_NETWORK_OPTIONS:	return "RPC_NT_INVALID_NETWORK_OPTIONS";
	case RPC_NT_NO_CALL_ACTIVE: 			return "RPC_NT_NO_CALL_ACTIVE";
	case RPC_NT_CALL_FAILED:				return "RPC_NT_CALL_FAILED";
	case RPC_NT_CALL_FAILED_DNE:			return "RPC_NT_CALL_FAILED_DNE";
	case RPC_NT_PROTOCOL_ERROR: 			return "RPC_NT_PROTOCOL_ERROR";
	case RPC_NT_UNSUPPORTED_TRANS_SYN:		return "RPC_NT_UNSUPPORTED_TRANS_SYN";
	case RPC_NT_UNSUPPORTED_TYPE:			return "RPC_NT_UNSUPPORTED_TYPE";
	case RPC_NT_INVALID_TAG:				return "RPC_NT_INVALID_TAG";
	case RPC_NT_INVALID_BOUND:				return "RPC_NT_INVALID_BOUND";
	case RPC_NT_NO_ENTRY_NAME:				return "RPC_NT_NO_ENTRY_NAME";
	case RPC_NT_INVALID_NAME_SYNTAX:		return "RPC_NT_INVALID_NAME_SYNTAX";
	case RPC_NT_UNSUPPORTED_NAME_SYNTAX:	return "RPC_NT_UNSUPPORTED_NAME_SYNTAX";
	case RPC_NT_UUID_NO_ADDRESS:			return "RPC_NT_UUID_NO_ADDRESS";
	case RPC_NT_DUPLICATE_ENDPOINT: 		return "RPC_NT_DUPLICATE_ENDPOINT";
	case RPC_NT_UNKNOWN_AUTHN_TYPE: 		return "RPC_NT_UNKNOWN_AUTHN_TYPE";
	case RPC_NT_MAX_CALLS_TOO_SMALL:		return "RPC_NT_MAX_CALLS_TOO_SMALL";
	case RPC_NT_STRING_TOO_LONG:			return "RPC_NT_STRING_TOO_LONG";
	case RPC_NT_PROTSEQ_NOT_FOUND:			return "RPC_NT_PROTSEQ_NOT_FOUND";
	case RPC_NT_PROCNUM_OUT_OF_RANGE:		return "RPC_NT_PROCNUM_OUT_OF_RANGE";
	case RPC_NT_BINDING_HAS_NO_AUTH:		return "RPC_NT_BINDING_HAS_NO_AUTH";
	case RPC_NT_UNKNOWN_AUTHN_SERVICE:		return "RPC_NT_UNKNOWN_AUTHN_SERVICE";
	case RPC_NT_UNKNOWN_AUTHN_LEVEL:		return "RPC_NT_UNKNOWN_AUTHN_LEVEL";
	case RPC_NT_INVALID_AUTH_IDENTITY:		return "RPC_NT_INVALID_AUTH_IDENTITY";
	case RPC_NT_UNKNOWN_AUTHZ_SERVICE:		return "RPC_NT_UNKNOWN_AUTHZ_SERVICE";
	case EPT_NT_INVALID_ENTRY:				return "EPT_NT_INVALID_ENTRY";
	case EPT_NT_CANT_PERFORM_OP:			return "EPT_NT_CANT_PERFORM_OP";
	case EPT_NT_NOT_REGISTERED: 			return "EPT_NT_NOT_REGISTERED";
	case RPC_NT_NOTHING_TO_EXPORT:			return "RPC_NT_NOTHING_TO_EXPORT";
	case RPC_NT_INCOMPLETE_NAME:			return "RPC_NT_INCOMPLETE_NAME";
	case RPC_NT_INVALID_VERS_OPTION:		return "RPC_NT_INVALID_VERS_OPTION";
	case RPC_NT_NO_MORE_MEMBERS:			return "RPC_NT_NO_MORE_MEMBERS";
	case RPC_NT_NOT_ALL_OBJS_UNEXPORTED:	return "RPC_NT_NOT_ALL_OBJS_UNEXPORTED";
	case RPC_NT_INTERFACE_NOT_FOUND:		return "RPC_NT_INTERFACE_NOT_FOUND";
	case RPC_NT_ENTRY_ALREADY_EXISTS:		return "RPC_NT_ENTRY_ALREADY_EXISTS";
	case RPC_NT_ENTRY_NOT_FOUND:			return "RPC_NT_ENTRY_NOT_FOUND";
	case RPC_NT_NAME_SERVICE_UNAVAILABLE:	return "RPC_NT_NAME_SERVICE_UNAVAILABLE";
	case RPC_NT_INVALID_NAF_ID: 			return "RPC_NT_INVALID_NAF_ID";
	case RPC_NT_CANNOT_SUPPORT: 			return "RPC_NT_CANNOT_SUPPORT";
	case RPC_NT_NO_CONTEXT_AVAILABLE:		return "RPC_NT_NO_CONTEXT_AVAILABLE";
	case RPC_NT_INTERNAL_ERROR: 			return "RPC_NT_INTERNAL_ERROR";
	case RPC_NT_ZERO_DIVIDE:				return "RPC_NT_ZERO_DIVIDE";
	case RPC_NT_ADDRESS_ERROR:				return "RPC_NT_ADDRESS_ERROR";
	case RPC_NT_FP_DIV_ZERO:				return "RPC_NT_FP_DIV_ZERO";
	case RPC_NT_FP_UNDERFLOW:				return "RPC_NT_FP_UNDERFLOW";
	case RPC_NT_FP_OVERFLOW:				return "RPC_NT_FP_OVERFLOW";
	case RPC_NT_NO_MORE_ENTRIES:			return "RPC_NT_NO_MORE_ENTRIES";
	case RPC_NT_SS_CHAR_TRANS_OPEN_FAIL:	return "RPC_NT_SS_CHAR_TRANS_OPEN_FAIL";
	case RPC_NT_SS_CHAR_TRANS_SHORT_FILE:	return "RPC_NT_SS_CHAR_TRANS_SHORT_FILE";
	case RPC_NT_SS_IN_NULL_CONTEXT: 		return "RPC_NT_SS_IN_NULL_CONTEXT";
	case RPC_NT_SS_CONTEXT_MISMATCH:		return "RPC_NT_SS_CONTEXT_MISMATCH";
	case RPC_NT_SS_CONTEXT_DAMAGED: 		return "RPC_NT_SS_CONTEXT_DAMAGED";
	case RPC_NT_SS_HANDLES_MISMATCH:		return "RPC_NT_SS_HANDLES_MISMATCH";
	case RPC_NT_SS_CANNOT_GET_CALL_HANDLE:	return "RPC_NT_SS_CANNOT_GET_CALL_HANDLE";
	case RPC_NT_NULL_REF_POINTER:			return "RPC_NT_NULL_REF_POINTER";
	case RPC_NT_ENUM_VALUE_OUT_OF_RANGE:	return "RPC_NT_ENUM_VALUE_OUT_OF_RANGE";
	case RPC_NT_BYTE_COUNT_TOO_SMALL:		return "RPC_NT_BYTE_COUNT_TOO_SMALL";
	case RPC_NT_BAD_STUB_DATA:				return "RPC_NT_BAD_STUB_DATA";
	case RPC_NT_CALL_IN_PROGRESS:			return "RPC_NT_CALL_IN_PROGRESS";
	case RPC_NT_NO_MORE_BINDINGS:			return "RPC_NT_NO_MORE_BINDINGS";
	case RPC_NT_GROUP_MEMBER_NOT_FOUND: 	return "RPC_NT_GROUP_MEMBER_NOT_FOUND";
	case EPT_NT_CANT_CREATE:				return "EPT_NT_CANT_CREATE";
	case RPC_NT_INVALID_OBJECT: 			return "RPC_NT_INVALID_OBJECT";
	case RPC_NT_NO_INTERFACES:				return "RPC_NT_NO_INTERFACES";
	case RPC_NT_CALL_CANCELLED: 			return "RPC_NT_CALL_CANCELLED";
	case RPC_NT_BINDING_INCOMPLETE: 		return "RPC_NT_BINDING_INCOMPLETE";
	case RPC_NT_COMM_FAILURE:				return "RPC_NT_COMM_FAILURE";
	case RPC_NT_UNSUPPORTED_AUTHN_LEVEL:	return "RPC_NT_UNSUPPORTED_AUTHN_LEVEL";
	case RPC_NT_NO_PRINC_NAME:				return "RPC_NT_NO_PRINC_NAME";
	case RPC_NT_NOT_RPC_ERROR:				return "RPC_NT_NOT_RPC_ERROR";
	case RPC_NT_UUID_LOCAL_ONLY:			return "RPC_NT_UUID_LOCAL_ONLY";
	case RPC_NT_SEC_PKG_ERROR:				return "RPC_NT_SEC_PKG_ERROR";
	case RPC_NT_NOT_CANCELLED:				return "RPC_NT_NOT_CANCELLED";
	case RPC_NT_INVALID_ES_ACTION:			return "RPC_NT_INVALID_ES_ACTION";
	case RPC_NT_WRONG_ES_VERSION:			return "RPC_NT_WRONG_ES_VERSION";
	case RPC_NT_WRONG_STUB_VERSION: 		return "RPC_NT_WRONG_STUB_VERSION";
	case RPC_NT_INVALID_PIPE_OBJECT:		return "RPC_NT_INVALID_PIPE_OBJECT";
	case RPC_NT_INVALID_PIPE_OPERATION: 	return "RPC_NT_INVALID_PIPE_OPERATION";
	case RPC_NT_WRONG_PIPE_VERSION: 		return "RPC_NT_WRONG_PIPE_VERSION";
	case RPC_NT_PIPE_CLOSED:				return "RPC_NT_PIPE_CLOSED";
	case RPC_NT_PIPE_DISCIPLINE_ERROR:		return "RPC_NT_PIPE_DISCIPLINE_ERROR";
	case RPC_NT_PIPE_EMPTY: 				return "RPC_NT_PIPE_EMPTY";
	case RPC_NT_INVALID_ASYNC_HANDLE:		return "RPC_NT_INVALID_ASYNC_HANDLE";
	case RPC_NT_INVALID_ASYNC_CALL: 		return "RPC_NT_INVALID_ASYNC_CALL";
	case RPC_NT_SEND_INCOMPLETE:			return "RPC_NT_SEND_INCOMPLETE";
	case STATUS_ACPI_INVALID_OPCODE:		return "STATUS_ACPI_INVALID_OPCODE";
	case STATUS_ACPI_STACK_OVERFLOW:		return "STATUS_ACPI_STACK_OVERFLOW";
	case STATUS_ACPI_ASSERT_FAILED: 		return "STATUS_ACPI_ASSERT_FAILED";
	case STATUS_ACPI_INVALID_INDEX: 		return "STATUS_ACPI_INVALID_INDEX";
	case STATUS_ACPI_INVALID_ARGUMENT:		return "STATUS_ACPI_INVALID_ARGUMENT";
	case STATUS_ACPI_FATAL: 				return "STATUS_ACPI_FATAL";
	case STATUS_ACPI_INVALID_SUPERNAME: 	return "STATUS_ACPI_INVALID_SUPERNAME";
	case STATUS_ACPI_INVALID_ARGTYPE:		return "STATUS_ACPI_INVALID_ARGTYPE";
	case STATUS_ACPI_INVALID_OBJTYPE:		return "STATUS_ACPI_INVALID_OBJTYPE";
	case STATUS_ACPI_INVALID_TARGETTYPE:	return "STATUS_ACPI_INVALID_TARGETTYPE";
	case STATUS_ACPI_INCORRECT_ARGUMENT_COUNT:
											return "STATUS_ACPI_INCORRECT_ARGUMENT_COUNT";
	case STATUS_ACPI_ADDRESS_NOT_MAPPED:	return "STATUS_ACPI_ADDRESS_NOT_MAPPED";
	case STATUS_ACPI_INVALID_EVENTTYPE: 	return "STATUS_ACPI_INVALID_EVENTTYPE";
	case STATUS_ACPI_HANDLER_COLLISION: 	return "STATUS_ACPI_HANDLER_COLLISION";
	case STATUS_ACPI_INVALID_DATA:			return "STATUS_ACPI_INVALID_DATA";
	case STATUS_ACPI_INVALID_REGION:		return "STATUS_ACPI_INVALID_REGION";
	case STATUS_ACPI_INVALID_ACCESS_SIZE:	return "STATUS_ACPI_INVALID_ACCESS_SIZE";
	case STATUS_ACPI_ACQUIRE_GLOBAL_LOCK:	return "STATUS_ACPI_ACQUIRE_GLOBAL_LOCK";
	case STATUS_ACPI_ALREADY_INITIALIZED:	return "STATUS_ACPI_ALREADY_INITIALIZED";
	case STATUS_ACPI_NOT_INITIALIZED:		return "STATUS_ACPI_NOT_INITIALIZED";
	case STATUS_ACPI_INVALID_MUTEX_LEVEL:	return "STATUS_ACPI_INVALID_MUTEX_LEVEL";
	case STATUS_ACPI_MUTEX_NOT_OWNED:		return "STATUS_ACPI_MUTEX_NOT_OWNED";
	case STATUS_ACPI_MUTEX_NOT_OWNER:		return "STATUS_ACPI_MUTEX_NOT_OWNER";
	case STATUS_ACPI_RS_ACCESS: 			return "STATUS_ACPI_RS_ACCESS";
	case STATUS_ACPI_INVALID_TABLE: 		return "STATUS_ACPI_INVALID_TABLE";
	case STATUS_ACPI_REG_HANDLER_FAILED:	return "STATUS_ACPI_REG_HANDLER_FAILED";
	case STATUS_ACPI_POWER_REQUEST_FAILED:	return "STATUS_ACPI_POWER_REQUEST_FAILED";
	case STATUS_CTX_WINSTATION_NAME_INVALID:return "STATUS_CTX_WINSTATION_NAME_INVALID";
	case STATUS_CTX_INVALID_PD: 			return "STATUS_CTX_INVALID_PD";
	case STATUS_CTX_PD_NOT_FOUND:			return "STATUS_CTX_PD_NOT_FOUND";
	case STATUS_CTX_CDM_CONNECT:			return "STATUS_CTX_CDM_CONNECT";
	case STATUS_CTX_CDM_DISCONNECT: 		return "STATUS_CTX_CDM_DISCONNECT";
	case STATUS_CTX_CLOSE_PENDING:			return "STATUS_CTX_CLOSE_PENDING";
	case STATUS_CTX_NO_OUTBUF:				return "STATUS_CTX_NO_OUTBUF";
	case STATUS_CTX_MODEM_INF_NOT_FOUND:	return "STATUS_CTX_MODEM_INF_NOT_FOUND";
	case STATUS_CTX_INVALID_MODEMNAME:		return "STATUS_CTX_INVALID_MODEMNAME";
	case STATUS_CTX_RESPONSE_ERROR: 		return "STATUS_CTX_RESPONSE_ERROR";
	case STATUS_CTX_MODEM_RESPONSE_TIMEOUT: return "STATUS_CTX_MODEM_RESPONSE_TIMEOUT";
	case STATUS_CTX_MODEM_RESPONSE_NO_CARRIER:
											return "STATUS_CTX_MODEM_RESPONSE_NO_CARRIER";
	case STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE:
							 				return "STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE";
	case STATUS_CTX_MODEM_RESPONSE_BUSY:	return "STATUS_CTX_MODEM_RESPONSE_BUSY";
	case STATUS_CTX_MODEM_RESPONSE_VOICE:	return "STATUS_CTX_MODEM_RESPONSE_VOICE";
	case STATUS_CTX_TD_ERROR:				return "STATUS_CTX_TD_ERROR";
	case STATUS_CTX_LICENSE_CLIENT_INVALID: return "STATUS_CTX_LICENSE_CLIENT_INVALID";
	case STATUS_CTX_LICENSE_NOT_AVAILABLE:	return "STATUS_CTX_LICENSE_NOT_AVAILABLE";
	case STATUS_CTX_LICENSE_EXPIRED:		return "STATUS_CTX_LICENSE_EXPIRED";
	case STATUS_CTX_WINSTATION_NOT_FOUND:	return "STATUS_CTX_WINSTATION_NOT_FOUND";
	case STATUS_CTX_WINSTATION_NAME_COLLISION:
											return "STATUS_CTX_WINSTATION_NAME_COLLISION";
	case STATUS_CTX_WINSTATION_BUSY:		return "STATUS_CTX_WINSTATION_BUSY";
	case STATUS_CTX_BAD_VIDEO_MODE: 		return "STATUS_CTX_BAD_VIDEO_MODE";
	case STATUS_CTX_GRAPHICS_INVALID:		return "STATUS_CTX_GRAPHICS_INVALID";
	case STATUS_CTX_NOT_CONSOLE:			return "STATUS_CTX_NOT_CONSOLE";
	case STATUS_CTX_CLIENT_QUERY_TIMEOUT:	return "STATUS_CTX_CLIENT_QUERY_TIMEOUT";
	case STATUS_CTX_CONSOLE_DISCONNECT: 	return "STATUS_CTX_CONSOLE_DISCONNECT";
	case STATUS_CTX_CONSOLE_CONNECT:		return "STATUS_CTX_CONSOLE_CONNECT";
	case STATUS_CTX_SHADOW_DENIED:			return "STATUS_CTX_SHADOW_DENIED";
	case STATUS_CTX_WINSTATION_ACCESS_DENIED:
											return "STATUS_CTX_WINSTATION_ACCESS_DENIED";
	case STATUS_CTX_INVALID_WD: 			return "STATUS_CTX_INVALID_WD";
	case STATUS_CTX_WD_NOT_FOUND:			return "STATUS_CTX_WD_NOT_FOUND";
	case STATUS_CTX_SHADOW_INVALID: 		return "STATUS_CTX_SHADOW_INVALID";
	case STATUS_CTX_SHADOW_DISABLED:		return "STATUS_CTX_SHADOW_DISABLED";
	case STATUS_RDP_PROTOCOL_ERROR: 		return "STATUS_RDP_PROTOCOL_ERROR";
	case STATUS_CTX_CLIENT_LICENSE_NOT_SET: return "STATUS_CTX_CLIENT_LICENSE_NOT_SET";
	case STATUS_CTX_CLIENT_LICENSE_IN_USE:	return "STATUS_CTX_CLIENT_LICENSE_IN_USE";
	case STATUS_PNP_BAD_MPS_TABLE:			return "STATUS_PNP_BAD_MPS_TABLE";
	case STATUS_PNP_TRANSLATION_FAILED: 	return "STATUS_PNP_TRANSLATION_FAILED";
	case STATUS_PNP_IRQ_TRANSLATION_FAILED: return "STATUS_PNP_IRQ_TRANSLATION_FAILED";
	default:
		return NULL;
	}
}
#endif	// DBG
