/*
	maindlg.c

	Virtual Floppy Disk drive control panel
    Copyright (C) 2003 Kenji Kato
*/

#include "sysincl.h"
#include "vfdctl.h"
#include "vfdutil.h"
#include "vfdwin.h"
#include "resource.h"

//
//	child dialogs enumeration
//
enum {
	VFD_CHILD_DRIVE = 0,
	VFD_CHILD_ASSOC,
	VFD_CHILD_LINK,
	VFD_CHILD_ABOUT,
	VFD_CHILD_MAX
};

//
//	Constant table for child dialog creation
//
static const struct _CHILD_TABLE {
	UINT	nTitleStr;
	UINT	nDlgTempl;
	DLGPROC	pDlgProc;
}
ChildTable[VFD_CHILD_MAX] = {
	{ IDS_TAB_DRIVER,	IDD_DRIVER,	DriverProc	},
	{ IDS_TAB_ASSOC,	IDD_ASSOC,	AssocProc	},
	{ IDS_TAB_LINK,		IDD_LINK,	LinkProc	},
	{ IDS_TAB_ABOUT,	IDD_ABOUT,	AboutProc	},
};

//
//	child dialogs
//
int	 nChildIdx = 0;
HWND hChildWnd[VFD_CHILD_MAX] = { 0 };

//
//	window message handlers
//
static void OnInitDialog(HWND hDlg);
static void OnTabNotify	(LPNMHDR pNMHDR);
static void OnVfdWakeup	(HWND hDlg);
static void OnVfdRefresh(HWND hDlg, WPARAM reason);

//
//	Main Dialog Procedure
//
BOOL CALLBACK MainProc(
	HWND hDlg,
	UINT msg,
	WPARAM wParam,
	LPARAM lParam) 
{
	static UINT uWakeUpMsg = 0;
	static UINT uDeviceMsg = 0;

    switch (msg) { 
    case WM_INITDIALOG:
		uWakeUpMsg = RegisterWindowMessage(VFD_WAKEUP_MSG_STR);
		uDeviceMsg = RegisterWindowMessage(VFD_BROADCAST_MSG_STR);

		OnInitDialog(hDlg);
		break;

	case WM_CLOSE:
		DestroyWindow(hDlg);
		break;

	case WM_NOTIFY:
		if (wParam == IDC_TAB_CONTROL) {
			OnTabNotify((LPNMHDR)lParam);
		}
		break;

	case WM_DESTROY:
		PostQuitMessage(0);
		break;

	default:
		if (msg == uWakeUpMsg) {
			OnVfdWakeup(hDlg);
		}
		else if (msg == uDeviceMsg) {
			OnVfdRefresh(hDlg, wParam);
		}
		break;
    } 

    return 0; 
}

//
//	Main Dialog Initialize
//
void OnInitDialog(
	HWND	hDlg)
{
	TCITEM	item;
	char	title[50];
	HWND	hTab;
	RECT	rect;
	int		i;
	HICON	hIcon;
	HMENU	hMenu;

	//
	//	Prepare icon
	//
	hIcon = LoadImage(hAppInstance, MAKEINTRESOURCE(IDI_VFD_CONFIG), IMAGE_ICON, 16, 16, 0);

	if (hIcon) {
		SendMessage(hDlg, WM_SETICON, ICON_SMALL, (LPARAM)hIcon);
	}

	//
	//	Adjust system menu (required for Windows NT)
	//
	hMenu = GetSystemMenu(hDlg, FALSE);

	if (hMenu) {
		DeleteMenu(hMenu, SC_MAXIMIZE, MF_BYCOMMAND);
		DeleteMenu(hMenu, SC_SIZE, MF_BYCOMMAND);
	}

	//
	//	Prepare tab control
	//
	hTab = GetDlgItem(hDlg, IDC_TAB_CONTROL);

	if (hTab == NULL) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnMainInitDialog : GetDlgItem - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	memset(&item, 0, sizeof(item));
	item.mask = TCIF_TEXT;
	item.pszText = title;

	for (i = 0; i < VFD_CHILD_MAX; i++) {
		if (LoadString(hAppInstance, ChildTable[i].nTitleStr, title, sizeof(title)) == 0) {
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : LoadString - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}

		if (TabCtrl_InsertItem(hTab, i, &item) == -1) {
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : TabCtrl_InsertItem - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}
	}

	//
	//	Prepare child dialogs
	//

	//	Calculate Tab control's client area

	if (!GetWindowRect(hTab, &rect)) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnMainInitDialog : GetWindowRect - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}


	{
		POINT pt;
		
		pt.x = rect.left;
		pt.y = rect.top;

		if (!ScreenToClient(hDlg, &pt)) {
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : ScreenToClient - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}

		rect.left = pt.x;
		rect.top = pt.y;

		pt.x = rect.right;
		pt.y = rect.bottom;

		if (!ScreenToClient(hDlg, &pt)) {
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : ScreenToClient - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}

		rect.right = pt.x;
		rect.bottom = pt.y;
	}

	TabCtrl_AdjustRect(hTab, FALSE, &rect);

	//	Create each child dialog

	for (i = 0; i < VFD_CHILD_MAX; i++) {
		hChildWnd[i] = CreateDialog(
			hAppInstance,
			MAKEINTRESOURCE(ChildTable[i].nDlgTempl),
			hDlg,
			ChildTable[i].pDlgProc);

		if (hChildWnd[i] == NULL) {
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : CreateDialog - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}

		if (!SetWindowPos(
			hChildWnd[i],
			HWND_TOP,
			rect.left,
			rect.top,
			0,
			0,
			SWP_NOZORDER | SWP_NOSIZE))
		{
			DWORD err = GetLastError();

			DEBUG_TRACE1("OnMainInitDialog : SetWindowPos - %s", ErrMsg(err));

			ShowErrorMessage(err, IDS_ERR_APP_INIT);

			return;
		}
	}

	//
	//	Get latest driver status
	//
	OnVfdRefresh(hDlg, VFD_OPERATION_NONE);

	//
	//	Show initial child
	//
	TabCtrl_SetCurSel(hTab, 0);
	ShowWindow(hChildWnd[0], SW_SHOW);
	SetFocus(hChildWnd[0]);

	return;
}

//
//	Switch child dialog according to currently selected tab
//
void OnTabNotify(LPNMHDR pNMHDR)
{
	if (pNMHDR->code == TCN_SELCHANGE) {

		int idx  = TabCtrl_GetCurSel(pNMHDR->hwndFrom);

		if (idx >= 0 && idx < VFD_CHILD_MAX && idx != nChildIdx) {

			// selected tab is actually changed, so switch child dialog
			ShowWindow(hChildWnd[idx], SW_SHOW);
			ShowWindow(hChildWnd[nChildIdx], SW_HIDE);

			nChildIdx = idx;

			SetFocus(pNMHDR->hwndFrom);
		}
	}
}

//	bring vfdwin window to the top
//	called in response to UM_VFDWIN_WAKEUP

void OnVfdWakeup(HWND hDlg)
{
	ShowWindow(hDlg, SW_SHOW);
	SetForegroundWindow(hDlg);
	PostMessage(hDlg, WM_SYSCOMMAND, SC_RESTORE, 0);
}

//	refreshes state text and child windows

void OnVfdRefresh(HWND hDlg, WPARAM reason)
{
	DWORD	current_state;
	DWORD	ret;
	TCHAR	msg[50];
	UINT	msgid;

	//	get current driver state

	ret = VfdGetDriverState(&current_state);

	if (ret != ERROR_SUCCESS) {
		DEBUG_TRACE1("OnVfdRefresh : VfdGetDriverState - %s", ret);
		return;
	}

	switch (current_state) {
	case SERVICE_STOPPED:
		msgid = IDS_DRIVER_STOPPED;
		break;

	case SERVICE_START_PENDING:
		msgid = IDS_DRIVER_START_PENDING;
		break;

	case SERVICE_STOP_PENDING:
		msgid = IDS_DRIVER_STOP_PENDING;
		break;

	case SERVICE_RUNNING:
		msgid = IDS_DRIVER_RUNNING;
		break;

	case SERVICE_CONTINUE_PENDING:
		msgid = IDS_DRIVER_CONTINUE_PENDING;
		break;

	case SERVICE_PAUSE_PENDING:
		msgid = IDS_DRIVER_PAUSE_PENDING;
		break;

	case SERVICE_PAUSED:
		msgid = IDS_DRIVER_PAUSED;
		break;

	case VFD_NOT_INSTALLED:
		msgid = IDS_DRIVER_NOT_INSTALLED;
		break;

	default:
		current_state = 0;
		msgid = IDS_DRIVER_UNKNOWN_STATE;
		break;
	}

	if (!LoadString(hAppInstance, msgid, msg, sizeof(msg))) {
		DEBUG_TRACE2("OnVfdRefresh : LoadString(%lu) - %s",
			msgid, ErrMsg(GetLastError()));
		sprintf(msg, "VfdWin Message %lu", msgid);
	}

	if (!SetDlgItemText(hDlg, IDC_CURRENT_STATE, msg)) {
		DEBUG_TRACE1(
			"UpdateStateText : SetDlgItemText(IDC_CURRENT_STATE) - %s",
			ErrMsg(GetLastError()));
	}

	//	Refresh the Drive dialog
	OnDriveRefresh(hChildWnd[VFD_CHILD_DRIVE], reason, current_state);
}
// End Of File
